/*
    Bear Engine - Level editor

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/level_editor.cpp"
 * \brief Implementation of the bf::level_editor class.
 * \author Julien Jorge.
 */
#include "bf/level_editor.hpp"

#include "bf/config_frame.hpp"
#include "bf/gui_level.hpp"
#include "bf/image_pool.hpp"
#include "bf/level_file_xml_reader.hpp"
#include "bf/level_file_xml_writer.hpp"
#include "bf/main_frame.hpp"
#include "bf/path_configuration.hpp"
#include "bf/splash_screen.hpp"
#include "bf/wx_facilities.hpp"

#include <wx/tooltip.h>
#include <claw/logger.hpp>
#include <claw/exception.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::level_editor::level_editor()
  : m_main_frame(NULL)
{
  srand(time(NULL));
} // level_editor::level_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the configuration of the program.
 */
bf::configuration& bf::level_editor::get_config()
{
  return m_config;
} // level_editor::get_config()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the configuration of the program.
 */
const bf::configuration& bf::level_editor::get_config() const
{
  return m_config;
} // level_editor::get_config()

/*----------------------------------------------------------------------------*/
/**
 * \brief Configure the program.
 */
void bf::level_editor::configure()
{
  config_frame dlg(NULL);

  if ( dlg.ShowModal() == wxID_OK )
    update_image_pool();
} // level_editor::configure()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the image pool.
 */
void bf::level_editor::update_image_pool() const
{
  image_pool::get_instance().clear();

  std::list<std::string>::const_iterator it;

  for ( it=path_configuration::get_instance().data_path.begin();
        it!=path_configuration::get_instance().data_path.end(); ++it )
    image_pool::get_instance().scan_directory(*it);
} // level_editor::update_image_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the item classes pool.
 */
const bf::item_class_pool& bf::level_editor::get_item_class_pool() const
{
  return m_class_pool;
} // level_editor::get_item_class_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a level.
 * \param path The path to the level file.
 */
void bf::level_editor::compile( const wxString& path ) const
{
  level_file_xml_reader reader;
  level* lvl(NULL);

  try
    {
      lvl = reader.load( get_item_class_pool(), path );

      if ( check_level(*lvl) )
        {
          compile_level(*lvl, path);
          delete lvl;
        }
      else
        {
          delete lvl;
          CLAW_EXCEPTION("Invalid level.");
        }
    }
  catch(...)
    {
      delete lvl;
      throw;
    }
} // level_editor::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update a level.
 * \param path The path to the level file.
 */
void bf::level_editor::update( const wxString& path ) const
{
  level_file_xml_reader reader;
  level_file_xml_writer writer;
  level* lvl(NULL);

  try
    {
      lvl = reader.load( get_item_class_pool(), path );

      std::ofstream f( wx_to_std_string(path).c_str() );
      writer.save( f, *lvl );
      delete lvl;
    }
  catch(...)
    {
      delete lvl;
      throw;
    }
} // level_editor::update()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile a level.
 * \param lvl The level to compile.
 * \param path The path to the level file.
 * \return true if the compilation went ok.
 */
void bf::level_editor::compile_level
( const level& lvl, const wxString& path ) const
{
  std::string std_path( wx_to_std_string(path) );
  std::size_t pos = std_path.rfind(".lvl");

  if ( pos != std::string::npos )
    std_path = std_path.substr(0, pos);

  std_path += ".cl";

  std::ofstream f( std_path.c_str() );

  if (f)
    {
      compiled_file cf(f);
      lvl.compile(cf);
    }
  else
    throw CLAW_EXCEPTION("Can't open the level file.");
} // level_editor::compile_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the application is initializing.
 */
bool bf::level_editor::do_init_app()
{
  splash_screen* splash = splash_screen::create(wxT("level-splash.png"));

  if (splash!=NULL)
    {
      splash->get_version_label().SetForegroundColour( *wxWHITE );
      splash->get_status_label().SetForegroundColour( *wxWHITE );
      splash->get_status_label().SetLabel(_("Loading item class files..."));
      wxYield();
    }

  init_config();

  if (splash!=NULL)
    {
      splash->get_status_label().SetLabel(_("Loading images..."));
      wxYield();
    }

  update_image_pool();

  m_main_frame = new main_frame;
  m_main_frame->Show();

  if (splash!=NULL)
    {
      splash->get_status_label().SetLabel(_("Loading levels..."));
      wxYield();
    }

  load_levels();

  if (splash!=NULL)
    splash->Destroy();

  return true;
} // level_editor::do_init_app()

/*----------------------------------------------------------------------------*/
/**
 * \brief Minimal initialisation for command line usage.
 */
bool bf::level_editor::do_command_line_init()
{
  init_config();

  return !path_configuration::get_instance().item_class_path.empty();
} // level_editor::do_command_line_init()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load and apply the configuration.
 */
void bf::level_editor::init_config()
{
  m_config.load();

  if ( path_configuration::get_instance().item_class_path.empty()
       || path_configuration::get_instance().data_path.empty() )
    configure();

  m_class_pool.scan_directory
    ( path_configuration::get_instance().item_class_path );
} // level_editor::init_config()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if a level is valid.
 * \param lvl The level to check.
 */
bool bf::level_editor::check_level( level& lvl ) const
{
  level::invalid_items_list items;
  level::invalid_items_list::const_iterator it;

  lvl.check(items);

  for (it=items.begin(); it!=items.end(); ++it)
    claw::logger << claw::log_error << "layer " << it->second << ", item "
                 << it->first->get_class().get_class_name() << " bottom="
                 << it->first->get_rendering_parameters().get_bottom()
                 << " left="
                 << it->first->get_rendering_parameters().get_left() << ": "
                 << it->first->get_rendering_parameters().get_error()
                 << std::endl;

  return items.empty();
} // level_editor::check_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the levels passed on the command line.
 */
void bf::level_editor::load_levels()
{
  for (int i=1; i<argc; ++i)
    try
      {
        m_main_frame->load_level( argv[i] );
      }
    catch( std::ios_base::failure& e )
      {
        claw::logger << claw::log_error << e.what()
                     << "\nCreating a new level." << std::endl;
        m_main_frame->new_level( argv[i] );
      }
    catch( std::exception& e )
      {
        claw::logger << claw::log_error << e.what() << std::endl;
      }
} // level_editor::load_levels()
