/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file level_information.cpp
 * \brief Implementation of the ptb::level_information class.
 * \author Sebastien Angibaud
 */
#include "ptb/level_information.hpp"

#include "ptb/game_variables.hpp"
#include "engine/game.hpp"
#include "engine/variable/variable.hpp"

#include <algorithm>
#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>
#include <libintl.h>

/*----------------------------------------------------------------------------*/
const std::string ptb::level_information::s_level_file_field = "level_file";
const std::string ptb::level_information::s_locked_informations_field =
  "locked_informations";
const std::string ptb::level_information::s_unlocked_informations_field =
  "unlocked_informations";
const std::string ptb::level_information::s_playability_field = "playability";
const std::string ptb::level_information::s_score_format_field = "score_format";
const std::string ptb::level_information::s_score_ordering = "score_ordering";
const std::string ptb::level_information::s_unlocked = "unlocked";

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::level_information::level_information()
{

} // level_information::level_information()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::level_information::level_information(const std::string& name)
  : m_id(name), m_playability(playability_type::one_or_two_players),
    m_score_format("%v (%p)")
{
  m_score_table.load
    ( bear::engine::game::get_instance().get_custom_game_file(m_id) );

  // get the thumbnail
  std::string thumb_name = m_id;

  claw::text::replace( thumb_name, std::string(" /"), std::string("_") );
  std::transform
    ( thumb_name.begin(), thumb_name.end(), thumb_name.begin(), tolower );

  m_thumb_filename = "gfx/thumb/" + thumb_name + ".png";
} // level_information::level_information()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of the level.
 * \param field The considered field.
 * \param The value of the field.
 */
void ptb::level_information::load
(const std::string& field, const std::string& value)
{
  if ( field == s_level_file_field )
    m_filename = value;
  else if ( field == s_locked_informations_field )
    m_locked_informations = value;
  else if ( field == s_unlocked_informations_field )
    m_unlocked_informations = value;
  else if ( field == s_playability_field )
    m_playability = playability_type::from_string( value );
  else if ( field == s_score_format_field )
    m_score_format = gettext(value.c_str());
  else if ( field == s_score_ordering )
    m_score_table.set_score_ordering( value == "max" );
  else if ( field == s_unlocked )
    game_variables::set_mini_game_status( m_id, (value == "true") );
  else
    claw::logger << claw::log_error
                 << "The field '" << field << "' is unknown."  << std::endl;
} // level_information::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::level_information::is_valid() const
{
  return (!m_id.empty()) && (!m_thumb_filename.empty());
} // level_information::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the level is unlocked.
 */
bool ptb::level_information::is_unlocked() const
{
  return game_variables::get_all_mini_game_unlocked_status()
    || game_variables::get_mini_game_status( m_id );
} // level_information::is_unlocked()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the identifier of the level.
 */
const std::string& ptb::level_information::get_id() const
{
   return m_id;
} // level_information::get_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the filename of the level.
 */
const std::string& ptb::level_information::get_filename() const
{
   return m_filename;
} // level_information::get_filename()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the filename of thumb.
 * \param f The filename.
 */
void ptb::level_information::set_thumb_filename(const std::string& f)
{
  m_thumb_filename = f;
} // frame_play_level_information::set_thumb_filename()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the filename of thumb.
 */
const std::string& ptb::level_information::get_thumb_filename() const
{
  return m_thumb_filename;
} // frame_play_level_information::get_thumb_filename()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set informations when the game is locked.
 * \param info Informations.
 */
void ptb::level_information::set_locked_informations(const std::string& info)
{
  m_locked_informations = info;
} // frame_play_level_information::set_locked informations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set informations when the game is unlocked.
 * \param info Informations.
 */
void ptb::level_information::set_unlocked_informations(const std::string& info)
{
  m_unlocked_informations = info;
} // frame_play_level_information::set_unlocked informations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get informations of the level.
 */
const std::string& ptb::level_information::get_informations() const
{
  if ( is_unlocked() )
    return m_unlocked_informations;
  else
    return m_locked_informations;
} // frame_play_level_information::get_informations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get records informations of the level.
 */
std::string ptb::level_information::get_record_informations() const
{
  std::string result;

  for (score_table::const_iterator it=m_score_table.begin();
       it!=m_score_table.end(); ++it)
    result += it->format(m_score_format) + '\n';

  return result;
} // frame_play_level_information::get_record_informations()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the level has already been finished.
 */
bool ptb::level_information::is_finished() const
{
  return !m_score_table.empty();
} // frame_play_level_information::is_finished()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the playability of the level : number of players.
 */
ptb::playability_type::value_type
ptb::level_information::get_playability() const
{
  return m_playability;
} // level_information::get_playability()
