
/* Web Polygraph       http://www.web-polygraph.org/
 * (C) 2003-2006 The Measurement Factory
 * Licensed under the Apache License, Version 2.0 */

#include "xstd/xstd.h"

#include "xstd/h/iomanip.h"
#include "xstd/h/string.h"
#include "xstd/Assert.h"

#include "xstd/Checksum.h"

bool Checksum::equal(const Checksum &s) const {
	Assert(set() && s.set());
	return memcmp(theDigest, s.theDigest, sizeof(theDigest)) == 0;
}

ostream &Checksum::print(ostream &os) const {
	if (set()) {
		const ios_fmtflags flags = os.flags();
		const char fill = os.fill('0');
		os << hex << setfill('0');
		for (int i = 0; i < size(); ++i)
			os << setw(2) << (unsigned int)theDigest[i];
		os.fill(fill);
		os.flags(flags);
		return os;
	}
	return os << "<none>";
}

/*
 * Derived from RFC 1321, Appendix 3.  Copyright (C) 1991-2, RSA Data
 * Security, Inc. Created 1991.  All rights reserved.  License to copy and
 * use this software is granted provided that it is identified as the "RSA
 * Data Security, Inc. MD5 Message-Digest Algorithm" in all material
 * mentioning or referencing this software or this function.  License is
 * also granted to make and use derivative works provided that such works
 * are identified as "derived from the RSA Data Security, Inc. MD5
 * Message-Digest Algorithm" in all material mentioning or referencing the
 * derived work.  RSA Data Security, Inc. makes no representations
 * concerning either the merchantability of this software or the
 * suitability of this software for any particular purpose. It is provided
 * "as is" without express or implied warranty of any kind.  These notices
 * must be retained in any copies of any part of this documentation and/or
 * software.
 */

#define S11 7
#define S12 12
#define S13 17
#define S14 22
#define S21 5
#define S22 9
#define S23 14
#define S24 20
#define S31 4
#define S32 11
#define S33 16
#define S34 23
#define S41 6
#define S42 10
#define S43 15
#define S44 21

const unsigned char ChecksumAlg::ThePadding[64] = {
	0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

// ROTATE_LEFT rotates x left n bits.
#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

// FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4.
// Rotation is separate from addition to prevent recomputation.
#define FF(a, b, c, d, x, s, ac) { \
	(a) += F ((b), (c), (d)) + (x) + (unsigned int)(ac); \
	(a) = ROTATE_LEFT ((a), (s)); \
	(a) += (b); \
}

#define GG(a, b, c, d, x, s, ac) { \
	(a) += G ((b), (c), (d)) + (x) + (unsigned int)(ac); \
	(a) = ROTATE_LEFT ((a), (s)); \
	(a) += (b); \
}

#define HH(a, b, c, d, x, s, ac) { \
	(a) += H ((b), (c), (d)) + (x) + (unsigned int)(ac); \
	(a) = ROTATE_LEFT ((a), (s)); \
	(a) += (b); \
}

#define II(a, b, c, d, x, s, ac) { \
	(a) += I ((b), (c), (d)) + (x) + (unsigned int)(ac); \
	(a) = ROTATE_LEFT ((a), (s)); \
	(a) += (b); \
}


ChecksumAlg::ChecksumAlg() {
	reset();
}

void ChecksumAlg::reset() {
	theCounts[0] = theCounts[1] = 0;

	theState[0] = 0x67452301;
	theState[1] = 0xefcdab89;
	theState[2] = 0x98badcfe;
	theState[3] = 0x10325476;
	
	theSum.reset();
}


// MD5 block update operation. Continues an MD5 message-digest
// operation, processing another message block, and updating the context.
void ChecksumAlg::update(const unsigned char *input, const unsigned inputLen) {
	Assert(!theSum.set());

	// Compute number of bytes mod 64
	unsigned int index = 
		(unsigned int)((theCounts[0] >> 3) & 0x3F);

	// Update number of bits
	if ((theCounts[0] += ((unsigned int)inputLen << 3)) < ((unsigned int)inputLen << 3))
		theCounts[1]++;

	theCounts[1] += ((unsigned int)inputLen >> 29);

	const unsigned int partLen = 64 - index;

	// transform as many times as possible.
	unsigned int i = 0;
	if (inputLen >= partLen) {
		memcpy(&theBuffer[index], input, partLen);
		transform(theBuffer);

		for (i = partLen; i + 63 < inputLen; i += 64)
			transform(&input[i]);

		index = 0;
	}

	// Buffer remaining input
	memcpy(&theBuffer[index], &input[i], inputLen-i);
}

void ChecksumAlg::update(const char *buf, Size size) {
	update((const unsigned char *)buf, (unsigned int)size);
}

// MD5 finalization. Ends an MD5 message-digest operation, writing the
// the message digest and zeroizing the context.
void ChecksumAlg::final() {
	unsigned char bits[8];
	unsigned int index, padLen;

	// save number of bits
	encode(bits, theCounts, 8);

	// pad out to 56 mod 64.
	index = (unsigned int)((theCounts[0] >> 3) & 0x3f);
	padLen = index < 56 ? (56 - index) : (120 - index);
	update(ThePadding, padLen);

	// append length (before padding)
	update(bits, 8);

	// store state in digest
	encode(theSum.theDigest, theState, sizeof(theSum.theDigest));
	theSum.isSet = true;

	// zeroize sensitive information
	memset(theState, 0, sizeof(theState));
	memset(theCounts, 0, sizeof(theCounts));
	memset(theBuffer, 0, sizeof(theBuffer));
}

// MD5 basic transformation. Transforms state based on block.
void ChecksumAlg::transform(const unsigned char block[64]) {
	unsigned int a = theState[0];
	unsigned int b = theState[1];
	unsigned int c = theState[2];
	unsigned int d = theState[3];

	unsigned int x[16];
	decode(x, block, 64);

	/* Round 1 */
	FF (a, b, c, d, x[ 0], S11, 0xd76aa478); /* 1 */
	FF (d, a, b, c, x[ 1], S12, 0xe8c7b756); /* 2 */
	FF (c, d, a, b, x[ 2], S13, 0x242070db); /* 3 */
	FF (b, c, d, a, x[ 3], S14, 0xc1bdceee); /* 4 */
	FF (a, b, c, d, x[ 4], S11, 0xf57c0faf); /* 5 */
	FF (d, a, b, c, x[ 5], S12, 0x4787c62a); /* 6 */
	FF (c, d, a, b, x[ 6], S13, 0xa8304613); /* 7 */
	FF (b, c, d, a, x[ 7], S14, 0xfd469501); /* 8 */
	FF (a, b, c, d, x[ 8], S11, 0x698098d8); /* 9 */
	FF (d, a, b, c, x[ 9], S12, 0x8b44f7af); /* 10 */
	FF (c, d, a, b, x[10], S13, 0xffff5bb1); /* 11 */
	FF (b, c, d, a, x[11], S14, 0x895cd7be); /* 12 */
	FF (a, b, c, d, x[12], S11, 0x6b901122); /* 13 */
	FF (d, a, b, c, x[13], S12, 0xfd987193); /* 14 */
	FF (c, d, a, b, x[14], S13, 0xa679438e); /* 15 */
	FF (b, c, d, a, x[15], S14, 0x49b40821); /* 16 */

	/* Round 2 */
	GG (a, b, c, d, x[ 1], S21, 0xf61e2562); /* 17 */
	GG (d, a, b, c, x[ 6], S22, 0xc040b340); /* 18 */
	GG (c, d, a, b, x[11], S23, 0x265e5a51); /* 19 */
	GG (b, c, d, a, x[ 0], S24, 0xe9b6c7aa); /* 20 */
	GG (a, b, c, d, x[ 5], S21, 0xd62f105d); /* 21 */
	GG (d, a, b, c, x[10], S22,  0x2441453); /* 22 */
	GG (c, d, a, b, x[15], S23, 0xd8a1e681); /* 23 */
	GG (b, c, d, a, x[ 4], S24, 0xe7d3fbc8); /* 24 */
	GG (a, b, c, d, x[ 9], S21, 0x21e1cde6); /* 25 */
	GG (d, a, b, c, x[14], S22, 0xc33707d6); /* 26 */
	GG (c, d, a, b, x[ 3], S23, 0xf4d50d87); /* 27 */

	GG (b, c, d, a, x[ 8], S24, 0x455a14ed); /* 28 */
	GG (a, b, c, d, x[13], S21, 0xa9e3e905); /* 29 */
	GG (d, a, b, c, x[ 2], S22, 0xfcefa3f8); /* 30 */
	GG (c, d, a, b, x[ 7], S23, 0x676f02d9); /* 31 */
	GG (b, c, d, a, x[12], S24, 0x8d2a4c8a); /* 32 */

	/* Round 3 */
	HH (a, b, c, d, x[ 5], S31, 0xfffa3942); /* 33 */
	HH (d, a, b, c, x[ 8], S32, 0x8771f681); /* 34 */
	HH (c, d, a, b, x[11], S33, 0x6d9d6122); /* 35 */
	HH (b, c, d, a, x[14], S34, 0xfde5380c); /* 36 */
	HH (a, b, c, d, x[ 1], S31, 0xa4beea44); /* 37 */
	HH (d, a, b, c, x[ 4], S32, 0x4bdecfa9); /* 38 */
	HH (c, d, a, b, x[ 7], S33, 0xf6bb4b60); /* 39 */
	HH (b, c, d, a, x[10], S34, 0xbebfbc70); /* 40 */
	HH (a, b, c, d, x[13], S31, 0x289b7ec6); /* 41 */
	HH (d, a, b, c, x[ 0], S32, 0xeaa127fa); /* 42 */
	HH (c, d, a, b, x[ 3], S33, 0xd4ef3085); /* 43 */
	HH (b, c, d, a, x[ 6], S34,  0x4881d05); /* 44 */
	HH (a, b, c, d, x[ 9], S31, 0xd9d4d039); /* 45 */
	HH (d, a, b, c, x[12], S32, 0xe6db99e5); /* 46 */
	HH (c, d, a, b, x[15], S33, 0x1fa27cf8); /* 47 */
	HH (b, c, d, a, x[ 2], S34, 0xc4ac5665); /* 48 */

	/* Round 4 */
	II (a, b, c, d, x[ 0], S41, 0xf4292244); /* 49 */
	II (d, a, b, c, x[ 7], S42, 0x432aff97); /* 50 */
	II (c, d, a, b, x[14], S43, 0xab9423a7); /* 51 */
	II (b, c, d, a, x[ 5], S44, 0xfc93a039); /* 52 */
	II (a, b, c, d, x[12], S41, 0x655b59c3); /* 53 */
	II (d, a, b, c, x[ 3], S42, 0x8f0ccc92); /* 54 */
	II (c, d, a, b, x[10], S43, 0xffeff47d); /* 55 */
	II (b, c, d, a, x[ 1], S44, 0x85845dd1); /* 56 */
	II (a, b, c, d, x[ 8], S41, 0x6fa87e4f); /* 57 */
	II (d, a, b, c, x[15], S42, 0xfe2ce6e0); /* 58 */
	II (c, d, a, b, x[ 6], S43, 0xa3014314); /* 59 */
	II (b, c, d, a, x[13], S44, 0x4e0811a1); /* 60 */
	II (a, b, c, d, x[ 4], S41, 0xf7537e82); /* 61 */
	II (d, a, b, c, x[11], S42, 0xbd3af235); /* 62 */
	II (c, d, a, b, x[ 2], S43, 0x2ad7d2bb); /* 63 */
	II (b, c, d, a, x[ 9], S44, 0xeb86d391); /* 64 */

	theState[0] += a;
	theState[1] += b;
	theState[2] += c;
	theState[3] += d;

	// Zeroize sensitive information.
	memset(x, 0, sizeof(x));
}

// encodes input (unsigned int) into output (unsigned char)
// assumes len is a multiple of 4
void ChecksumAlg::encode(unsigned char *output, const unsigned int *input, unsigned int len) {
	unsigned int i, j;
	for (i = 0, j = 0; j < len; i++, j += 4) {
		output[j] = (unsigned char)(input[i] & 0xff);
		output[j+1] = (unsigned char)((input[i] >> 8) & 0xff);
		output[j+2] = (unsigned char)((input[i] >> 16) & 0xff);
		output[j+3] = (unsigned char)((input[i] >> 24) & 0xff);
	}
}

// decodes input (unsigned char) into output (unsigned int)
// assumes len is a multiple of 4
void ChecksumAlg::decode(unsigned int *output, const unsigned char *input, unsigned int len) {
	unsigned int i, j;
	for (i = 0, j = 0; j < len; i++, j += 4) {
		output[i] = ((unsigned int)input[j]) | (((unsigned int)input[j+1]) << 8) |
			(((unsigned int)input[j+2]) << 16) | (((unsigned int)input[j+3]) << 24);
	}
}
