/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Folders : Gtk.TreeView {
    Accounts accounts;
    AccountInfo local_info;
    Gtk.TreeStore store;
    Gtk.TreePath? selected_folder = null;
    Gtk.TreePath? drop_folder = null;
    int blink_counter = -1;

    public Postler.Messages messages { get; set; }
    public string? selected_location { get; private set; }

    public Postler.AccountInfo? get_selected_account () {
        return_val_if_fail (messages != null, null);
        Gtk.TreeIter iter;
        if (get_selection ().get_selected (null, out iter)) {
            AccountInfo account_info;
            store.get (iter, Columns.INFO, out account_info);
            return account_info;
        }
        return null;
    }

    enum Columns {
        ICON,
        NAME,
        DISPLAY_NAME,
        ELLIPSIZE,
        LOCATION,
        INFO,
        TOPLEVEL_MONITOR,
        FOLDER_MONITOR,
        UNREAD_MONITOR,
        FOLDER_TYPE
    }

    void on_drag_data_received (Gdk.DragContext context, int x, int y,
        Gtk.SelectionData selection_data, uint info, uint time_) {

        Gtk.TreePath path;
        Gtk.TreeViewDropPosition pos;
        if (get_dest_row_at_pos (x, y, out path, out pos)) {
              Gtk.TreeIter iter;
              var tree_model = get_model ();
              string destination_path;
              tree_model.get_iter_from_string (out iter, path.to_string ());
              tree_model.get (iter, Columns.LOCATION, out destination_path);
              foreach (string uri in selection_data.get_uris ()) {
                  try {
                      string location = Filename.from_uri (uri, null);
                      if (location.has_prefix (destination_path + "/cur/"))
                          return;

                      string new_location = Postler.Messages.update_filename (location,
                          destination_path + "/cur/");
                      var source = GLib.File.new_for_path (location);
                      var destination = GLib.File.new_for_path (new_location);
                      source.move (destination, GLib.FileCopyFlags.NONE);
                  }
                  catch (Error error) {
                      GLib.critical ("Dropping failed: %s", error.message);
                      return;
                  }
              }
        }
    }

    bool on_drag_motion (Gdk.DragContext context, int x, int y, uint time_) {
        if (selected_folder == null) {
            Gtk.TreeModel model;
            Gtk.TreeIter iter;
            get_selection ().get_selected (out model, out iter);
            selected_folder = model.get_path (iter);
        }
        Gtk.TreePath path;
        Gtk.TreeViewDropPosition pos;
        if (get_dest_row_at_pos (x, y, out path, out pos)) {
            var selection = get_selection ();
            selection.select_path (path);
        }
        return true;
    }

    void on_drag_leave (Gdk.DragContext context, uint time_) {
        get_selection ().select_path (selected_folder);
        selected_folder = null;
    }

    bool on_drag_drop (Gdk.DragContext context, int x, int y, uint time_) {
        if (selected_folder == null) {
            Gtk.TreeModel model;
            Gtk.TreeIter iter;
            get_selection ().get_selected (out model, out iter);
            selected_folder = model.get_path (iter);
        }
        Gtk.TreePath path;
        Gtk.TreeViewDropPosition pos;
        if (get_dest_row_at_pos (x, y, out path, out pos)) {
            var selection = get_selection ();
            selection.select_path (path);
            drop_folder = path;
            if (selected_folder.compare (drop_folder) != 0)
                GLib.Timeout.add (350, blink);
        }
        return true;
    }

    bool blink () {
        blink_counter++;
        if (blink_counter > 5) {
            drop_folder = null;
            blink_counter = -1;
            get_selection ().select_path (selected_folder);
            selected_folder = null;
            return false;
        }
        if ((blink_counter % 2) == 0) {
            get_selection ().unselect_all ();
            return true;
        } else {
            get_selection ().select_path (drop_folder);
            return true;
        }
    }

    public Folders (Accounts accounts) {
        /* Drag and Drop */
        Gtk.drag_dest_set (this, Gtk.DestDefaults.ALL, {}, Gdk.DragAction.MOVE);
        Gtk.drag_dest_add_uri_targets (this);
        drag_data_received.connect (on_drag_data_received);
        drag_motion.connect (on_drag_motion);
        drag_leave.connect (on_drag_leave);
        drag_drop.connect (on_drag_drop);

        this.accounts = accounts;

        /* Local folders */
        local_info = new AccountInfo ();
        local_info.display_name = _("Local");
        local_info.type = AccountType.MAILDIR;
        local_info.path = null;
        unowned string mail = Environment.get_variable ("MAILDIR");
        if (mail != null && mail != "")
            local_info.path = mail;
        else {
            string local_mail = Environment.get_home_dir () + "/Mail";
            if (FileUtils.test (local_mail, FileTest.IS_DIR))
                local_info.path = local_mail;
            else {
                local_mail = Environment.get_home_dir () + "/Maildir";
                if (FileUtils.test (local_mail, FileTest.IS_DIR))
                    local_info.path = local_mail;
            }
        }

        store = new Gtk.TreeStore (10,
            typeof (string), typeof (string), typeof (string),
            typeof (int), typeof (string),
            typeof (AccountInfo), typeof (GLib.FileMonitor),
            typeof (GLib.FileMonitor), typeof (GLib.FileMonitor), typeof (FolderType));
        set_model (store);
        insert_column_with_attributes (-1, "Icon",
            new Gtk.CellRendererPixbuf (), "stock-id", Columns.ICON);
        insert_column_with_attributes (-1, "Folder",
            new Gtk.CellRendererText (), "markup", Columns.DISPLAY_NAME,
                                         "ellipsize", Columns.ELLIPSIZE);
        GLib.Idle.add (populate);

        store.set_sort_func (Columns.FOLDER_TYPE, folder_compare);
        store.set_sort_column_id (Columns.FOLDER_TYPE, Gtk.SortType.ASCENDING);
        store.sort_column_changed ();
    }

    public int folder_compare (Gtk.TreeModel model, Gtk.TreeIter a, Gtk.TreeIter b) {
        FolderType type_a;
        FolderType type_b;

        model.get (a, Columns.FOLDER_TYPE, out type_a);
        model.get (b, Columns.FOLDER_TYPE, out type_b);

        if (type_a < type_b)
            return -1;
        if (type_a > type_b)
            return 1;
        if (type_a == type_b) {
            string name_a;
            string name_b;

            model.get (a, Columns.NAME, out name_a);
            model.get (b, Columns.NAME, out name_b);
            return GLib.strcmp (name_a, name_b);
        }

        assert_not_reached ();
    }

    void unread_count_update (Gtk.TreeIter iter, File msg_dir, string label) {
        try {
            var msg_enumerator = msg_dir.enumerate_children ("", 0, null);
            int unread = 0;
            FileInfo info;
            while ((info = msg_enumerator.next_file (null)) != null)
                unread++;
            string escaped = GLib.Markup.escape_text (label);
            if (unread == 0)
                store.set (iter, Columns.DISPLAY_NAME, "%s".printf (escaped));
            else if (msg_dir.get_path ().has_suffix ("INBOX/new"))
                store.set (iter, Columns.DISPLAY_NAME,
                           "<b>%s (%d)</b>".printf (escaped, unread));
            else
                store.set (iter, Columns.DISPLAY_NAME,
                           "%s (%d)".printf (escaped, unread));
        } catch (GLib.Error error) {
            GLib.critical (_("Failed to monitor folder \"%s\": %s"),
                           msg_dir.get_path (), error.message);
        }
    }

    void unread_monitor_changed (File msg_dir, string label) {
        string location = msg_dir.get_path ();
        location = location.slice (0, -4); /* - /new */

        Gtk.TreeIter account_iter;
        if (!store.iter_children (out account_iter, null))
            return;
        do {
            string existing_location;
            store.get (account_iter, Columns.LOCATION, out existing_location);
            if (existing_location == location) {
                unread_count_update (account_iter, msg_dir, label);
                break;
            }
            string account_location = existing_location.slice (0, - 6);
            if (location.has_prefix (account_location)) {
                Gtk.TreeIter iter;
                if (get_folder_iter (location, account_iter, out iter))
                    unread_count_update (iter, msg_dir, label);
                break;
            }
        } while (store.iter_next (ref account_iter));
    }

    public bool populate () {
        store.clear ();

        var account_infos = accounts.get_infos ().copy ();
        if (local_info.path != null)
            account_infos.prepend (local_info);
        return populate_accounts (account_infos);
    }

    bool get_folder_iter (string location, Gtk.TreeIter parent_iter,
        out Gtk.TreeIter iter) {

        string existing_location;
        store.get (parent_iter, Columns.LOCATION, out existing_location);

        if (existing_location == location) {
            if (&iter != null)
                iter = parent_iter;
            return true;
        }

        Gtk.TreeIter folder_iter;
        if (!store.iter_children (out folder_iter, parent_iter))
            return false;

        bool found = false;
        do {
            found = get_folder_iter (location,folder_iter, out iter);

        } while (!found && store.iter_next (ref folder_iter));

        return found;
    }

    public static string decode_foldername (string name) {
        /* Modified UTF-7 according to RFC 2060 */
        string decoded = name.replace ("&", "+").replace (",", "/");
        try {
            return GLib.convert (decoded, -1, "UTF-8", "UTF-7", null);
        }
        catch (Error error) {
            if (error is ConvertError.ILLEGAL_SEQUENCE)
                GLib.warning ("Incomplete folder name conversion: %s", name);
            else
                GLib.warning ("Failed to convert folder name: %s", error.message);
        }
        return name;
    }

    public bool populate_accounts (GLib.List<AccountInfo> infos) {
        bool need_update = false;

        /* foreach (unowned AccountInfo account_info in infos) { */
        for (unowned GLib.List<AccountInfo> info_iter = infos;
            info_iter != null; info_iter = info_iter.next) {
            AccountInfo account_info = info_iter.data;
            if (account_info.type == AccountType.SEARCH) {
                store.insert_with_values (null, null, -1,
                    Columns.ICON, STOCK_FOLDER_SAVED_SEARCH,
                    Columns.NAME, account_info.name,
                    Columns.DISPLAY_NAME, account_info.name,
                    Columns.ELLIPSIZE, Pango.EllipsizeMode.MIDDLE,
                    Columns.INFO, account_info,
                    Columns.LOCATION, account_info.path,
                    Columns.FOLDER_TYPE, FolderType.GENERIC);
                continue;
            }

            try {
                var folder_dir = File.new_for_path (account_info.path);

                Gtk.TreeIter account_iter;
                bool existing_iter = false;
                if (store.iter_children (out account_iter, null)) {
                    do {
                        AccountInfo other_info;
                        store.get (account_iter,
                            Columns.INFO, out other_info);
                        if (account_info == other_info) {
                            existing_iter = true;
                            break;
                        }
                    } while (store.iter_next (ref account_iter));
                }
                if (existing_iter) {
                    store.set (account_iter,
                                Columns.ICON, Gtk.STOCK_DIRECTORY,
                                Columns.LOCATION, null);
                    Gtk.TreeIter iter;
                    while (store.iter_children (out iter, account_iter))
                        store.remove (iter);
                }
                else {
                    var monitor = folder_dir.monitor_directory (0, null);
                    monitor.changed.connect ((monitor, file, other, event) => {
                        var account_infos = new GLib.List<AccountInfo> ();
                        account_infos.prepend (account_info);
                        populate_accounts (account_infos);
                    });
                    store.insert_with_values (out account_iter, null, -1,
                        Columns.ICON, Gtk.STOCK_DIALOG_ERROR,
                        Columns.NAME, account_info.name,
                        Columns.DISPLAY_NAME, account_info.display_name,
                        Columns.ELLIPSIZE, Pango.EllipsizeMode.MIDDLE,
                        Columns.LOCATION, null,
                        Columns.INFO, account_info,
                        Columns.TOPLEVEL_MONITOR, monitor);
                }

                if (!FileUtils.test (account_info.path + "/INBOX", FileTest.EXISTS))
                    continue;

                var folder_enumerator = folder_dir.enumerate_children (
                    FILE_ATTRIBUTE_STANDARD_NAME, 0, null);
                FileInfo info;
                while ((info = folder_enumerator.next_file (null)) != null) {
                    string name = info.get_name ();

                    if (name in account_info.hide)
                        continue;

                    var status_dir = folder_dir.resolve_relative_path (name);
                    var new_dir = status_dir.resolve_relative_path ("new");
                    if (!new_dir.query_exists (null))
                        continue;

                    Gtk.TreeIter parent_iter = account_iter;
                    Gtk.TreeIter folder_iter;
                    string folder_name = name.replace ("~~", "~");

                    /* A . or ~- separates parent and subfolder */
                    unowned string? separator = null;
                    if ("~-" in name)
                        separator = "~-";
                    else if ("." in name)
                        separator = ".";
                    if (separator != null) {
                        string folder_dir_path = folder_dir.get_path ();
                        string[] name_pieces = name.split (separator);
                        string parent = name_pieces[name_pieces.length - 2];
                        folder_name = name_pieces[name_pieces.length - 1];
                        string parent_path = name.substring (0,
                            name.length - folder_name.length - separator.length);
                        if (get_folder_iter (account_info.path + "/" + parent_path,
                            account_iter, out folder_iter))
                            parent_iter = folder_iter;
                        else {
                            store.insert_with_values (out parent_iter, account_iter, -1,
                                Columns.ICON, Gtk.STOCK_DIRECTORY,
                                Columns.NAME, parent,
                                Columns.DISPLAY_NAME, parent,
                                Columns.ELLIPSIZE, Pango.EllipsizeMode.MIDDLE,
                                Columns.LOCATION, folder_dir_path + "/" + parent_path,
                                Columns.INFO, account_info,
                                Columns.FOLDER_TYPE, FolderType.GENERIC);
                        }
                    } else if (get_folder_iter (account_info.path + "/" + name,
                                                account_iter, null))
                        continue;

                    if (name == "INBOX") {
                        var msg_dir = folder_dir.resolve_relative_path (
                            account_info.path + "/" + name + "/new");
                        var monitor = msg_dir.monitor_directory (0, null);
                        string label = account_info.display_name;
                        monitor.changed.connect ((monitor, file, other, event) => {
                            unread_monitor_changed (msg_dir, label);
                        });

                        account_info.folders[FolderType.INBOX] = name;
                        store.set (account_iter,
                            Columns.ICON, STOCK_INBOX,
                            Columns.NAME, label,
                            Columns.LOCATION, account_info.path + "/" + name,
                            Columns.UNREAD_MONITOR, monitor,
                            Columns.FOLDER_TYPE, FolderType.INBOX);
                        unread_count_update (account_iter, msg_dir, label);
                        continue;
                    }

                    var account_dir = folder_dir.resolve_relative_path (name);
                    var monitor = account_dir.monitor_directory (0, null);
                    monitor.changed.connect ((monitor, file, other, event) => {
                        var account_infos = new GLib.List<AccountInfo> ();
                        account_infos.prepend (account_info);
                        populate_accounts (account_infos);
                    });

                    unowned MailFolder folder = account_info.get_localized_folder (name);
                    folder_iter = new Gtk.TreeIter ();
                    if (folder.role != null
                     && get_folder_iter (account_info.path + "/" + folder.role,
                                         account_iter, out folder_iter)) {
                        store.remove (folder_iter);
                        need_update = true;
                    }

                    string display_name = _(folder.label) ?? decode_foldername (folder_name);
                    store.insert_with_values (out folder_iter,
                        folder.localized != null ? account_iter : parent_iter, -1,
                        Columns.ICON, folder.stock_id ?? Gtk.STOCK_DIRECTORY,
                        Columns.NAME, display_name,
                        Columns.ELLIPSIZE, Pango.EllipsizeMode.MIDDLE,
                        Columns.LOCATION, account_info.path + "/" + name,
                        Columns.INFO, account_info,
                        Columns.FOLDER_MONITOR, monitor,
                        Columns.FOLDER_TYPE, folder.type);
                    var msg_dir = folder_dir.resolve_relative_path (
                        account_info.path + "/" + name + "/new");
                    monitor = msg_dir.monitor_directory (0, null);
                    monitor.changed.connect ((monitor, file, other, event) => {
                        unread_monitor_changed (msg_dir, display_name);
                    });
                    unread_count_update (folder_iter, msg_dir, display_name);
                    store.set (folder_iter, Columns.UNREAD_MONITOR, monitor);
                }

                /* Look for missing special folders or create them */
                Gtk.TreeIter folder_iter = new Gtk.TreeIter ();
                account_info.verify_folders ((type, folder) => {
                    if (get_folder_iter (account_info.path + "/" + folder.role,
                                         account_iter, out folder_iter)) {
                        account_info.folders[type] = folder.role;
                        store.set (folder_iter,
                                   Columns.ICON, folder.stock_id ?? Gtk.STOCK_DIRECTORY,
                                   Columns.NAME, folder.label,
                                   Columns.DISPLAY_NAME, folder.label,
                                   Columns.FOLDER_TYPE, folder.type);
                        need_update = true;
                    } else {
                        string missing = account_info.path + "/" + folder.role;
                        DirUtils.create_with_parents (missing + "/new", 0700);
                        DirUtils.create (missing + "/cur", 0700);
                        DirUtils.create (missing + "/tmp", 0700);
                    }
                });

                if (account_info.display_name == _("Inbox"))
                    expand_row (store.get_path (account_iter), false);
            }
            catch (GLib.Error error) {
                GLib.critical (_("Failed to read folder \"%s\": %s"),
                    account_info.path, error.message);
            }
            if (need_update)
                accounts.update ();
        }
        return false;
    }

    public bool select_folder (string folder, Gtk.TreeIter? parent_iter=null)
        requires (messages != null) {
        if (folder == "")
            return false;
        Gtk.TreeIter iter;
        if (!store.iter_children (out iter, parent_iter))
            return false;
        do {
            string location;
            store.get (iter, Columns.LOCATION, out location);
            if (location == folder) {
                get_selection ().select_iter (iter);
                display_folder_iter (iter);
                return true;
            }
            if (select_folder (folder, iter))
                return true;
        } while (store.iter_next (ref iter));
        return false;
    }

    void display_folder_iter (Gtk.TreeIter iter)
        requires (messages != null) {
        string location;
        AccountInfo account_info;
        store.get (iter, Columns.LOCATION, out location,
                         Columns.INFO, out account_info);
        selected_location = location;
        if (location != null) {
            messages.populate (location, account_info);
            messages.grab_focus ();
            if (account_info.type == AccountType.SEARCH)
                return;
            try {
                var folder_dir = File.new_for_path (location + "/new");
                var enumerator = folder_dir.enumerate_children (
                    FILE_ATTRIBUTE_STANDARD_NAME, 0);
                FileInfo file_info;
                while ((file_info = enumerator.next_file ()) != null) {
                    var msg = File.new_for_path (
                        location + "/new/" + file_info.get_name ());
                    var destination = File.new_for_path (
                        location + "/cur/" + file_info.get_name ());
                    msg.move (destination, FileCopyFlags.NONE);
                }
            } catch (Error error) {
                GLib.warning (_("Failed to select folder \"%s\": %s"),
                    location, error.message);
            }
        }
        else
            messages.display_error (_("No messages were fetched yet"),
                _("Click Receive Mail in the toolbar.\n"
                + "If it doesn't work, right-click the account name and\n"
                + "select Account Properties to check the account details."));
    }

    void use_folder (Gtk.TreeIter iter, FolderType ftype) {
        string location = "";
        AccountInfo account_info;
        store.get (iter, Columns.LOCATION, out location,
                         Columns.INFO, out account_info);
        string folder = (location.split (account_info.name + "/")) [1];
        account_info.set_folder (ftype, folder);
        accounts.update ();
        populate ();
    }
    
    void import_folder (Gtk.TreeIter iter) {
        var dialog = new Gtk.FileChooserDialog (_("Import Archived Mailbox"),
            get_toplevel () as Gtk.Window, Gtk.FileChooserAction.OPEN);
        dialog.add_buttons (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
            Gtk.STOCK_OPEN, Gtk.ResponseType.OK);
        dialog.set_default_response (Gtk.ResponseType.OK);
        int response = dialog.run ();
        var mailbox_archive = dialog.get_file ();
        dialog.destroy ();
        if (response != Gtk.ResponseType.OK)
            return;

        string location;
        AccountInfo account_info;
        store.get (iter, Columns.LOCATION, out location,
                         Columns.INFO, out account_info);
        try {
            string path = location + "/cur/";
            var stream = new DataInputStream (mailbox_archive.read (null));
            GLib.StringBuilder body = new GLib.StringBuilder ();
            string line;
            while ((line = stream.read_line (null, null)) != null) {
                if (line.has_prefix ("From ")) {
                    if (body.len == 0)
                        continue;

                    string filename = Postler.Messages.generate_filename (path, "S");
                    FileUtils.set_contents (filename, body.str, -1);
                    FileUtils.chmod (filename, 0700);
                    body = new GLib.StringBuilder ();
                }
                body.append (line + "\n");
            }
            if (body.len != 0) {
                string filename = Postler.Messages.generate_filename (path, "S");
                FileUtils.set_contents (filename, body.str, -1);
                FileUtils.chmod (filename, 0700);
            }
            messages.populate (location, account_info);
        } catch (GLib.Error error) {
            GLib.critical (_("Failed to empty folder \"%s\": %s"),
                location, error.message);
        }
    }

    void empty_folder (Gtk.TreeIter iter) {
        string location;
        store.get (iter, Columns.LOCATION, out location);
        try {
            string[] folders = { "cur", "new" };
            foreach (var folder in folders) {
                var folder_dir = File.new_for_path (location + "/" + folder);
                var folder_enumerator = folder_dir.enumerate_children (
                    FILE_ATTRIBUTE_STANDARD_NAME, 0, null);
                GLib.FileInfo info;
                while ((info = folder_enumerator.next_file (null)) != null) {
                    var file = folder_dir.resolve_relative_path (info.get_name ());
                    file.delete (null);
                }
            }
            messages.clear ();
        }
        catch (GLib.Error error) {
            GLib.critical (_("Failed to empty folder \"%s\": %s"),
                location, error.message);
        }
    }

    public override bool popup_menu () {
        var menu = new Gtk.Menu ();
        menu.deactivate.connect ((menu) => {
            GLib.Idle.add (() => {menu.destroy (); return false; });
        });

        Gtk.MenuItem menuitem;
        menuitem = new Gtk.MenuItem.with_mnemonic (_("Open in New _Window"));
        menuitem.activate.connect ((menuitem) => {
            Gtk.TreeIter iter;
            if (get_selection ().get_selected (null, out iter)) {
                string? location;
                AccountInfo account_info;
                store.get (iter, Columns.LOCATION, out location,
                                 Columns.INFO, out account_info);
                var bureau = new Bureau ();
                GLib.Idle.add (() => {
                    bureau.folders.select_folder (account_info.path + "/INBOX");
                    return false;
                });
                bureau.show ();
             }
        });
        menu.append (menuitem);

        bool is_account = false;
        bool is_local = false;
        bool is_search = false;
        Gtk.TreeIter selected_iter;
        if (get_selection ().get_selected (null, out selected_iter)) {
            Gtk.TreeIter iter;
            if (!store.iter_parent (out iter, selected_iter))
                is_account = true;

            AccountInfo? account_info;
            store.get (selected_iter, Columns.INFO, out account_info);
            is_local = account_info == local_info;
            is_search = account_info.type == AccountType.SEARCH;
        }

        if (!is_search) {
            menu.append (new Gtk.SeparatorMenuItem ());

            menuitem = new Gtk.MenuItem.with_mnemonic (_("_Import Archived Mailbox"));
            menuitem.activate.connect ((menuitem) => {
                 Gtk.TreeIter iter;
                 if (get_selection ().get_selected (null, out iter))
                     import_folder (iter);
            });
            menu.append (menuitem);
            menuitem = new Gtk.MenuItem.with_mnemonic (_("_Empty Folder"));
            menuitem.activate.connect ((menuitem) => {
                 Gtk.TreeIter iter;
                 if (get_selection ().get_selected (null, out iter))
                     empty_folder (iter);
            });
            menu.append (menuitem);
        }

        if (!is_account && !is_local) {
            menu.append (new Gtk.SeparatorMenuItem ());

            menuitem = new Gtk.MenuItem.with_mnemonic (_("_Use as..."));
            var use_as_menu = new Gtk.Menu ();
            for (int i = 1; i < FolderType.GENERIC; i++) {
                var type = (FolderType)i;
                var folder_item = new Gtk.ImageMenuItem.from_stock (
                    type.get_stock_id () ?? Gtk.STOCK_DIRECTORY, null);
                folder_item.set_always_show_image (true);
                folder_item.label = type.get_label ();
                use_as_menu.append (folder_item);
                folder_item.activate.connect ((folder_item) => {
                    Gtk.TreeIter iter;
                   if (get_selection ().get_selected (null, out iter))
                         use_folder (iter, type);
                });
            }
            menuitem.submenu = use_as_menu;
            menu.append (menuitem);

            menuitem = new Gtk.MenuItem.with_mnemonic (_("_Hide Folder"));
            menuitem.activate.connect ((menuitem) => {
                 Gtk.TreeIter iter;
                 if (get_selection ().get_selected (null, out iter)) {
                     string? location;
                     AccountInfo? account_info;
                     store.get (iter,
                         Columns.LOCATION, out location,
                         Columns.INFO, out account_info);
                     if (account_info != null) {
                         string name = location.rstr ("/").substring (1, -1);
                         if (!account_info.hide.contains (name))
                             account_info.hide += "," + name;
                         accounts.update ();
                         populate ();
                     }
                 }
            });
            menu.append (menuitem);
        }
        else if (!is_local) {
            menu.append (new Gtk.SeparatorMenuItem ());

            menuitem = new Gtk.MenuItem.with_mnemonic (_("Account _Properties"));
            if (is_search)
                menuitem.label = _("Saved Search _Properties");
            menuitem.activate.connect ((menuitem) => {
                Gtk.TreeIter iter;
                 if (get_selection ().get_selected (null, out iter)) {
                     AccountInfo? account_info;
                     store.get (iter, Columns.INFO, out account_info);
                     AccountSetup.edit_account (account_info).done.connect (
                         (setup, info) => {
                         if (info == null)
                             accounts.remove_info (account_info);
                         else {
                             accounts.update ();
                             new Postler.Client ().fetch (info.name);
                         }
                         populate ();
                     } );
                 }
            });
            menu.append (menuitem);
        }

        menu.show_all ();

        var event = Gtk.get_current_event ();
        menu.attach_to_widget (this, (widget, menu) => { });
        menu.popup (null, null, null, event.button.button, event.get_time ());
        return true;
    }

    public override bool button_release_event (Gdk.EventButton event) {
        Gtk.TreeIter iter;
        if (get_selection ().get_selected (null, out iter)) {
            if (event.button == 3) {
                /* popup_menu (); */
                bool handled;
                GLib.Signal.emit_by_name (this, "popup-menu", &handled);
            }
            else
                display_folder_iter (iter);
        }
        return base.button_release_event (event);
    }

    public override void row_activated (Gtk.TreePath path, Gtk.TreeViewColumn column) {
        Gtk.TreeIter iter;
        if (store.get_iter (out iter, path)) {
            display_folder_iter (iter);
        }
    }
}
