/***************************************************************************
                          gui_taglisting.cpp  -  description
                             -------------------
    begin                : Thu May 3 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "gui_taglisting.h"
#include "configuration.h"
#include "jobmanager.h"
#include "job_modifytracksindatabase.h"
#include "job_getextradata.h"
#include "gui.h"
#include "lvi_track.h"
#include "lvi_musicbrainzresult.h"
#include "tagger.h"
#include "guess_dialog.h"
#ifdef HAVE_LIBMUSICBRAINZ
#include "musicbrainzquery_dialog.h"
#endif /*HAVE_LIBMUSICBRAINZ*/
#include "pixmapcache.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include <qaccel.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qmessagebox.h>

#include <prokyon.h>


#define EDIT_MODE    1
#define VIEWING_MODE 2

#define MAXGENRE 147

// *******************
// *** PUBLIC PART ***
// *******************

// ##################################################
// # constructor
// ##################################################
GUI_TagListing::GUI_TagListing(QWidget *parent, QListView *_tracklist, const char *name)
: QFrame(parent, name), tracklist(_tracklist) {
    int lineSpace = fontMetrics().lineSpacing();

    QGridLayout *lay0 = new QGridLayout(this, 4, 5);

    lay0->addRowSpacing(3, lineSpace / 2);
    lay0->addColSpacing(0, lineSpace / 2);
    lay0->addColSpacing(2, lineSpace / 2);
    lay0->addColSpacing(4, lineSpace / 2);

    QButtonGroup *infobox = new QButtonGroup(_("File Information"), this);
    QGridLayout *lay1 = new QGridLayout(infobox, 7, 10);

    lay1->addRowSpacing(0, lineSpace);
    lay1->addRowSpacing(2, lineSpace / 2);
    lay1->addRowSpacing(4, lineSpace / 2);
    lay1->addRowSpacing(6, lineSpace / 2);

    lay1->addColSpacing(0, lineSpace / 2);
    lay1->addColSpacing(5, 16);
    lay1->addColSpacing(9, lineSpace / 2);

    lay1->setColStretch(7, 1);

    QLabel *filelabel = new QLabel("<b>" + QString(_("File")) + ": </b>", infobox);
    lay1->addWidget(filelabel, 1, 1, 2);

    labelFile = new QLabel( " ", infobox);
    lay1->addWidget(labelFile, 1, 2);

    QLabel *labelOf = new QLabel(" " + QString(_("of")) + " ", infobox);
    lay1->addWidget(labelOf, 1, 3);
    labelSum = new QLabel("0", infobox);
    lay1->addWidget(labelSum, 1, 4);
    QLabel *namelabel = new QLabel("<b>" + QString(_("Name")) + ": </b>", infobox);
    lay1->addWidget(namelabel, 1, 6);

    stackFilename = new QWidgetStack(infobox);
    labelFilename = new QLabel("", infobox);
    labelFilename->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackFilename->addWidget(labelFilename, 1);
    editFilename = new QLineEdit("", infobox);
    editFilename->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackFilename->addWidget(editFilename, 2);
    stackFilename->raiseWidget(1);
    lay1->addWidget(stackFilename, 1, 7);
    buttonFilename = new ExtendedPushButton("", infobox);
    QToolTip::add
        ( buttonFilename, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
    lay1->addWidget(buttonFilename, 1, 8);

    connect(editFilename, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_filenameEdit(const QString&)));
    connect(buttonFilename, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *pathlabel = new QLabel("<b>" + QString(_("Path")) + ": </b>", infobox);
    lay1->addWidget(pathlabel, 3, 1, 2);
    labelPath = new QLabel("", infobox);
    labelPath->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    lay1->addMultiCellWidget(labelPath, 3, 3, 2, 7);

    QLabel *infolabel = new QLabel("<b>" + QString(_("Info")) + ": </b>", infobox);
    lay1->addWidget(infolabel, 5, 1, 2);
    labelInfo = new QLabel("", infobox);
    labelInfo->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    lay1->addMultiCellWidget(labelInfo, 5, 5, 2, 7);

    lay0->addWidget(infobox, 0, 1);

    QButtonGroup *tagbox = new QButtonGroup(_("Tag Information"), this);
    QGridLayout *lay3 = new QGridLayout(tagbox, 7, 12);

    lay3->addRowSpacing(0, lineSpace);
    lay3->addRowSpacing(6, lineSpace / 2);
    lay3->addColSpacing(0, lineSpace / 2);
    lay3->addColSpacing(6, 10);
    lay3->addColSpacing(11, lineSpace / 2);

    lay3->setColStretch(4, 1);

    QLabel *labtitle = new QLabel("<b>" + QString(_("Title")) + ": </b>", tagbox);
    lay3->addWidget(labtitle, 1, 1, 2);
    stackTitle = new QWidgetStack(tagbox);
    labelTitle = new QLabel("", tagbox);
    labelTitle->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackTitle->addWidget(labelTitle, 1);
    editTitle = new QLineEdit("", tagbox);
    editTitle->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    //    editTitle->setFrame(false);
    stackTitle->addWidget(editTitle, 2);
    stackTitle->raiseWidget(1);
    lay3->addMultiCellWidget(stackTitle, 1, 1, 2, 9);
    buttonTitle = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonTitle, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
    lay3->addWidget(buttonTitle, 1, 10);

    connect(editTitle, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_titleEdit(const QString&)));
    connect(buttonTitle, SIGNAL(clicked()),
            this, SLOT(slot_titleButton()));
    connect(buttonTitle, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labartist = new QLabel("<b>" + QString(_("Artist")) + ": </b>", tagbox);
    lay3->addWidget(labartist, 2, 1, 2);
    stackArtist = new QWidgetStack(tagbox);
    labelArtist = new QLabel("", tagbox);
    labelArtist->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackArtist->addWidget(labelArtist, 1);
    editArtist = new QLineEdit("", tagbox);
    editArtist->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    //    editArtist->setFrame(false);
    stackArtist->addWidget(editArtist, 2);
    stackArtist->raiseWidget(1);
    lay3->addMultiCellWidget(stackArtist, 2, 2, 2, 9);
    buttonArtist = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonArtist, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
    lay3->addWidget(buttonArtist, 2, 10);

    connect(editArtist, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_artistEdit(const QString&)));
    connect(buttonArtist, SIGNAL(clicked()),
            this, SLOT(slot_artistButton()));
    connect(buttonArtist, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labalbum = new QLabel("<b>" + QString(_("Album")) + ": </b>", tagbox);
    lay3->addWidget(labalbum, 3, 1, 2);
    stackAlbum = new QWidgetStack(tagbox);
    labelAlbum = new QLabel("", tagbox);
    labelAlbum->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackAlbum->addWidget(labelAlbum, 1);
    editAlbum = new QLineEdit("", tagbox);
    editAlbum->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    //    editAlbum->setFrame(false);
    stackAlbum->addWidget(editAlbum, 2);
    stackAlbum->raiseWidget(1);
    lay3->addMultiCellWidget(stackAlbum, 3, 3, 2, 4);
    buttonAlbum = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonAlbum, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
    lay3->addWidget(buttonAlbum, 3, 5);

    connect(editAlbum, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_albumEdit(const QString&)));
    connect(buttonAlbum, SIGNAL(clicked()),
            this, SLOT(slot_albumButton()));
    connect(buttonAlbum, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labyear = new QLabel("<b>" + QString(_("Year")) + ": </b>", tagbox);
    lay3->addWidget(labyear, 3, 8, 2);
    stackYear = new QWidgetStack(tagbox);
    labelYear = new QLabel("", tagbox);
    stackYear->addWidget(labelYear, 1);
    editYear = new QLineEdit("", tagbox);
    //    editYear->setFrame(false);
    stackYear->addWidget(editYear, 2);
    stackYear->raiseWidget(1);
    stackYear->setFixedWidth(60);
    lay3->addWidget(stackYear, 3, 9);
    buttonYear = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonYear, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );

    lay3->addWidget(buttonYear, 3, 10);

    connect(editYear, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_yearEdit(const QString&)));
    connect(buttonYear, SIGNAL(clicked()),
            this, SLOT(slot_yearButton()));
    connect(buttonYear, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labcomment = new QLabel("<b>" + QString(_("Comment")) + ": </b>", tagbox);
    lay3->addWidget(labcomment, 4, 1, 2);
    stackComment = new QWidgetStack(tagbox);
    labelComment = new QLabel("", tagbox);
    labelComment->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackComment->addWidget(labelComment, 1);
    editComment = new QLineEdit("", tagbox);
    editComment->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    //    editComment->setFrame(false);
    stackComment->addWidget(editComment, 2);
    stackComment->raiseWidget(1);
    lay3->addMultiCellWidget(stackComment, 4, 4, 2, 9);
    buttonComment = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonComment, _( "Left click to apply change to all selected tracks.\n Right click for special options."));
    lay3->addWidget(buttonComment, 4, 10);

    connect(editComment, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_commentEdit(const QString&)));
    connect(buttonComment, SIGNAL(clicked()),
            this, SLOT(slot_commentButton()));
    connect(buttonComment, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labtrack = new QLabel("<b>" + QString(_("Track")) + ": </b>", tagbox);
    lay3->addWidget(labtrack, 5, 8, 2);
    stackTrack = new QWidgetStack(tagbox);
    labelTrack = new QLabel("", tagbox);
    stackTrack->addWidget(labelTrack, 1);
    spinTrack = new QSpinBox(0, 255, 1, tagbox);
    spinTrack->setSpecialValueText(" ");
    spinTrack->setButtonSymbols(QSpinBox::PlusMinus);
    stackTrack->addWidget(spinTrack, 2);
    stackTrack->raiseWidget(1);
    stackTrack->setFixedWidth(60);
    lay3->addWidget(stackTrack, 5, 9, 1);
    buttonTrack = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonTrack, _( "Left click to apply change to all selected tracks.\n Right click for special options.") );
    lay3->addWidget(buttonTrack, 5, 10);

    connect(spinTrack, SIGNAL(valueChanged(int)), this, SLOT(slot_tracknumberChanged(int)));
    connect(buttonTrack, SIGNAL(clicked()),
            this, SLOT(slot_trackButton()));
    connect(buttonTrack, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    QLabel *labgenre = new QLabel("<b>" + QString(_("Genre")) + ": </b>", tagbox);
    lay3->addWidget(labgenre, 5, 1, 2);
    stackGenre = new QWidgetStack(tagbox);
    labelGenre = new QLabel("", tagbox);
    labelGenre->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackGenre->addWidget(labelGenre, 1);
    comboGenre = new QComboBox(tagbox);
    comboGenre->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackGenre->addWidget(comboGenre, 2);
    stackGenre->raiseWidget(1);

    for(int i=0; i <= MAXGENRE+1; i++)
        comboGenre->insertItem(Tagger::getGenreAlpha(i), i);
    comboGenre->setCurrentItem(0);

    stackGenre->setFixedWidth(160);
    lay3->addWidget(stackGenre, 5, 2, 1);
    buttonGenre = new ExtendedPushButton("", tagbox);
    QToolTip::add
        ( buttonGenre, _( "Left click to apply change to all selected tracks.\n Right click for special options." ) );
    lay3->addWidget(buttonGenre, 5, 3);

    connect(comboGenre, SIGNAL(activated(int)),
            this, SLOT(slot_genreCombo(int)));
    connect(buttonGenre, SIGNAL(clicked()),
            this, SLOT(slot_genreButton()));
    connect(buttonGenre, SIGNAL(contextMenu(ExtendedPushButton*, const QPoint&)),
            this, SLOT(slot_buttonContext(ExtendedPushButton*, const QPoint&)));

    lay0->addWidget(tagbox, 1, 1);

    QGroupBox *commandbox = new QGroupBox(1, Qt::Horizontal, _("Commands"), this);
    commandbox->setInsideMargin( 2 );
    commandbox->setInsideSpacing( 0 );
    QFrame *commandframe = new QFrame(commandbox);
    QVBoxLayout *lay4 = new QVBoxLayout(commandframe);

    QIconSet iconset1 = pixmapcache->getFadeIconSet("lvi_changed.png");
    buttonEdit = new QPushButton(iconset1, _("Edit"), commandframe);
    buttonEdit->setMaximumHeight(iconset1.pixmap( QIconSet::Small,QIconSet::Normal ).height() + 7);
    lay4->addWidget(buttonEdit);
    #ifdef HAVE_LIBMUSICBRAINZ

    QIconSet iconset2 = pixmapcache->getFadeIconSet("ask_musicbrainz.png");
    buttonMusicbrainz = new QPushButton(iconset2, _("MusicBrainz"), commandframe);
    buttonMusicbrainz->setMaximumHeight(iconset2.pixmap( QIconSet::Small,QIconSet::Normal ).height() + 7);
    lay4->addWidget(buttonMusicbrainz);
    #endif

    QIconSet iconset3 = pixmapcache->getFadeIconSet("idea.png");
    buttonGuess = new QPushButton(iconset3, _("Guess"), commandframe);
    buttonGuess->setMaximumHeight(iconset3.pixmap( QIconSet::Small,QIconSet::Normal ).height() + 7);
    lay4->addWidget(buttonGuess);
    QIconSet iconset4 = pixmapcache->getFadeIconSet("filesave.png");
    buttonApply = new QPushButton(iconset4, _("Save"), commandframe);
    buttonApply->setMaximumHeight(iconset4.pixmap( QIconSet::Small,QIconSet::Normal ).height() + 7);
    lay4->addWidget(buttonApply);
    QIconSet iconset5 = pixmapcache->getFadeIconSet("action_discard.png");
    buttonDiscard = new QPushButton(iconset5, _("Discard"), commandframe);
    buttonDiscard->setMaximumHeight(iconset5.pixmap( QIconSet::Small,QIconSet::Normal ).height() + 7);
    lay4->addWidget(buttonDiscard);

    lay0->addWidget(commandbox, 1, 3);

    connect(buttonEdit, SIGNAL(clicked()),
            this, SLOT(slot_editbutton()));
    connect(buttonGuess, SIGNAL(clicked()),
            this, SLOT(slot_menuGuessFilename()));
    #ifdef HAVE_LIBMUSICBRAINZ

    connect(buttonMusicbrainz, SIGNAL(clicked()),
            this, SLOT(slot_musicbrainzbutton()));
    #endif

    connect(buttonApply, SIGNAL(clicked()),
            this, SLOT(slot_applybutton()));
    connect(buttonDiscard, SIGNAL(clicked()),
            this, SLOT(slot_discardbutton()));

    QButtonGroup *albumInformationBox = new QButtonGroup(_("Album Information"), this);
    QToolTip::add
        ( albumInformationBox, _( "Additional information about the album \nthe current track is part of" ));
    QGridLayout *lay2 = new QGridLayout(albumInformationBox, 4, 4);

    lay2->addRowSpacing(0, lineSpace);
    lay2->addRowSpacing(3, lineSpace / 2);
    lay2->addColSpacing(0, lineSpace / 2);
    lay2->addColSpacing(3, lineSpace / 2);
    lay2->setRowStretch(4, 1);

    QLabel *labAlbumCategory = new QLabel("<b>" + QString(_("Category")) + ":</b>", albumInformationBox);
    lay2->addMultiCellWidget(labAlbumCategory, 1, 1, 1, 2);

    stackAlbumCategory = new QWidgetStack(albumInformationBox);
    labelAlbumCategory = new QLabel("", albumInformationBox);
    stackAlbumCategory->addWidget(labelAlbumCategory, 1);

    comboAlbumCategory = new QComboBox(albumInformationBox);
    // Do not replyce these hardcoded values by the constants
    // defined in prokyon.h. Otherwise xgettext will not be
    // able to extract these strings for translation.
    comboAlbumCategory->insertItem(_("Normal"));
    comboAlbumCategory->insertItem(_("Comedy"));
    comboAlbumCategory->insertItem(_("Radio Drama"));
    comboAlbumCategory->insertItem(_("Musical"));
    comboAlbumCategory->insertItem(_("Sampler"));
    comboAlbumCategory->insertItem(_("Soundtrack"));

    stackAlbumCategory->addWidget(comboAlbumCategory, 2);
    stackAlbumCategory->raiseWidget(1);

    lay2->addWidget(stackAlbumCategory, 2, 1);

    buttonAlbumCategory = new ExtendedPushButton("", albumInformationBox, 0);
    QToolTip::add
        ( buttonAlbumCategory, _( "Left click to apply change to all selected tracks." ) );
    lay2->addWidget(buttonAlbumCategory, 2, 2);

    lay0->addWidget(albumInformationBox, 0, 3);

    connect(comboAlbumCategory, SIGNAL(activated(int)), this, SLOT(slot_albumCategory(int)));
    connect(buttonAlbumCategory, SIGNAL(clicked()), this, SLOT(slot_albumCategoryButton()));

    QButtonGroup *personalbox = new QButtonGroup(_("Personal Information"), this);
    QGridLayout *lay5 = new QGridLayout(personalbox, 3, 9);

    lay5->addRowSpacing(0, lineSpace);
    lay5->addRowSpacing(2, lineSpace / 2);
    lay5->addColSpacing(0, lineSpace / 2);
    lay5->addColSpacing(2, lineSpace / 2);
    lay5->addColSpacing(4, lineSpace / 2);
    lay5->addColSpacing(6, lineSpace / 2);
    lay5->addColSpacing(8, lineSpace / 2);

    lay5->setColStretch(3, 1);

    QLabel *notesLabel = new QLabel("<b>" + QString(_("Notes")) + ": </b>", personalbox);
    notesLabel->setAlignment(Qt::AlignRight);
    lay5->addWidget(notesLabel, 1, 1);

    stackNotes = new QWidgetStack(personalbox);
    labelNotes = new QLabel("", personalbox);
    labelNotes->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackNotes->addWidget(labelNotes, 1);
    editNotes = new QLineEdit("", personalbox);
    editNotes->setSizePolicy( QSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred ) );
    stackNotes->addWidget(editNotes, 2);
    stackNotes->raiseWidget(1);
    lay5->addWidget(stackNotes, 1, 3);

    connect(editNotes, SIGNAL(textChanged(const QString&)),
            this, SLOT(slot_notesEdit(const QString&)));

    QLabel *ratingLabel = new QLabel("<b>" + QString(_("Rating")) + ": </b>", personalbox);
    ratingLabel->setAlignment(Qt::AlignRight);
    lay5->addWidget(ratingLabel, 1, 5);

    stackRating = new QWidgetStack(personalbox);
    labelRating = new QLabel("", personalbox);
    stackRating->addWidget(labelRating, 1);
    spinRating = new QComboBox(false, personalbox);
    for(int i=0;i<=5;i++) {
        spinRating->insertItem(pixmapcache->get
                               (QString::number(i) + "star.png"), QString::number(i), i);
    }
    stackRating->addWidget(spinRating, 2);
    stackRating->raiseWidget(1);
    lay5->addWidget(stackRating, 1, 7);

    connect(spinRating, SIGNAL(activated(int)), this, SLOT(slot_ratingChanged(int)));

    lay0->addMultiCellWidget(personalbox, 2, 2, 1, 3);

    QAccel *accel = new QAccel(this);
    accel->connectItem(accel->insertItem(CTRL+Key_P),this,
                       SLOT(slot_selPrevTrack()));
    accel->connectItem(accel->insertItem(CTRL+Key_N),this,
                       SLOT(slot_selNextTrack()));

    buttonEdit->setEnabled(false);
    buttonGuess->setEnabled(false);
    #ifdef HAVE_LIBMUSICBRAINZ

    buttonMusicbrainz->setEnabled(false);
    #endif /*HAVE_LIBMUSICBRAINZ*/

    buttonApply->setEnabled(false);
    buttonDiscard->setEnabled(false);

    buttonFilename->hide();
    buttonTitle->hide();
    buttonArtist->hide();
    buttonAlbum->hide();
    buttonYear->hide();
    buttonTrack->hide();
    buttonGenre->hide();
    buttonComment->hide();
    buttonAlbumCategory->hide();

    editLock = 0;
    editMode = false;
    doNothing = false;

}

void GUI_TagListing::slot_setConnectionState(bool state) {
    if(state == false && editMode) {
        switchMode(VIEWING_MODE, false);
        notifySelectionChanged();
    }
}

// ##################################################
// # set switch for the ability to edit tags
// ##################################################
void GUI_TagListing::notifyAddEditLock() {
    editLock++;
    buttonEdit->setEnabled((currentTrack >= 0) && editLock == 0 && editEnabled);
}

void GUI_TagListing::notifyRemoveEditLock() {
    editLock--;
    buttonEdit->setEnabled((currentTrack >= 0) && editLock == 0 && editEnabled);
}

void GUI_TagListing::notifyEnableEdit(bool b) {
    buttonEdit->setEnabled(b);
    editEnabled = b;
}

void GUI_TagListing::getSelectedItems() {
    selectedItems.clear();
    QListViewItem *curr = tracklist->firstChild();
    while(curr) {
        if(curr->isSelected())
            selectedItems.append(curr);
        curr = curr->nextSibling();
    }
}

// ##################################################
// # selection has changed
// ##################################################
void GUI_TagListing::notifySelectionChanged() {
    if(doNothing)
        return;
    if(!editMode)
        getSelectedItems();
    if(selectedItems.count() == 1 && selectedItems.at(0) != tracklist->currentItem()) {
        tracklist->setCurrentItem(selectedItems.at(0));
        tracklist->setSelected(selectedItems.at(0), true);
        #if (QT_VERSION >= 0x030201)

        tracklist->setSelectionAnchor(selectedItems.at(0));
        #endif

    };
    notifyCurrentChanged(0);
}

void GUI_TagListing::notifyCurrentChanged(QListViewItem *lvi) {
    if(doNothing)
        return;
    if(!editMode)
        getSelectedItems();
    int selectcount = (selectedItems.count());

    if(selectcount == 0 && tracklist->currentItem() != 0) {
        tracklist->setSelected(tracklist->currentItem(), true);
        if(!editMode)
            getSelectedItems();
        selectcount = 1;
    }

    if(!editMode) {
        buttonEdit->setEnabled((selectcount != 0) && editLock == 0 && editEnabled);
        buttonApply->setEnabled(false);
        #ifdef HAVE_LIBMUSICBRAINZ

        buttonMusicbrainz->setEnabled(false);
        #endif /*HAVE_LIBMUSICBRAINZ*/

        buttonDiscard->setEnabled(false);
    }

    applyCurrentTrack();

    #ifdef HAVE_MEXTRAS
    //**** comment this section to suppress musicextras data refresh

    // Only if we're outside of the edit mode.
    // In edit mode the MusicExtras data is not displayed anyway,
    // and the generated jobs come in the way of MusicBrainz queries.
    if (!editMode) {
        LVI_Track *item = static_cast<LVI_Track*>(gui->getListing()->list()->currentItem());
        if (item) {
            jobman->addJob(new Job_GetExtraData( MEXTRAS_GET, *item->getOriginalTrack() ));
        }
    }

    //******* end of section *****/
    #endif /* HAVE_MEXTRAS */
}

// ##################################################
// # sorting has changed (editmode only)
// ##################################################
void GUI_TagListing::notifySortingInEditModeChanged() {
    selectedItems.clear();
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        selectedItems.append(item);
        item = item->nextSibling();
    }

    applyCurrentTrack();
}

// ##################################################
// # destructor
// ##################################################
GUI_TagListing::~GUI_TagListing() {}

// ********************
// *** PRIVATE PART ***
// ********************

// ##################################################
// # draw information for current track
// ##################################################
void GUI_TagListing::applyCurrentTrack() {
    if( verbose == 4 ) {
        qWarning( "GUI_TagListing::applyCurrentTrack() starts");
        dumpSelectedItems( &selectedItems );
    }

    currentTrack = selectedItems.find(tracklist->currentItem());

    QString num;
    num.sprintf("%d", selectedItems.count());
    labelSum->setText(num);
    labelFile->setText(QString::number(currentTrack+1));

    if(currentTrack >= 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));
        TRACK *track = lvi->getOriginalTrack();

        labelInfo->setText(Tagger::getTypeAsString(track) + ", " +
                           Tagger::getModeAsString(track) + ", " +
                           Tagger::getSizeAsString(track));

        int xgenre = Tagger::genreToAlphaIndex(lvi->getGenre());

        setAlbumCategoryFromComment(lvi->getComment());

        if(!editMode) {
            QString s;
            labelFilename->setText(lvi->getFilename());
            labelPath->setText(lvi->getPath());
            labelTitle->setText(lvi->getTitle());
            labelArtist->setText(lvi->getArtist());
            labelAlbum->setText(lvi->getAlbum());
            if(lvi->getTracknumber() != 0) {
                labelTrack->setText(s.sprintf("%02d", lvi->getTracknumber()));
            } else {
                labelTrack->setText("");
            }
            labelYear->setText(lvi->getYear());
            labelComment->setText(lvi->getComment());
            labelGenre->setText(Tagger::getGenre(lvi->getGenre()));
            labelNotes->setText(lvi->getNotes());
            labelRating->setText(s.sprintf("%d", lvi->getRating()));
            labelRating->setPixmap(pixmapcache->get
                                   (QString::number(lvi->getRating()) + "star.png"));
        } else {
            labelFilename->setText(lvi->getFilename());
            labelPath->setText(lvi->getPath());
            editFilename->setText(lvi->getFilename());
            nonEditPath = lvi->getPath();
            if(lvi->getType() == MEDIUM_CDROM || !lvi->getIsAvailable()) {
                stackFilename->raiseWidget(1);
                buttonFilename->hide();
            } else {
                stackFilename->raiseWidget(2);
                buttonFilename->show();
            }

            #ifdef HAVE_LIBMUSICBRAINZ
            if(lvi->getIsAvailable()) {
                buttonMusicbrainz->setEnabled(true);
            } else {
                buttonMusicbrainz->setEnabled(false);
            }
            #endif /*HAVE_LIBMUSICBRAINZ*/
            editTitle->setText(lvi->getTitle());
            editArtist->setText(lvi->getArtist());
            editAlbum->setText(lvi->getAlbum());
            spinTrack->setValue(lvi->getTracknumber());
            editYear->setText(lvi->getYear());
            editComment->setText(lvi->getComment());
            comboGenre->setCurrentItem(xgenre);
            editNotes->setText(lvi->getNotes());
            spinRating->setCurrentItem(lvi->getRating());
        }
        tracklist->ensureItemVisible(lvi);
    } else {
        labelFilename->setText("");
        labelPath->setText("");
        labelInfo->setText("");
        labelTitle->setText("");
        labelArtist->setText("");
        labelAlbum->setText("");
        labelTrack->setText("");
        labelYear->setText("");
        labelComment->setText("");
        labelGenre->setText("");
        labelAlbumCategory->setText("");
        labelNotes->setText("");
        labelRating->setText("");
        labelRating->setPixmap(0);
    }

    if ( verbose == 4 )
        qWarning( "GUI_TagListing::applyCurrentTrack() stops");
}

// ##################################################
// # set album category from comment
// ##################################################
void GUI_TagListing::setAlbumCategoryFromComment(QString comment) {
    int widget = 1;

    if(comment == SAMPLER) {
        comboAlbumCategory->setCurrentItem(4);
    } else if(comment == SOUNDTRACK) {
        comboAlbumCategory->setCurrentItem(5);
    } else if(comment == COMEDY) {
        comboAlbumCategory->setCurrentItem(1);
    } else if(comment == DRAMA) {
        comboAlbumCategory->setCurrentItem(2);
    } else if(comment == MUSICAL) {
        comboAlbumCategory->setCurrentItem(3);
    } else {
        comboAlbumCategory->setCurrentItem(0);
        widget = 2;
    }

    labelAlbumCategory->setText(comboAlbumCategory->currentText());

    if(editMode) {
        labelComment->setText(comment);
        if(widget != stackComment->id(stackComment->visibleWidget())) {
            editComment->setText(comment);
            stackComment->raiseWidget(widget);
        }
    }
}

// ##################################################
// # switch between view- and editmode
// ##################################################
void GUI_TagListing::switchMode(int mode, bool saving) {

    int widgetlayer;

    if ( verbose == 4 )
        qWarning( "GUI_TagListing::switchMode( mode=%s, saving=%s ) starts",
                  ( mode == EDIT_MODE ) ? "EDIT_MODE" : "VIEWING_MODE",
                  saving ? "true" : "false" );

    if ( mode == VIEWING_MODE ) { // restore previous tooltip
        QToolTip::remove
            ( this );
        QToolTip::add
            ( this, oldtip);
    };

    if ( mode == EDIT_MODE ) {  // save previous tooltip
        oldtip = QToolTip::textFor( this, QPoint(0,0));
        QToolTip::remove
            ( this );
    };

    if(!saving) {
        buttonLastClicked = buttonFilename;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonTitle;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonArtist;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonAlbum;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonTrack;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonYear;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonGenre;
        slot_menuDiscardChangesALL();
        buttonLastClicked = buttonComment;
        slot_menuDiscardChangesALL();
    }

    if(mode == EDIT_MODE)
        widgetlayer = 2;
    else
        widgetlayer = 1;

    if(mode == VIEWING_MODE)
        stackFilename->raiseWidget(widgetlayer);
    stackTitle->raiseWidget(widgetlayer);
    stackArtist->raiseWidget(widgetlayer);
    stackAlbum->raiseWidget(widgetlayer);
    stackYear->raiseWidget(widgetlayer);
    stackTrack->raiseWidget(widgetlayer);
    stackGenre->raiseWidget(widgetlayer);
    stackComment->raiseWidget(widgetlayer);
    stackAlbumCategory->raiseWidget(widgetlayer);
    stackNotes->raiseWidget(widgetlayer);
    stackRating->raiseWidget(widgetlayer);

    if(mode == EDIT_MODE) {
        buttonFilename->show();
        buttonTitle->show();
        buttonArtist->show();
        buttonAlbum->show();
        buttonYear->show();
        buttonTrack->show();
        buttonGenre->show();
        buttonComment->show();
        buttonAlbumCategory->show();

        buttonEdit->setEnabled(false);
        buttonGuess->setEnabled(true);
        #ifdef HAVE_LIBMUSICBRAINZ

        buttonMusicbrainz->setEnabled(true);
        #endif /*HAVE_LIBMUSICBRAINZ*/

        buttonApply->setEnabled(true);
        buttonDiscard->setEnabled(true);

        doNothing = true;
        gui->getListing()->setEditMode(true);
        tracklist->setSelected(selectedItems.at(currentTrack), true);
        doNothing = false;

        gui->setStatusInfo(" " + QString(_("Mode")) + ": " + /* TRANSLATORS: Mode 'tagedit' */ QString(_("tagedit")), 5);

        // save panels status to restore them once edit is done
        statusleft = gui->getSelector()->isVisible();
        statusright = gui->getPlaylisting()->isVisible();
        statusbottom = gui->getTagListing()->isVisible();
        autohide = (config->getAutoHide() == 1);

        editMode = true;
        if(autohide) {
            gui->getSelector()->hide();
            gui->getSourcer()->hide();
            gui->getPlaylisting()->hide();
            gui->getSwitchBar()->setEnabled(false);
        }
        // always show the bottom bar when in edit mode
        gui->getTagListing()->show();
    } else {
        buttonFilename->hide();
        buttonTitle->hide();
        buttonArtist->hide();
        buttonAlbum->hide();
        buttonYear->hide();
        buttonTrack->hide();
        buttonGenre->hide();
        buttonComment->hide();
        buttonAlbumCategory->hide();

        if(!saving)
            buttonEdit->setEnabled(true);
        buttonGuess->setEnabled(false);
        #ifdef HAVE_LIBMUSICBRAINZ

        buttonMusicbrainz->setEnabled(false);
        #endif /*HAVE_LIBMUSICBRAINZ*/

        buttonApply->setEnabled(false);
        buttonDiscard->setEnabled(false);

        gui->setStatusInfo(" " + QString(_("Mode")) + ": " + QString(_("normal")), 5);
        editMode = false;
        if(autohide) {
            if(statusleft)
                gui->getSelector()->show();
            if(statusleft)
                gui->getSourcer()->show();
            if(statusright)
                gui->getPlaylisting()->show();
            gui->getSwitchBar()->setEnabled(true);
        }
        if(!statusbottom)
            gui->getTagListing()->hide();

        //    LVI_Track *track = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack))));
        LVI_Track *track = dynamic_cast<LVI_Track*>(tracklist->currentItem());
        gui->getListing()->setEditMode(false);
        notifySelectionChanged();
        if(track->getIsShowed()) {
            tracklist->setCurrentItem(track);
            //      currentTrack = selectedItems.find(track);
        }
    }

    applyCurrentTrack();

    if ( verbose == 4 )
        qWarning( "GUI_TagListing::switchMode ends");
}

// ##################################################
// # callbacks for the 5 normal buttons
// ##################################################

void GUI_TagListing::slot_editbutton() {
    switchMode(EDIT_MODE, true);
}

void GUI_TagListing::slot_applybutton() {
    bool updateSamplerTab = false;
    bool updateFavouritesTab = false;

    QList<TRACK> *trackList = new QList<TRACK>;
    QList<QString> *origFilenameList = new QList<QString>;

    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);
        if(!lvi->getIsAvailable())
            lvi->setFilename(lvi->getOriginalTrack()->filename);
        if(lvi->hasFilenameChanged() ||
                lvi->hasTitleChanged() || lvi->hasArtistChanged() || lvi->hasAlbumChanged() ||
                lvi->hasYearChanged() || lvi->hasGenreChanged() || lvi->hasCommentChanged() ||
                lvi->hasTracknumberChanged() || lvi->hasNotesChanged() || lvi->hasRatingChanged()) {

            if(lvi->hasCommentChanged() || (lvi->hasAlbumChanged() && !isNormalAlbum(lvi->getComment())))
                updateSamplerTab = true;

            if(lvi->hasArtistChanged())
                updateFavouritesTab = true;

            TRACK *origTrack = lvi->getOriginalTrack();
            TRACK *changedTrack = new TRACK;
            changedTrack->id           = origTrack->id;
            changedTrack->path         = origTrack->path;
            changedTrack->filename     = lvi->getFilename();
            changedTrack->medium       = origTrack->medium;
            changedTrack->lastModified = origTrack->lastModified;
            changedTrack->hasChanged   = (lvi->hasTitleChanged() || lvi->hasArtistChanged() || lvi->hasAlbumChanged() ||
                                          lvi->hasYearChanged() || lvi->hasGenreChanged() || lvi->hasCommentChanged() ||
                                          lvi->hasTracknumberChanged() || lvi->hasNotesChanged() || lvi->hasRatingChanged() ||
                                          origTrack->hasChanged);
            changedTrack->mimetype     = origTrack->mimetype;
            changedTrack->version      = origTrack->version;
            changedTrack->layer        = origTrack->layer;
            changedTrack->mode         = origTrack->mode;
            changedTrack->bitrate      = origTrack->bitrate;
            changedTrack->samplerate   = origTrack->samplerate;
            changedTrack->length       = origTrack->length;
            changedTrack->size         = origTrack->size;
            changedTrack->artist       = lvi->getArtist().stripWhiteSpace();
            changedTrack->title        = lvi->getTitle().stripWhiteSpace();
            changedTrack->album        = lvi->getAlbum().stripWhiteSpace();
            changedTrack->tracknumber  = lvi->getTracknumber();
            changedTrack->year         = lvi->getYear();
            changedTrack->genre        = lvi->getGenre();
            changedTrack->comment      = lvi->getComment().stripWhiteSpace();
            changedTrack->notes        = lvi->getNotes();
            changedTrack->rating       = lvi->getRating();
            trackList->append(changedTrack);

            QString *fname = new QString(origTrack->filename);
            origFilenameList->append(fname);
        }
        lvi->applyChanges();
        item = item->nextSibling();
    }

    buttonLastClicked = buttonFilename;
    slot_menuDiscardChangesALL();
    jobman->addJob(new Job_ModifyTracksInDatabase(trackList, origFilenameList, updateSamplerTab, updateFavouritesTab));

    switchMode(VIEWING_MODE, true);
}

void GUI_TagListing::slot_discardbutton() {
    switchMode(VIEWING_MODE, false);
}

// ##################################################
// # callbacks for the editfields in editmode
// ##################################################
void GUI_TagListing::slot_artistEdit(const QString& artist) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       )  lvi->setArtist(artist);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_filenameEdit(const QString& filename) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setFilename(filename);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_titleEdit(const QString& title) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setTitle(title);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_albumEdit(const QString& album) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setAlbum(album);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_tracknumberChanged(int tracknumber) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setTracknumber(tracknumber);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_yearEdit(const QString& year) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setYear(year);
}

void GUI_TagListing::slot_commentEdit(const QString& comment) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setComment(comment);
    else
        Q_ASSERT(lvi);
    setAlbumCategoryFromComment(comment);
}

void GUI_TagListing::slot_genreCombo(int genre) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setGenre(Tagger::alphaToGenreIndex(genre));
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_albumCategory(int id) {
    if(id < 6) // 6 = PushButton
    {
        QString comment = getCommentFromId(id);

        if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
           ) lvi->setComment(comment);
        else
            Q_ASSERT(lvi);
        setAlbumCategoryFromComment(comment);
    }
}

void GUI_TagListing::slot_notesEdit(const QString& notes) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setNotes(notes);
    else
        Q_ASSERT(lvi);
}

void GUI_TagListing::slot_ratingChanged(int rating) {
    if ( LVI_Track* lvi = dynamic_cast<LVI_Track*>(tracklist->selectedItem())
       ) lvi->setRating(rating);
    else
        Q_ASSERT(lvi);
}

QString GUI_TagListing::getCommentFromId(int id) {
    QString comment = NORMAL;
    if(id == 4)
        comment = SAMPLER;
    if(id == 5)
        comment = SOUNDTRACK;
    if(id == 1)
        comment = COMEDY;
    if(id == 2)
        comment = DRAMA;
    if(id == 3)
        comment = MUSICAL;

    return comment;
}

// ##################################################
// # callbacks for the extended edit buttons
// ##################################################
void GUI_TagListing::slot_artistButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setArtist(editArtist->text());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_titleButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setTitle(editTitle->text());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_albumButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setAlbum(editAlbum->text());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_trackButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setTracknumber(spinTrack->value());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_autotrackButton() {
    int cur_tracknum;
    QListViewItem *item = tracklist->firstChild();
    //  cur_tracknum = dynamic_cast<LVI_Track*>(item)->getTracknumber();
    //  if (cur_tracknum <= 0)
    cur_tracknum = 1;
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setTracknumber(cur_tracknum++);
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_yearButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setYear(editYear->text());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_commentButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setComment(editComment->text());
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_genreButton() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setGenre(Tagger::alphaToGenreIndex(comboGenre->currentItem()));
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_albumCategoryButton() {
    QString comment = getCommentFromId(comboAlbumCategory->currentItem());
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        dynamic_cast<LVI_Track*>(item)->setComment(comment);
        item = item->nextSibling();
    }
    setAlbumCategoryFromComment(comment);
}

// ##################################################
// # context menu in editmode
// ##################################################
void GUI_TagListing::slot_buttonContext(ExtendedPushButton *but, const QPoint& p) {
    buttonLastClicked = but;

    QPopupMenu *menu = new QPopupMenu(this);

    if(but == buttonFilename) {
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Standard Filename"), this, SLOT(slot_menuStandardFilename()), 0, 1);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Andromeda Filename"), this, SLOT(slot_menuAndromedaFilename()), 0, 2);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Custom Filename"), this, SLOT(slot_menuCustomFilename()), 0, 3);
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 4);
        menu->insertSeparator();
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Standard Filename to ALL"), this, SLOT(slot_menuStandardFilenameALL()), 0, 5);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Andromeda Filename to ALL"), this, SLOT(slot_menuAndromedaFilenameALL()), 0, 6);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Custom Filename to ALL"), this, SLOT(slot_menuCustomFilenameALL()), 0, 7);
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 8);
        menu->insertSeparator();
        menu->insertItem(pixmapcache->get
                         ("help.png"), _("More Info"), this, SLOT(slot_menuHelpAndromeda()), 0, 9);
    }
    if(but == buttonTitle || but == buttonArtist || but == buttonAlbum || but == buttonComment) {
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("First-Letter Upcase"), this, SLOT(slot_menuFLUpcase()), 0, 1);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Downcase"), this, SLOT(slot_menuDowncase()), 0, 2);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Guess from Filename"), this, SLOT(slot_menuGuessFilename()), 0, 3);
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 4);
        menu->insertSeparator();
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("First-Letter Upcase to ALL"), this, SLOT(slot_menuFLUpcaseALL()), 0, 5);
        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Downcase to ALL"), this, SLOT(slot_menuDowncaseALL()), 0, 6);
        #ifdef HAVE_LIBMUSICBRAINZ

        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Query MusicBrainz for ALL"), this, SLOT(slot_menuMusicBrainzALL()), 0, 7);
        #endif

        menu->insertItem(pixmapcache->get
                         ("action_writetags.png"), _("Guess from Filename to ALL"), this, SLOT(slot_menuGuessFilenameALL()), 0, 8);
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 9);
    }
    if(but == buttonTrack || but == buttonYear || but == buttonGenre) {
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes"), this, SLOT(slot_menuDiscardChanges()), 0, 1);
        menu->insertSeparator();
        if(but == buttonTrack)
            menu->insertItem(pixmapcache->get
                             ("action_writetags.png"), _("Auto Numbering By Listing Position"), this, SLOT(slot_autotrackButton()), 0, 3);
        menu->insertItem(pixmapcache->get
                         ("action_discard.png"), _("Discard Changes to ALL"), this, SLOT(slot_menuDiscardChangesALL()), 0, 2);
    }

    menu->exec(p, 0);
}

// ##################################################
// # callbacks for the menu-entries in context menu
// ##################################################
void GUI_TagListing::slot_menuDiscardChanges() {
    LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));

    if(buttonLastClicked == buttonFilename)
        editFilename->setText(lvi->getOriginalTrack()->filename);
    else if(buttonLastClicked == buttonTitle)
        editTitle->setText(lvi->getOriginalTrack()->title);
    else if(buttonLastClicked == buttonArtist)
        editArtist->setText(lvi->getOriginalTrack()->artist);
    else if(buttonLastClicked == buttonAlbum)
        editAlbum->setText(lvi->getOriginalTrack()->album);
    else if(buttonLastClicked == buttonTrack)
        spinTrack->setValue(lvi->getOriginalTrack()->tracknumber);
    else if(buttonLastClicked == buttonYear)
        editYear->setText(lvi->getOriginalTrack()->year);
    else if(buttonLastClicked == buttonGenre) {
        lvi->setGenre(lvi->getOriginalTrack()->genre);
        int xgenre = Tagger::genreToAlphaIndex(lvi->getGenre());
        comboGenre->setCurrentItem(xgenre);
    } else if(buttonLastClicked == buttonComment)
        editComment->setText(lvi->getOriginalTrack()->comment);
}

void GUI_TagListing::slot_menuDiscardChangesALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        if(buttonLastClicked == buttonFilename)
            lvi->setFilename(lvi->getOriginalTrack()->filename);
        else if(buttonLastClicked == buttonTitle)
            lvi->setTitle(lvi->getOriginalTrack()->title);
        else if(buttonLastClicked == buttonArtist)
            lvi->setArtist(lvi->getOriginalTrack()->artist);
        else if(buttonLastClicked == buttonAlbum)
            lvi->setAlbum(lvi->getOriginalTrack()->album);
        else if(buttonLastClicked == buttonYear)
            lvi->setYear(lvi->getOriginalTrack()->year);
        else if(buttonLastClicked == buttonTrack)
            lvi->setTracknumber(lvi->getOriginalTrack()->tracknumber);
        else if(buttonLastClicked == buttonGenre)
            lvi->setGenre(lvi->getOriginalTrack()->genre);
        else if(buttonLastClicked == buttonComment)
            lvi->setComment(lvi->getOriginalTrack()->comment);

        item = item->nextSibling();
    }
    slot_menuDiscardChanges();
}

void GUI_TagListing::slot_menuFLUpcase() {
    if(buttonLastClicked == buttonTitle) {
        QString s = editTitle->text();
        s = s.lower();
        for(uint i=0; i<s.length(); i++)
            if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                s[i] = s[i].upper();
        editTitle->setText(s);
    } else if(buttonLastClicked == buttonArtist) {
        QString s = editArtist->text();
        s = s.lower();
        for(uint i=0; i<s.length(); i++)
            if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                s[i] = s[i].upper();
        editArtist->setText(s);
    } else if(buttonLastClicked == buttonAlbum) {
        QString s = editAlbum->text();
        s = s.lower();
        for(uint i=0; i<s.length(); i++)
            if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                s[i] = s[i].upper();
        editAlbum->setText(s);
    } else if(buttonLastClicked == buttonComment) {
        QString s = editComment->text();
        s = s.lower();
        for(uint i=0; i<s.length(); i++)
            if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                s[i] = s[i].upper();
        editComment->setText(s);
    }
}

void GUI_TagListing::slot_menuFLUpcaseALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        if(buttonLastClicked == buttonTitle) {
            QString s = lvi->getTitle();
            s = s.lower();
            for(uint i=0; i<s.length(); i++)
                if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                    s[i] = s[i].upper();
            lvi->setTitle(s);
        } else if(buttonLastClicked == buttonArtist) {
            QString s = lvi->getArtist();
            s = s.lower();
            for(uint i=0; i<s.length(); i++)
                if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                    s[i] = s[i].upper();
            lvi->setArtist(s);
        } else if(buttonLastClicked == buttonAlbum) {
            QString s = lvi->getAlbum();
            s = s.lower();
            for(uint i=0; i<s.length(); i++)
                if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                    s[i] = s[i].upper();
            lvi->setAlbum(s);
        } else if(buttonLastClicked == buttonComment) {
            QString s = lvi->getComment();
            s = s.lower();
            for(uint i=0; i<s.length(); i++)
                if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                    s[i] = s[i].upper();
            lvi->setComment(s);
        }

        item = item->nextSibling();
    }
    slot_menuFLUpcase();
}

void GUI_TagListing::slot_menuDowncase() {
    if(buttonLastClicked == buttonTitle) {
        QString s = editTitle->text();
        s = s.lower();
        editTitle->setText(s);
    } else if(buttonLastClicked == buttonArtist) {
        QString s = editArtist->text();
        s = s.lower();
        editArtist->setText(s);
    } else if(buttonLastClicked == buttonAlbum) {
        QString s = editAlbum->text();
        s = s.lower();
        editAlbum->setText(s);
    } else if(buttonLastClicked == buttonComment) {
        QString s = editComment->text();
        s = s.lower();
        editComment->setText(s);
    }
}

void GUI_TagListing::slot_menuDowncaseALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        if(buttonLastClicked == buttonTitle) {
            QString s = lvi->getTitle();
            s = s.lower();
            lvi->setTitle(s);
        } else if(buttonLastClicked == buttonArtist) {
            QString s = lvi->getArtist();
            s = s.lower();
            lvi->setArtist(s);
        } else if(buttonLastClicked == buttonAlbum) {
            QString s = lvi->getAlbum();
            s = s.lower();
            lvi->setAlbum(s);
        } else if(buttonLastClicked == buttonComment) {
            QString s = lvi->getComment();
            s = s.lower();
            lvi->setComment(s);
        }
        item = item->nextSibling();
    }
    slot_menuDowncase();
}

void GUI_TagListing::slot_musicbrainzbutton() {
    #ifdef HAVE_LIBMUSICBRAINZ
    MusicBrainzQuery_Dialog* musicBrainzQueryDialog =
        new MusicBrainzQuery_Dialog( this, nonEditPath + "/" + editFilename->text() );
    if(musicBrainzQueryDialog->exec() == QDialog::Accepted) {
        LVI_MusicBrainzResult *result = musicBrainzQueryDialog->getSelected();
        if(result) {
            editArtist->setText(result->getArtist());
            editAlbum->setText(result->getAlbum());
            editTitle->setText(result->getTitle());
            spinTrack->setValue(result->getTracknumber());
        }
    }
    delete musicBrainzQueryDialog;
    #endif
}

void GUI_TagListing::slot_menuMusicBrainzALL() {
    #ifdef HAVE_LIBMUSICBRAINZ
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);
        tracklist->setCurrentItem(item);

        MusicBrainzQuery_Dialog* musicBrainzQueryDialog =
            new MusicBrainzQuery_Dialog( this, nonEditPath + "/" + editFilename->text() );
        musicBrainzQueryDialog->setModal(true);
        int clickedButtonCode = musicBrainzQueryDialog->exec();
        if(clickedButtonCode == QDialog::Accepted) {
            LVI_MusicBrainzResult *selectedResult = musicBrainzQueryDialog->getSelected();
            if(selectedResult) {
                editArtist->setText(selectedResult->getArtist());
                editAlbum->setText(selectedResult->getAlbum());
                editTitle->setText(selectedResult->getTitle());
                spinTrack->setValue(selectedResult->getTracknumber());
            }
        }
        delete musicBrainzQueryDialog;
        if (clickedButtonCode == MusicBrainzQuery_Dialog::SkipRemaining)
            return;
        item = item->nextSibling();
    }
    #endif
}

void GUI_TagListing::slot_menuGuessFilename() {
    Guess_Dialog* guess_Dialog = new Guess_Dialog( this, editFilename->text(), nonEditPath);
    int result = guess_Dialog->exec();
    if(result == QDialog::Accepted) {
        if (!guess_Dialog->artistname->text().isEmpty())
            editArtist->setText(guess_Dialog->artistname->text());
        if (!guess_Dialog->albumname->text().isEmpty())
            editAlbum->setText(guess_Dialog->albumname->text());
        if (!guess_Dialog->titlename->text().isEmpty())
            editTitle->setText(guess_Dialog->titlename->text());
        if (!guess_Dialog->track->text().isEmpty())
            spinTrack->setValue(guess_Dialog->track->text().toInt());
    }
    delete( guess_Dialog);  // call dtor. Do not remove
}

void GUI_TagListing::slot_menuGuessFilenameALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);
        tracklist->setCurrentItem(item);
        Guess_Dialog* guess_Dialog = new Guess_Dialog( this, lvi->getFilename(), lvi->getPath() );
        int result = guess_Dialog->exec();
        if(result == QDialog::Accepted) {
            if (!guess_Dialog->artistname->text().isEmpty())
                lvi->setArtist(guess_Dialog->artistname->text());
            if (!guess_Dialog->albumname->text().isEmpty())
                lvi->setAlbum(guess_Dialog->albumname->text());
            if (!guess_Dialog->titlename->text().isEmpty())
                lvi->setTitle(guess_Dialog->titlename->text());
            if (!guess_Dialog->track->text().isEmpty())
                lvi->setTracknumber(guess_Dialog->track->text().toInt());
            if (lvi->isSelected()) {
                if (!guess_Dialog->artistname->text().isEmpty())
                    editArtist->setText(guess_Dialog->artistname->text());
                if (!guess_Dialog->albumname->text().isEmpty())
                    editAlbum->setText(guess_Dialog->albumname->text());
                if (!guess_Dialog->titlename->text().isEmpty())
                    editTitle->setText(guess_Dialog->titlename->text());
                if (!guess_Dialog->track->text().isEmpty())
                    spinTrack->setValue(guess_Dialog->track->text().toInt());
            }
        }
        delete( guess_Dialog);  // call dtor. Do not remove
        if (result == Guess_Dialog::SkipRemaining)
            return;
        item = item->nextSibling();
    }
}

void GUI_TagListing::slot_menuStandardFilename() {
    if(editArtist->text().isEmpty() || editTitle->text().isEmpty())
        return;
    else {
        QString ending = editFilename->text();
        ending = ending.remove(0, ending.findRev('.')).lower();
        editFilename->setText(filter_ext2(editArtist->text().lower() + " - " + editTitle->text().lower() + ending));
    }
}

void GUI_TagListing::slot_menuStandardFilenameALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        if(!lvi->getArtist().isEmpty() && !lvi->getTitle().isEmpty()) {
            QString ending = lvi->getFilename();
            ending = ending.remove(0, ending.findRev('.')).lower();
            lvi->setFilename(filter_ext2(lvi->getArtist().lower() + " - " + lvi->getTitle().lower() + ending));
        }

        item = item->nextSibling();
    }
    slot_menuStandardFilename();
}

void GUI_TagListing::slot_menuAndromedaFilename() {
    if(editArtist->text().isEmpty() || editTitle->text().isEmpty())
        return;
    else {
        QString ending = editFilename->text();
        ending = ending.remove(0, ending.findRev('.')).lower();

        QString s = editAlbum->text();
        s = s.lower();
        QString acro = "";
        int acrocount = 0;
        for(uint i=0; i<s.length(); i++)
            if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                acro[acrocount++] = s[i].upper();

        LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));
        if(lvi->getTracknumber() != 0) {
            QString s = "";
            s.sprintf("%02d", lvi->getTracknumber());
            acro+=s;
        }

        editFilename->setText(filter_ext2(editArtist->text().lower() + " - " + acro + " - " + editTitle->text().lower() + ending));
    }
}

void GUI_TagListing::slot_menuHelpAndromeda() {
    QMessageBox::information(this, _("About Andromeda Filename"),
                             _("The Andromeda option assigns a filename to a track, so it is lexically in "
                               "the right order including the order of tracknumbers. It was implemented from the "
                               "great PHP-Script <b>Andromeda</b>.<br>"
                               "An Andromeda filename consists of the artist name, followed by the album abbreviation, a track number and "
                               " finally title name.<br>"
                               "By consistently using Andromeda file name, a simple alphanumeric sort will classify your tracks "
                               "by artist, album and track number.") );
}

void GUI_TagListing::slot_menuAndromedaFilenameALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        if (!lvi->getArtist().isEmpty() && !lvi->getTitle().isEmpty()) {
            QString ending = lvi->getFilename();
            ending = ending.remove(0, ending.findRev('.')).lower();
            QString s = lvi->getAlbum();
            s = s.lower();
            QString acro = "";
            int acrocount = 0;
            for(uint i=0; i<s.length(); i++)
                if(i==0 || (!s[i-1].isLetterOrNumber() && !(i>1 && s[i-1].isPunct() && !s[i-2].isSpace())))
                    acro[acrocount++] = s[i].upper();

            if(lvi->getTracknumber() != 0) {
                QString s = "";
                s.sprintf("%02d", lvi->getTracknumber());
                acro+=s;
            }

            lvi->setFilename(filter_ext2(lvi->getArtist().lower() + " - " + acro + " - " + lvi->getTitle().lower() + ending));
        }

        item = item->nextSibling();
    }
    slot_menuAndromedaFilename();
}

void GUI_TagListing::slot_menuCustomFilename() {
    if(editArtist->text().isEmpty() || editTitle->text().isEmpty())
        return;
    else {
        QString ending = editFilename->text();
        ending = ending.remove(0, ending.findRev('.')).lower();

        LVI_Track *lvi = dynamic_cast<LVI_Track*>(selectedItems.at(currentTrack));

        QString rename_template = config->getCustomRename();
        if (! rename_template.isEmpty() ) {
            QString num = "";
            num.sprintf("%02d", lvi->getTracknumber());

            rename_template.replace("%N", num);
            rename_template.replace("%A", editArtist->text());
            rename_template.replace("%T", editTitle->text());
            rename_template.replace("%S", editAlbum->text());
            rename_template.replace("%Y", editYear->text());
            editFilename->setText(filter_ext2( rename_template + ending ));
        }
    }
}

void GUI_TagListing::slot_menuCustomFilenameALL() {
    QListViewItem *item = tracklist->firstChild();
    while(item != 0) {
        LVI_Track *lvi = dynamic_cast<LVI_Track*>(item);

        QString rename_template = config->getCustomRename();
        if (! rename_template.isEmpty() ) {
            QString ending = lvi->getFilename();
            ending = ending.remove(0, ending.findRev('.')).lower();

            QString num = "";
            num.sprintf("%02d", lvi->getTracknumber());

            rename_template.replace("%N", num);
            rename_template.replace("%A", lvi->getArtist());
            rename_template.replace("%T", lvi->getTitle());
            rename_template.replace("%S", lvi->getAlbum());
            rename_template.replace("%Y", lvi->getYear());
            lvi->setFilename(filter_ext2( rename_template + ending ));
        }

        item = item->nextSibling();
    }
    slot_menuCustomFilename();
}




void GUI_TagListing::slot_selPrevTrack() {
    QListViewItem *item;
    if (!selectedItems.isEmpty() && selectedItems.count() == 1)
        item = selectedItems.getFirst();
    else if (tracklist->currentItem())
        item = tracklist->currentItem();
    else
        item = tracklist->firstChild();

    if (item) {
        item = item->itemAbove();
        if (item) {
            tracklist->clearSelection();
            tracklist->setSelected(item,true);
            tracklist->setCurrentItem(item);
            tracklist->ensureItemVisible(item);
        }
    }
}

void GUI_TagListing::slot_selNextTrack() {
    QListViewItem *item;
    if (!selectedItems.isEmpty() && selectedItems.count() == 1)
        item = selectedItems.getFirst();
    else if (tracklist->currentItem())
        item = tracklist->currentItem();
    else
        item = tracklist->firstChild();

    if (item) {
        item = item->itemBelow();
        if (item) {
            tracklist->clearSelection();
            tracklist->setSelected(item,true);
            tracklist->setCurrentItem(item);
            tracklist->ensureItemVisible(item);
        }
    }
}

QString GUI_TagListing::filter_ext2(QString filename) {

    // avoid filename using "/" and non printable character..

    for (int i=0; i<filename.length(); i++)
        if ( !filename[i].isPrint() || filename[i] == QChar('/') )
            filename[i]=QChar('_');
    return filename;
}
