#include <QtCore>
#include <QtGui>
#include <QString>
#include <QSet>
#include <QMap>
#include <QButtonGroup>

#include "psiplugin.h"
#include "stanzafilter.h"
#include "stanzasender.h"
#include "stanzasendinghost.h"
#include "optionaccessor.h"
#include "optionaccessinghost.h"
#include "eventfilter.h"

#define constMessage "message"
#define constTime "ltime"
#define constTid "ltid"
#define constTimeKey "ltimekey"
#define constTidKey "ltidkey"
#define constAllMsg "allunread"
#define constNotifyOn "notifyon"
#define constVersion "0.3.8"

class GmailNotifyPlugin : public QObject, public PsiPlugin, public EventFilter,
        public StanzaFilter, public StanzaSender, public OptionAccessor
{
        Q_OBJECT
        Q_INTERFACES(PsiPlugin StanzaFilter StanzaSender EventFilter OptionAccessor)
public:
        GmailNotifyPlugin();
        virtual QString name() const;
        virtual QString shortName() const;
        virtual QString version() const;
        virtual QWidget* options();
        virtual bool enable();
        virtual bool disable();
        // OptionAccessor
        virtual void setOptionAccessingHost(OptionAccessingHost* host);
        virtual void optionChanged(const QString& option);

        virtual void applyOptions();
        virtual void restoreOptions();

        virtual void setStanzaSendingHost(StanzaSendingHost *host);
        virtual bool incomingStanza(int account, const QDomElement& stanza);

        virtual bool processEvent(int account, QDomElement& e);
        virtual bool processMessage(int account, const QString& fromJid, const QString& body, const QString& subject);

private:
        bool enabled;
        StanzaSendingHost* stanzaSender;
        OptionAccessingHost* psiOptions;
        int id;
        QHash<int,QString> accounts;
        //QSet<QString> jidSet;
        QMap<QString,QString> lastCheck;
        QMap<QString,QString> lastTid;
        QString message;
        QTextEdit *editMessage;
        QCheckBox *notifyOnButton;
        bool allUnread;
        QRadioButton *onlyNewMsg;
        QRadioButton *allUneradMsg;
        bool notifyOn;
};

Q_EXPORT_PLUGIN(GmailNotifyPlugin);


GmailNotifyPlugin::GmailNotifyPlugin():enabled(false),id(0),allUnread(true){
        message =	"From: %1 %2\n"\
				"Header: %3\n"\
				"Url: http://mail.google.com/mail/";
	editMessage = 0;
        onlyNewMsg = 0;
        allUneradMsg = 0;
        notifyOn = true;
}

//PsiPlugin
QString GmailNotifyPlugin::name() const{
        return "GMail Notification Plugin";
}

QString GmailNotifyPlugin::shortName() const{
        return "gmailnotify";
}

QString GmailNotifyPlugin::version() const{
        return constVersion;
}

bool GmailNotifyPlugin::enable(){
        enabled = true;
        //Read default values
        QVariant vMessage(message);
        vMessage= psiOptions->getPluginOption(constMessage);
        if (!vMessage.isNull()) {
                message= vMessage.toString();
        }
        QVariant vNotify(notifyOn);
        vNotify= psiOptions->getPluginOption(constNotifyOn);
        if (!vNotify.isNull()) {
                notifyOn= vNotify.toBool();
        }
        QVariant vFlag(allUnread);
        vFlag= psiOptions->getPluginOption(constAllMsg);
        if (!vFlag.isNull()) {
                allUnread= vFlag.toBool();
        }
        if (!allUnread){
            lastCheck.clear();
            lastTid.clear();
            QStringList ltimekey;
            QVariant vTimeKey(ltimekey);
            vTimeKey = psiOptions->getPluginOption(constTimeKey);
            QStringList ltime;
            QVariant vTime(ltime);
            vTime= psiOptions->getPluginOption(constTime);
            if (!vTimeKey.isNull() && !vTime.isNull()) {
                    ltime= vTime.toStringList();
                    ltimekey = vTimeKey.toStringList();
                    int key = 0;
                    foreach(QString value,ltime){
                        lastCheck.insert(ltimekey.at(key++),value);
                    }
            }
            QStringList ltid;
            QVariant vTid(ltid);
            vTid= psiOptions->getPluginOption(constTid);
            QStringList ltidkey;
            QVariant vTidKey(ltidkey);
            vTidKey = psiOptions->getPluginOption(constTidKey);
            if (!vTidKey.isNull() && !vTid.isNull()) {
                    ltid = vTid.toStringList();
                    ltidkey = vTidKey.toStringList();
                    int key = 0;
                    foreach(QString value,ltid){
                        lastTid.insert(ltidkey.at(key++),value);
                    }
            }
        }
        return true;
}

bool GmailNotifyPlugin::disable(){
        enabled = false;
        return true;
}



bool GmailNotifyPlugin::processMessage(int account, const QString& fromJid, const QString& body, const QString& subject){
    Q_UNUSED(account);
    Q_UNUSED(fromJid);
    Q_UNUSED(body);
    Q_UNUSED(subject);
    return false;
}

bool GmailNotifyPlugin::processEvent(int account, QDomElement& e){
    if (enabled && accounts.keys().contains(account)) {
        QDomElement msg = e.lastChildElement();
        QString from = msg.attribute("from");
        QString to = msg.attribute("to");
        if ( to == from){
            from.replace(QRegExp("(.*)/.*"),"\\1/gmail");
            msg.setAttribute("from",from);
        }
    }
    return false;
}


void GmailNotifyPlugin::setStanzaSendingHost(StanzaSendingHost *host){
    stanzaSender = host;
}

bool GmailNotifyPlugin::incomingStanza(int account, const QDomElement& stanza){
    bool blocked = false;
    if (enabled) {
        //первоначальный запрос нотификаций
        if (stanza.tagName() == "iq"){
            QDomNode n = stanza.firstChild();
            if (!n.isNull()){
                QDomElement i = n.toElement();
                //проверяем поддержку расширения сервером
                if (stanza.attribute("type") == "result" && !i.isNull() && i.tagName() == "query" && i.attribute("xmlns") == "http://jabber.org/protocol/disco#info"){
                    for (QDomNode child = i.firstChild(); !child.isNull(); child = child.nextSibling()) {
                        QDomElement feature = child.toElement();
                        //отсеиваем клиентов с поддержкой нотификации
                        if (!feature.isNull() && feature.tagName()=="identity" && feature.attribute("category") != "server") break;
                        if( !feature.isNull() && feature.tagName()=="feature" && feature.attribute("var") == "google:mail:notify" && feature.attribute("node") == ""){
                            //получаем jid
                            QString jid = stanza.attribute("to");
                            QString clearJid = jid.split("/").at(0);
                            accounts.insert(account, clearJid);
                            //устанавливаем в настройках, получение нотификаций в true/false
                            QString reply = QString("<iq type=\"set\" to=\"%1\" id=\"%2\"><usersetting xmlns=\"google:setting\"><autoacceptrequests value=\"false\"/>"\
                                          "<mailnotifications value=\"%3\"/></usersetting></iq>").arg(clearJid).arg(++id).arg(notifyOn?"true":"false");
                            stanzaSender->sendStanza(account, reply);
                            if (notifyOn == true){
                                //шлём запрос в гугл на получение нотификаций
                                reply = QString("<iq type='get' from='%1' to='%2' id='mail-request-%5'>"\
                                    "<query xmlns='google:mail:notify' %3 %4/></iq>").arg(jid)
                                        .arg(clearJid)
                                        .arg((lastCheck.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-time='%1'").arg(lastCheck.value(QString("%1").arg(account))))
                                        .arg((lastTid.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-tid='%1'").arg(lastTid.value(QString("%1").arg(account))))
                                        .arg(++id);
                                stanzaSender->sendStanza(account, reply);
                            }
                            break;
                        }
                    }
                } else if ((accounts.empty() || accounts.keys().contains(account)) && stanza.attribute("type") == "set" && !i.isNull() && i.tagName() == "new-mail" && i.attribute("xmlns") == "google:mail:notify"){
                    //получили сообщение о новом письме
                    //шлём ответ
                    QString from = stanza.attribute("to");
                    QString to = from.split("/").at(0);
                    QString iqId = stanza.attribute("id");
                    QString reply = QString("<iq type='result' from='%1' to='%2' id='%3' />").arg(from).arg(to).arg(iqId);
                    stanzaSender->sendStanza(account, reply);
                    //и теперь шлём запрос письма
                    reply = QString("<iq type='get' from='%1' to='%2' id='mail-request-%3'>"\
                        "<query xmlns='google:mail:notify' %4 %5/></iq>")
                            .arg(from)
                            .arg(to)
                            .arg(id)
                            .arg((lastCheck.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-time='%1'").arg(lastCheck.value(QString("%1").arg(account))))
                            .arg((lastTid.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-tid='%1'").arg(lastTid.value(QString("%1").arg(account))));
                    stanzaSender->sendStanza(account, reply);
                    //блокируем дальшейшую обработку станзы
                    blocked = true;
                } else if(stanza.attribute("type") == "result" && !i.isNull() && i.tagName() == "mailbox" && i.attribute("xmlns") == "google:mail:notify") {
                    //получили новые письма
                    lastCheck.insert(QString("%1").arg(account), i.attribute("result-time"));
                    QString from = stanza.attribute("to");
                    QDomElement lastmail = i.firstChildElement("mail-thread-info");
                    if (!lastmail.isNull()){
                        lastTid.insert(QString("%1").arg(account), lastmail.attribute("tid"));
                    }
                    //сохраняем данные о послединих письмах
                    if (!allUnread){
                        psiOptions->setPluginOption(constTime, QVariant(lastCheck.values()));
                        psiOptions->setPluginOption(constTid, QVariant(lastTid.values()));
                        psiOptions->setPluginOption(constTimeKey, QVariant(lastCheck.keys()));
                        psiOptions->setPluginOption(constTidKey, QVariant(lastTid.keys()));
                    }
                    for (QDomNode child = i.lastChild(); !child.isNull(); child = child.previousSibling()) {
                        //url не валидный
                        QString url = child.toElement().attribute("url");
                        QString subject = child.firstChildElement("subject").text();
                        QString snippet = child.firstChildElement("snippet").text();
                        QDomElement senders = child.firstChildElement("senders");
                        QString name , email;
                        for (QDomNode sender = senders.firstChild(); !sender.isNull(); sender = sender.nextSibling()){
                            QDomElement adresser = sender.toElement();
                            if (!adresser.isNull() && adresser.attribute("originator")=="1"){
                                name = adresser.attribute("name");
                                email = adresser.attribute("address");
                            }
                        }
                        QString reply = QString("<message to=\"%1\" id=\"new-message-%2\" >"\
                            "<subject>%3</subject>"\
							"<body>%4</body>"\
                                                        "</message>").arg(from).arg(++id).arg((subject=="")?"\n":subject).
								arg(message.arg(name).arg(email).arg(snippet));
                        stanzaSender->sendStanza(account, reply);
                    }
                }
            }
        }
    }
    return blocked;
}

QWidget* GmailNotifyPlugin::options()
{
	if (!enabled) {
		return 0;
	}
	QWidget *optionsWid = new QWidget();

	editMessage= new QTextEdit(optionsWid);
	editMessage->setPlainText(message);
        notifyOnButton = new QCheckBox(tr("Notification On/Off"),optionsWid);
        QVBoxLayout *vbox= new QVBoxLayout(optionsWid);
        QButtonGroup *bgroup = new QButtonGroup(optionsWid);
        allUneradMsg = new QRadioButton(tr("Show all unread messages"),optionsWid);
        onlyNewMsg = new QRadioButton(tr("Show new messages only"),optionsWid);
        if (allUnread){
            allUneradMsg->setChecked(true);
        } else {
            onlyNewMsg->setChecked(true);
        }
        notifyOnButton->setChecked(notifyOn);
        vbox->addWidget(notifyOnButton);
        bgroup->addButton(allUneradMsg);
        bgroup->addButton(onlyNewMsg);
        QHBoxLayout *hbox= new QHBoxLayout();
        hbox->addWidget(allUneradMsg);
        hbox->addWidget(onlyNewMsg);
        vbox->addLayout(hbox);
        QGridLayout *layout = new QGridLayout();
	layout->addWidget(new QLabel("Message:\n%1 - name\n%2 - e-mail\n%3 - message", optionsWid), 0, 0);
	layout->addWidget(editMessage, 0, 1);
        vbox->addLayout(layout);
	return optionsWid;
}

void GmailNotifyPlugin::setOptionAccessingHost(OptionAccessingHost* host)
{
	psiOptions = host;
}

void GmailNotifyPlugin::optionChanged(const QString& option)
{
	Q_UNUSED(option);
}

void GmailNotifyPlugin::applyOptions() {
        if (!editMessage && !onlyNewMsg && !allUneradMsg) {
		return;
	}
	QVariant vMessage(editMessage->toPlainText());
	psiOptions->setPluginOption(constMessage, vMessage);
	message = vMessage.toString();
        QVariant vRead(!onlyNewMsg->isChecked());
        psiOptions->setPluginOption(constAllMsg, vRead);
        allUnread = vRead.toBool();

        QVariant vNotifyOn(notifyOnButton->isChecked());
        psiOptions->setPluginOption(constNotifyOn, vNotifyOn);
        notifyOn = vNotifyOn.toBool();
        foreach(int account,accounts.keys()){
            QString reply = QString("<iq type=\"set\" to=\"%1\" id=\"%2\"><usersetting xmlns=\"google:setting\">"\
                "<mailnotifications value=\"%3\"/></usersetting></iq>").arg(accounts.value(account)).arg(++id).arg(notifyOn?"true":"false");
            stanzaSender->sendStanza(account, reply);

            if (notifyOn == true){
            //шлём запрос в гугл на получение нотификаций
                reply = QString("<iq type='get' to='%1' id='mail-request-%4'>"\
                  "<query xmlns='google:mail:notify' %2 %3/></iq>")
                  .arg(accounts.value(account))
                  .arg((lastCheck.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-time='%1'").arg(lastCheck.value(QString("%1").arg(account))))
                  .arg((lastTid.value(QString("%1").arg(account),"")=="")?QString(""):QString("newer-than-tid='%1'").arg(lastTid.value(QString("%1").arg(account))))
                  .arg(++id);
                stanzaSender->sendStanza(account, reply);
            }
        }


}

void GmailNotifyPlugin::restoreOptions() {
        if (editMessage == 0 || onlyNewMsg == 0 || allUneradMsg == 0) {
                return;
        }
        QVariant vMessage(message);
        vMessage = psiOptions->getPluginOption(constMessage);
        if (!vMessage.isNull()) {
		editMessage->setPlainText(vMessage.toString());
	}
        QVariant vNotifyOn(notifyOn);
        vNotifyOn = psiOptions->getPluginOption(constNotifyOn);
        if (!vNotifyOn.isNull()) {
                notifyOnButton->setChecked(vNotifyOn.toBool());
        }
        QVariant vFlag(allUnread);
        vFlag= psiOptions->getPluginOption(constAllMsg);
        if (!vFlag.isNull()) {
            if (vFlag.toBool()){
                allUneradMsg->setChecked(true);
            }else{
                onlyNewMsg->setChecked(true);
            }
        }
}

#include "gmailnotifyplugin.moc"
