///////////////////////////////////////////////////////////////
//  Copyright Christopher Kormanyos 2002 - 2011.
//  Copyright 2011 John Maddock. Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_
//
// This work is based on an earlier work:
// "Algorithm 910: A Portable C++ Multiple-Precision System for Special-Function Calculations",
// in ACM TOMS, {VOL 37, ISSUE 4, (February 2011)} (C) ACM, 2011. http://doi.acm.org/10.1145/1916461.1916469

#ifdef _MSC_VER
#  define _SCL_SECURE_NO_WARNINGS
#endif

#include <boost/detail/lightweight_test.hpp>
#include <boost/array.hpp>
#include "test.hpp"

#if !defined(TEST_MPF_50) && !defined(TEST_MPF) && !defined(TEST_BACKEND) && !defined(TEST_MPZ) && !defined(TEST_CPP_DEC_FLOAT) && !defined(TEST_MPFR) && !defined(TEST_MPFR_50) && !defined(TEST_MPQ)
#  define TEST_MPF_50
//#  define TEST_MPF
#  define TEST_BACKEND
#  define TEST_CPP_DEC_FLOAT

#ifdef _MSC_VER
#pragma message("CAUTION!!: No backend type specified so testing everything.... this will take some time!!")
#endif
#ifdef __GNUC__
#pragma warning "CAUTION!!: No backend type specified so testing everything.... this will take some time!!"
#endif

#endif

#if defined(TEST_MPF_50)
#include <boost/multiprecision/gmp.hpp>
#endif
#if defined(TEST_MPFR_50)
#include <boost/multiprecision/mpfr.hpp>
#endif
#ifdef TEST_BACKEND
#include <boost/multiprecision/concepts/mp_number_archetypes.hpp>
#endif
#ifdef TEST_CPP_DEC_FLOAT
#include <boost/multiprecision/cpp_dec_float.hpp>
#endif

template <class T>
void test()
{
   static const boost::array<const char*, 101u> data =
   {{
           "0",
           "1.77245385090551602729816748334114518279754945612238712821380778985291128459103218137495065673854466541622682362428257066623615286572442260252509370960278706846203769865310512284992517302895082622893209537926796280017463901535147972051670019018523401858544697449491264031392177552590621640541933250090639840761373347747515343366798978936585183640879545116516173876005906739343179133280985484624818490205465485219561325156164746751504273876105610799612710721006037204448367236529661370809432349883166842421384570960912042042778577806869476657000521830568512541339663694465418151071669388332194292935706226886522442054214994804992075648639887483850593064021821402928581123306497894520362114907896228738940324597819851313487126651250629326004465638210967502681249693059542046156076195221739152507020779275809905433290066222306761446966124818874306997883520506146444385418530797357425717918563595974995995226384924220388910396640644729397284134504300214056423343303926175613417633632001703765416347632066927654181283576249032690450848532013419243598973087119379948293873011126256165881888478597787596376136",
           "2.50662827463100050241576528481104525300698674060993831662992357634229365460784197494659583837805726611600997266520387964486632361812673618095785565596614793191343548045812373113732780431301993880264715023446550489842580232941386277966868361303367582384418115607849009560004137515027529165146122584386186492069906092666409118801573969725237770253874824784863348864259750658951309573876342725192688040202857185924973630809978452538550099964101929825864558813716963467480152897452330473583474775619896089188931497718648548480469365216002979523919601823809165790411297975297712884880166754700558630196824142164233836361751921305615135984170982837036342343558097687424487526178010922324284883919536725153974582825630336511862808790203551930088145604026487771415650048376723275094136806424717481635184471307383168191953698843723340165605500555063029823385865249561125896634362311039438867484401970215424844692673432572876116452243380115662703163531763018031226825617668891776095502388960424467990521700581770868994182958846287294933254842394882655811767896697230383860684402997328870793925904767307522570843",
           "3.06998012383946546543865487466779458212212931325292345360920786339428774168895114940961908166467541760759370742449261902703454294050994162164179758776674921695210543321431976582792749695268039936031183310664759907107096641183589015949857072626264281130269822115622663836956231385381805315027562932655928781117811850728886383553772535550181070778545447895443834405205758132853217474114166659516983365541988620856642054467498643554872029717170398214071471942791253118514487638811304670059307541182798357443181785843119089299569008318867284816553359946310778010700675312493633986577498099096587006610844378385188518085716548910095121883158502955712612589941408660542373794977570528185247073824741671612893506514342410201362222059007498227945177352958046758148141756202994673600560820713848287574799339070054752116909136251765469324207685490757710781971553227808407026035595494277776874273936176792674139758867775825077183338566625556426910971465911191049352588901212917090887839548860059100937478465817844825147395110789153650851340493729565322385491250242078198566481445683133486898404370694024388072677",
           "3.54490770181103205459633496668229036559509891224477425642761557970582256918206436274990131347708933083245364724856514133247230573144884520505018741920557413692407539730621024569985034605790165245786419075853592560034927803070295944103340038037046803717089394898982528062784355105181243281083866500181279681522746695495030686733597957873170367281759090233032347752011813478686358266561970969249636980410930970439122650312329493503008547752211221599225421442012074408896734473059322741618864699766333684842769141921824084085557155613738953314001043661137025082679327388930836302143338776664388585871412453773044884108429989609984151297279774967701186128043642805857162246612995789040724229815792457477880649195639702626974253302501258652008931276421935005362499386119084092312152390443478305014041558551619810866580132444613522893932249637748613995767041012292888770837061594714851435837127191949991990452769848440777820793281289458794568269008600428112846686607852351226835267264003407530832695264133855308362567152498065380901697064026838487197946174238759896587746022252512331763776957195575192752273",
           "3.96332729760601101334502876512018303152235975639619479270725806219673871802657165722877418785963403123274056479329535362288655768235647487599476465537517349983459959396942125116402730874711739662452418728544662333652033991382097872017643021546533893998444782653493302314148486254376557154660429700133894837513481248619445021033294987995072089685196890751303705448870314051631971912137432499217943513505107459328579034234471969034380444279483175756904340168935694146646969758408612915912693951504143726276325727089094957428376128777490747011483653095038326498799910297470352498638922548951534285262912166233490129232297588127154122989932582549543207430381783967345221251717788180706706197608063733887726205509696294918128904282730298886608490931339746624769514837195042182809390984886538469878790432756760740967852006981627342718373868089930899834470763073879097175124340898791818850834951894187606842759251386865815997713188572196024482151907891258713869124247005410786490332294268446455782911914942752568277824622418634516620092362780994044583559067632108995042187023154200764184950878131638370936876",
           "4.34160752734960595617808776374477460723265336870727556185727396662220923171777437150069768940869308257867815441035461967651472282081172615279256504443662226606580627885152042042903335792297058916252842051925089290836831715454034263505813680769125291515266958692652530407528011558306762163905143207811598688431871170088962518292286615299738096190444738533203161905492043851016820053962091086343470680514156330656554108208679889001485483033545415301817702473378043508312559202170115728182897990337699623761986171416615021158179762123766012269021534563660979972864906653441725316241881374716900480598276640840394699718984639649897058274667150152948516142801074260038108590401332448730361188129358652703927446859710621067457990618378017703891730724396666299819790072397688972686459430503026856861736069218065420528347106866972580915029753621255304587561985717307629932540410639210302482882847873727015801634901755832519840544096758977294231047035556754992897011888753217953080197537610448726652746992121685068257440261763964980703424121083058605934982281210054435774069392790646709404966217005552197344493",
           "4.68947209983475140722248849561783002846415813208241069816339837584532567997445983205028355189765592648498726557587869684603896433394456735512692292415879610033794342501630732934996209978740746985164578019897211786300244708711916089199260058311084265831293902281791295846169702839730909351250374703986364508165540235333505281067384239291594323539816814423865582516577000094943855922812941268841597446572000848645702560088494072507948045804364964153467630734509663721026050762887316466331441449595084971862668493255933265611225629677595775797966272446890047531313544059153687362017301654537115283160575298256857111770084802622759507512748413868494678293024737576577634559012937002405564693237564782661478500575417457048864770102766387138956802222948987452086048547098754786648793023622941962569186040944009281487954082049335826413270469685003200962898734900372045181956983558503256265211391593721293841419380653006415407706145131700572194167555007645631927616801157793244790467083152103173240943823769137606544644820547634530641496507929783339735192083140914072272339899107017006649210880028740845785609",
           "5.01325654926200100483153056962209050601397348121987663325984715268458730921568394989319167675611453223201994533040775928973264723625347236191571131193229586382687096091624746227465560862603987760529430046893100979685160465882772555933736722606735164768836231215698019120008275030055058330292245168772372984139812185332818237603147939450475540507749649569726697728519501317902619147752685450385376080405714371849947261619956905077100199928203859651729117627433926934960305794904660947166949551239792178377862995437297096960938730432005959047839203647618331580822595950595425769760333509401117260393648284328467672723503842611230271968341965674072684687116195374848975052356021844648569767839073450307949165651260673023725617580407103860176291208052975542831300096753446550188273612849434963270368942614766336383907397687446680331211001110126059646771730499122251793268724622078877734968803940430849689385346865145752232904486760231325406327063526036062453651235337783552191004777920848935981043401163541737988365917692574589866509684789765311623535793394460767721368805994657741587851809534615045141685",
           "5.31736155271654808189450245002343554839264836836716138464142336955873385377309654412485197021563399624868047087284771199870845859717326780757528112880836120538611309595931536854977551908685247868679628613780388840052391704605443916155010057055570205575634092348473792094176532657771864921625799750271919522284120043242546030100396936809755550922638635349548521628017720218029537399842956453874455470616396455658683975468494240254512821628316832398838132163018111613345101709588984112428297049649500527264153712882736126128335733420608429971001565491705537624018991083396254453215008164996582878807118680659567326162644984414976226945919662451551779192065464208785743369919493683561086344723688686216820973793459553940461379953751887978013396914632902508043749079178626138468228585665217457521062337827429716299870198666920284340898374456622920993650561518439333156255592392072277153755690787924987985679154772661166731189921934188191852403512900642169270029911778526840252900896005111296249042896200782962543850728747098071352545596040257730796919261358139844881619033378768497645665435793362789128409",
           "5.60499121639792869931128243386880089385432377521078326495835339365228337765188797433638433752458537394171467057445629904555420755398315175666449466777454516186127276489666213119783783503829250827797595051437870440938931280680358959301715436419833364761665112542860561623096658225538557287000997305661352632293104827706037023135997795899396723258283708659284220534304000728311717413345472853792896822724499122866947121737000944102509490771269197783584170349935127887498647119562411695900883847124507613781432804229894058444676290534036738474342530360927632101836180145945813951643250885687876144098262191352431818136969137533863587873139725102593606754797723773610593636883665209271759639631465381618953021812982067002849796433240989364353192362974216155114226322430340671719312492223369487631048057665757861195096925695551107847621963544401115555199917577802008717751423237289036673475431186363543556380872803928482717387196307480474684969943012499128760245113496312609854517190311704797289830560651678579059756085781560282375298034806741180400732788584921019507521738918616603376508365622727087150932",
           "5.87856438167412823505442087789707097199843492103867437517314210715732455817816855246687780176492612349299503748836474526021467611461493586217913925495851405816296682706960791383913340375739628111898270614813266755200083564799407766788027485532332352024027224698199060815938197172865670956402289304658624389748945536687871488565787601288863899811544146566179125054361345491299666333144296548463580152591320203647485383402648565396566035890953872530543987300700877916661520962595410666627445873003711787722452859856569260494472375128936533790154715039734447084354515787738062350432618897439009684320850395595213875985248081961123885179420668160222435104348738172793787396737481265016580386986993101160673821650068377182350730255005754757604683510524357896861879706617761317908260663342673848298530050583191985919675390877532981022238129601627016291632330926779809631293070560375804444455958596407594955858475081851191996245268822870776361657887467004531332263395611291120089229481068789229695018187723686024313640325575962035428327239557079268995825494033666401736558372076816062703768818475224366442780",
           "6.13996024767893093087730974933558916424425862650584690721841572678857548337790229881923816332935083521518741484898523805406908588101988324328359517553349843390421086642863953165585499390536079872062366621329519814214193282367178031899714145252528562260539644231245327673912462770763610630055125865311857562235623701457772767107545071100362141557090895790887668810411516265706434948228333319033966731083977241713284108934997287109744059434340796428142943885582506237028975277622609340118615082365596714886363571686238178599138016637734569633106719892621556021401350624987267973154996198193174013221688756770377036171433097820190243766317005911425225179882817321084747589955141056370494147649483343225787013028684820402724444118014996455890354705916093516296283512405989347201121641427696575149598678140109504233818272503530938648415370981515421563943106455616814052071190988555553748547872353585348279517735551650154366677133251112853821942931822382098705177802425834181775679097720118201874956931635689650294790221578307301702680987459130644770982500484156397132962891366266973796808741388048776145354",
           "6.39067324283344282612593456066836977441869915362953508565729525367923342752058036792717988019560288813706968456779323116609349360225813396901702110619796078057334552019514797946669652098699860634363540091309264934180808145340898043417996593792792270890934539784162252743214627553825160296688912049057995757459733133778428381528832147374042143201083422723391419017423476330077549613969668943960552183346770788992856348755201584958184758568943497685412776464637452851023132948222137176500877587016169499594616364239121570837276604948806785595382545249190023683014214186698190428219840682236280784558646634466507658233281348215201532077497422563499058676631131322841280086826370841460898715011137512038685864392079526912553523177019272440755510704279946872616455555313590656257840391404433847924673482299505689689769822347893968864046186358674105429643912021090605610906309627700793415194628412350783590691473324467581637253579290246840759262280795786394666829277098636383739586574194509405681992353099095656288666217128758044223382834658781601679058686559406948665993762673723079305381020139309264407402",
           "6.63191504395654222093463225517095044257167358492883535520176025342395774211712657468200830942121109644363941340107615485128085049415855416468332623498613542536495098385995084442339062608583978638604918143347913056760626947758803721255941289939544782980624105947726287394246015934530636662452762740891034120692720860989191872617116876270993379145692068344563800080161568845297279023268580882621139265056001709415926357601143767499984660052292050156852098465716279953503805000122483365981348049448669145674005237424902758331184292667285733831838736385479377968893647208473593030642721808734551034183742049017463013327984017500270415348903044357529006803230836576763451403141885042163557880315832238470993889819388689962997828261446921618788229788339158160824684571810429272556119989956914418082964164895325888955223056217505031535326694543563995771654860377141087298652099371042888622930700365091746883791244230649741108473410207660026523417093735663777713562847430184433298925557785602803469943746593977417910571201562354064143775475359422531818578919533441219169564634992179085747739421396934533170253",
           "6.86468424647826745014070025117717964024823451584566770298700199608686799810550334183271384036184024738713075357138304363500729701192009148330990514337822154917664169841019331985106373480242557119822580196465316734207821863081970176402541887949909079651180860550135778414330866329785285584237686124445664832783599634024854488870218943262246632754953077685614238917113547631797236909830439953169012672877303232625810370072296875077876422022480554261365112845624378648826827015798512264531843127157809664230543124721508342792582642801227649598361094857362199437229919720183995231808663615464777549808840564458629976163064258768399634097749419103019327558046204392219063653339901570969874642832022669018679996962548734797438261313737916279431144129292768130662584134756962916924264725125780137451698441993315631510366262043104603857813311015159104411207656587159892683007990478175317339407414253752547566811439103970532316615414849940518882874013736359501398823560697095317586441184708051608329999378177452455843424936412317738990534790002839870054268627101977849744783551897006721190846394315014179836658",
           "7.08981540362206410919266993336458073119019782448954851285523115941164513836412872549980262695417866166490729449713028266494461146289769041010037483841114827384815079461242049139970069211580330491572838151707185120069855606140591888206680076074093607434178789797965056125568710210362486562167733000362559363045493390990061373467195915746340734563518180466064695504023626957372716533123941938499273960821861940878245300624658987006017095504422443198450842884024148817793468946118645483237729399532667369685538283843648168171114311227477906628002087322274050165358654777861672604286677553328777171742824907546089768216859979219968302594559549935402372256087285611714324493225991578081448459631584914955761298391279405253948506605002517304017862552843870010724998772238168184624304780886956610028083117103239621733160264889227045787864499275497227991534082024585777541674123189429702871674254383899983980905539696881555641586562578917589136538017200856225693373215704702453670534528006815061665390528267710616725134304996130761803394128053676974395892348477519793175492044505024663527553914391150385504545",
           "7.30801444381621929576410414169825272457257572273051004511973439898925887398217614148081823016200949202834301901341123741998769012029715026961550074765354322325117408706019312795809700115432061001422232366558866090103569420172751971374783210350559722862260759626145820691043338967446065368729759698186882375361190232710825561214648603121419802078232806394885971508914082308000592725279784010670779537521209568997392415395010518735400320816396054990550737425086661554419518324971399666539322809263579915158239535250650034449433728950585403021312904469201906868572755498527768200566105436948987083446413896510763908286033307148367944294928994814851182637242369700047960905551015550891703778083820346244133873451184266447103836670847395501001367479270898875772403881434563877248062713559640372882766006167571899610660259996723172304669594755157014583499099899215181509673430723657636479581940652958938443019241663200400679385786760329883423015546482354218805580324551932049528493376425537320109696781135375077474891741199783559527857010245972108769520093587637938249980386334004821761518122012885593787752",
           "7.51988482389300150724729585443313575902096022182981494988977072902688096382352592483978751513417179834802991799561163893459897085438020854287356696789844379574030644137437119341198341293905981640794145070339651469527740698824158833900605083910102747153254346823547028680012412545082587495438367753158559476209718277999227356404721909175713310761624474354590046592779251976853928721629028175578064120608571557774920892429935357615650299892305789477593676441150890402440458692356991420750424326859688267566794493155945645441408095648008938571758805471427497371233893925893138654640500264101675890590472426492701509085255763916845407952512948511109027030674293062273462578534032766972854651758610175461923748476891009535588426370610655790264436812079463314246950145130169825282410419274152444905553413922149504575861096531170020496816501665189089470157595748683377689903086933118316602453205910646274534078020297718628349356730140346988109490595289054093680476853006675328286507166881273403971565101745312606982548876538861884799764527184647967435303690091691151582053208991986612381777714301922567712528",
           "7.72594721818665232400474478573212164113820580488963719814529844615257171391366083837598391841280737832475169393182272541886671862529201489762301217381727573506969107619318907033743713523584964748522235434860386240308259592226625308659565532071953554996921178294803994916620561170523298903408385287659414301910401193614716492481152548641726819012461794117027835912991336708840836650710281453954679059886797070453447940199379600879845764678851272615250256962113900000364647452410778972648375707943133926939195015381201426487954498462993056064744870715359730063675217952027028557542103700913113678148959057734510658771520791956879881016752781141774540974358264943959912543543231872348745595316009779251124673800869204892703157616225827295653493619371958957677652500482852408284671621135986394278188285088465620289846607059129785050607459357597788813170941394726408533348925805369815717709891710786025165701419078226562572328981485933748567540138562641027503508306197019193198804817270177795577482266694943462207325431350513542160949582976461379550638607760760045294712529920901600994240565773585313185966",
           "7.92665459521202202669005753024036606304471951279238958541451612439347743605314331445754837571926806246548112958659070724577311536471294975198952931075034699966919918793884250232805461749423479324904837457089324667304067982764195744035286043093067787996889565306986604628296972508753114309320859400267789675026962497238890042066589975990144179370393781502607410897740628103263943824274864998435887027010214918657158068468943938068760888558966351513808680337871388293293939516817225831825387903008287452552651454178189914856752257554981494022967306190076652997599820594940704997277845097903068570525824332466980258464595176254308245979865165099086414860763567934690442503435576361413412395216127467775452411019392589836257808565460597773216981862679493249539029674390084365618781969773076939757580865513521481935704013963254685436747736179861799668941526147758194350248681797583637701669903788375213685518502773731631995426377144392048964303815782517427738248494010821572980664588536892911565823829885505136555649244837269033240184725561988089167118135264217990084374046308401528369901756263276741873753",
           "8.12240393759049993775056085790344905374064693901884539671441829845338578772903259406144188243219398035169276840513008383162269420554517811059936012081975520344394583550625149140104890709858952675303897227339074601057818074118589698571155136079180679720382723235231538458633009534010518571057778133890494075838487054606092393806209145857916456072372240571169803355429160587649756506023316970860684505132706954885924145348134169632738593180936266392964765757145617588713808060798434336980045883860242149289652316891255507207532033909817967022156020135008789172223827513417867030366597530330645580768779589871599830015247516272684746457878444123615590973536303130500232072733480785298450742941376024222195955073652956648240243919431159576650278660779563988680861574614265500071902828098808112054130736935301171188377689518808421589113550324566036499684989624896195759485888011688106014339156094756792157734665908674538715149602342862415095116193802949560661722409343540501008020488626114518430540516030303410754550714099923563850234789351977675338732716128448507828986158960243138593921163553099984790877",
           "8.31354547584695967940566378712267275288624539467359186240950261388420109063530469298372814586792322543394723656678356561805557643176058539909258806409526952143256265316829288574367192300134911326662579540672482872991356622029261017993364337446668505438648868283461277399259103747660496390168344220787056266691284920945026070198958124599742654165567628853365640939746555309453490999458540074120276825684247685153740702582674467303878126564742370625497947819552642763989933468008133993752611603522750223144711829552595043853200184607806501342069297514482919782863001913353586468996478801683403541333200250999752731449803793607515150980663437341133381305911283161078249341037269510562017916894714417589792091216017912110414165892331827272726552875986295229205099562827308056238370455191631964062828836731217300323551683849308479366652512262425881556198411502851577353998426212961146886854599036602811970862235743112454698389727605325232872049778977783694191972600155385422071256805381015864689257913905946358386236370406339271062529911829230871245337410607604318231151206871615890518289557823950960924210",
           "8.50039005178969681405606694066857144339871874464321576831825225912653747873113563539576206921739603496466768191005963101509608216565231580166435976841183722328004091691395570837807106693569643325702021135074219386315687196397182977310724857959339705673410615655661738050498092907663921364482163693412990352699674010982361328463115845819153103736613515518538534444209396391044515527047534370066166976494484027905402462724778219101349242704419346472667550031765101749789603575074461809215731212887738126264415774177469700439565715044625979289026800701352092929127409916129794527211068956300531815748065200493298365734410099333662178356881545890691882401846675346795963611125059579096807842008824231421616080054119488676338739204001665175181321425845571967990452486774866455322197409200481385043451140315730188098217149173808292210439135828385844496694628703028478095465983540068867270220327383626367373083798876294946716176284955504565982296829544751335108774407823029718852800476212574258039209896227934401044333736396538381519446030490401670705477490565602459130058989097840050168516571928831986962913",
           "8.68321505469921191235617552748954921446530673741455112371454793324441846343554874300139537881738616515735630882070923935302944564162345230558513008887324453213161255770304084085806671584594117832505684103850178581673663430908068527011627361538250583030533917385305060815056023116613524327810286415623197376863742340177925036584573230599476192380889477066406323810984087702033640107924182172686941361028312661313108216417359778002970966067090830603635404946756087016625118404340231456365795980675399247523972342833230042316359524247532024538043069127321959945729813306883450632483762749433800961196553281680789399437969279299794116549334300305897032285602148520076217180802664897460722376258717305407854893719421242134915981236756035407783461448793332599639580144795377945372918861006053713723472138436130841056694213733945161830059507242510609175123971434615259865080821278420604965765695747454031603269803511665039681088193517954588462094071113509985794023777506435906160395075220897453305493984243370136514880523527929961406848242166117211869964562420108871548138785581293418809932434011104394688986",
           "8.86226925452758013649083741670572591398774728061193564106903894926455642295516090687475328369272332708113411812141285333118076432862211301262546854801393534231018849326552561424962586514475413114466047689633981400087319507675739860258350095092617009292723487247456320156960887762953108202709666250453199203806866738737576716833994894682925918204397725582580869380029533696715895666404927423124092451027327426097806625780823733757521369380528053998063553605030186022241836182648306854047161749415834212106922854804560210213892889034347383285002609152842562706698318472327090755358346941660971464678531134432612210271074974024960378243199437419252965320109107014642905616532489472601810574539481143694701622989099256567435633256253146630022328191054837513406248465297710230780380976108695762535103896379049527166450331111533807234830624094371534989417602530732221927092653986787128589592817979874979976131924621101944551983203223646986420672521501070282116716519630878067088168160008518827081738160334638270906417881245163452254242660067096217994865435596899741469365055631280829409442392988937981880682",
           "9.03777677270990269726949426172862835196387072970673604230446563698972563128792797577778210699007499500445574250898762044933778750351774122552995617564832932129894429850247153326323323596747185514513829114667599621242841895828094634599104752390660626316359597361160805506977060785864672892966797271805786612860244448725471833865343085096665002696889160030343332226214610521511200144375391579536397373156980251473081426535067176615086187889982851260244145348339114794114481076655003899438879933457078775812975837503314934547103980296764470525861438620896571785240248887907778823513254337650341006992728965834432443948838718849505116031704247005966435163013464172640987633315418180257980416126510536269126886862377598402609093638096342316879109441931256004415158100347181648434500606990925357893191947800032622820087068319734681796146574385134374500536260087114364120328174363307505811192446227573869975931538764994397186624871438876822552201130556306754697296734741569342977980309931897617697257730057945805570021974106384510091603635687409774141632668637120038927426728226929784072977105651339473101367",
           "9.20994037151839639631596462400338374636638793975877036082762359018286322506685344822885724499402625282278112227347785708110362882152982486492539276330024765085631629964295929748378249085804119808093549931994279721321289923550767047849571217878792843390809466346867991510868694156145415945082688797967786343353435552186659150661317606650543212335636343686331503215617274398559652422342499978550950096625965862569926163402495930664616089151511194642214415828373759355543462916433914010177922623548395072329545357529357267898707024956601854449660079838932334032102025937480901959732494297289761019832533135155565554257149646730285365649475508867137837769824225981627121384932711584555741221474225014838680519543027230604086666177022494683835532058874140274444425268608984020801682462141544862724398017210164256350727408755296407972623056472273132345914659683425221078106786482833330622821808530378022419276603327475231550015699876669280732914397733573148057766703638751272663518646580177302812435397453534475442185332367460952554021481188695967156473750726234595699444337049400460695213112082073164218031",
           "9.37894419966950281444497699123566005692831626416482139632679675169065135994891966410056710379531185296997453115175739369207792866788913471025384584831759220067588685003261465869992419957481493970329156039794423572600489417423832178398520116622168531662587804563582591692339405679461818702500749407972729016331080470667010562134768478583188647079633628847731165033154000189887711845625882537683194893144001697291405120176988145015896091608729928306935261469019327442052101525774632932662882899190169943725336986511866531222451259355191551595932544893780095062627088118307374724034603309074230566321150596513714223540169605245519015025496827736989356586049475153155269118025874004811129386475129565322957001150834914097729540205532774277913604445897974904172097094197509573297586047245883925138372081888018562975908164098671652826540939370006401925797469800744090363913967117006512530422783187442587682838761306012830815412290263401144388335110015291263855233602315586489580934166304206346481887647538275213089289641095269061282993015859566679470384166281828144544679798214034013298421760057481691571217",
           "9.54495610016641218053318050888027529952710202531153199575774210745792169686775867122286336893013980770883110393522762476818358876088129524710547725582400752350379006684967820956995284778747313445279111803934019554639715103722176378053227974233654201464826899178346960385701319012743023002834221825212678330062853593931422954382790959273783367681208829081125750296573583945273785727616528225035230730367043489573228646448365807546311686441424497747445818035366984113867781703025697773957490231025374107987159920796065423503962296590063470579872505263841733246548152591941651333013912614638133883335123920373070975004252305213546254181990284206802141740422901313199964786740911915995115427439794642825914605300283815175742959582878011011823566253034650282755730842497543625168601556462908567644285914941630229285900004349592874758990965409149767611455899099763319215107166406191621967564685710927764812630422088568711352107239482256315420275932243379732104022438119314824626037473517483829358567542790510299355289069031782291584296420055747908391149240325877115257851162524812619999740512151814664737655",
           "9.70812956277849627020013008432023090142391629970752953381907255677408677683608283777972133699852226057384221251532058096143107203837634058545789370004650021408813594346758795590127624936380909477521162346782666694927945494282229717053636144889370566148233104188061190841182087686652430168640066046228725465172358875908141846376474355637636250095973439621439980693326786799959495956257525100473767694685408934461084692899974993028686930219291782447513786152395692604500383372485738542065647909407318214440351068221656470065118040045018410800167501279347597149519022186563837419966281229875188530501814945342053480368694378460223735611289594593646472862387163956627509412278526670700766402504793826839677765281076427340901353768173748184477638624215647772178191872811543334258915265427131978394362810703976093768885428630849932102391562462424663355002600929871388002261215265969294828417306845017134019844876207436524153517810279329274165301203983428519852972984397790790147707779427760025639355001932966608597649497333189984495365421369654196836993842047830874262598283161232047618963368717192374581237",
           "9.86860538583256911820721127887475338860580572014113804764764567126280797502596779665406841324694419448453464475175038076268453561722477590781196193451866802150809759490712608888093911291954287843130175265867101964469755453143964084226755674039351003953563412948117884754289084798586264882426396661496381950383124624978217697985543751868445611832757526512148116901823991753587937386414616186527759915861182575983960867544511432164162055576224548617161606250500799648602173880407153261952724142638011741327947253437647196929757560958075420545787575757761080599830662321444781770561880447820833072140574469797429581558718245198502380676502327269925525873736136551448095090807729668471638614047172598418830416390391932045518785962567615347617805861430300764479148941742520703366317243140861995853488523921623877514753981220597859917397820846973107706661150411207144792858833362409021546109104485305007696344300462841655088641041139133938632803072811438751371559900802700904742289494188612587210415338399019832109459618353269954649298194490921810882187710461971651611477483705188781040460993037832290727754",
           "10.0265130985240020096630611392441810120279469624397532665196943053691746184313678997863833535122290644640398906608155185794652944725069447238314226238645917276537419218324949245493112172520797552105886009378620195937032093176554511186747344521347032953767246243139603824001655006011011666058449033754474596827962437066563647520629587890095108101549929913945339545703900263580523829550537090077075216081142874369989452323991381015420039985640771930345823525486785386992061158980932189433389910247958435675572599087459419392187746086401191809567840729523666316164519190119085153952066701880223452078729656865693534544700768522246054393668393134814536937423239074969795010471204368929713953567814690061589833130252134604745123516081420772035258241610595108566260019350689310037654722569886992654073788522953267276781479537489336066242200222025211929354346099824450358653744924415775546993760788086169937877069373029150446580897352046265081265412705207212490730247067556710438200955584169787196208680232708347597673183538514917973301936957953062324707158678892153544273761198931548317570361906923009028337",
           "10.1819721846243116158211195253926465666143347779815073047979911288728454119358550456874471060202706503091805338588626998024526166707650425773136658258074400903070578475129191134262714581830920726995815514653824789384673555040322558822640155690142138250981661144460598431698331929864441089428405069965236237150722105664811612049206741649626761309984600129580251270318577455853101292940548425839036066861121228690420716119923444705834233799961792271274742610225908970795751081615038981535248648458311885095799828145600663740483135495656806225462943772037314913924014868167062651948437081797920497967655996219852832742745294377907352391106793565373082008386931837077219804533025205245429016269305924868300059131465990367513110617768155977075582917392828200481639719171646996937843435515414397834073681939587838302796801187221313907962667869992989319909211964361393188990384040622902615977028693735275429763987359474700539603288721055646644016768647749162193038655134281042688011723482785471924105498640148127126652745316007086053077591814555887669601195725983108276147651242913586147914229810924624624361",
           "10.3350931404633683691947436472707444709708693615480182604123950370334349544443021679231379500921649218800863593926760528756661011078126321844492668291462478393476503125536591219070509735210428947761875792828673301887541398513337078226608138436426593790404989932058023039769630384800611394806279966525193241106630459881453504032664228967399745528185257298594581458987077801324925958107100694837096169968690367946833033746754352596262639045628825434791015203759110279895685375377254959267384242428992179590079538654788434703331515082118259553412887391533698850248031662366010669929529784194795489078820235616519309873494295540734349932360576182477685064907138632368679604675882383935711506161826635113153997912713265877050477273997328780264216580134895532814312026671290109568535316587756940503116601114428904476514966934012890148992229020985082631236739332830184773141236150086303025054599442020885304151228567409725486746114685532611428843970696553723054870710919524255238928541738356450696516302034137957025179252274911492305284875580251537498242752152958407117066303014561567387290966125697410459079",
           "10.4859783938191844505415935988555812169983910386669556187311537807971601489415184480294324534243750962935064587431065554132339655732601450466677346771766404683525586939467074949776376839516117900277707447519492965025773007813570350841194233890993183787851607015508682597768217867560501711087754801339252398186169563368609851974003387177803015256072875765438214167126280799321733391301521412815249368328101551446918784488181577223454487520647934057527710710187315407383842241034812304414618212426764289725514443697625658320934830119552552349454613146318727307440539214183275364282577270889732517818803743197835075972826771689559984211698450313448053374104683526842305753510193148049529361498811259623906343956382898276922773672099874559380274456904526382655867821184634279087424431765532414870700646132525635436192820560367779661664927428988733468931949806424994253052465407314901679252685856376776344476864227019344231976962348967817311235507614203388405955265951309377513783090550287722753128747261754207014515315881341170594023794216162541770211605362973081229737656368414853828067103315720270750873",
           "10.6347231054330961637890049000468710967852967367343227692828467391174677075461930882497039404312679924973609417456954239974169171943465356151505622576167224107722261919186307370995510381737049573735925722756077768010478340921088783231002011411114041115126818469694758418835306531554372984325159950054383904456824008648509206020079387361951110184527727069909704325603544043605907479968591290774891094123279291131736795093698848050902564325663366479767626432603622322669020341917796822485659409929900105452830742576547225225667146684121685994200313098341107524803798216679250890643001632999316575761423736131913465232528996882995245389183932490310355838413092841757148673983898736712217268944737737243364194758691910788092275990750377595602679382926580501608749815835725227693645717133043491504212467565485943259974039733384056868179674891324584198730112303687866631251118478414455430751138157584997597135830954532233346237984386837638370480702580128433854005982355705368050580179201022259249808579240156592508770145749419614270509119208051546159383852271627968976323806675753699529133087158672557825682",
           "10.7814158709708600973470860025697540134833131069132002021875135200480437122785122611113481677522707878078270440052147356092727695527920302171183192478655274869472668345393510398722323555179529207792958553208776624344340893028683838808544904189559530608618288330229561660785494464076778823090275262667649435818366046674470097351430635081878043122218079020162033184280808128488592626175069016310564044843363282384351603178223984429085151346774445891706273636784622362866483557972819012892787057761108736681050757369877123799164633273904139629258646240624506200184033024483531204256854941841214015611888414080581230610166978791113905519323893487804024647095097676311979368148451812686908830353037070405782315727089541764339584093637958422138796375184575441595432133925090181949043536072178753049486872204759334059020258970699357154378751215836144836352430031659044940188023466979385706234766354871873384990411026120082971281373943799958524209806200235642459562870710282507653952117525804197309040193746636407624935115304309270078857483676519373602411048897743124249850703751765985347181558029527375643215",
           "10.9261393381382494183949485187745355142392643001293200037804399336898503279246222062168613731063277954463829865410973870577192080366472062964040509440033278743615823412810808078908898807870613199997974098405643999383690286991545964650196935814017029950722564391559018285106664794338502744786047104831103687602494259961816398342935032870065474763913436754495569767978627808003615457573685732588153239217166014299098858554629191013913370517654023841228586351241202877223367260494855558385058038046339898004510183855157527283289679410125378588229499416948431610345648489884658061910043475079418189307489137285545541133794341838958556245644032832230955632382851912531436083565766604594268028883405369333102861811110938475059430428576302304134018133552423994855168237965050769662481212059909922125002130317511819126236377187306085364203705624983640591322488295607424619160166746461204981597459574970519980426620414393119798825541827904061106147335114790118012457207318406183818422346913078125642646008877987007918624713747892824771389677121777686596898164466209882065864762628266271132718624356253452793196",
           "11.0689707511584805117811434659236005479075122572654030261378132541766399464801053452268915436339505264869099960261186473580786841783940799119071862844203935756997737196827013181251824477181865522696656125604284386736805412679099995442756521262882968085901005737598504818120062960533623452879232321601862728439436565082413065419068437974996041360159534876356676912365960559399896203071153029799663110896121451617844812242705440244059051070332217484066565235938948140400311999208370258263622580614733135218175221722605324220534894428960790855444760163600390939087544087363052520657480331091788922029648714976820617337239621908921583216442263332160508780261589523500129489988435510612958327244523134689294673236630152564918827636287844901928132768223334355784295928691382783374669003085509906226346278070859886444582772115212199422165986886962139084337282264595688658520928296714120909731973712859435413818042451694188522026870751629872095477433593107333415373943667221266057108367464732484461448541683512088757459353259534085109191456436195932802460979824143610395941363551604400219577421540555259221554",
           "11.2099824327958573986225648677376017877086475504215665299167067873045667553037759486727686750491707478834293411489125980911084151079663035133289893355490903237225455297933242623956756700765850165559519010287574088187786256136071791860343087283966672952333022508572112324619331645107711457400199461132270526458620965541207404627199559179879344651656741731856844106860800145662343482669094570758579364544899824573389424347400188820501898154253839556716834069987025577499729423912482339180176769424901522756286560845978811688935258106807347694868506072185526420367236029189162790328650177137575228819652438270486363627393827506772717574627945020518721350959544754722118727376733041854351927926293076323790604362596413400569959286648197872870638472594843231022845264486068134343862498444673897526209611533151572239019385139110221569524392708880223111039983515560401743550284647457807334695086237272708711276174560785696543477439261496094936993988602499825752049022699262521970903438062340959457966112130335715811951217156312056475059606961348236080146557716984203901504347783723320675301673124545417430186",
           "11.3492422124642983926102856582985015285395067466832927043380356548562390804853608284602050743042816031793047600978062552284631473271698301080420930678002935270860213755533376410950434831683129266708417862791056849585318715091316511224772804579306952692391203428022706911552905897215022011255923056920968598357445951686767158007175858503283841137547847702169068881757041416256682026875119665861386894957609106355261396135163282225259885976498039746087649186814655905823222925759171803884806349350968882196634773164961078791229532476208824156202390581863307496926197095334795614010135645721792945689410813947143894261828309379674182914788493173415091362063280686801906811659162851572848118095721313328315435166989609801810101070813864563570801067282282192253552931615123833493584040857696592187534301264584146553957310718712255292573453576205261351861700915480639583301267167270018138941290820062263217901164511455735107338870881079416882936221444752202024321144178837275657426737380444900404799010786716522164241960362132578101367581203971478969200704781556667157536062389987737624943537002727294733986",
           "11.4868138076131153078093551672152110823792896720438206208589815428358866831391271502916151740432833498923733353034882653574261402114885623063517466286983135609311347428616058487641764580040391132399123735354188924995726084316672645564644522404101010145113496826905661533851564869311026284567807835770793084267376582965078917724760751368098985709220493218989081195410084107253702771346532326060479533778255009456352092653858186317227040032545274297836762746748179412232929420302325855877594918256439049163258491252026402004889017462470314144712637559338394573924897045814379676299403438700340396558709369818377199922811836650582080165464955651223303653264959752987238908036388800978603698543078745746928727371682109974731915482666302457811317513367974821492659153756397368790555646602353958513206947086398520237613898204615597814371058763275824648368925762130837683909032578524176198944131412130278847693768941107158139354113448426375117837953121182384305427053092552066260458822741364621133976801759853058864467919702658753108948232787444346415009083332605651415456404729714544798893173653905698148137",
           "11.6227571644752653007277348109508691363662508507756756751036701167061120453681652358829612013449512186720692331837632298664953191556594884903747923711091159798037951823211201052540116242903833376452779227088949634320335447650281101165924512030256613650508677820241405437704762762206782047959767675141661427645411489541320052642562158115270383536284327256491550000081996990946292465147038958140512483013436893674439652156220554266153922101105181022227884639488325310071375913459006514297537440977563220662148148970327960819157453757205132071390336936162228686072898193957868554143074268366176486204705456695741921244062641977969446800006474326691825663289023526630549872030867738441378997270522928994756399267446297804673669222384728768703789276705361232508520458717382592703150193394825544156320863346704391789739396085519417964187584947528508961792799045137827190950383493042007176116232841260864577432961637844272929218361324891683925506702245396833672138534821860627603036634022415963773401638269124800251684954328516467828750283529560003793822429188759689400075614871516086231035596052689743737025",
           "11.7571287633482564701088417557941419439968698420773487503462842143146491163563371049337556035298522469859900749767294905204293522292298717243582785099170281163259336541392158276782668075147925622379654122962653351040016712959881553357605497106466470404805444939639812163187639434573134191280457860931724877949789107337574297713157520257772779962308829313235825010872269098259933266628859309692716030518264040729497076680529713079313207178190774506108797460140175583332304192519082133325489174600742357544490571971313852098894475025787306758030943007946889416870903157547612470086523779487801936864170079119042775197049616392224777035884133632044487020869747634558757479347496253003316077397398620232134764330013675436470146051001150951520936702104871579372375941323552263581652132668534769659706010116638397183935078175506596204447625920325403258326466185355961926258614112075160888891191719281518991171695016370238399249053764574155272331577493400906266452679122258224017845896213757845939003637544737204862728065115192407085665447911415853799165098806733280347311674415363212540753763695044873288556",
           "11.8899818928180330400350862953605490945670792691885843781217741865902161540797149716863225635789020936982216943798860608686596730470694246279842939661255204995037987819082637534920819262413521898735725618563398700095610197414629361605292906463960168199533434796047990694244545876312967146398128910040168451254044374585833506309988496398521626905559067225391111634661094215489591573641229749765383054051532237798573710270341590710314133283844952727071302050680708243994090927522583874773808185451243117882897718126728487228512838633247224103445095928511497949639973089241105749591676764685460285578873649870047038769689276438146236896979774764862962229114535190203566375515336454212011859282419120166317861652908888475438671284819089665982547279401923987430854451158512654842817295465961540963637129827028222290355602094488202815512160426979269950341228922163729152537302269637545655250485568256282052827775416059744799313956571658807344645572367377614160737274101623235947099688280533936734873574482825770483347386725590354986027708834298213375067720289632698512656106946260229255485263439491511281063",
           "12.0213668967023250475121483705231298685008638313363655540949682389613264866896132906306098642187026618532213215211701448267391188368047667853571572694775536356753264174403972755252732243831827044176139209024924394250581368618462046261557956041495153611139167826243672443648055383775943303041688490663671373136155053510360540267843423005332353653056262845689595224998209096627100780784418138991680078849703487894125005007545215782339870314429009558182449825490593249314052067759681658697789673578678949212809806351598964422253804794344853941490798950989758297119239095904492398955585999875857616109880042018712293986158989479506589338238909339482879806698524858243147005443655293460070165320341112926446082098108538465624611435423462989443724332948895564374322613862644840100579844319808606840135928070026153435302458230552287194862404115544729553555860135767866296645359954013075856567898559441124011177102556776336026301858215217613058011231118563314739369153408486077998475470304773205610152830759374129202640779824926230644346880253659061205608241669503029250691241581669322113673069664040300361434",
           "12.1513313969589473215460257145568651201497518877897922428050262513288347242121973389754626809925134955645428561432377800707082867615886711829398230527046072149152117757833141185434477765869902183850205483276767851442668275870416792912369988091698336520343468857746751071883637173754271209585483303314697136442637145609733795560874016392837967424129624669796698075496202528858880383635774883501185883268804218639948489991597161620364526639036982352505612319544441359778240876504358105343789015081880171993801331134760442658432233574812308195804501595367608510691945957205786397773541197798945975331217200866768614798768369352312293373100870145040568859670724160372893783528980570376945737242122861351956826347861801637766043000626205823928280244639867984099618586640713739887577174825475613963518050410729443551908819114265689677577478145683280934590715455378138742260694148005560374430740814239204990726910962840148903954657341678891978442315098243895522200158153790667123362932646639672146391300060334021960329885543548050873714607049607780426796959929658969497274971918388832149882433776112378434610",
           "12.2799204953578618617546194986711783284885172530116938144368314535771509667558045976384763266587016704303748296979704761081381717620397664865671903510669968678084217328572790633117099878107215974412473324265903962842838656473435606379942829050505712452107928846249065534782492554152722126011025173062371512447124740291554553421509014220072428311418179158177533762082303253141286989645666663806793346216795448342656821786999457421948811886868159285628588777116501247405795055524521868023723016473119342977272714337247635719827603327546913926621343978524311204280270124997453594630999239638634802644337751354075407234286619564038048753263401182285045035976563464216949517991028211274098829529896668645157402605736964080544888823602999291178070941183218703259256702481197869440224328285539315029919735628021900846763654500706187729683074196303084312788621291123362810414238197711110749709574470717069655903547110330030873335426650222570764388586364476419741035560485166836355135819544023640374991386327137930058958044315661460340536197491826128954196500096831279426592578273253394759361748277609755229071",
           "12.4071769563386121910871723833880162795828461928567098974966545289703789921372252696246545971698126579135877653699779946636530700600709582176756559672195094792342638905717358599494762112026557836025246676548757396012224731074603580436169013312966381300981288214643884821974524286813435148379353275063447888532961343423260740356759285255609628548615681581561321713204134717540225393296689839237372943143825839653692927609315322726052991713273927559728897504704226043113857065570762959566602644918216789694969199672638429429945004464808633659900365281397958778937764586125792705750168571832536005054994358820565709437950496363494452954047921238695415144815274982050006786314548526164253480435527360117258227218473895919440988655875440528203125946747677251876874785141679432309253336655217406754914545493066933803303046355614733012876287373212014898518464354302511069792971558150198002542994517182497196658469446954272237277648451310578098894153010149839496340312748322929392343542401192635791443342446849357926898503374322883315593972409393470519281160983565963805711107788379316117321935018451317463295",
           "12.5331413731550025120788264240552262650349337030496915831496178817114682730392098747329791918902863305800498633260193982243316180906336809047892782798307396595671774022906186556866390215650996940132357511723275244921290116470693138983434180651683791192209057803924504780002068757513764582573061292193093246034953046333204559400786984862618885126937412392431674432129875329475654786938171362596344020101428592962486815404989226269275049982050964912932279406858481733740076448726165236791737387809948044594465748859324274240234682608001489761959800911904582895205648987648856442440083377350279315098412071082116918180875960652807567992085491418518171171779048843712243763089005461162142441959768362576987291412815168255931404395101775965044072802013243885707825024188361637547068403212358740817592235653691584095976849421861670082802750277531514911692932624780562948317181155519719433742200985107712422346336716286438058226121690057831351581765881509015613412808834445888047751194480212233995260850290885434497091479423143647466627421197441327905883948348615191930342201498664435396962952383653761285421",
           "12.6578523191369022687748451647920884211351618089655179455231284731368243638831930092692097344570437079901479457687357718509226590879392775115842265975597049724341563061418697031268607830263630749575108233222183077768581512869483056352211926808706010251598533367951222352547093385705143351301673698003798847470899037622815846893505998436154809431208770604783851594929676185652397778605631719747005946930957513095696933347983859316339644080864356270066190757023433598780191812725848826161342208170182063208964958296027351135838121377963924650053695239000481943588371643194130257120532938135164021072988780496867855135805902766925269096358874433009794964404505124093744345096904976475426376615294176138495927979455477696384319750513407338903997493563537351640190122018222196334825106936172049588584462292486129117684021653955548443028996098787136328122645959968477838803895341464736220878601918003630919411584046348014738162733741711244046308733521756823739827300238391611818552174369697043280135230378038036669764857912765848567411839492389221128518450325850349211951118365174902595493189143091128924607",
           "12.7813464856668856522518691213367395488373983072590701713145905073584668550411607358543597603912057762741393691355864623321869872045162679380340422123959215611466910403902959589333930419739972126872708018261852986836161629068179608683599318758558454178186907956832450548642925510765032059337782409811599151491946626755685676305766429474808428640216684544678283803484695266015509922793933788792110436669354157798571269751040316991636951713788699537082555292927490570204626589644427435300175517403233899918923272847824314167455320989761357119076509049838004736602842837339638085643968136447256156911729326893301531646656269643040306415499484512699811735326226264568256017365274168292179743002227502407737172878415905382510704635403854488151102140855989374523291111062718131251568078280886769584934696459901137937953964469578793772809237271734821085928782404218121122181261925540158683038925682470156718138294664893516327450715858049368151852456159157278933365855419727276747917314838901881136398470619819131257733243425751608844676566931756320335811737311881389733198752534744615861076204027861852881480",
           "12.9036588082705845248371543632003751697456908407741692410784368848786088725089065725375873804520087666996942746345726834840639975755506362717246534706442221166298621207504273622834658505486588760226060303402875348318760592759886407055350301443543632102422510097863033684320708847504885397025676888805608608575632588352908208382177260510976770708247524111302977184221988730373831487689478022518795416243743368791870779402875990432576302206741559354874008433704301896418503322037242979413682693250947191295393881165307605408058206874512520008288998516843589530555818279993680055875466221805842339665434759158713770816962051147931208116808824595295514375045899970471981704078070597035475020898965145851209954427460157152268663408082085125094326139800815655980081696754509782998928240270234248524845103756193355528876937225771208802709529190373396585475974586241329821820837801569449366833520227139445910446436020810058939318594350345270816751238237726135838608689569520523122450244404454913108932211269247818789816060744177821446724166244434311897023559474757455995497570838056986157112960941696803729504",
           "13.0248225820488178685342632912343238216979601061218266855718218998666276951533231145020930682260792477360344632310638590295441684624351784583776951333098667981974188365545612612871000737689117674875852615577526787251049514636210279051744104230737587454580087607795759122258403467492028649171542962343479606529561351026688755487685984589921428857133421559960948571647613155305046016188627325903041204154246899196966232462603966700445644910063624590545310742013413052493767760651034718454869397101309887128595851424984506347453928637129803680706460369098293991859471996032517594872564412415070144179482992252118409915695391894969117482400145045884554842840322278011432577120399734619108356438807595811178234057913186320237397185513405311167519217318999889945937021719306691805937829150908057058520820765419626158504132060091774274508926086376591376268595715192288979762123191763090744864854362118104740490470526749755952163229027693188269314110667026497869103566625965385924059261283134617995824097636505520477232078529189494211027236324917581780494684363016330732220817837194012821489865101665659203348",
           "13.1448695675323697034354136539313425180677406990253396231413553532158284729201772491011000710677650219054903693738656249197132353617817043139899355573740668204032909567434296038746304150062741309218977952141761472466065644454165057415693463642194136073230607947203262919479440729879071393287017368357538247791027372780044529165958694345741149609916405379143996738502906726395310549057499135876771945647726253389292849661459433071570826833703064597528372476669325238379807256995707509496936873280377426680968490544204144850652418667102858872780887677263338414336218554299503848227359603371483052714037382060524446989585125868194539667288699342590529933341716843693870456228542919094333819496744852442099842703290379050058483746581999303382422470703068869484513287507363016362925163918396245228476769779932001778978178333151597445957345163009522871315976591629005355442432626542153837481717190476686292454901373614541009740405739756345322959707818426567664529141400786956674079108340389783956916039981285613130422619726646882192158229279130067633381058950064006933912115038270746168224845260024281286513",
           "13.2638300879130844418692645103419008851433471698576707104035205068479154842342531493640166188424221928872788268021523097025617009883171083293666524699722708507299019677199016888467812521716795727720983628669582611352125389551760744251188257987908956596124821189545257478849203186906127332490552548178206824138544172197838374523423375254198675829138413668912760016032313769059455804653716176524227853011200341883185271520228753499996932010458410031370419693143255990700761000024496673196269609889733829134801047484980551666236858533457146766367747277095875593778729441694718606128544361746910206836748409803492602665596803500054083069780608871505801360646167315352690280628377008432711576063166447694198777963877737992599565652289384323757645957667831632164936914362085854511223997991382883616592832979065177791044611243501006307065338908712799154330972075428217459730419874208577724586140073018349376758248846129948221694682041532005304683418747132755542712569486036886659785111557120560693988749318795483582114240312470812828755095071884506363715783906688243833912926998435817149547884279386906634051",
           "13.3817331184947121778387694156683631366934609178535869224734646839530858830882778845431320165557527082867167385741007241233972349668341335799303120558949874914731632596672292536486840252369343075578062661394846729290886120806335230931470676125584836958188275362750631959355148413735909208659653125785489545299880650571571989789816736276472548899331360413686815340230072317761139617108693201596677053371411718241422778714572140731264458353536124425397143067303995240635567710414918725517399711583125030933885900593041624549493909270279061801044266813541511476059360070953132414715897099793064187618904913446179695444823627572975718616435530412231897626821211591630120456854888471296525545355847536852618755937846705639827645780754110681069739162703086029559215644121294709889120500305321671981859220528418810561091642459738043327836329007402390418543140043048855442197107222952090170835732458531444797752568765756885481280298531950424829111511736236703060978513393719587267334516212509718247535160103206224947196863863264871876290808068433102976710291998849009357718081807090905471789659277124839575212",
           "13.4986063691111464245935803057163445333446588880954505218387843221811866477814509702726201994868384090085699517429739877130502525078353243003832198065078057206125867558943180721358516952593492926100016791442634700742497189485375590347895515367419024810678201127613962063650779588258427129303547853247317558007027886497918350521735648476962160113303483169507138322563300579235728423638552959320773585796896051727660519635215799533389594349417041464679040078438099479890297375500511781991113346083206493978277223781060900609203238286306467871141396991230768088662632124270257464480586088316850690383023345309392492732989705825016919526022563362954082266181904414071139631058950012713540527049102477938879051900769524120919579312018938311603558253092272074870766079137716555254983943200583248719785173513772917080626726235739995775731701263869010544645677727292023394850886826745691463821983518385802531448450511994014078687762847948523441279737079750421003933691499255395954393282954286628119050279442723264746413216878905804503870657461514746916596067609999603045774298320322180133249625717533859092149",
           "13.6144763601762444379926521651240108493489429948062675392899749011656367981822455178853834668853591318227100171777478230478862293910435061059034482102617694499353623610929608083252219728860071814777496237554962825789860436664713007380885823257620705055571016537859083323332155283573726935538232458058826315899029704394102678491521465094178345832292296568693637792747393915301530583743993444025896838065118897343043328452916631236017814449542928117422626902025859794220329730506629124433815526321617718341415931415683837830843770074490481176348422362162861812661346849873918814224705384024355691424864367748319171681413896659635516678441852608440834144162761248958567982479001381585906063881204351628751599665209292094281446522809539191841105420891950899912208472122984294731662074195162813392306707005283151329632622622723286645815527491117005404146816741085203466335075802165744347567892995323123529815454414242781188328567365653543913086263285794736485546877191137857066664712691746005042143722635223400766012268118857311779477233465292274708583508374133748593182509627918787117455241091542819813359",
           "13.7293684929565349002814005023543592804964690316913354059740039921737359962110066836654276807236804947742615071427660872700145940238401829666198102867564430983532833968203866397021274696048511423964516039293063346841564372616394035280508377589981815930236172110027155682866173265957057116847537224889132966556719926804970897774043788652449326550990615537122847783422709526359447381966087990633802534575460646525162074014459375015575284404496110852273022569124875729765365403159702452906368625431561932846108624944301668558516528560245529919672218971472439887445983944036799046361732723092955509961768112891725995232612851753679926819549883820603865511609240878443812730667980314193974928566404533803735999392509746959487652262747583255886228825858553626132516826951392583384852945025156027490339688398663126302073252408620920771562662203031820882241531317431978536601598095635063467881482850750509513362287820794106463323082969988103776574802747271900279764712139419063517288236941610321665999875635490491168684987282463547798106958000567974010853725420395569948956710379401344238169278863002835967332",
           "13.8433071145943080802284382623942725643459634316370295832402490811584690521837914916840395377602432304386346321516022467700817063042586192089471811215559491096430603765490105342479464734028031685014017988597008866099563434747509151656359907403458022227708855288944495192855866834164923404580317751114810233440974813740088485452355904551218828581183511828392819322800423143611251652965268254587551056785058961962030307952250857929384076417852708688910729480334385710261322295627380646587526400594088017012104793668923859704481465512725231143313046840031532848517753228102569245860823651766421422437423742518931925679240338180398466042655884199628267998290371715502966743259677385997995351151244386978277483823663420757491844034607904266068469818830496093618976553620949696221798505339106720245367168877405376645269053274617909080172273096809275302452571198400347118006550023013802032386296495237116432503947861118574507348019315687686050292633217476405553408632695073744388341821217781854190838168727859124017786167368413748894627539001773837225674001211236781270826204317868935713403376698315709386501",
           "13.9563155783525897163028298734019451091984537120228258009450472188267318438067135063202368534135430885083946298549306330756477989556178368665622436754180093476324880937959506547854538083370019416256575776948316707917842310903894619857591393548227911657101468449311608259623964684665016119377046647048049581386101096936670413063222259162339202191273628263372563875815041957309547200658623166127371954284217563921132238888463255849732942787630189806285692737078454631222616459782467155377806421107402656800251572053070998411099760060179547316216902478018563990476197410484317348357273331772258891407220597320756586444959820203436442764666649896682610015685305787123195940452350461973507211933252869415122355860564522872963830841829314900168567809701229312807570848373079796018588703327116777837136272491150569217486925114594308036945621245791848046986349492932929571328689798094081442393417715210813349185572333277619344160965594558435011430506853671006276522611925991500789685436288350432338677840599188216306552667639876278614865474090999682755277088033035543967117665417559338997296158961230538684255",
           "14.0684162995042542216674654868534900853924743962472320944901951275359770399233794961508506556929677794549617967276360905381168930018337020653807687724763883010138302750489219880498862993622224095549373405969163535890073412613574826759778017493325279749388170684537388753850910851919272805375112411195909352449662070600051584320215271787478297061945044327159674754973100028483156776843882380652479233971600254593710768026548221752384413741309489246040289220352899116307815228866194939899432434878525491558800547976779979683367688903278732739389881734067014259394063217746106208605190496361134584948172589477057133531025440786827852253824524160548403487907421272973290367703881100721669407971269434798443550172625237114659431030829916141687040666884696235625814564129626435994637907086882588770755812283202784446386224614800747923981140905500960288869620470111613554587095067550976879563417478116388152425814195901924622311843539510171658250266502293689578285040347337973437140124945630951972283147130741281963393446164290359192448952378935001920557624942274221681701969732105101994763264008622253735683",
           "14.1796308072441282183853398667291614623803956489790970257104623188232902767282574509996052539083573233298145889942605653298892229257953808202007496768222965476963015892248409827994013842316066098314567630341437024013971121228118377641336015214818721486835757959593011225113742042072497312433546600072511872609098678198012274693439183149268146912703636093212939100804725391474543306624788387699854792164372388175649060124931797401203419100884488639690168576804829763558693789223729096647545879906533473937107656768729633634222862245495581325600417464454810033071730955572334520857335510665755434348564981509217953643371995843993660518911909987080474451217457122342864898645198315616289691926316982991152259678255881050789701321000503460803572510568774002144999754447633636924860956177391322005616623420647924346632052977845409157572899855099445598306816404917155508334824637885940574334850876779996796181107939376311128317312515783517827307603440171245138674643140940490734106905601363012333078105653542123345026860999226152360678825610735394879178469695503958635098408901004932705510782878230077100909",
           "14.2899797929645988880465370254439290463492686909646365784756651098966688981604710380521849261282420820080718993444184921140050946072295733777832552069831871497137709875484943005092844538329601721866028257517525127400112068229653107318786632214087321828354194286744189213884282476820178532929026410946336549250316411573632399544976436381304928255592757344082191936540818455775028244579956105065883479913799108021321113869783952904182528280242051379454218665561448727584030626615636178822706926814162499131369332023155252856087861647757703882420420084149242345266216149260241941567215880661724399293386262636274265942504563029111804888220594251821006906781849767633589028769011495517300679816156698760917682079890665846007154923427386849193920295060178471954901369186035029069580960675884569751521212531321426347035716993287569668349433572526231363707210075671937119893845403234179825305216364116472043430327447622673443562491607514186170086950732726413599236608453920182597620790203604341499252603323429275866821091530657037950087409657380807571921060480257877831704466578263052537696771142883837667113",
           "14.3994831552013127711282056937351191001504093681728491932833832593979434994003430971466998201017992112805329010979828688660650916169722207860723532331591812462512916668982966008728885975214700393366748922112781560669442805260850747279232150993153692443149737581771637951619052246592314537389589547596925358885851353137852240888588511143973759301432800806312328067688373837578111007355067907342413561415117098106229593114120795475094598890952936412656420669510625019084604237747484641036400610081271857751008108979431957256358635365498727651936976456111883065483148617986465457792466402273815217241817163501352405823111497054361243886482686577961223780674009485714921257768842405600740516816638273826990939617065159558546972545307457270500896948955482426026947281134410487716638801651876698608693876104397820386663227664888754028406305427233767301780066332483940467863387166158701953351518332400189178702153049021261024299343478982543905250789402003398874430729393561464367541193884402280665307850918818544192856877565661864230591257923543269259287766045586392906627455350388206936516877511705006505488",
           "14.5081600415254637821968107724614159379846815250015966632796986302946639359084284397562134794977873371225673488011687447300524372638777391491271044047013009672846765153462292968052361902815148016221806543555516416870983561046485697445468525590089690561736144515682350187091487327533880302736617743526917603555494921673130404909292620569107730648244440333842777280190694749887082191823918012255416939347228605499563575333082299453802606427072078361952092566154166904748856523531473204373911618483373304549076618565708179979210316820872299882348744668938414281930060202535519438925657354409407529756410826043868530687455675261372284869586600498214736517437102860130625263737648428644588668437974421283838793145810608177937448518286819323950509354081962798121406813386052183361387375237142429814580328643754914782720125219937145864600464244591497580867134545593652723923600055555437968222771839658325503488203860238775496018875409219820657867877489532429308420073127288519458470515585824162233002589242653827817278322155993680288963941768325132608929167231803325441134141466079523533004874074960297398614",
           "14.6160288876324385915282082833965054491451514454610200902394687979785177479643522829616364603240189840566860380268224748399753802405943005392310014953070864465023481741203862559161940023086412200284446473311773218020713884034550394274956642070111944572452151925229164138208667793489213073745951939637376475072238046542165112242929720624283960415646561278977194301782816461600118545055956802134155907504241913799478483079002103747080064163279210998110147485017332310883903664994279933307864561852715983031647907050130006889886745790117080604262580893840381373714551099705553640113221087389797416689282779302152781657206661429673588858985798962970236527448473940009592181110203110178340755616764069248826774690236853289420767334169479100200273495854179775154480776286912775449612542711928074576553201233514379922132051999344634460933918951031402916699819979843036301934686144731527295916388130591787688603848332640080135877157352065976684603109296470843761116064910386409905698675285107464021939356227075015494978348239956711905571402049194421753904018717527587649996077266800964352303624402577118757550",
           "14.7231074538527950476608918508913485210193928960438010156179716284477839899656296406728660773561791479300392640548844362423376122815685515054605750659875846934795898267446279294839998923205284269250511553190404067373365705229426342396747030473106863384763359235329631176350397962888021561481413516660292977014970409803172385929807914633372249932127302974588451306885951998924516950174494016508742886729576894104195203891310671564861400064652564905406159871623321437476286879872259083518836208723342177364371256650666793948295098750435124566209886842002880143316847324199429697285209003531473719741904014483758049310397090719533728097260541450500261317080346355382466832396046849587639939461239781658290728113330095841108422751523386313998448194981539672857289889090299694644812449713503451035671101363587351933041787935726164774892898031020620075494054191750748238175302842601156452243294124366476294688491666489019848361123258726182091191116324806207133930412068583323189688687308435693004227410778794481033787991938082837748996778982096020946799825701362360467247804846582233346459021849435486394414",
           "14.8294128592903342463108620791209811162154136750542777284601531691224280305317915513146089525434202545204740516997954296244402311661887141413394532911541491603071708609940180070016261017915762557510278422457087524038813467378351272004581598457324467747373030877616178977537315248779792746942842469899245159612381076176060345997493587854288545462033896684057639832634271247921538572263389626921342927751757718681583536839938535751092658431596624136867197010308317749181005637423090876543873298238850113918296817041690209369633579425326504751223838947047626606883955090485790288429021610217101252189548742578539077253367329082514118063765427191960206852366367930126992346062168587627226173914689723039724555064028133389228017987680521282524145956765953246086551097796047782148791998851951221419883319817993999053278752506962511065678830257218401748568500747333591882058597483002531463654177181720203177842917132707292556291289892843397045703928170161651798774202385266858024628481292908429107158745626477532254897607152843489836358490936853541807395474017791353789095258403694863885903856428866385139618",
           "14.9349616137730772518742205278502213908354666769768829412154208873914099415699826430905240500055519452566108757946045864392050766773281378318859154769239371884763198843033880284347786378856604370045769280031694584512791649322990279975840486835965869228399135416024953621164643532400693170358493033550812154255730604795122027373166382306083361422620489180286334685541856384068029463428651261328445748342560926139564774421725776053820917605563776256951832599871875185183132937072360851486597203051765984337363986545459745401275505488192448731938753525598111844098064910555199796449249197864727749286292996338718305630998619908744286323602224649686053449748987583874785327499948813912602137839272202488683363930706928086605262922617223633664247999608682512707009348695830289960455591428363355858586332036652696876145134314146004076987087103822970399358514255536649174532275760391584685618135079679840634515711782148326692482011033477006288036228152078965813370535741600755067904328039271925962758975702936379448509324264036066209770450856159966672496255612322665295729865499903170425957082810506551820498",
           "15.0397696477860030144945917088662715180419204436596298997795414580537619276470518496795750302683435966960598359912232778691979417087604170857471339357968875914806128827487423868239668258781196328158829014067930293905548139764831766780121016782020549430650869364709405736002482509016517499087673550631711895241943655599845471280944381835142662152324894870918009318555850395370785744325805635115612824121714311554984178485987071523130059978461157895518735288230178080488091738471398284150084865371937653513358898631189129088281619129601787714351761094285499474246778785178627730928100052820335178118094485298540301817051152783369081590502589702221805406134858612454692515706806553394570930351722035092384749695378201907117685274122131158052887362415892662849390029026033965056482083854830488981110682784429900915172219306234004099363300333037817894031519149736675537980617386623663320490641182129254906815604059543725669871346028069397621898119057810818736095370601335065657301433376254680794313020349062521396509775307772376959952905436929593487060738018338230316410641798397322476355542860384513542506",
           "15.1438523405392099196696941526680757548021305645205291142290551070202545969870806060651771792665589490055461678011089894664863226391200155446121001161264758488222957146306674042780025430540877651260123695207749882929804823786072408198311834035741094722904703994169074927527266054117152533255107532619942528281884626828101541096773239563535294095652383393617113444649817226044992144468812474731988014146393516008194986121551417317503118242518823714684606734325443399645596959933855453707336231367089097896345978026922587762215370241399967794400041904056720435869378522780399983882254234030896553717060098405070969432588607227990450595751600391582489120049548888787130887917656163272514952919890907132626716845497133112466950843147056634292546916486178521992255691504244921986125000809530997425200716615625657674011024011250858696591746091527644535689099940477872177573673286692409322089652390467243801596188075140733650997142370201841877916119994009448748653226635048181819256281784782880874622975340845523108222703652277923638344521781640569055426282343260980758922970221905010015543051560514662433274",
           "15.2472245463115253370040636354083283666887786708714985639777668988830176955007183975627913778852644689341469297550544080185597077331715777153346360021007242495125392214774851264280715894226066393672643745253857483137929607374086675166192281961765358804770989187738591443219457532796617012663647789609431774075740567432008038961013943937307134814420860312306044163504297043835283963554752015900991148417739158394289066018572722217110593352847089201745718938357549727957979705487561467680940950578965108092971745951832261598491593612429568718722411875570411732507363965110669584642166670193068764600247928278340651032872526563501469069542627171043818450750969393729921266850657068206236571831577209919077311206015198636883915870987141318717046234082051220681758363352349761727350749598802358775860890853785340760259057433823505166593652654890358333190798767847250596060513312651361844271657521510045055877766996925314299833842766830666961654150747864536564429858555272922415110649733857674682620223561453248275694304210618254529601023775975491296511721889498789622849848564452286751061016082497088131584",
           "15.3499006191973273271932743733389729106106465662646172680460393169714387084447557470480954083233770880379685371224630951351727147025497081082089879388337460847605271660715988291396374847634019968015591655332379953553548320591794507974928536313132140565134911057811331918478115692690902657513781466327964390558905925364443191776886267775090535389272723947721917202602879066426608737057083329758491682770994310428321027233749321777436014858585199107035735971395626559257243819405652335029653770591399178721590892921559544649784504159433642408276679973155389005350337656246816993288749049548293503305422189192594259042858274455047560941579251477856306294970704330271186897488785264092623536912370835806446753257171205100681111029503749113972588676479023379074070878101497336800280410356924143787399669535027376058454568125882734662103842745378855390985776613904203513017797747138888437136968088396337069879433887912538591669283312778213455485732955595524676294450606458545443919774430029550468739232908922412573697555394576825425670246864782661192745625121039099283240722841566743449202185347012194036339",
           "15.4518944363733046480094895714642432822764116097792743962905968923051434278273216767519678368256147566495033878636454508377334372505840297952460243476345514701393821523863781406748742704716992949704447086972077248061651918445325061731913106414390710999384235658960798983324112234104659780681677057531882860382080238722943298496230509728345363802492358823405567182598267341768167330142056290790935811977359414090689588039875920175969152935770254523050051392422780000072929490482155794529675141588626785387839003076240285297590899692598611212948974143071946012735043590405405711508420740182622735629791811546902131754304158391375976203350556228354908194871652988791982508708646374469749119063201955850224934760173840978540631523245165459130698723874391791535530500096570481656934324227197278855637657017693124057969321411825957010121491871519557762634188278945281706669785161073963143541978342157205033140283815645312514465796297186749713508027712528205500701661239403838639760963454035559115496453338988692441465086270102708432189916595292275910127721552152009058942505984180320198848113154717062637193",
           "15.5532194199919290973914478271101509782761473427556110871326128562635425877948922439065276333739001292422317859939226322742544533781551368669476914135913677167870569871992237487074670090250710391585542749455379676350917711973517231870663274531838871780463639607198142971742006771650734656927254936190760891498433170178096075966621078837969943641749478412326387664953246784721461426256026142642356786959544769333983224947181138296945030840334549653119000316260143077956508210419522473362562194412180815212162799927642323378945617932239804210721035735661553698229406968222610664812153679540697776845028123042956550110363507444357710659863250974690713371372128989567244046052322175878868097778138345843971318806979435481065172044184336853266870505659167792988132948583467977404968694959221486656240243417746389624264522689502159242881137811206933660406168854853625265097649099854630342455703384775379760129637369884518317406075014753125436006487026676158299533450354777908097205026564092618416813042451075357865075958584347010560603909401672674530473016380690156057342486458799820056849296157803970758151",
           "15.6538885577994285336615322473375966991656095021435737363246099636760165366202196502339470259559417674183944684300887447920118633727001251716571523443596918848444675379732072543560564133448336770198458947224033362634921875944735296488201053135882187385898631210771646381454433721775116899072292857866394011427204808403618078909248941183413487233021402017272788966706121586469579730897499975748868490518551337318808652816220884250194097501134581362589451191335370815112714612837896607752466579681001969484402775834313370422194201064429923758665189369487888128968954146148578820551889951620171166054882708759173934381781451338459291737123198762967272009330015190135251071755806152473036989743744301736751164450587318893784207514998879026913548107957822584996296731105979982593758231329776234669559571495246544966721659816165141685360821041442768655874261766471614879723279695129605131434774277154659938308116466122952625429526707433155213455467498548671879756157972690042396634723002326783535035168700456971324492035298595293576869795744389366846556161860716979752870417581809090754427851499367674362389",
           "15.7539144225679246209830112311867562489275000482982979660467978944730262881956936027247186896733459505337734582610770093614470487327551772081823931905589550005108629069012648755348782514228521197088346998693251351381154586496257153314478419466512135062477857636559546187276312380852109985976404783376323163746995587475215213229589580070616867837675337919606283901588189302295510731616001258677000217918798250564696325480908966629389997769717889909303798070409227548291667767301493099723866749466998823232247966469030308331085073105913646005885136696884725098040065704027444469765094470600930146845779530531161938075234915622678545819344667609350674211649145259963974499132458151510972295814587234589477382477724361318542752546281937647008728052174217050786598039517009852677136114528982242252373617954020044735443912957185708403871758652762532215592658585695740093871059422733899587935643096714258513937804831673775551390957596911031295963904326636986587614829199748400378284270489821119258092593559132400132757457303025648114989713809051372859786312670557557455024113430415444733469786577609605818156",
           "15.8533091904240440533801150604807321260894390255847791708290322487869548721062866289150967514385361249309622591731814144915462307294258995039790586215006939993383983758776850046561092349884695864980967491417864933460813596552839148807057208618613557599377913061397320925659394501750622861864171880053557935005392499447778008413317995198028835874078756300521482179548125620652788764854972999687177405402042983731431613693788787613752177711793270302761736067574277658658787903363445166365077580601657490510530290835637982971350451510996298804593461238015330599519964118988140999455569019580613714105164866493396051692919035250861649195973033019817282972152713586938088500687115272282682479043225493555090482203878517967251561713092119554643396372535898649907805934878016873123756393954615387951516173102704296387140802792650937087349547235972359933788305229551638870049736359516727540333980757675042737103700554746326399085275428878409792860763156503485547649698802164314596132917707378582313164765977101027311129848967453806648036945112397617833423627052843598016874809261680305673980351252655348374751",
           "15.9520846581496442456835073500703066451779451051014841539242701086762015613192896323745559106469019887460414126185431359961253757915198034227258433864250836161583392878779461056493265572605574360603888584134116652015717511381633174846503017116671061672690227704542137628252959797331559476487739925081575856685236012972763809030119081042926665276791590604864556488405316065408861219952886936162336641184918936697605192640548272076353846488495049719651439648905433484003530512876695233728489114894850158179246113864820837838500720026182528991300469647511661287205697325018876335964502449498974863642135604197870197848793495324492868083775898735465533757619639262635723010975848105068325903417106605865046292138037866182138413986125566393404019074389870752413124723753587841540468575699565237256318701348228914889961059600076085302269512336986876298095168455531799946876677717621683146126707236377496395703746431798350019356976580256457555721053870192650781008973533558052075870268801533388874712868860234888763155218624129421405763678812077319239075778407441953464485710013630549293699630738008836738523",
           "16.0502522595242423047658196757944506380174575705759839278253230321445250829639481910961471931453116564538738497439478557175108609739886591120839569490500534822875304849736326488209685631259797055597162087990625950864597269364766964393819102203843223113243118426671582284256752979792222658121065967489791739897901556464359571005944335246989093673473307922127953030743811956112527855213800208534921642131401366328104759081292508449281353427532625724955074014554055167375410461346511957866858023371199864971286184642392511378752324547642788064420065819699737323404789938130114304157002506318247766787276624518432056105705328622750000206435275106450682964574301892472715679021266368586162764998585907621481532861299023329709215002403001723015543872591121222930217538067739593739604698257682609428654874555491602494919829715215979019400567591170934142751177256545981785724618392334660345395676518512531709484372866958613593771143352725257746445919068135186009397759278633269127719426191365945433194069995854695522773330552489914970861258642696680947645047295192118341734641074275298677776747132725952692040",
           "16.1478230807732358804057911156359975578643269683364195005563293802325983679008984974142648714564442396471730040316476902098875094502600153583773509818474040254222873787323736489817652011370386406266250541511480202000072979598788155627873931210965950356003142252258291831141254744970185472236595284729309004225710563295709356523567610266397233789715267714804425396668330717742603267969954704057766975076430627334443597598422164729733865234176455146920250925398679022857258803104782468690438474584298601058258695871408692587584948563444892617899745726543714797413684357745555329038056593572733989924153298466060559153513812177648291034169614177512283257755959087939669505720323242559668932499003542818847651691092512987952257177634187849222278857431112238824103335198289105119122009380272498965826349738709572199722545419521302521285254070487073289909851344138939360922891594389079867498143873584750623589649101531017564978708138390620735321513779148712154176474583208544330503651654716904907116042682578169464107303714557388993154579350605533888231564568424188460142954331592433579506586890005695885131",
           "16.2448078751809998755011217158068981074812938780376907934288365969067715754580651881228837648643879607033855368102601676632453884110903562211987202416395104068878916710125029828020978141971790535060779445467814920211563614823717939714231027215836135944076544647046307691726601906802103714211555626778098815167697410921218478761241829171583291214474448114233960671085832117529951301204663394172136901026541390977184829069626833926547718636187253278592953151429123517742761612159686867396009176772048429857930463378251101441506406781963593404431204027001757834444765502683573406073319506066129116153755917974319966003049503254536949291575688824723118194707260626100046414546696157059690148588275204844439191014730591329648048783886231915330055732155912797736172314922853100014380565619761622410826147387060234237675537903761684317822710064913207299936997924979239151897177602337621202867831218951358431546933181734907743029920468572483019023238760589912132344481868708100201604097725222903686108103206060682150910142819984712770046957870395535067746543225689701565797231792048627718784232710619996958175",
           "16.3412170769233839606963694201612486691212128081356542739834914529370956581095536199949227667563196669751881720330847867749383217493562824508538192834810436530795251403748280815986888683090444963568544342830253998350516075153587022434123870840098964485733650438270480924257439534833868943235183119239136909416092084248849001763157362588633568969048428387870886333191418072195613026763062715704979389513606318621766300613512341969898200938319702408657590012474789777007123467808768282135872999150490149110158277006503250724953661837621574840115483372782860303277647284047119064007533593648396594330123081102605014990883472685835045389875767425672113541605338848610128464772662573977617761213414690520480262408650017838431407962017900140324046253143316145470737963355177005851379248147035370433387833420553321888460391167558380062662113254804107448975955978227341116496076743206260952106365143827443135137267145015186532545755133066280336384345677654367847031873202744469936264238113240447117511360963000412145400308169673233984915808549522819952866570114619293558801485679439444177553637843192666077022",
           "16.4370608141699779533272474043612773161336096733770488551072483958825670966030102770671692026351162758060426915952129701085867802729318106406251695179863885464546615845148228047143539122431375925160247632834389582278394024330480487104411090951724620355644351152169170545009407362559853715960254174609992367516445002937337915452344034027633137804671869009243012191216273628143810422653645542449220632441502889291380255597379975105906010494299091436098925393391453500383425028779847395212818216089937829167091842984754080301999529208965785570901563511025237838322249140592553123941513540557406487062630678801556860907709013663182736461628834763074768624936750014768974286417533180445348738169351441886657201423938621061945983927061958409374247097943865562851972717122204967234909902165207404812597672553085816768537588350650139595857211821910608937442187518672584714044022382415383735643991420907608195400673486843199910879900829598881279040861315931094524028380978241183376771806462591007552168307790120399348941306657109189741410463773559051033330925696056841916167522766228211393262242421521807262490",
           "16.5323489215027158446880825763336276539766654254272088764809875681257710901986514691251337281589597719030480979213983912333851607240138763154510692762929645134059390271108447868644173448984649545846075027064353386413401987057840078259547584916364797582515654545908036217306641693761885158675919161061304000673810487248591975444958643217446048240102727684132123680636339276356809494454912918773518940016680535088516253123764670882202459282812565757616402645579638779334204722728987624467180608880772319027534713555789150175287611585325048568498467487723808179090257348438123807077849594237725002857165318916587733119054901238012585296647106745143265490190013740659650259863719019289349488568359299562334530428982968033376892598005685239623058868742620448370226790430347712128643558758014939672725409586893221245647244200231825250935867451601474997763673733348363862073700502793354719022161931458435920378330557182605440069878979638381549353114629450034341931427150761577849342839121295284737993268351881523490532278330519022627550209271545051013287036978208888323804801374289519895876593198148288412392",
           "16.6270909516939193588113275742453455057724907893471837248190052277684021812706093859674562917358464508678944731335671312361111528635211707981851761281905390428651253063365857714873438460026982265332515908134496574598271324405852203598672867489333701087729773656692255479851820749532099278033668844157411253338256984189005214039791624919948530833113525770673128187949311061890698199891708014824055365136849537030748140516534893460775625312948474125099589563910528552797986693601626798750522320704550044628942365910519008770640036921561300268413859502896583956572600382670717293799295760336680708266640050199950546289960758721503030196132687468226676261182256632215649868207453902112403583378942883517958418243203582422082833178466365454545310575197259045841019912565461611247674091038326392812565767346243460064710336769861695873330502452485176311239682300570315470799685242592229377370919807320562394172447148622490939677945521065046574409955795556738838394520031077084414251361076203172937851582781189271677247274081267854212505982365846174249067482121520863646230241374323178103657911564790192184842",
           "16.7212961868837069596057427052330107838993265239842708646860851842703326371466460550285293861073853336430183927150625107744808082403658130218547877377223408626609338367841882539006706085525641360251738080907390644127125042595230279880637143368202331046893342048365464875596596339640431519843405480605327189233621288814329341353255673008710460439879747397502025941747471573127868412038250150005699065690619898644879967858901930028888717317423009915633225475656314991246247179300655948036458387420845855288877301555946710757857081888614708196039221016144384209347600898967572348354997689536324424925550167367649124956548098079407104673133738019223267238283865985764961736835718647168600253346670325018729936886914389649147361585260144938350502362779387053821926782839707071611980639055740818753798912061341260095450383120266085280004484309135011098574017285860156410583019562874766816141902023632532127548071497727038984896353454326117060152368514028177470621582059370306418123530554777252590192842881894048535435254976906742189935635359698594705800588739642243011198473083262224352068070947521888133964",
           "16.8149736491937860979338473016064026815629713256323497948750601809568501329556639230091530125737561218251440117233688971366626226619494552699934840033236354855838182946899863935935135051148775248339278515431361132281679384204107687790514630925950009428499533762858168486928997467661567186100299191698405789687931448311811106940799338769819016977485112597785266160291200218493515224003641856137869046817349736860084136521100283230752847231380759335075251104980538366249594135868723508770265154137352284134429841268968217533402887160211021542302759108278289630550854043783744185492975265706362843229478657405729545441090741260159076361941917530778082026439317132083178091065099562781527891889439614485685906543894620100854938929972296809305957708892264846534267896729102201515793747667010846289314417299727358358529077708665332354286589063320334666559975273340602615325426971186711002042629355909063066914261841178544815216158892244142405490982903749738628073534048893782956355157093511439186949168195503573717926825734468084712589410442022354120219836575476305852256521675584981012952509686818126145279",
           "16.9081321108119800703237241338254436690744863062204119333243683660599775226753565893209713389767766163161342358021127981271381134650847016286405604432987298022336098638563576267935354939301846494369816655337783246226209880075011315863553918585959135040655522896546602388719639509065495093614444330145112109456097752772698468793609700122810958534616000768715917714125275538225625413748533473685169190832331617113246579894504494307284312332165945233891016849490928121382853036960384597592081111721835630475340517447933339362704327966786541965781642013462403621512902647219230788448655730199762241068447349463838875501396349840327689919737332595302012041802495467706531267072999713901403393223067607382639770962103669304655143862584052038284208763832029931753811022041122025100304848953045363327544205876760304756720535938198106981928743202962212743831014622551526319592553193580573329441708624935281603762989949456873712545791427861490109968314910368234559629841124132807220595321318685022115111820471491845170412521830502033004342105561816928208532788908700951934092518100776316736470776442731692226738",
           "17.0007801035793936281121338813371428867974374892864315366365045182530749574622712707915241384347920699293353638201192620301921643313046316033287195368236744465600818338279114167561421338713928665140404227014843877263137439279436595462144971591867941134682123131132347610099618581532784272896432738682598070539934802196472265692623169163830620747322703103707706888841879278208903105409506874013233395298896805581080492544955643820269848540883869294533510006353020349957920715014892361843146242577547625252883154835493940087913143008925195857805360140270418585825481983225958905442213791260106363149613040098659673146882019866732435671376309178138376480369335069359192722225011915819361568401764846284323216010823897735267747840800333035036264285169114393598090497354973291064439481840096277008690228063146037619643429834761658442087827165677168899338925740605695619093196708013773454044065476725273474616759775258989343235256991100913196459365908950267021754881564605943770560095242514851607841979245586880208866747279307676303889206098080334141095498113120491826011797819568010033703314385766397392583",
           "17.0929259281098732308568909665181706090496358902951216052253094189082758184705768795728909606854105818308409911054301261308877905438819451701225816708775064163858458811220345957117443399486918793485162274149097723524277233813071363121925271943987039923392571390254618731212423397735256468820990712184553680703600120940437674047217109360209678344268629771031575160683542930850707582417669810552350662093148256275597877537621451125081219588556687019419120479317429823172107590283143446497248490970649768152464725806707130059311761908605648006129152794658287289985756220544649174887465478997809837771313116429454050449336877009922282134713999375548742678240072088620118531202157406755008724496781361076606758611169943346574996481406674965729735501106625955985540926899693816341047327437159278090340558983247348348622180363741547423011416478126650485047347378185921229539834418983942666680991963426108094492973587943493986808103947127419030775318104983559336517699874867642430990267257306908318775712545165790980870833770357342634597715966182879292691159156387989718754333799048530940386842225448232251458",
           "17.1845776624693504399278552427660121759014355872173486225007203824024372840436910407549731315312720641661312015514244338979423498353490793511266853712442843525886341505446387699669856777170226001440090937353259650728051196433419270760984455065871632331371517893432279015787092579337531691612539497018543844203336945831070057415570861416522661341142055293762783441277787020381532046055783982924019170148071336852167967942723659657524174803803803905087308384416784431284819770363195053820968301578868801810080541678810607866404802296225462621786603189632403039176608470841164659717593262258932759900746120866956926679585397696382726469136456383993490336779413736927110713121992182458332821162549136160976369409523883489900369202140222731089221360463387016825261552399416009215359001253955577936829021948708246676341112942870776695271407974775582902477226815721865733052145151608560413760572173015439460907252134033256691370749495569198954137521453923556632774376333569072404958969821817487022660731421354245851272428708325787898296874181411803997899332362008586913072670711064770984500148015011826261837",
           "17.2757431704407540851904450153949790380186687458654220684915230386503275825407063160604517288734546119505644391694970728295669021023983949819859759900830880079629779705059154865670128635551815720536815841099639525614851417689538084914680867424196821634089586241442130192038834512446570011888290802100133594182555986337723487386416534433468263224465623305274641330130540795411919391711267934371941107593205550581646424497484790190474300721785629523248847972136552642048161447129520586107875881613551369904113055686380180469790846706208691893855593511420750973530511165283555627597887745887134627634145517416050748158800890514425554639321377870923909855121500878983383395887478657227223962509627436500904372492098670589675883214674156834456454663362930882283624845127465731275714349781314030602882871329526710766294672136474269162102821374407938659858184740917810129868058466378108500499655709083903249046674366509191153761430075951411439469754773409282874508134728609913829574500910503209081050228705417132900939410588699788540167873020190018123072161219964179184690554184874930100756126982018853834025",
           "17.3664301093984238247123510549790984289306134748291022474290958664888369268710974860027907576347723303147126176414184787060588912832469046111702601777464890642632251154060816817161334316918823566501136820770035716334732686181613705402325472307650116606106783477061012163011204623322704865562057283124639475372748468035585007316914646119895238476177895413281264762196817540406728021584836434537388272205662532262621643283471955600594193213418166120727080989351217403325023680868046291273159196135079849504794468566646008463271904849506404907608613825464391989145962661376690126496752549886760192239310656336157879887593855859958823309866860061179406457120429704015243436160532979492144475251743461081570978743884248426983196247351207081556692289758666519927916028959075589074583772201210742744694427687226168211338842746789032366011901448502121835024794286923051973016164255684120993153139149490806320653960702333007936217638703590917692418814222701997158804755501287181232079015044179490661098796848674027302976104705585992281369648433223442373992912484021774309627757116258683761986486802220878937797",
           "17.4566459378143413191692225015426086274031136235582935243632385313858201877701704036484032520749642131133778450955342272025602510829158221972777670586061661701347328186526764805212080639525455982292015938346325289124302844332447725669617702921917808500525274187600536620801147819163613587837238728871997225991025080525180442534466864555824862414662339955733621810264862776187374419901285240431490175974449378317927398381519709919509933689563116335690204265534027606560764875449531405967998639356231702870487782798115026659692007825259723146752469612698767331463901780062493519689225278109283378310274389559679038834170680206517259537270404175432248633893701903077868034694197984698018454399995348712691476944938462701653906803877183551688371651410557476681964972950883316889946306025542494264115645449898893118484226513921500673490556592825150254681189771557834158711489644867165103165017573599086205249218767497287654393905135560681295542932352259490763907188060867843701402082704725079265886683176519491584308173914664681606671056250177292407244890772315688524722948557271250549147036005452988137053",
           "17.5463979224170035169103569936773167710489071842695682164094650343960555822548938246261708686464008628120698086564271575140642653268871532667049895917630355233940483632068661179612946301911395716185300516412585342889806163058970394576807852912357307669092680925494306692002896260519270415602285809070330544448934264866486383161101778807666439177712377349404344204981825461265916701713439907634881628142000030147481541566984916776985069974871350878105191169601874427236107028216631331508432342933927262432252048403053983936328555651202085666743721276666416053287908582708399019416116728290391041137776899514963685453226344913930595188919687985925439640490668381197141268324607645626999418743675707607782207977941235558303966153142486351061701922818541439990955033863706292565895764497302237144629129915168217734367589190606338115923850388544120876370105674692788127644053617727607207239081379150797391284871402801013281516570366080963892214472234112621858777932368224243266851672272297127593365190407239608295928071192401106453278389676417859068237527688061268702479082098130209555748133337115265799590",
           "17.6356931450223847051632626336912129159953047631160231255194263214719736745345056574006334052947783704789851124650942357806440283438448075865374177648755421744889004812088237415174002112721888433569481184443980026560025069439822330036408245659699705607208167409459718244781459151859701286920686791397587316924683661006361446569736280386659169943463243969853737516308403647389899899943288964539074045777396061094245615020794569618969810767286161759163196190210263374998456288778623199988233761901113536316735857956970778148341712538680960137046414511920334125306354736321418705129785669231702905296255118678564162795574424588337165553826200448066730531304621451838136219021244379504974116096097930348202146495020513154705219076501726427281405053157307369058563911985328395372478199002802154489559015174957595775902617263259894306671438880488104887489699278033942889387921168112741333336787578922278486757542524555357598873580646861232908497366240101359399679018683387336026768844320636768908505456317105807294092097672788610628498171867123780698747648210099920520967511623044818811130645542567309932834",
           "17.7245385090551602729816748334114518279754945612238712821380778985291128459103218137495065673854466541622682362428257066623615286572442260252509370960278706846203769865310512284992517302895082622893209537926796280017463901535147972051670019018523401858544697449491264031392177552590621640541933250090639840761373347747515343366798978936585183640879545116516173876005906739343179133280985484624818490205465485219561325156164746751504273876105610799612710721006037204448367236529661370809432349883166842421384570960912042042778577806869476657000521830568512541339663694465418151071669388332194292935706226886522442054214994804992075648639887483850593064021821402928581123306497894520362114907896228738940324597819851313487126651250629326004465638210967502681249693059542046156076195221739152507020779275809905433290066222306761446966124818874306997883520506146444385418530797357425717918563595974995995226384924220388910396640644729397284134504300214056423343303926175613417633632001703765416347632066927654181283576249032690450848532013419243598973087119379948293873011126256165881888478597787596376136",
   }};

   T pi = static_cast<T>("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913152");

   unsigned max_err = 0;
   for(unsigned k = 0; k < data.size(); k++)
   {
      T val = sqrt(pi * k);
      T e = relative_error(val, T(data[k]));
      unsigned err = e.template convert_to<unsigned>();
      if(err > max_err)
      {
         max_err = err;
      }
   }
   std::cout << "Max error was: " << max_err << std::endl;
   BOOST_TEST(max_err < 20);
}


int main()
{
#ifdef TEST_BACKEND
   test<boost::multiprecision::number<boost::multiprecision::concepts::number_backend_float_architype> >();
#endif
#ifdef TEST_MPF_50
   test<boost::multiprecision::mpf_float_50>();
   test<boost::multiprecision::mpf_float_100>();
#endif
#ifdef TEST_MPFR_50
   test<boost::multiprecision::mpfr_float_50>();
   test<boost::multiprecision::mpfr_float_100>();
#endif
#ifdef TEST_CPP_DEC_FLOAT
   test<boost::multiprecision::cpp_dec_float_50>();
   test<boost::multiprecision::cpp_dec_float_100>();
#ifndef SLOW_COMPLER
   // Some "peculiar" digit counts which stress our code:
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<65> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<64> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<63> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<62> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<61, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<60, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<59, long long, std::allocator<void> > > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<58, long long, std::allocator<void> > > >();
   // Check low multiprecision digit counts.
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<9> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<18> > >();
   // Check up to 1000 multiprecision digits.
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<500> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<1000> > >();
#endif
#endif
   return boost::report_errors();
}



