# host.py
# Copyright (C) 2008 Stefan J. Betz <stefan_betz@gmx.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""
Contains host related Functions & Classes...
"""

import os
import os.path
import subprocess
from tempfile import NamedTemporaryFile
from pyneighborhood import config
from pyneighborhood.errors import MountpointError
from pyneighborhood.misc import homeconfdir, print_debug

class host(object):
    """
    Base Class for Host's
    """

    def __init__(self):
        """
        Constructor
        """
        self.ip = ""
        self.name = ""
        self.dns_name = ""
        self.comment = ""
        self.workgroup = ""
        self.shares = {}

    def create_mountpoint(self, share):
        """
        Create a Mountpoint for Share
        """
        basedir = config.get("Main", "mount_directory")
        workgroupdir = os.path.join(basedir, self.workgroup)
        hostdir = os.path.join(basedir, self.workgroup, self.name)
        mountpoint = os.path.join(basedir, self.workgroup, self.name, share)
        if not os.access(basedir, os.R_OK + os.W_OK + os.X_OK):
            if not os.path.isdir(basedir):
                os.mkdir(basedir)
            else:
                raise MountpointError
        if not os.path.exists(workgroupdir):
            os.mkdir(workgroupdir)
        if not os.path.exists(hostdir):
            os.mkdir(hostdir)
        if not os.path.exists(mountpoint):
            os.mkdir(mountpoint)
        return mountpoint

    def mount(self, share, fstype, username=None, password=None, target=None):
        """
        Mounting Shares...
        """
        error = None
        options = []
        command = []
        if config.get("Main", "enable_sudo") == "True":
            for item in config.get("Main", "sudo").split(" "):
                command.append(item)
            del item
            command.append("--")
        if fstype == "CIFS":
            command.append(config.get("CIFS", "mount"))
            if len(config.get("CIFS", "options")) > 0:
                options.append(config.get("CIFS", "options"))
        elif fstype == "SMB":
            command.append(config.get("SMB", "mount"))
            if len(config.get("SMB", "options")) > 0:
                options.append(config.get("SMB", "options"))
        command.append("//" + self.ip + "/" + share)
        print_debug("target = "+str(target))
        if target != None:
            if os.path.exists(target):
                mountpoint = target
            else:
                error = "The specified mount directory does not exist!"
                return (target, error)
        else:
            mountpoint = self.create_mountpoint(share)
        command.append(mountpoint)
        if username == None or password == None:
            options.append("guest")
            options.append("sec=none")
        else:
            credentials = NamedTemporaryFile(dir=homeconfdir(),prefix="smbcreds")
            credentials.write('username=%s\n' % (username))
            credentials.write('password=%s\n' % (password))
            credentials.flush()
            options.append( "credentials=%s" % (credentials.name))
        if len(options) != 0:
            command.append("-o")
            command.append(",".join(options))
        print_debug(" ".join(command))
        process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        process.wait()
        try:
            credentials.close()
            del credentials
        except UnboundLocalError:
            pass
        error = process.stdout.read()
        if len(error) != 0:
            print_debug(error)
            return (mountpoint, error)
        else:
            return (mountpoint, None)
