/* -*- mode: C -*-
 *
 *       File:         recsel.c
 *       Date:         Fri Jan  1 23:12:38 2010
 *
 *       GNU recutils - recsel
 *
 */

/* Copyright (C) 2010 Jose E. Marchesi */

/* This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <xalloc.h>
#include <gettext.h>
#define _(str) gettext (str)

#include <rec.h>
#include <recutl.h>

/* Forward prototypes.  */
void recsel_parse_args (int argc, char **argv);
bool recsel_process_data (rec_db_t db);

/*
 * Global variables
 */

bool       recsel_print_values = false;
bool       recsel_print_row    = false;
char      *recutl_sex_str      = NULL;
rec_sex_t  recutl_sex          = NULL;
char      *recsel_fex_str      = NULL;
rec_fex_t  recsel_fex          = NULL;
char      *recutl_type         = NULL;
bool       recsel_collapse     = false;
bool       recsel_count        = false;
bool       recutl_insensitive  = false;
bool       recsel_descriptors  = false;
size_t     recutl_num          = -1;
rec_writer_mode_t recsel_write_mode = REC_WRITER_NORMAL;

/*
 * Command line options management.
 */

enum
{
  COMMON_ARGS,
  RECORD_SELECTION_ARGS,
  PRINT_ARG,
  PRINT_VALUES_ARG,
  PRINT_IN_A_ROW_ARG,
  COLLAPSE_ARG,
  COUNT_ARG,
  DESCRIPTOR_ARG,
  PRINT_SEXPS_ARG
};

static const struct option GNU_longOptions[] =
  {
    COMMON_LONG_ARGS,
    RECORD_SELECTION_LONG_ARGS,
    {"print", required_argument, NULL, PRINT_ARG},
    {"print-values", required_argument, NULL, PRINT_VALUES_ARG},
    {"print-row", required_argument, NULL, PRINT_IN_A_ROW_ARG},
    {"collapse", no_argument, NULL, COLLAPSE_ARG},
    {"count", no_argument, NULL, COUNT_ARG},
    {"include-descriptors", no_argument, NULL, DESCRIPTOR_ARG},
    {"print-sexps", no_argument, NULL, PRINT_SEXPS_ARG},
    {NULL, 0, NULL, 0}
  };

/*
 * Functions.
 */

void
recutl_print_help (void)
{
  /* TRANSLATORS: --help output, recsel synopsis.
     no-wrap */
  printf (_("\
Usage: recsel [OPTION]... [-t TYPE] [-n NUM | -e RECORD_EXPR] [-c | (-p|-P) FIELD_EXPR] [FILE]...\n"));

  /* TRANSLATORS: --help output, recsel arguments.
     no-wrap */
  fputs(_("\
Select and print rec data.\n"), stdout);

  puts ("");
  /* TRANSLATORS: --help output, recsel arguments.
     no-wrap */
  fputs (_("\
  -d, --include-descriptors           print record descriptors along with the matched\n\
                                        records.\n\
  -C, --collapse                      do not section the result in records with newlines.\n"),
         stdout);
  
  recutl_print_help_common ();

  puts ("");
  recutl_print_help_record_selection ();

  puts ("");
  /* TRANSLATORS: --help output, recsel output options.
     no-wrap */
  fputs (_("\
Output options:\n\
  -p, --print=FIELDS                  comma-separated list of fields to print for each\n\
                                        matching record.\n\
  -P, --print-values=FIELDS           as -p, but print only the values of the selected\n\
                                        fields.\n\
  -R, --print-row=FIELDS              as -P, but separate the values with spaces instead\n\
                                        of newlines.\n\
  -c, --count                         print a count of the matching records instead of\n\
                                        the records themselves.\n"), stdout);

  puts ("");
  /* TRANSLATORS: --help output, recsel special options.
     no-wrap */
  fputs (_("\
Special options:\n\
  -S, --print-sexps                   print the data in sexps instead of rec format.\n"),
         stdout);

  puts ("");
  /* TRANSLATORS: --help output, recsel examples.
     no-wrap */
  fputs (_("\
Examples:\n\
\n\
        recsel -t Friend -e \"Name ~ 'Smith'\" friends.rec\n\
        recsel -C -e \"#Email && Wiki = 'no'\" -P Email[0] gnupdf-hackers.rec\n"),
         stdout);

  puts ("");
  recutl_print_help_footer ();
}

void
recsel_parse_args (int argc,
                   char **argv)
{
  char c;
  int ret;

  while ((ret = getopt_long (argc,
                             argv,
                             RECORD_SELECTION_SHORT_ARGS
                             "SCdcp:P:R:",
                             GNU_longOptions,
                             NULL)) != -1)
    {
      c = ret;
      switch (c)
        {
        COMMON_ARGS_CASES
        RECORD_SELECTION_ARGS_CASES
        case DESCRIPTOR_ARG:
        case 'd':
          {
            recsel_descriptors = true;
            break;
          }
        case PRINT_SEXPS_ARG:
        case 'S':
          {
            recsel_write_mode = REC_WRITER_SEXP;
            break;
          }
        case PRINT_ARG:
        case 'p':
        case 'P':
        case 'R':
          {
            if (recsel_count)
              {
                recutl_fatal (_("cannot specify -[pPR] and also -c.\n"));
              }

            if (c == 'P')
              {
                recsel_print_values = true;
              }

            if (c == 'R')
              {
                recsel_print_values = true;
                recsel_print_row = true;
              }

            recsel_fex_str = xstrdup (optarg);

            if (!rec_fex_check (recsel_fex_str, REC_FEX_SUBSCRIPTS))
              {
                exit (EXIT_FAILURE);
              }

            /* Create the field expresion.  */
            recsel_fex = rec_fex_new (recsel_fex_str,
                                      REC_FEX_SUBSCRIPTS);
            if (!recsel_fex)
              {
                recutl_fatal (_("internal error creating the field expression.\n"));
              }

            break;
          }
        case COLLAPSE_ARG:
        case 'C':
          {
            recsel_collapse = true;
            break;
          }
        case COUNT_ARG:
        case 'c':
          {
            if (recsel_fex_str)
              {
                recutl_fatal (_("cannot specify -c and also -p.\n"));
                exit (EXIT_FAILURE);
              }

            recsel_count = true;
            break;
          }
        default:
          {
            exit (EXIT_FAILURE);
          }

        }
    }
}

bool
recsel_process_data (rec_db_t db)
{
  bool ret;
  int rset_size;
  rec_rset_t rset;
  rec_record_t record;
  int n_rset, written, num_rec;
  rec_writer_t writer;
  bool parse_status;
  bool wrote_descriptor;
  rec_rset_elem_t elem_rset;

  ret = true;

  writer = rec_writer_new (stdout);

  /* If the database contains more than one type of records and the
     user did'nt specify the recutl_type then ask the user to clear
     the request.  */
  if (!recutl_type && (rec_db_size (db) > 1))
    {
      recutl_fatal (_("several record types found.  Please use -t to specify one.\n"));
    }

  written = 0;
  for (n_rset = 0; n_rset < rec_db_size (db); n_rset++)
    {
      rset = rec_db_get_rset (db, n_rset);
      rset_size = rec_rset_num_records (rset);

      wrote_descriptor = false;

      /* Don't process empty record sets.  */
      if (rset_size == 0)
        {
          continue;
        }

      /* If the user specified a type, print the record set only if it
       * is of the given type.  */
      if (recutl_type
          && (!rec_rset_type (rset)
              || (strcmp (recutl_type, rec_rset_type (rset)) != 0)))
        {
          continue;
        }

      /* If the user didn't specify a type, print a record set if and
       * only if:
       *
       * - It is the default record set.
       * - The file contains just one record set.
       */

      if (!recutl_type
          && rec_rset_type (rset)
          && (rec_db_size (db) > 1))
        {
          continue;
        }
          
      /*  Process this record set.  */
      num_rec = -1;
      elem_rset = rec_rset_null_elem ();
      while (rec_rset_elem_p (elem_rset = rec_rset_next_record (rset, elem_rset)))
        {
          record = rec_rset_elem_record (elem_rset);
          num_rec++;

          /* Shall we skip this record?  */
          if (((recutl_num != -1) && (recutl_num != num_rec))
              || (recutl_sex_str && !(rec_sex_eval (recutl_sex, record, &parse_status)
                                      && parse_status)))
            {
              if (recutl_sex_str && (!parse_status))
                {
                  recutl_error (_("evaluating the selection expression.\n"));
                  return false;
                }
      
              continue;
            }

          /* Process this record.  */
          if (recsel_count)
            {
              /* We just count this record and continue.  */
              written++;
            }
          else
            {
              char *output = NULL;

              if (recsel_fex_str)
                {
                  output = recutl_eval_field_expression (recsel_fex,
                                                         record,
                                                         recsel_write_mode,
                                                         recsel_print_values,
                                                         recsel_print_row);
                }

              /* Insert a newline?  */
              if ((written != 0)
                  && (!recsel_collapse)
                  && (!recsel_fex_str || output))
                {
                  fprintf (stdout, "\n");
                }

              /* Write the record descriptor if required.  */
              if (recsel_descriptors
                  && !wrote_descriptor
                  && rec_rset_descriptor (rset))
                {
                  rec_write_record (writer, rec_rset_descriptor (rset), recsel_write_mode);
                  fprintf (stdout, "\n");
                  wrote_descriptor = true;
                }

              if (recsel_fex_str)
                {
                  /* Print the field expression.  */
                  if (output)
                    {
                      fprintf (stdout, "%s", output);
                    }
                }
              else
                {
                  rec_write_record (writer, record, recsel_write_mode);
                }

              written++;
            }
        }
    }

  if (recsel_count)
    {
      fprintf (stdout, "%d\n", written);
    }

  rec_writer_destroy (writer);

  return ret;
}

int
main (int argc, char *argv[])
{
  int res;
  rec_db_t db;

  res = 0;

  recutl_init ("recsel");

  /* Parse arguments.  */
  recsel_parse_args (argc, argv);

  /* Get the input data.  */
  db = recutl_build_db (argc, argv);
  if (!db)
    {
      res = 1;
    }

  /* Process the data.  */
  if (!recsel_process_data (db))
    {
      res = 1;
    }

  rec_db_destroy (db);

  return res;
}

/* End of recsel.c */
