///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/quadrature.h"
#include "rheolef/gauss_jacobi.h"
using namespace rheolef;
using namespace std;


/*
 * quadrature formulae
 * refs: G. Dhatt and G. Touzot,
 *       Une presentation de la methode des elements finis,
 *       Maloine editeur, Paris
 *	 Deuxieme edition,
 *	 1984,
 *	 page 288
 */

void
quadrature_on_geo::init_edge (quadrature_option_type opt)
{
  // -------------------------------------------------------------------------
  // special case : superconvergent patch recovery nodes & weights
  // -------------------------------------------------------------------------
  quadrature_option_type::family_type f = opt.get_family();
  if (f == quadrature_option_type::superconvergent) {
      f =  quadrature_option_type::gauss;
  }
  // -------------------------------------------------------------------------
  // special case : Gauss-Lobatto points
  // e.g. when using special option in riesz_representer
  // -------------------------------------------------------------------------
  if (f == quadrature_option_type::gauss_lobatto) {
    switch (opt.get_order()) {
     case 0 :
     case 1 :
	    // trapezes:
	    wx(x(Float(0), Float(0)), 0.5);
	    wx(x(Float(1), Float(0)), 0.5);
	    break;
     default:
	    error_macro ("unsupported Gauss-Lobatto("<<opt.get_order()<<")");
    }
    return;
  }
  // -------------------------------------------------------------------------
  // default & general case : Gauss points
  // -------------------------------------------------------------------------
  assert_macro (f == quadrature_option_type::gauss, 
	"unsupported quadrature family \"" << opt.get_family_name() << "\"");

  // Gauss-Legendre quadrature formulae 
  //  where Legendre = Jacobi(alpha=0,beta=0) polynoms

  size_type n = n_node_gauss(opt.get_order());
  vector<Float> zeta(n), omega(n);
  gauss_jacobi (n, 0, 0, zeta.begin(), omega.begin());
  for (size_type i = 0; i < n; i++)
    wx (x((1+zeta[i])/2), omega[i]/2);
}
