///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/cad_element.h"
using namespace rheolef;
using namespace std;

cad_element::size_type
cad_element::n_node_from_degree(size_type deg, size_type deg2) const
{
    switch (type()) {
      case p: return 1;
      case e: return deg+1;
      case t: return (deg+1)*(deg+2)/2;
      case q: return (deg+1)*(deg2+1);
      default: error_macro ("unexpected type"); return 0;
    }
}
cad_element::size_type
cad_element::n_face_from_degree(size_type deg, size_type deg2) const
{
    switch (type()) {
      case p: return 0;
      case e: return deg;
      case t: return deg*deg;
      case q: return deg*deg2;
      default: error_macro ("unexpected type"); return 0;
    }
}
void
cad_element::set_type (enum_type ty, size_type deg, size_type deg2)
{
    if (deg2 == numeric_limits<size_type>::max()) {
	deg2 = deg;
    }
    reference_element::set_type(ty);
    _deg  = deg;
    _deg2 = deg2;

    size_type sz = n_node_from_degree(deg,deg2);
    _points.resize(sz);
}
string
cad_element::name() const
{
    char c = reference_element::name();
    char buf[256];
    switch (type()) {
      case p: sprintf (buf, "%c", c); break;
      case e: 
      case t: sprintf (buf, "%c_%d", c, (int)degree()); break;
      case q: sprintf (buf, "%c_%d_%d", c, (int)degree(), (int)degree2()); break;
      default: error_macro ("unexpected type");
    }
    return buf;
}
// specific for Bezier:
// reference edge       = [0,1]
// reference triangle   = { 0 <= x <= 1, 0 <= y <= x }
// reference quadrangle = [0,1]^2
void
cad_element::ref_node_from_degree(size_type idx, size_type subdeg, size_type subdeg2, point& x_ref) const
{
    switch (type()) {
      case p: break;
      case e: x_ref[0] = Float(idx)/Float(subdeg); break;
      case t: {
	    size_type count = 0;
	    for (size_type i1 = 0; i1 <= subdeg; i1++) {
	      for (size_type i0 = 0; i0 <= subdeg-i1; i0++) {
		if (idx == count) {
            	    x_ref[0] = Float(i0)/Float(subdeg);
            	    x_ref[1] = Float(i1)/Float(subdeg);
		    return;
		}
		count++;
	      }
	    }
            error_macro ("node index " << idx 
		<< " out of range 0:" << (subdeg+1)*(subdeg+2)/2 - 1);
	    break;
      }
      case q: {
	    size_type i0 = idx % (subdeg+1);
	    size_type i1 = idx / (subdeg+1);
            x_ref[0] = Float(i0)/Float(subdeg);
            x_ref[1] = Float(i1)/Float(subdeg2);
	    break;
      }
      default: error_macro ("unexpected type");
    }
}
