///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// plotmtv output
//
// authors:
//      Nicolas.Roquet@imag.fr
//      Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 6 may 2001
//
#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
using namespace rheolef;
using namespace std;

static inline
void
mtv_point (ostream& s, const point& p, Float value)
{
    s << p[0] << " " << p[1] << " " << value << endl;
}

//======================================================

static
inline
void mtv_segment (ostream& s, const point& p, const point& q, Float z)
{
	p.put(s, 2); s << '\t' << z << endl;
	q.put(s, 2); s << '\t' << z << endl;
}
static
inline
void mtv_element (ostream& s, const geo_element& g, geo::const_iterator_node x, Float z, Float value)
{
	// cas d'un element 't'
	s << "% fillcolor = " << value << endl;
  	switch (g.name()) {
	  case 't' :
	    mtv_segment (s, x[g[0]], x[g[1]], z);
	    mtv_segment (s, x[g[2]], x[g[0]], z); 
	    break;
	  case 'q' :
	    x[g[0]].put(s, 2); s << '\t' << z << endl;
	    x[g[1]].put(s, 2); s << '\t' << z << endl;
	    x[g[2]].put(s, 2); s << '\t' << z << endl;
	    x[g[3]].put(s, 2); s << '\t' << z << endl;
	    x[g[0]].put(s, 2); s << '\t' << z << endl;
	    break;
	  default:
	    error_macro ("mtv_element: unexpected element `" << g.type() << "'");
	}
	s << endl;
}

// ============================================================================

static
void
plotmtv_2d_header (
    const string basename,
    const field& x, ostream& mtv, int digits10,
    bool grid, bool put_domains, bool fill, bool label, size_t isovalue_table_zero_index)
{
  const size_t niso = 50;
  const size_t default_isovalue_table_zero_index = 10;
  mtv << setprecision(digits10)
      << "#!plotmtv\n"
      << "$ DATA = CONTCURVE\n"
      << "% comment    = \"rheolef\"\n"
      << "% toplabel   = \"" << basename << "\"\n"
      << "% subtitle   = \"\"\n"
      << endl
      << "# scale and window\n"
      << "% equalscale\n";
  
  const point xmin = x.get_geo().xmin();
  const point xmax = x.get_geo().xmax();
  Float umin = x.min();
  Float umax = x.max();
  mtv << "% xmin= " << xmin[0] << " xmax= " << xmax[0]
      <<   " ymin= " << xmin[1] << " ymax= " << xmax[1]
      <<   " zmin= " << umin    << " zmax= " << umax << "\n"
      << endl;
  if (fill) mtv << "#";
  mtv << "%contstyle=1 # line style color drawing, with label values\n";
  if (!fill) mtv << "#";
  mtv << "%contstyle=2 # fill color painting\n"
      << "%contlabel=" << (label ? "true" : "false") << " # write labels when using line style contours\n"
      << endl
      << "%nsteps=" << niso << " # set constant value step\n";
  if (isovalue_table_zero_index == 0 || isovalue_table_zero_index == numeric_limits<size_t>::max()) {
      mtv << "#%contours=()  # specify also values associated to lines\n";
  } else {
      size_t nneg = isovalue_table_zero_index;
      size_t npos = niso - nneg;
      mtv << "%contours=( \\\n";
      for (size_t i = 0; i < nneg; i++)
         mtv << "\t" << umin - Float(i)*umin/Float(nneg) << " \\\n";
      mtv << "\t0\\\n";
      for (size_t i = 1; i <= npos; i++)
         mtv << "\t" << Float(i)*umax/Float(npos) << "\\\n";
      mtv << ")\n";
  }
  mtv << endl;
  if (!grid) mtv << "#";
  mtv << "%meshplot # show also grid mesh\n"
      << endl
      << "# for Postscript output: rectangle with background color (white) and solid fill\n"
      << "#@rectangle x1=1 y1=0 z1=0.0 x2=4 y2=5 z2=0.0\n"
      << "#@fillcolor=-1 \n"
      << "#@filltype=1\n"
      << endl;
}
// ============================================================================
int
field::plotmtv2d_P0 (
	const string& basename, bool execute, bool clean, bool verbose,
	bool grid, bool put_domains, bool fill, bool elevation) const 
{
  char filename [1000];
  sprintf (filename, "%s.mtv", basename.c_str());
  ofstream mtv (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int dim = dimension();
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  //
  // header
  //
  mtv << "#!plotmtv\n";
  mtv << "$ DATA = CURVE3D\n";
  mtv << "% comment    = \"rheolef\"\n";
  mtv << "% toplabel   = \"" << basename << "\"\n";
  mtv << "% subtitle   = \"\"\n";
  mtv << "% equalscale\n";
  mtv << "% linecolor  = 1\n";
  mtv << "% dfilltype = 1\n";
  mtv << endl;

  const point p1 = get_geo().xmin();
  const point p2 = get_geo().xmax();
 
  mtv << "% xmin= " << p1[0] << " xmax= " << p2[0]
      <<   " ymin= " << p1[1] << " ymax= " << p2[1]
      <<   " zmin= " << min() << " zmax= " << max() << "\n";
  mtv << endl;
 // if (fill) mtv << "#";
  mtv << "%contstyle=1 # line style color drawing, with label values\n";
  if (!fill) mtv << "#";
  mtv << "%contstyle=2 # fill color painting\n";
  mtv << endl;
  mtv << "%nsteps=50  # set constant value step\n";
  mtv << "#%contours=()  # specify also values associated to lines\n";
  mtv << endl;
  if (!grid) mtv << "#";
  mtv << "%meshplot # show also grid mesh\n";
  mtv << endl;
  mtv << "# for Postscript output: rectangle with background color (white) and solid fill\n";
  mtv << "#@ rectangle x1=1 y1=0 z1=0.0 x2=4 y2=5 z2=0.0  fillcolor=-1 filltype=1\n";
  mtv << endl;
  //
  // plot elements
  //
  const geo& m = get_geo();
  Float MAX = max(), MIN = min();
  Float ratio = 10./(MIN-MAX);
  
  geo::const_iterator iter_K = m.begin();
  geo::const_iterator last_K = m.end();
  vec<Float>::const_iterator iter_u = u.begin();
  while (iter_K != last_K)

      mtv_element (mtv, *iter_K++, m.begin_node(), *iter_u++,  ratio*(MIN-(*iter_u)));

  mtv.close();
  if (put_domains) 
   {
      mtv << "$ DATA = CURVE2D" << endl;
      m.put_mtv_domains(mtv,1);
   }
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string option = (elevation ? "-3d " : "");
      option += (put_domains ? "-plotall " : ""); 
      string command = "plotmtv " + option + basename + ".mtv";
      if (verbose) {
	  command += " 1>&2";
          clog << "! " << command << endl;
      } else {
          command += " >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}

//======================================================

int
field::plotmtv2d_P1 (
	const string& basename, bool execute, bool clean, bool verbose, 
	bool grid, bool put_domains, bool fill, bool elevation,
	bool label, size_type isovalue_table_zero_index) const
{
  const geo& m = get_geo();

  char filename [1000];
  sprintf (filename, "%s.mtv", basename.c_str());
  ofstream mtv (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  //
  // header
  //
  plotmtv_2d_header (basename, *this, mtv, digits10, grid, put_domains, fill, label,
		  isovalue_table_zero_index);
  //
  // plot elements
  //
  geo::const_iterator_node p = m.begin_node(); // random access
  for (geo::const_iterator iter_K = m.begin(); iter_K != m.end(); iter_K++) {

      const geo_element& K = *iter_K;
      for (size_type i = 0; i < K.size(); i++)
	  mtv_point (mtv, p[K[i]], at(K[i]));
      mtv << endl;
  }
  if (put_domains) 
   {
      mtv << "$ DATA = CURVE2D" << endl;
      m.put_mtv_domains(mtv,1);
   }
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string option = (elevation ? "-3d " : "");
      option += (put_domains ? "-plotall " : ""); 
      string command = "plotmtv " + option + basename + ".mtv";
      if (verbose) {
	  command += " 1>&2";
          clog << "! " << command << endl;
      } else {
          command += " >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}

int
field::plotmtv2d_P1d (
	const string& basename, bool execute, bool clean, bool verbose, 
	bool grid, bool put_domains, bool fill, bool elevation,
	bool label, size_type isovalue_table_zero_index) const
{
  char filename [1000];
  sprintf (filename, "%s.mtv", basename.c_str());
  ofstream mtv (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  //
  // header
  //
  plotmtv_2d_header (basename, *this, mtv, digits10, grid, put_domains, fill, label,
		  isovalue_table_zero_index);
  const geo& m = get_geo();
  geo::const_iterator iter_K = m.begin();
  geo::const_iterator last_K = m.end();
  geo::const_iterator_node p = m.begin_node(); // random access
  size_type K_idx = 0 ;
  while (iter_K != last_K)
    {
      const geo_element& K = *iter_K++;
      switch (K.name())
	{
	case 't' :
	case 'q' :
	  {
	    size_type n_dof = _V.get_basis().size(K.type()) ;
	    tiny_vector<size_type> dof(n_dof) ;
	    _V.set_dof(K, dof) ;
	    for (size_type i_loc_dof = 0; i_loc_dof < n_dof; i_loc_dof++)
	      mtv_point( mtv, _V.x_dof(K, i_loc_dof), at(dof(i_loc_dof)) ) ;
	    break ;
	  }
	default  :
	  break ;
	}
      mtv << endl;
      K_idx++ ;
    }
  if (put_domains) 
   {
      mtv << "$ DATA = CURVE2D" << endl;
      m.put_mtv_domains(mtv,1);
   }
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string option = (elevation ? "-3d " : "");
      option += (put_domains ? "-plotall " : ""); 
      string command = "plotmtv " + option + basename + ".mtv";
      if (verbose) {
	  command += " 1>&2";
          clog << "! " << command << endl;
      } else {
          command += " >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
int
field::plotmtv2d_P2 (
	const string& basename, bool execute, bool clean, bool verbose, 
	bool grid, bool put_domains, bool fill, bool elevation,
	bool label, size_type isovalue_table_zero_index) const
{
  string filename = basename + ".mtv";
  ofstream mtv (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  //
  // header
  //
  plotmtv_2d_header (basename, *this, mtv, digits10, grid, put_domains, fill, label,
		  isovalue_table_zero_index);
  //
  // plot isovalues
  //
  const geo& m = get_geo();
  geo::const_iterator iter_K = m.begin();
  geo::const_iterator last_K = m.end();
  geo::const_iterator_node p = m.begin_node(); // random access
  size_type K_idx = 0 ;
  while (iter_K != last_K)
    {
      const geo_element& K = *iter_K++;
      switch (K.name())
	{
	case 't' :
	  {
	    size_type n_dof = _V.get_basis().size(K.type()) ;
	    tiny_vector<size_type> dof(n_dof) ;
	    _V.set_dof(K, dof) ;
	      mtv_point( mtv, _V.x_dof(K, 0), at(dof(0)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 3), at(dof(3)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 5), at(dof(5)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 3), at(dof(3)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 1), at(dof(1)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 4), at(dof(4)) ) ;
	      mtv << endl;	      
	      mtv_point( mtv, _V.x_dof(K, 4), at(dof(4)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 2), at(dof(2)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 5), at(dof(5)) ) ;
	      mtv << endl;	      
	      mtv_point( mtv, _V.x_dof(K, 3), at(dof(3)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 4), at(dof(4)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 5), at(dof(5)) ) ;
	    break ;
	  }
	case 'q' :
	  {
	    size_type n_dof = _V.get_basis().size(K.type()) ;
	    tiny_vector<size_type> dof(n_dof) ;
	    _V.set_dof(K, dof) ;
	      mtv_point( mtv, _V.x_dof(K, 0), at(dof(0)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 4), at(dof(4)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 0), at(dof(0)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 7), at(dof(7)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 4), at(dof(4)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 1), at(dof(1)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 1), at(dof(1)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 5), at(dof(5)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 5), at(dof(5)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 2), at(dof(2)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 2), at(dof(2)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 6), at(dof(6)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 7), at(dof(7)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 3), at(dof(3)) ) ;
	      mtv << endl;
	      mtv_point( mtv, _V.x_dof(K, 8), at(dof(8)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 6), at(dof(6)) ) ;
	      mtv_point( mtv, _V.x_dof(K, 3), at(dof(3)) ) ;
	      mtv << endl;
	      break ;
	  }
	default  :
	  break ;
	}
      mtv << endl;
      K_idx++ ;
    }
  if (put_domains) 
   {
      mtv << "$ DATA = CURVE2D" << endl;
      m.put_mtv_domains(mtv,1);
   }
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string option = (elevation ? "-3d " : "");
      option += (put_domains ? "-plotall " : ""); 
      string command = "plotmtv " + option + basename + ".mtv";
      if (verbose) {
	  command += " 1>&2";
          clog << "! " << command << endl;
      } else {
          command += " >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
//
// plot 1D P1-field as elevation in 2D
//
int
field::plotmtv1d_P1 (
  const string& basename, 
  bool execute, bool clean, bool verbose) const
{
  string mtv_name = basename + ".mtv";
  ofstream mtv (mtv_name.c_str());
  if (verbose) clog << "! file \"" << mtv_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  //
  // header
  //
  mtv << "#!plotmtv\n"
      << "$ DATA = CURVE2D\n"
      << "% comment    = \"rheolef\"\n"
      << "% toplabel   = \"" << basename << "\"\n"
      << "% subtitle   = \"\"\n"
      << "% linecolor  = 1\n"
      ;
  const geo& g = get_geo();
  geo::const_iterator_node p = g.begin_node();
  geo::const_iterator last_K = g.end();
  for (geo::const_iterator iter_K = g.begin(); iter_K != last_K; iter_K++) {
      const geo_element& K = *iter_K;
      Float x0 = p[K[0]][0];
      Float x1 = p[K[1]][0];
      Float y0 = at(K[0]);
      Float y1 = at(K[1]);
      mtv << x0 << " " << y0 << endl
	  << x1 << " " << y1 << endl 
	  << endl;
  }
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string command;
      if (verbose) {
	  command = "plotmtv " + basename + ".mtv 1>&2";
          clog << "! " << command << endl;
      } else {
	  command = "plotmtv " + basename + ".mtv >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
//
// plot 1D P2-field as elevation in 2D
//
int
field::plotmtv1d_P2 (
  const string& basename, 
  bool execute, bool clean, bool verbose) const
{
  string mtv_name = basename + ".mtv";
  ofstream mtv (mtv_name.c_str());
  if (verbose) clog << "! file \"" << mtv_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  mtv << setprecision(digits10);
  //
  // header
  //
  mtv << "#!plotmtv\n"
      << "$ DATA = CURVE2D\n"
      << "% comment    = \"rheolef\"\n"
      << "% toplabel   = \"" << basename << "\"\n"
      << "% subtitle   = \"\"\n"
      << "% linecolor  = 1\n"
      ;
  const geo& g = get_geo();
  tiny_vector<size_type> dof;
  geo::const_iterator last_K = g.end();
  for (geo::const_iterator iter_K = g.begin(); iter_K != last_K; iter_K++) {
      const geo_element& K = *iter_K;
      _V.set_dof(K, dof) ;
      Float y0 = at(dof(0));
      Float ym = at(dof(2));
      Float y1 = at(dof(1));
      point p0 = _V.x_dof(K, 0);
      point pm = _V.x_dof(K, 2);
      point p1 = _V.x_dof(K, 1);
      mtv << p0[0] << " " << y0 << endl
	  << pm[0] << " " << ym << endl 
	  << endl
	  << pm[0] << " " << ym << endl 
	  << p1[0] << " " << y1 << endl 
	  << endl;
  }
  mtv.close();
  //
  // run plotmtv
  //
  int status = 0;
  if (execute) {
      string command;
      if (verbose) {
	  command = "plotmtv " + basename + ".mtv 1>&2";
          clog << "! " << command << endl;
      } else {
	  command = "plotmtv " + basename + ".mtv >/dev/null";
      }
      status = system (command.c_str());
  }
  //
  // clear plotmtv data
  //
  if (clean) {
      string command = "/bin/rm -f " + basename + ".mtv";
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
