///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// contents:
//      * graphic output: 
//         - plotmtv: 2d-P0, 2d-P1, 1d-P1
//         - vtk:     3d-P1
//      * cut by plane: get dim-1 field and geo
//        by projecting in 1d-P1 (not yet in 2d-P1)
//
// authors:
//      Nicolas.Roquet@imag.fr
//      Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 28 jan 1998
//
#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/iofem.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
#include "rheolef/field-visu.h" // visu options
using namespace std;
namespace rheolef { 

// available drivers are now listed here: 
// next step is to use dynamically loaded modules...
// ex. field_visu_driver (u, "mayavi", opt)
//     v = field_visu_filter (u, "cut", opt)
//
int field_visu_mayavi (const field&, const field_visu_option_type&);
//
// graphic suite = big switch !
//
ostream& 
operator << (ostream& s, const field& u)
{
  field_visu_option_type opt (s);

  // new renders treat directly all options
  if (opt.format [iorheo::mayavi]) {
    field_visu_mayavi (u, opt);
    return s;
  }

  if (u.n_component() <= 1) {
      opt.velocity    = false;
      opt.deformation = false;
  }
  field v;

  if (opt.iso) {
      geo g_iso = u.iso(opt.basename, opt.execute, opt.clean, opt.verbose, opt.isovalue);
      // graphic options are applied to geo instead of field:
      return s << g_iso;
  } 
  if (opt.cut) {

    if (u.get_approx() != "P1" &&
        u.get_approx() != "P2")
      error_macro("field: " << u.get_approx()
			    << " cut output not implemented.");
    v = u.cut (opt.basename, opt.execute, opt.clean, opt.verbose, opt.normal, opt.origin);
    opt.basename = v.get_geo().name();
  } else {
    v = u;
  }
  if (opt.velocity) {
    // vector-valued case
    if (opt.autovscale) opt.vscale*=u.get_geo().hmin()/euclidian_norm(u).max();
    if (opt.format [iorheo::gnuplot]) {
    	check_macro(v.dimension()==2,"Only 2D vector plots supported with gnuplot, use plotmtv instead");
	if (u.n_component() != v.dimension()) {
        	error_macro("field: unexpected "
                  << u.n_component() << " component field for "
	          << v.dimension()   << "D plotmtv velocity output.");
        }
	field::plot_options gopt;
	gopt.execute=opt.execute; gopt.clean=opt.clean; gopt.verbose=opt.verbose; 
	gopt.vscale=opt.vscale; gopt.meshplot=opt.grid;
        u.gnuplot2d_velocity(opt.basename, gopt);
    } else if (opt.format [iorheo::plotmtv]) {
        switch (v.dimension()) {
	  case 2:
	  case 3: {
  		if (u.n_component() != v.dimension()) {
                  error_macro("field: unexpected " 
	            << u.n_component() << " component field for "
		    << v.dimension()   << "D plotmtv velocity output.");
	        }
		u.plotmtv_velocity (opt.basename, opt.execute, opt.clean, opt.verbose, 
			opt.vscale, opt.domains);
	        return s;
	  } // 2d, 3d
	  default: { 
              error_macro("field: " 
	       << v.dimension() << "D plotmtv velocity output not implemented.");
          }
        } // swicth dimension
    } else if (opt.format [iorheo::vtk]) {

        switch (v.dimension()) {
	  case 2:
	  case 3: {
  		if (u.n_component() != v.dimension()) {
                  error_macro("field: unexpected " 
	            << u.n_component() << " component field for "
		    << v.dimension()   << "D vtk velocity output.");
	        }
		u.vtk_velocity (opt.basename, opt.execute, opt.clean, opt.verbose, opt.vscale);
	        return s;
	  } // 2d, 3d
	  default: { 
              error_macro("field: " 
	       << v.dimension() << "D vtk velocity output not implemented.");
          }
        } // swicth dimension

    } else {
          error_macro("field: velocity output format (fmt=" << opt.format << ") not implemented.");
    } // switch
    return s;
  } // velocity

  if (opt.deformation) {
    // vector-valued case
    if (opt.format [iorheo::plotmtv]) {
        switch (v.dimension()) {
	  case 2:
	  case 3: {
  		if (u.n_component() != v.dimension()) {
                  error_macro("field: unexpected " 
	            << u.n_component() << " component field for "
		    << v.dimension()   << "D plotmtv deformation output.");
	        }
		u.plotmtv_deformation_P1 (opt.basename, opt.execute, opt.clean, opt.verbose, opt.vscale);
	        return s;
	  } // 2d, 3d
	  default: { 
              error_macro("field: " 
	       << v.dimension() << "D plotmtv deformation output not implemented.");
          }
        } // swicth dimension
    
    } else if (opt.format [iorheo::vtk]) {

        switch (v.dimension()) {
	  case 2:
	  case 3: {
  		if (u.n_component() != v.dimension()) {
                  error_macro("field: unexpected " 
	            << u.n_component() << " component field for "
		    << v.dimension()   << "D vtk deformation output.");
	        }
		u.vtk_deformation (opt.basename, opt.execute, opt.clean, opt.verbose, opt.vscale);
	        return s;
	  } // 2d, 3d
	  default: { 
              error_macro("field: " 
	       << v.dimension() << "D vtk deformation output not implemented.");
          }
        } // swicth dimension

    } else {
          error_macro("field: deformation output format (fmt=" << opt.format << ") not implemented.");
    }
    return s;
  } // deformation

  // scalar-valued case
  if (opt.format [iorheo::gnuplot]) {
      switch (v.dimension()) {
	case 1:
	  v.gnuplot1d (opt.basename, opt.execute, opt.clean, opt.verbose);
	  return s;
	case 2:
	 {
	  field::plot_options gopt;
	  gopt.execute=opt.execute; gopt.clean=opt.clean; gopt.verbose=opt.verbose; 
	  v.gnuplot2d_elevation (opt.basename, gopt);
	  return s;
	 }
        default:
	      error_macro("field: " << v.dimension()
		<< "D gnuplot output not implemented.");
	      return s;
      } // switch dim
  
  } else if (opt.format [iorheo::plotmtv]) {

      switch (v.dimension()) {
	case 1: {
	  if (v.get_approx() == "P2") {
              v.plotmtv1d_P2 (opt.basename, opt.execute, opt.clean, opt.verbose);
	      return s;
	  } else if (v.get_approx() == "P1") {
              v.plotmtv1d_P1 (opt.basename, opt.execute, opt.clean, opt.verbose);
	      return s;
	  } else {
	      error_macro("field: " << v.get_approx()
		    << " 1D plotmtv output not implemented.");
	  } // 1d
	} 
	case 2: {
	  if (v.get_approx() == "P2") {
	      v.plotmtv2d_P2 (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid,
	      		opt.domains, opt.fill, opt.elevation, opt.label, opt.n_isovalue_negative);
	      return s;
	  } else if (v.get_approx() == "P1d") {
	      v.plotmtv2d_P1d (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid,
	      		opt.domains, opt.fill, opt.elevation,
			opt.label, opt.n_isovalue_negative);
	      return s;
	  } else if (v.get_approx() == "P1") {
	      v.plotmtv2d_P1 (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid,
	      		opt.domains, opt.fill, opt.elevation,
			opt.label, opt.n_isovalue_negative);
	      return s;
	  } else if (v.get_approx() == "P0") {
	      v.plotmtv2d_P0 (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid,
	      		opt.domains, opt.fill, opt.elevation);
	      return s;
	  } else {
	      error_macro("field: " << v.get_approx()
				    << " 2D plotmtv output not implemented.");
	  } // 2d
        }
        default: { 
            error_macro("field: " 
	     << v.dimension() << "D plotmtv output not implemented.");
          }
      } // swicth dimension
      return s;

  } else if (opt.format [iorheo::vtk]) {

      if (v.dimension() == 2) {
	if (v.get_approx() == "P1" ||
	    v.get_approx() == "P2") {
	    if (opt.elevation) {
	        v.vtk_elevation (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid, opt.fill);
	    } else {
	        v.vtk2d_Pk (opt.basename, opt.execute, opt.clean, opt.verbose, opt.grid, opt.fill);
	    }
	} else {
	  error_macro("field: " << v.get_approx()
			        << " 2D vtk output not implemented.");
	}
      } else {
	if (v.get_approx() == "P1" ||
	    v.get_approx() == "P2") {
	  v.vtk3d_Pk (opt.basename, opt.execute, opt.clean, opt.verbose);
	} else {
	  error_macro("field: " << v.get_approx()
			        << " 3D vtk output not implemented.");
	}
      }
      return s;
 
  } else if (opt.format [iorheo::bamg])     { return v.put_bamg_metric(s); 
  } else if (opt.format [iorheo::mmg3d])    { return v.put_mmg3d_metric(s); 
  } else if (opt.format [iorheo::gmsh])     { return v.put_gmsh(s); 
  } else if (opt.format [iorheo::cemagref]) { return v.put_cemagref(s); 
  } else if (opt.format [iorheo::rheo]) { 
        if (opt.cut) {
          // case of a cut by plane and text output: save also cutted mesh
          orheostream os (opt.basename, "geo");
	  os << rheo << v.get_geo();
        }
        return v.put(s);

  } else {

      error_macro("field: output format (fmt=" << opt.format << ") not implemented.");
      return s;
  }
  return s;
}
// ===========================================================================
// field_visu_option_type members
// ===========================================================================
field_visu_option_type::field_visu_option_type (ostream& s) {

     format  = iorheo::flags(s) & iorheo::format_field;

     execute     = iorheo::getexecute(s);
     clean       = iorheo::getclean(s);
     verbose     = iorheo::getverbose(s);
     
     basename = iorheo::getbasename(s);
     if (basename.length() == 0) basename = "output";
       
     color           = iorheo::getcolor(s);
     gray            = iorheo::getgray(s);
     black_and_white = iorheo::getblack_and_white(s);
     cut         = iorheo::getcut(s);
     iso         = iorheo::getiso(s);
     fill        = iorheo::getfill(s);
     stereo      = iorheo::getstereo(s);
     grid        = iorheo::getgrid(s);
     domains	 = iorheo::getdomains(s);
     volume      = iorheo::getvolume(s);
     elevation   = iorheo::getelevation(s);
     velocity    = iorheo::getvelocity(s);
     deformation = iorheo::getdeformation(s);
     label       = iorheo::getlabel(s);
   
     isovalue = iorheo::getisovalue(s);
     n_isovalue = iorheo::getn_isovalue(s);
     n_isovalue_negative = iorheo::getn_isovalue_negative(s);
     vscale     = iorheo::getvectorscale(s);
     autovscale = iorheo::getautovscale(s);

     origin  = iofem::getorigin(s);
     normal  = iofem::getnormal(s);
}
void
field_visu_option_type::dump (ostream& os) const {
    os << "dump field_visu_option_type:" << endl
       << "  format	" << format << endl
       << "   iorheo::vtk "    << iorheo::vtk    << endl
       << "   iorheo::mayavi " << iorheo::mayavi << endl
       << "  execute	" << execute << endl
       << "  clean	" << clean << endl
       << "  verbose	" << verbose << endl
       << "  basename	" << basename << endl
       << "  cut	" << cut << endl
       << "  iso	" << iso << endl
       << "  fill	" << fill << endl
       << "  grid	" << grid << endl
       << "  domains	" << domains << endl
       << "  volume	" << volume << endl
       << "  elevation	" << elevation << endl
       << "  velocity	" << elevation << endl
       << "  deformation " << deformation << endl
       << "  label	" << label << endl
       << "  n_isovalue	" << n_isovalue_negative << endl
       << "  n_isovalue_negative	" << n_isovalue_negative << endl
       << "  isovalue	" << isovalue << endl
       << "  vscale	" << vscale << endl
       << "  autovscale	" << autovscale << endl
       << "  origin     " << origin << endl
       << "  normal     " << normal << endl
       << endl;
}
}// namespace rheolef
