# ifndef _RHEO_FORM_H
# define _RHEO_FORM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/skit.h"
#include "rheolef/csr.h"
#include "rheolef/space.h"
#include "rheolef/form_diag.h"
namespace rheolef { 

#define form Form /* otherwise, conflict with GNU C++ in <stream.h> */

/*Class:form
NAME:  @code{form} - representation of a finite element operator
DESCRIPTION:       
  The form class groups four sparse matrix, associated to 
  a bilinear form on finite element space.
  This class is represented by four sparse matrix,
  associated to unknown and blocked degrees of freedom 
  of origin and destination spaces
  (@pxref{space class}).

EXAMPLE:
  The operator A associated to a bilinear form a(.,.) by
  the relation (Au,v) = a(u,v) could be applied by using
  a sample matrix notation A*u, as shown by the following code:
  @example
      geo m("square");
      space V(m,"P1");
      form a(V,V,"grad_grad");
      field x(V);
      x = fct;
      field y = a*x;
      cout << plotmtv << y;
  @end example

  This block-matrix operation is equivalent to:
  @example
     y.u = a.uu*x.u + a.ub*x.b;
     y.b = a.bu*x.u + a.bb*x.b;
  @end example

SEE ALSO: class "space", class "field", class "csr"
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Nicolas.Roquet@imag.fr
   | Pierre.Saramito@imag.fr
DATE:   2 july 1997
METHODS: @form
End:
*/

//<form:  
class form {
public :
// typedefs:

    typedef csr<Float>::size_type size_type;

// allocator/deallocator:

    form ();
    form (const space& X, const space& Y);
    	//! locked_boundaries means that special vector BCs are applied, 
	//! see space::lock_components()
    form (const space& X, const space& Y, const std::string& op_name, 
    	bool locked_boundaries=false);
    form (const space& X, const space& Y, const std::string& op_name, const domain& d);
	//! Currently weighted forms in P2 spaces use a quadrature formula which is
	//! limited to double precision floats.
    form (const space& X, const space& Y, const std::string& op_name, const field& wh,
	bool use_coordinate_system_weight=true);
    form (const space& X, const space& Y, const std::string& op_name, 
	const domain& d, const field& wh, bool use_coordinate_system_weight=true);
    form (const space& X, const std::string& op_name);
    form (const form_diag& X);

// accessors:

    const geo&   get_geo() const;
    const space& get_first_space() const;
    const space& get_second_space() const;
    size_type nrow() const;
    size_type ncol() const;
    bool for_locked_boundaries() const { return _for_locked_boundaries; } 

// linear algebra:

    Float operator () (const class field&, const class field&) const;
    friend class field operator * (const form& a, const class field& x);
    field trans_mult (const field& x) const;
    friend form trans(const form&); 
    friend form operator * (const Float& lambda, const form&);
    friend form operator + (const form&, const form&);
    friend form operator - (const form&);
    friend form operator - (const form&, const form&);
    friend form operator * (const form&, const form&);
    friend form operator * (const form&, const form_diag&);
    friend form operator * (const form_diag&, const form&);

// input/output:

    friend std::ostream& operator << (std::ostream& s, const form& a);
    friend class form_manip;

// data
protected:
    space      X_;
    space      Y_;
    bool       _for_locked_boundaries;
public :
    csr<Float> uu;
    csr<Float> ub;
    csr<Float> bu;
    csr<Float> bb;
};
form form_nul(const space& X, const space& Y) ;
//>form:

// also compatibility with field_component:
//    v = a*u[1]
class field operator * (const form& a, const class field_component& x);

// ------------ inline'd -----------------------------------

inline
form::form ()
: X_(), Y_(), _for_locked_boundaries(false), uu(), ub(), bu(), bb()
{}
inline 
form::form (const space& X, const space& Y)
: X_(X), Y_(Y), _for_locked_boundaries(false), uu(), ub(), bu(), bb()
{
    X_.freeze();
    Y_.freeze();
}
inline
const space& 
form::get_first_space() const
{
  return X_;
}
inline
const space& 
form::get_second_space() const
{
  return Y_;
}
inline
const geo&
form::get_geo() const 
{
    return X_.get_geo();
}
inline
form::size_type
form::nrow() const 
{
    return Y_.size();
}
inline
form::size_type
form::ncol() const 
{
    return X_.size();
}
}// namespace rheolef
# endif /* _RHEO_FORM_H */
