///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// visualisation of geometries using vtk
//
// authors:
//    	Pierre.Saramito@imag.fr
//
// date: 12 may 1997; changes 4 may 2001
//
#include "rheolef/georep.h"
#include "rheolef/geo-visu-aux.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
using namespace std;
namespace rheolef { 

#ifdef _COMMENT_
// from vtk-3.1.2/common/vtkCellType.h:
#define VTK_EMPTY_CELL 0
#define VTK_VERTEX 1
#define VTK_POLY_VERTEX 2
#define VTK_LINE 3
#define VTK_POLY_LINE 4
#define VTK_TRIANGLE 5
#define VTK_TRIANGLE_STRIP 6
#define VTK_POLYGON 7
#define VTK_PIXEL 8
#define VTK_QUAD 9
#define VTK_TETRA 10
#define VTK_VOXEL 11
#define VTK_HEXAHEDRON 12
#define VTK_WEDGE 13
#define VTK_PYRAMID 14

#define VTK_PARAMETRIC_CURVE 51
#define VTK_PARAMETRIC_SURFACE 52
#endif // _COMMENT_

static
size_t
vtk_cell_type (const geo_element& K) {
  switch (K.name()) {
    case 'p' : return 1; break;
    case 'e' : return 3; break;
    case 't' : return 5; break;
    case 'q' : return 9; break;
    case 'T' : return 10; break;
    case 'H' : return 12; break;
    case 'P' : return 13; break; // i.e. prism = wedge for vtk
    default : {
	   error_macro ("cell type `" << K.name() << "' not yet implemented in VTK format -- sorry.");
	   return 0;
        }
    }
}
// -----------------------------------------------------------------------------
// get/put vtk file format
// -----------------------------------------------------------------------------

istream&
georep::get_vtk_polydata (istream& poly)
{
    bool verbose    = iorheo::getverbose(clog);
    bool clean      = iorheo::getclean(clog);
    string basename = iorheo::getbasename(poly);

    _name    = basename;
    _dim     = 3;
    _version = 1;
    //
    // load the coordinates
    // 		POINTS <np> float
    //          {xi yi zi} i=0..np-1
    size_type np;
    if (!scatch(poly,"POINTS"))
       error_macro ("geo: unexpected vtk polydata input file.");
    poly >> np;
    _x.resize(np);
    scatch(poly,"float");
    get_nodes (poly, begin_node(), end_node(), _xmin, _xmax, 3);
    //
    // load the domain connectivity
    //	  LINES <ne> <ntot>
    //    2  {s1 s2} j=0..ne-1
    // or
    //    POLYGONS <ne> <ntot>
    //    p  {s1 .. sp} j=0..ne-1
    //
    string mark;
    poly >> ws >> mark;
    if (mark == "VERTICES") { // skip unused paragraph in polydata v3.0
        size_type n, n2, d, idx;
        poly >> n >> n2;
        for (size_type i = 0; i < n; i++) poly  >> d >> idx;
        poly >> ws >> mark;
    }
    size_type dom_dim;
    if (mark == "POLYGONS")      dom_dim = 2;
    else if (mark == "LINES")    dom_dim = 1;
    else error_macro ("geo: unexpected `" << mark 
	 << "' in vtk polydata input.");
    //
    // third, load the domain connectivity
    //
    reset_counters ();
    _count_geo     [0] = n_node();
    _count_element [0] = n_node();
    size_type ne, ntot;
    poly >> ne >> ntot;
    resize(ne);
    georep::iterator iter_e = begin();
    georep::iterator last_e = end();
    while (poly.good() && iter_e != last_e) {
        size_type p;
        poly >> ws >> p;
        (*iter_e).set_type(p, dom_dim);
        for (size_type i = 0 ; i < p; i++)
            poly >> (*iter_e)[i] ;
        _count_geo     [(*iter_e).dimension()]++;
        _count_element [(*iter_e).type()]++;
        ++iter_e;
    }
    return poly;
}
// tested only in 2d:
ostream&
georep::put_vtk_polydata (ostream& out) const
{
    check_macro (dimension() <= 2, "vtk polydata does not support 3d mesh");

    out << setprecision(15)
        << "# vtk DataFile Version 1.0" << endl
        << "automatically generated by rheolef" << endl
        << "ASCII" << endl
	<< "DATASET POLYDATA" << endl
	<< "POINTS " << n_node() << " float" << endl;

    size_type d = dimension();
    for (const_iterator_node iter_p = begin_node(); iter_p != end_node(); iter_p++) {
        out << *iter_p << endl ;
    }
    size_type n_line   =   n_triangle() +   n_quadrangle();
    size_type n_value  = 3*n_triangle() + 4*n_quadrangle() + n_line;
    out << "POLYGONS " << n_line << " " << n_value << endl;
    for (const_iterator i = begin(); i != end(); i++) {
        const geo_element& K = *i;
	if (K.size() != 3 && K.size() != 4) continue;
	out << K.size();
	for (size_type j = 0; j < K.size(); j++) {
            out << " " << K[j];
	}
        out << endl;
    }
    out << endl;
    return out;
}
ostream& georep::put_vtk (ostream& vtk, bool lattice) const
{
  if (lattice) {
      return put_vtk_P2_iso_P1 (vtk);
  }
  vtk << "# vtk DataFile Version 1.0\n";
  vtk << "Unstructured Grid\n";
  vtk << "ASCII\n";
  vtk << endl;
  vtk << "DATASET UNSTRUCTURED_GRID\n";
  vtk << "POINTS " << n_node() << " float\n";
  vtk << endl;

  const_iterator_node iter_p = begin_node();
  const_iterator_node last_p = end_node();
  while (iter_p != last_p)
      vtk << *iter_p++ << endl;
  vtk << endl;
  
  size_type nb_connectivity_entries = size();
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      nb_connectivity_entries += (*iter_e++).size();

  vtk << "CELLS " << size() << " " << nb_connectivity_entries << endl;
  iter_e = begin();
  while (iter_e != last_e) {
      const geo_element& e = *iter_e++;
      vtk << e.size() << "\t" ;
      for (size_type j=0 ; j < e.size(); j++)
	vtk << e[j] << " " ;
      vtk << endl ;
  }
  vtk << endl;
  vtk << "CELL_TYPES " << size() << endl;
  iter_e = begin();
  while (iter_e != last_e) {
      vtk << vtk_cell_type(*iter_e++) << endl;
  }
  return vtk;
}
ostream& georep::put_vtk_P2_iso_P1 (ostream& vtk) const
{
  may_have_version_2();
  vtk << "# vtk DataFile Version 1.0\n";
  vtk << "Unstructured Grid\n";
  vtk << "ASCII\n";
  vtk << endl;
  vtk << "DATASET UNSTRUCTURED_GRID\n";
  size_type n_internal = 0;
  if (dimension() == 2) {
    n_internal = n_quadrangle();
  } if (dimension() == 3) {
    n_internal = n_prism() + n_hexaedra();
  }
  vtk << "POINTS " << n_node() + n_edge() + n_internal << " float\n";
  vtk << endl;

  // put nodes
  for (const_iterator_node i = begin_node(); i != end_node(); i++)
      vtk << *i << endl;
  vtk << endl;

  // put middle-edge nodes
  vector<size_type> v1(n_edge());
  vector<size_type> v2(n_edge());
  for (const_iterator i = begin(); i != end(); i++) {
      const geo_element& K = *i;
      for (size_type i = 0; i < K.n_edge(); i++) {
         v1 [K.edge(i)] = K.subgeo_vertex (1, i, 0);
         v2 [K.edge(i)] = K.subgeo_vertex (1, i, 1);
      }
  }
  const_iterator_node p = begin_node();
  for (size_type j = 0; j < n_edge(); j++) {
      vtk << 0.5*(p[v1[j]] + p[v2[j]]) << "\n" ;
  }
  // put internal-nodes
  if (dimension() == 2 && n_quadrangle() != 0) {
    for (const_iterator i = begin(); i != end(); i++) {
      const geo_element& K = *i;
      if (K.type() != geo_element::q) continue;
      vtk << 0.25*(p[K[0]] + p[K[1]] + p[K[2]] + p[K[3]]) << "\n" ;
    }
  }
  vtk << endl ;

  // count number of elements and entries
  size_type sz    = 0;
  size_type ndata = 0;
  for (const_iterator i = begin(); i != end(); i++) {
      const geo_element& K = *i;
      switch (K.name()) {
	case 'p' : sz += 1; ndata +=  2; break;  // 1 p
	case 'e' : sz += 2; ndata +=  6; break;  // 2 e
	case 't' : sz += 4; ndata += 16; break;  // 4 t
	case 'q' : sz += 4; ndata += 20; break;  // 4 q
	case 'T' : sz += 8; ndata += 40; break;  // 8 T
        default  : error_macro ("P2-iso-P1: element `" 
			<< K.name() << "' not supported -- sorry.");
                   break;
     }
  }
  vtk << "CELLS " << sz << " " << ndata << endl;
  size_type nn = n_node();
  size_type ne = n_edge();
  size_type E [12];
  for (const_iterator i = begin(); i != end(); i++) {
      const geo_element& K = *i;
      for (size_type j = 0; j < K.n_edge(); j++) {
	E [j] = nn + K.edge(j);
      }
      switch (K.name()) {
	case 'p' : {
	    vtk << "1\t" << K[0] << endl;
	    break;
	}
	case 'e' : { // 2 e
	    vtk << "2\t" << K[0] << " " << E[0] << endl
	        << "2\t" << E[0] << " " << K[1] << endl;
	    break;
	}
	case 't' : { // 4 t
	    vtk << "3\t" << K[0] << " " << E[0] << " " << E[2] << endl
	        << "3\t" << K[1] << " " << E[1] << " " << E[0] << endl
	        << "3\t" << K[2] << " " << E[2] << " " << E[1] << endl
	        << "3\t" << E[0] << " " << E[1] << " " << E[2] << endl;
	    break;
	}
	case 'q' : { // 4 q
	    // Hyp: triangle are numbered first; create internal node
            size_type I0 = nn + ne + K.index() - n_triangle();
	    vtk << "4\t" << K[0] << " " << E[0] << " " << I0 << " " << E[3] << endl
	        << "4\t" << K[1] << " " << E[1] << " " << I0 << " " << E[0] << endl
	        << "4\t" << K[2] << " " << E[2] << " " << I0 << " " << E[1] << endl
	        << "4\t" << K[3] << " " << E[3] << " " << I0 << " " << E[2] << endl;
	    break;
	}
	case 'T' : { // 8 T (creates internal edge E2-E4)
	    vtk << "4\t" << K[0] << " " << E[0] << " " << E[2] << " " << E[3] << endl
	        << "4\t" << K[1] << " " << E[1] << " " << E[0] << " " << E[4] << endl
	        << "4\t" << K[2] << " " << E[2] << " " << E[1] << " " << E[5] << endl
	        << "4\t" << K[3] << " " << E[4] << " " << E[3] << " " << E[5] << endl
	        << "4\t" << E[2] << " " << E[4] << " " << E[0] << " " << E[1] << endl
	        << "4\t" << E[2] << " " << E[4] << " " << E[3] << " " << E[0] << endl
	        << "4\t" << E[2] << " " << E[4] << " " << E[5] << " " << E[3] << endl
	        << "4\t" << E[1] << " " << E[4] << " " << E[5] << " " << E[2] << endl
		;
	    break;
	}
        default : break;
      }
  }
  vtk << endl;
  vtk << "CELL_TYPES " << sz << endl;
  for (const_iterator i = begin(); i != end(); i++) {
      const geo_element& K = *i;
      size_type ncell     = 0;
      size_type cell_type = 0;
      switch (K.name()) {
	case 'p' : ncell = 1; cell_type = 1; break;
	case 'e' : ncell = 2; cell_type = 3; break;
	case 't' : ncell = 4; cell_type = 5; break;
	case 'q' : ncell = 4; cell_type = 9; break;
	case 'T' : ncell = 8; cell_type = 10; break;
        default : break;
      }
      for (size_type k = 0; k < ncell; k++) vtk << cell_type << endl;
  }
  vtk << endl;
  return vtk;
}
// slit each element as a local shrink: nodes are de-multiplied
// this is the only solution to render general P1d solutions with vtk
ostream& georep::put_vtk_P1d (ostream& vtk) const
{
  size_type np = 0;
  for (const_iterator iter_e = begin(); iter_e != end(); iter_e++) {
      np += (*iter_e).size();
  }
  vtk << "# vtk DataFile Version 1.0" << endl
      << "Unstructured Grid" << endl
      << "ASCII" << endl
      << endl
      << "DATASET UNSTRUCTURED_GRID" << endl
      << "POINTS " << np << " float" << endl
      << endl;

  const_iterator_node x = begin_node();
  for (const_iterator iter_e = begin(); iter_e != end(); iter_e++) {
      const geo_element& K = *iter_e;
      for (size_type j=0 ; j < K.size(); j++) {
          vtk << x[K[j]] << endl;
      }
  }
  vtk << endl;
  
  size_type nb_connectivity_entries = np + size();
  vtk << "CELLS " << size() << " " << nb_connectivity_entries << endl;
  size_t ip = 0;
  for (const_iterator iter_e = begin(); iter_e != end(); iter_e++) {
      const geo_element& K = *iter_e;
      vtk << K.size() << "\t";
      for (size_type j=0 ; j < K.size(); j++) {
	vtk << ip++ << " ";
      }
      vtk << endl ;
  }
  vtk << endl;
  vtk << "CELL_TYPES " << size() << endl;
  for (const_iterator iter_e = begin(); iter_e != end(); iter_e++) {
      vtk << vtk_cell_type(*iter_e) << endl;
  }
  return vtk;
}
// -----------------------------------------------------------------------------
// run vtk drivers
// -----------------------------------------------------------------------------
static const point vtk_color [] = {
	point( 1, 0.49, 0.25), // flesh
	point( 1, 0, 0),  // red
	point( 0, 1, 0),  // green
	point( 0, 0, 1),  // blue
        point( 1, 1, 0),  // yellow
	point( 1, 0, 1),  // magenta
	point( 0, 1, 1),  // cyan
	point(0.8900, 0.1500, 0.2100),  // alizarin_crimson
	point(0.6400, 0.5800, 0.5000),  // beige
	point(1.0000, 0.3800, 0.0100)  // cadmium_orange
};
static const georep::size_type vtk_n_color = sizeof(vtk_color)/sizeof(point);

static 
inline
string 
vtk_text_position(georep::size_type vtk_i_color) {
    return "10 " + itos(20*(vtk_i_color+1));
}
static void make_tcl_actor (ostream& tcl, const char* x,
                  const string& basename, georep::size_type color_idx, 
                  bool execute, bool clean, bool verbose, 
                  bool fill, bool shrink, bool tube, 
		  bool ball, bool full, bool cut,
		  const point& origin, const point& normal)
{
  bool use_edge_actor = full || tube;
  bool use_std_actor = fill || !use_edge_actor;
  bool use_transparency    = use_std_actor && use_edge_actor;
  bool has_origin = (origin[0] != numeric_limits<Float>::max());

  if (color_idx >= vtk_n_color) {
    warning_macro ("not enough colors for domain #" << color_idx
	<< "`" << x << "': skipping");
    return;
  }
  tcl << "#\n";
  tcl << "# load " << x << "\n";
  tcl << "#\n";
  tcl << "vtkUnstructuredGridReader " << x << "_reader;\n";
  tcl << "    " << x << "_reader SetFileName \"" << basename << ".vtk\"\n";
  tcl << "    " << x << "_reader Update;\n";
  if (cut) {
    if (strcmp (x, "mesh") == 0) {
      tcl << "#\n";
      tcl << "# build the cutting plane\n";
      tcl << "#\n";
      tcl << "vtkPlane plane;\n";
      if (has_origin)
          tcl << "    eval plane SetOrigin $origin;\n";
      else
	  tcl << "    eval plane SetOrigin [[mesh_reader GetOutput] GetCenter];\n";
      tcl << "    eval plane SetNormal $normal;\n";
    } 
    tcl << "#\n";
    tcl << "# " << x << " in the cutting plane\n";
    tcl << "#\n";
    tcl << "vtkCutter " << x << "_plane_cut;\n";
    tcl << "    " << x << "_plane_cut SetInput [" << x << "_reader GetOutput];\n";
    tcl << "    " << x << "_plane_cut SetCutFunction plane;\n";
    tcl << "vtkDataSetMapper " << x << "_cut_mapper;\n";
    tcl << "    " << x << "_cut_mapper SetInput [" << x << "_plane_cut GetOutput];\n";
    tcl << "vtkActor " << x << "_cut_actor;\n";
    tcl << "    " << x << "_cut_actor SetMapper " << x << "_cut_mapper;\n";
    tcl << "    eval [" << x << "_cut_actor GetProperty] SetColor " << vtk_color [color_idx] << ";\n";
    tcl << "    [" << x << "_cut_actor GetProperty] SetRepresentationToWireframe;\n";
  }
  if (use_std_actor) {
    tcl << "#\n";
    tcl << "# " << x << " volume\n";
    tcl << "#\n";
    if (!shrink) {
      if (!cut) {
        tcl << "vtkDataSetMapper " << x << "_map_std;\n"
            << "    " << x << "_map_std SetInput [" << x << "_reader GetOutput];\n";
      } else {
        tcl << "vtkGeometryFilter " << x << "_polydata\n"
            << "    " << x << "_polydata SetInput [" << x << "_reader GetOutput];\n"
            << "vtkClipPolyData " << x << "_clipper;\n"
            << "    " << x << "_clipper SetInput [" << x << "_polydata GetOutput];\n"
            << "    " << x << "_clipper SetClipFunction plane;\n"
            << "vtkDataSetMapper " << x << "_map_std;\n"
            << "    " << x << "_map_std SetInput [" << x << "_clipper GetOutput];\n";
      }
    } else {
        tcl << "vtkShrinkFilter " << x << "_shrink;\n";
        tcl << "    " << x << "_shrink SetInput [" << x << "_reader GetOutput];\n";
        tcl << "    " << x << "_shrink SetShrinkFactor 0.9;\n";
      if (!cut) {
        tcl << "vtkDataSetMapper " << x << "_map_std;\n";
        tcl << "    " << x << "_map_std SetInput [" << x << "_shrink GetOutput];\n";
      } else {
        tcl << "vtkClipPolyData " << x << "_clip_std;\n";
        tcl << "    " << x << "_clip_std SetInput [" << x << "_shrink GetOutput];\n";
        tcl << "    " << x << "_clip_std SetClipFunction plane;\n";
        tcl << "    " << x << "_clip_std SetValue 0.0;\n";
        tcl << "vtkDataSetMapper " << x << "_map_std;\n";
        tcl << "    " << x << "_map_std SetInput [" << x << "_clip_std GetOutput];\n";
      }
    }
    tcl << "vtkActor " << x << "_std_actor;\n";
    tcl <<   "    " << x << "_std_actor SetMapper " << x << "_map_std;\n";
    tcl << "    eval [" << x << "_std_actor GetProperty] SetColor " << vtk_color [color_idx] << ";\n";
    if (!fill) {
      tcl << "    [" << x << "_std_actor GetProperty] SetRepresentationToWireframe;\n";
    }
    if (use_transparency) {
      tcl << "    [" << x << "_std_actor GetProperty] SetOpacity 0.1;\n";
    }
  }
  if (use_edge_actor) {
    tcl << "#\n";
    tcl << "# " << x << " edge\n";
    tcl << "#\n";
    tcl << "vtkExtractEdges " << x << "_edges;\n";
    tcl << "    " << x << "_edges SetInput [" << x << "_reader GetOutput];\n";
    if (!tube) {
      if (!cut) {
        tcl << "vtkPolyDataMapper " << x << "_map_edges;\n";
        tcl << "    " << x << "_map_edges SetInput [" << x << "_edges GetOutput];\n";
      } else {
        tcl << "vtkClipPolyData " << x << "_clip_edges;\n";
        tcl << "    " << x << "_clip_edges SetInput [" << x << "_edges GetOutput];\n";
        tcl << "    " << x << "_clip_edges SetClipFunction plane;\n";
        tcl << "    " << x << "_clip_edges SetValue 0.0;\n";
        tcl << "vtkDataSetMapper " << x << "_map_edges;\n";
        tcl << "    " << x << "_map_edges SetInput [" << x << "_clip_edges GetOutput];\n";
      }
    } else {
      tcl << "vtkTubeFilter " << x << "_tubes;\n";
      tcl << "    " << x << "_tubes SetInput [" << x << "_edges GetOutput];\n";
      tcl << "    " << x << "_tubes SetRadius 0.02;\n";
      tcl << "    " << x << "_tubes SetNumberOfSides 6;\n";
      if (!cut) {
        tcl << "vtkPolyDataMapper " << x << "_map_edges;\n";
        tcl << "    " << x << "_map_edges SetInput [" << x << "_tubes GetOutput];\n";
      } else {
        tcl << "vtkClipPolyData " << x << "_clip_edges;\n";
        tcl << "    " << x << "_clip_edges SetInput [" << x << "_tubes GetOutput];\n";
        tcl << "    " << x << "_clip_edges SetClipFunction plane;\n";
        tcl << "    " << x << "_clip_edges SetValue 0.0;\n";
        tcl << "vtkDataSetMapper " << x << "_map_edges;\n";
        tcl << "    " << x << "_map_edges SetInput [" << x << "_clip_edges GetOutput];\n";
      }
    }
    tcl << "vtkActor " << x << "_edge_actor\n";
    tcl << "    " << x << "_edge_actor SetMapper " << x << "_map_edges;\n";
    if (!tube) {
      tcl << "    eval [" << x << "_edge_actor GetProperty] SetColor " << vtk_color[color_idx] << ";\n";
    } else {
      tcl << "    eval [" << x << "_edge_actor GetProperty] SetColor 0.2 0.63 0.79; # peacock\n";
      tcl << "    [" << x << "_edge_actor GetProperty] SetSpecularColor 1 1 1;\n";
      tcl << "    [" << x << "_edge_actor GetProperty] SetSpecular 0.3;\n";
      tcl << "    [" << x << "_edge_actor GetProperty] SetSpecularPower 20;\n";
      tcl << "    [" << x << "_edge_actor GetProperty] SetAmbient 0.2;\n";
      tcl << "    [" << x << "_edge_actor GetProperty] SetDiffuse 0.8;\n";
    }
  }
  if (ball) {
    tcl << "#\n";
    tcl << "# " << x << " vertices\n";
    tcl << "#\n";
    tcl << "vtkSphereSource " << x << "_sphere;\n";
    tcl << "  " << x << "_sphere SetRadius 0.080;\n";
    tcl << "  " << x << "_sphere SetThetaResolution 12;\n";
    tcl << "  " << x << "_sphere SetPhiResolution 12;\n";
    tcl << "vtkGlyph3D " << x << "_ball;\n";
    tcl << "  " << x << "_ball SetInput [" << x << "_reader GetOutput];\n";
    tcl << "  " << x << "_ball SetSource [" << x << "_sphere GetOutput];\n";
    if (!cut) {
      tcl << "vtkPolyDataMapper " << x << "_sphere_mapper;\n";
      tcl << "  " << x << "_sphere_mapper SetInput [" << x << "_ball GetOutput];\n";
    } else {
      tcl << "vtkClipPolyData " << x << "_clip_ball;\n";
      tcl << "    " << x << "_clip_ball SetInput [" << x << "_ball GetOutput];\n";
      tcl << "    " << x << "_clip_ball SetClipFunction plane;\n";
      tcl << "    " << x << "_clip_ball SetValue 0.0;\n";
      tcl << "vtkPolyDataMapper " << x << "_sphere_mapper;\n";
      tcl << "  " << x << "_sphere_mapper SetInput [" << x << "_clip_ball GetOutput];\n";
    }
    tcl << "vtkActor " << x << "_ball_actor;\n";
    tcl << "  " << x << "_ball_actor SetMapper " << x << "_sphere_mapper;\n";
    tcl << "  eval [" << x << "_ball_actor GetProperty] SetColor 1 0.4118 0.7059; # hot_pink;\n";
    tcl << "  [" << x << "_ball_actor GetProperty] SetSpecularColor 1 1 1;\n";
    tcl << "  [" << x << "_ball_actor GetProperty] SetSpecular 0.3;\n";
    tcl << "  [" << x << "_ball_actor GetProperty] SetSpecularPower 20;\n";
    tcl << "  [" << x << "_ball_actor GetProperty] SetAmbient 0.2;\n";
    tcl << "  [" << x << "_ball_actor GetProperty] SetDiffuse 0.8;\n";
  }
  if (use_std_actor) {
    tcl << "ren1 AddActor " << x << "_std_actor;\n";
  }
  if (use_edge_actor) {
    tcl << "ren1 AddActor " << x << "_edge_actor;\n";
  }
  if (ball) {
    tcl << "ren1 AddActor " << x << "_ball_actor;\n";
  }
  if (cut) {
    tcl << "ren1 AddActor " << x << "_cut_actor;\n";
  }
  tcl << "#\n"
      << "# " << x << " text\n"
      << "#\n"
      << "vtkTextMapper " << x << "_text_mapper;\n"
      << "    " << x << "_text_mapper SetInput \"" << x << "\";\n"

      //
      // Version 4.2 of VTK introduces several modifications
      // on a couple of classes. Now for instance, fonts are
      // managed with a single class. In order to stay compatible
      // with older and newer versions, we use the code below
      //

      << "    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {\n"
      << "        vtkTextProperty " << x << "_text_property\n"
      << "        " << x << "_text_property SetFontFamilyToTimes;\n"
      << "        " << x << "_text_property SetFontSize $font_size;\n"
      << "        " << x << "_text_mapper SetTextProperty " << x << "_text_property\n"
      << "    } else {\n"
      << "        " << x << "_text_mapper SetFontFamilyToTimes;\n"
      << "        " << x << "_text_mapper SetFontSize $font_size;\n"
      << "    }\n"
      << "vtkActor2D " << x << "_text_actor;\n"
      << "    " << x << "_text_actor SetMapper " << x << "_text_mapper;\n"
      << "    [" << x << "_text_actor GetProperty] SetColor " << vtk_color [color_idx] << ";\n"
      << "    " << x << "_text_actor SetPosition " << vtk_text_position (color_idx) << ";\n"
      << "ren1 AddActor2D " << x << "_text_actor;\n"
    ;
}
int georep::vtk_tcl (const string& basename, bool execute, bool clean, 
		  bool verbose, bool fill, bool shrink, bool tube, 
		  bool ball, bool full, bool cut, bool split,
		  const point& origin, const point& normal) const
{
  bool has_origin = (origin[0] != numeric_limits<Float>::max());
  char filename [1000];
  //
  // output geometry for vtkUnstructuredGridReader
  //
  sprintf (filename, "%s.vtk", basename.c_str());
  ofstream vtk (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  if (!split) {
	put_vtk (vtk);
  } else {
	put_vtk_P2_iso_P1 (vtk);
  }
  vtk.close();
  //
  // plot domains
  //
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  while (iter != last) {
      const domain& dom = *iter;
      sprintf (filename, "%s.%s.vtk", basename.c_str(), dom.name().c_str());
      vtk.open(filename);
      if (verbose) clog << "! file \"" << filename << "\" created.\n";
      dom.put_vtk (vtk, begin_node(), end_node());
      vtk.close();
      ++iter;
  }
  //
  // output tcl script for vtk executable
  //
  sprintf (filename, "%s.tcl", basename.c_str());
  ofstream tcl (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  tcl << setprecision(digits10);
 
  tcl << "# this is a tcl version of " << basename << ".vtk visualization\n";
  tcl << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_interactor.tcl\"\n";
  tcl << endl;
  tcl << "#\n";
  tcl << "# edit parameters\n";
  tcl << "#\n";
  if (cut && has_origin) {
    tcl << "set origin         \"" << origin << "\";\n";
  }
  if (cut) {
    tcl << "set normal         \"" << normal << "\";\n";
  }
  tcl << "set axes_color     \"0. 0. 0.\";\n";
  tcl << "set font_size      15;\n";
  tcl << "#\n";
  tcl << "# create rendering stuff\n";
  tcl << "#\n";
  tcl << "vtkRenderer ren1;\n";
  tcl << "vtkRenderWindow render_window;\n";
  tcl << "    render_window AddRenderer ren1;\n";
  tcl << "vtkRenderWindowInteractor iren;\n";
  tcl << "    iren SetRenderWindow render_window;\n";
  make_tcl_actor (tcl, "mesh", basename, 0, execute, clean, verbose, fill, shrink, tube, ball, full,
		  cut, origin, normal);
  iter = _domlist.begin();
  last = _domlist.end();
  size_type color_idx = 0;
  while (iter != last) {
      const domain& dom = *iter;
      char dom_basename [1000];
      sprintf (dom_basename, "%s.%s", basename.c_str(), dom.name().c_str());
      make_tcl_actor (tcl, dom.name().c_str(), dom_basename, color_idx+1, 
			   execute, clean, verbose, fill, shrink, tube, ball, 
			   full, cut, origin, normal);
      color_idx = (color_idx+1) % vtk_n_color;
      ++iter;
  }
  tcl << "#\n"
      << "# axis\n"
      << "#\n"
      << "vtkCubeAxesActor2D axes_actor\n"
      << "    axes_actor SetInput [mesh_reader GetOutput]\n"
      << "    axes_actor SetCamera [ren1 GetActiveCamera]\n"
      << "    axes_actor SetLabelFormat \"%6.4g\"\n"

      //
      // Version 4.2 of VTK introduces several modifications
      // on a couple of classes. Now for instance, fonts are
      // managed with a single class. In order to stay compatible
      // with older and newer versions, we use the code below
      //

      << "    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {\n"
      << "        vtkTextProperty axes_text_property\n"
      << "        axes_text_property BoldOn\n"
      << "        axes_text_property SetFontFamilyToTimes;\n"
      << "        axes_actor SetAxisTitleTextProperty axes_text_property\n"
      << "        axes_actor SetAxisLabelTextProperty axes_text_property\n"
      << "    } else {\n"
      << "        axes_actor BoldOn\n"
      << "        axes_actor SetFontFamilyToTimes\n"
      << "    }\n"
      << "    axes_actor SetFlyModeToOuterEdges\n"
      << "    axes_actor SetFontFactor 1.0\n"
      << "    eval [axes_actor GetProperty] SetColor $axes_color\n"
      << "ren1 AddProp  axes_actor\n"
      << "#\n"
      << "# render the image\n"
      << "#\n"
      << "render_window SetSize 500 500;\n"
    ;
  if (fill) {
    tcl << "ren1 SetBackground 0.2 0.3 0.4;\n";
  } else {
    tcl << "ren1 SetBackground 1 1 1; \n";
  }
  tcl << "if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {\n";
  tcl << "    iren AddObserver UserEvent {wm deiconify .vtkInteract};\n";
  tcl << "} else {\n";
  tcl << "    iren SetUserMethod {wm deiconify .vtkInteract};\n";
  tcl << "}\n";
  tcl << "render_window Render;\n";
  tcl << "#\n";
  tcl << "# prevent the tk window from showing up then start the event loop\n";
  tcl << "#\n";
  tcl << "wm withdraw .\n";
  tcl.close();
  //
  // run vtk
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "vtk %s.tcl", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear vtk data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.vtk %s.tcl", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
      iter = _domlist.begin();
      last = _domlist.end();
      while (iter != last) {
          const domain& dom = *iter;
          sprintf (command, "/bin/rm -f %s.%s.vtk", basename.c_str(), dom.name().c_str());
          if (verbose) clog << "! " << command << endl;
          status = system (command);
          ++iter;
      }
  }
  return status;
}
}// namespace rheolef
