#ifndef _RHEO_PIOLA_H
#define _RHEO_PIOLA_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

// Piola transformation:
//  F: hat_K  --> K
//     hat_x |--> x = F(hat_x)
// where F is polynomial Pk
// now just k=1 is supported (no isoparametric transformation yet)
//
// The polynoms of the transformation are evaluated one time
// for all at the nodes of the quadrature formulae on each
// type of reference element: hat_xq in hat_K
//
// author: Pierre Saramito
//
// date: 22 june 2009
//
#include "rheolef/space.h"
#include "rheolef/basis_on_lattice.h"
#include "rheolef/tensor.h"
namespace rheolef { 

class piola_on_quadrature : basis_on_quadrature {
public:
  typedef size_t size_type;
  piola_on_quadrature();
  piola_on_quadrature (const quadrature& quad, const space& Xh);
  void set (const quadrature& quad, const space& Xh);

  // F_K (hat_xq)
  point transformation (const geo_element& K, size_type q) const;
  // DF (hat_xq)
  void jacobian_transformation (const geo_element& K, size_type q, tensor& DF) const;
  Float det_jacobian_transformation (const tensor& DF, size_type map_dim) const;
  Float det_jacobian_transformation (const geo_element& K, size_type q) const;
  // DF (hat_x)
  void jacobian_transformation (const geo_element& K, const point& hat_x, tensor& DF) const;


  Float weight_coordinate_system (const geo_element& K, size_type q) const;
  size_type coordinate_dimension () const;
  void set_use_coordinate_system_weight (bool use);
  bool use_coordinate_system_weight () const;
  void set_use_coordinate_system_dual_weight (bool use);
  bool use_coordinate_system_dual_weight () const;
protected:
  space _Xh;
  bool  _use_coordinate_system_weight;
  bool  _use_coordinate_system_dual_weight;
};

class piola {
public:
  typedef size_t size_type;
  piola();
  piola (const space& Xh);
  void set (const space& Xh);

  // F_K (hat_x)
  point transformation (const geo_element& K, const point& x) const;
  // DF (hat_x)
  Float det_jacobian_transformation (const tensor& DF, size_type map_dim) const;
  Float det_jacobian_transformation (const geo_element& K, const point& hat_x) const;
  void jacobian_transformation (const geo_element& K, const point& hat_x, tensor& DF) const;
//  void piola_basis (const basis& b, const geo_element& K, const point& hat_x, std::vector<Float>& p) const;
   //! take into account non-Lagrange interpolation polynomials

    // evaluate the basis and derivative of basis functions at a given point of reference element
    Float reference_eval (const geo_element& hat_K, const tiny_vector<Float>& dof, const point& hat_x, 
        size_type i_comp = 0) const;
    Float reference_d_dxi_eval (size_type i, const geo_element& hat_K,
        const tiny_vector<Float>& dof, const point& hat_x,
        size_type i_comp) const;

  Float weight_coordinate_system (const geo_element& K, const point& hat_x) const;
  size_type coordinate_dimension () const;
  void set_use_coordinate_system_weight (bool use);
  bool use_coordinate_system_weight () const;
  void set_use_coordinate_system_dual_weight (bool use);
  bool use_coordinate_system_dual_weight () const;
protected:
  space _Xh;
  bool  _use_coordinate_system_weight;
  bool  _use_coordinate_system_dual_weight;
};

inline
Float
det_jacobian_piola_transformation (const tensor& DF, piola::size_type map_dim,
	piola::size_type coordinate_dimension)
{
    if (coordinate_dimension == map_dim) {
      return DF.determinant (map_dim);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_dim) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
            << map_dim << " in " << coordinate_dimension << "D mesh.");
    }
}

// map non-Lagrange reference basis functions to element basis function 
template<class InputIterator,class OutputIterator>
InputIterator 
piola_basis (const space& X, const geo_element& K, const basis& b, const tensor& DF, 
	InputIterator first_in, InputIterator last_in, OutputIterator first_out)
{
  if (b.family()==reference_element::Lagrange) return first_in;
  else if (b.family()==reference_element::Hermite) 
   {
   	Float det=det_jacobian_piola_transformation(DF, K.dimension(),
		X.get_global_geo().dimension());
	tiny_vector<piola::size_type> idx;
	X.set_dof(K, idx);
	tiny_vector<int> orientation;
	X.dof_orientation(K, orientation);
   	for (piola::size_type i=0; first_in!=last_in; first_in++, i++)
	 {
	 	if (b.dof_family(K, i)==reference_element::Lagrange)
		  first_out[i]=*first_in;
		else // Hermite
		  first_out[i]=orientation[i]*det*(*first_in); 
	 }
   }
  else error_macro("Unknown family of finite elements, " << b.family());
  return first_out;
}

}// namespace rheolef
#endif // _RHEO_PIOLA_H
