# ifndef _RHEOLEF_FIELD_H
# define _RHEOLEF_FIELD_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/skit.h"
#include "rheolef/space.h"
#include "rheolef/scalar_traits.h"

namespace rheolef {
//<field:  
/*Class:field
NAME:  @code{field} - piecewise polynomial finite element field
DESCRIPTION:       
  @noindent
  Store degrees of freedom associated to a mesh and
  a piecewise polynomial approximation, with respect
  to the numbering defined by the underlying @ref{space class}.

  @noindent
  This class contains two vectors, namely unknown and blocked
  degrees of freedoms, and the associated finite element space.
  Blocked and unknown degrees of freedom can be set by using
  domain name indexation:
  @example
        geo omega_h ("circle");
        space Vh (omega_h, "P1");
        Vh.block ("boundary");
        field uh (Vh);
        uh ["boundary"] = 0;
  @end example
  Interpolation of a function @code{u} in a field @code{uh} with respect to 
  the interpolation writes:
  @example
        Float u (const point&);
        uh = interpolate (Vh, u);
  @end example

EXAMPLE:
  Here is a complete example
  using the field class:
@example
        Float u (const point& x) @{ return x[0]*x[1]; @}
        int main() @{
          geo omega_h ("square");
          space Vh (omega_h, "P1");
          field uh (Vh);
          uh = interpolate (Vh, u);
          cout << plotmtv << u;
        @}
@end example
FEATURES:
   Algebra, such as x+y, x*y, x/y, lambda*x, ...are supported
   
   Transformations applies to all values of a field:
@example
   field vh = compose(fabs, uh);
   field wh = compose(atan2, uh, vh);
@end example
   The composition supports also general unary and binary class-functions.

   Vector-valued and tensor-valued field support is yet partial only.
   This feature will be more documented in the future.

AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
   | Jocelyn.Etienne@imag.fr
DATE:   2 july 1997  update: 22 may 2010
METHODS: @field
End:
*/
//<field:
template <class T>
class field_basic : public std::unary_function<basic_point<T>,T> {
public :
// typedefs:

    typedef typename vec<T>::size_type      size_type;
    typedef T                               value_type;
    typedef typename scalar_traits<T>::type float_type;
    typedef space_basic<float_type>         space_type;

    typedef typename vec<T>::communicator_type communicator_type;

// allocator/deallocator:
  
    field_basic();
    explicit field_basic (
	const space& V, 
	const Float& init_value = std::numeric_limits<Float>::max());

// accessors:

          vec<T>& u()       { return _u; }
          vec<T>& b()       { return _b; }
    const vec<T>& u() const { return _u; }
    const vec<T>& b() const { return _b; }

          T& operator[] (size_type idof);
    const T& operator[] (size_type idof) const;

    const space_basic<float_type>& get_space() const { return _V; }

// input/output:

    oparstream& put (oparstream& ops) const;

// data:
protected:
    space_basic<float_type> _V;
    vec<T>                  _u;
    vec<T>                  _b;
};
template <class T>
oparstream& operator << (oparstream& ops, const field_basic<T>& u);

// general field & function composition:
// v(x) := f(u(x)) <==> v := compose(f,u)
template<class T, class Function>
field_basic<T> compose (Function f, const field_basic<T>& u);

typedef field_basic<Float> field;
//>field:

// =================================================================================
// inlined
// =================================================================================
template <class T>
inline
field_basic<T>::field_basic ()
 : _V(),
   _u(),
   _b()
{
}
template <class T>
inline
T& 
field_basic<T>::operator[] (size_type idof)
{
    if (! _V.is_blocked(idof)) {
      return _u [_V.iub(idof)]; 
    } else {
      return _b [_V.iub(idof)];
    }
}
template <class T>
inline
const T&
field_basic<T>::operator[] (size_type idof) const
{
    if (! _V.is_blocked(idof)) {
      return _u [_V.iub(idof)];
    } else {
      return _b [_V.iub(idof)];
    }
}
template <class T>
inline
oparstream& operator << (oparstream& ops, const field_basic<T>& uh)
{
    return uh.put (ops);
}
template <class T, class Function>
inline
field_basic<T>
compose (Function f, const field_basic<T>& x)
{
    field_basic<T> y (x.get_space());
    std::transform(x.u().begin(), x.u().end(), y.u().begin(), f);
    std::transform(x.b().begin(), x.b().end(), y.b().begin(), f);
    return y;
}
#ifdef TODO
template <class T>
inline
field_basic<T>
compose (T (*f)(const T&), const field_basic<T>& x)
{
    field_basic<T> y (x.get_space());
    std::transform(x.u().begin(), x.u().end(), y.u().begin(), f);
    std::transform(x.b().begin(), x.b().end(), y.b().begin(), f);
    return y;
}
#endif // TODO

}// namespace rheolef
# endif /* _RHEOLEF_FIELD_H */
