///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// call to parmetis

#include "rheolef/config.h"
#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/geo_element.h"
#include "rheolef/array.h"
#include "rheolef/polymorphic_array.h"

#if defined(_RHEOLEF_HAVE_SCOTCH)
#include "geo_partition_scotch.h"
#elif defined(_RHEOLEF_HAVE_PARMETIS)
#include <parmetis.h>
#endif // _RHEOLEF_HAVE_PARMETIS

namespace rheolef {

array<polymorphic_array<geo_element>::size_type>
geo_partition (const polymorphic_array<geo_element>& elts, size_t map_dim)
{
  typedef polymorphic_array<geo_element>::size_type size_type;

  distributor ownership = elts.ownership();
  communicator comm = ownership.comm();
  size_type nproc = comm.size();

  array<size_type> partition (ownership);
  if (partition.par_size() < nproc) {
    // mesh is very small (nelts < nproc): parmetis does not support it !
    size_type first_ie = ownership.first_index();
    for (size_type ie = 0, nie = elts.size(); ie < nie; ie++) {
      partition [ie] = ie + first_ie;
    }
    return partition;
  }
  // here, mesh is not so small: sufficient for parmetis
  array<idxtype> part (ownership);
  std::vector<idxtype> elmdist (nproc+1);
  std::copy (
          ownership.begin(),
          ownership.end(),
  	  elmdist.begin());
  std::vector<idxtype> eptr (elts.par_size()+1);
  eptr [0] = 0;
  for (size_type ie = 0, nie = elts.size(); ie < nie; ie++) {
      const geo_element& K = elts [ie];
      eptr [ie+1] = eptr[ie] + K.size();
  }
  std::vector<idxtype> eind (eptr [elts.size()]);
  std::vector<idxtype>::iterator iter_eind = eind.begin();
  for (size_type ie = 0, nie = elts.size(); ie < nie; ie++) {
      const geo_element& K = elts [ie];
      for (size_type iloc = 0; iloc < K.size(); iloc++, iter_eind++) {
        *iter_eind = K[iloc];
      }
  }
  int wgtflag = 0;
  int numflag = 0;
  int mgcnum = map_dim;  
  int ncon = 1;  
  int nparts = nproc;  
  std::vector<float> tpwgts (nparts*ncon);
  std::fill (tpwgts.begin(), tpwgts.end(), 1./nparts);
  float ubvec [12];
  std::fill (ubvec, ubvec+ncon, 1.05);
  int options [10];
  options[0] = 1;
  const int pvm3_option_dbglvl = 1;
  const int pvm3_option_seed   = 2;
  options[pvm3_option_dbglvl] = 0; // otherwise: timming print to stdout...
  options[pvm3_option_seed]   = 0;
  int edgecut;
  MPI_Comm raw_comm = comm;
  idxtype *elmwgt = 0; 
#if defined(_RHEOLEF_HAVE_SCOTCH)
  geo_partition_scotch (
        elmdist.begin().operator->(),
	eptr.begin().operator->(),
	eind,
	elmwgt,
	&ncon,
	&mgcnum,
	&nparts, 
  	tpwgts.begin().operator->(),
	ubvec,
	&edgecut,
  	part.begin().operator->(),
	comm);
#elif defined(_RHEOLEF_HAVE_PARMETIS)
  ParMETIS_V3_PartMeshKway(
        elmdist.begin().operator->(),
	eptr.begin().operator->(),
  	eind.begin().operator->(),
	NULL,
	&wgtflag, 
        &numflag,
	&ncon,
	&mgcnum,
	&nparts,
  	tpwgts.begin().operator->(),
	ubvec,
	options,
	&edgecut,
  	part.begin().operator->(),
	&raw_comm);
#else // _RHEOLEF_HAVE_PARMETIS
# error either parmetis nor scotch partitionner founded
#endif // _RHEOLEF_HAVE_PARMETIS

  // convert int to size_type (64 bits : not the same sizeof...)
  std::copy (part.begin(), part.end(), partition.begin());
  return partition;
}

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
