# ifndef _SKIT_VECTOR_H
# define _SKIT_VECTOR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:vec
NAME: @code{vec} - dense vector 
@clindex vec
@clindex array
@clindex Float
DESCRIPTION:       
 @noindent 
 The class implements an array.
 A declaration whithout any parametrers correspond to an array with a null size:
 @example
      	vec<Float> x;
 @end example
 @noindent 
 Here, a @code{Float} array is specified.
 Note that the @code{Float} depends upon the configuration
 (@pxref{Installing}).
 The constructor can be invocated whith a size parameter:
 @example
     	vec<Float> x(n);
 @end example
 @noindent 
 Notes that the constructor can be invocated with an initializer:
 @example
     	vec<Float> y = x;
 @end example
 @noindent 
 or
 @example
     	Float lambda = 1.3;
     	vec<Float> y = lambda;
 @end example
 @noindent 
 Assignments are 
 @example
     	x = y;
 @end example
 @noindent 
 or 
 @example
     	x = lambda;
 @end example
 @noindent 
 Linear combinations are 
	@code{x+y},  
	@code{x-y},  
	@code{x*lambda},  
	@code{lambda*x}, 
  	@code{x/lambda}.
 
 @noindent 
 Others combinations are 
	@code{x*y}, 
	@code{x/y}, 
	@code{lambda/x}.

 @noindent 
 The euclidian norm is 
	@code{norm(x)} 
 while 
	@code{dot(x,y)}
 denotes the euclidian scalar product,
 
 @noindent 
 Input/output routines overload "<<" and ">>" operators:
 @example
     	cin  >> x;
 @end example
 and
 @noindent 
 @example
     	cout << x;
 @end example
 @noindent 
 @xref{iorheo class}.

NOTE:
 
 @noindent 
 Since the @code{vec<T>} class derives from the 
 @code{Array<T>} class, the @code{vec} class
 present also a STL-like container interface.
 
 Actually, the @code{vec<T>} implementation uses a STL 
 @code{vector<T>} class.

AUTHOR: 
    
    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:
    
    14 january 1997

End:
*/
// ==============================[ VEC CLASS ]===================================
# include "rheolef/array.h"

#ifdef _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

# include "blas-algorithm.h" // yassignr
namespace rheolef { 
    template <class X> class VExpr;

    template <class X, class T2>
    inline
    vec<T2>&
    assign (vec<T2>& y, const VExpr<X>& x)
    {
        unsigned int x_n = x.n();
	if (y.n() == 0 && x_n != 0) y.resize(x_n);
    	check_length (y, x);
	yassignx (y.begin(), y.end(), x.begin());
	return y;
    }   
}// namespace rheolef
#endif // _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

namespace rheolef { 
//<vec:
template <class T>
class vec : public Array<T> 
{
public:

    typedef          T                   element_type;
    typedef typename Array<T>::size_type size_type;

    // cstor, assignment
	explicit vec (unsigned int n = 0, const T& init_value = std::numeric_limits<T>::max()) 
	  : Array<T>(n, init_value) {}
	vec<T> operator = (T lambda);

    // accessors
	unsigned int size () const { return Array<T>::size(); }
	unsigned int n () const { return size(); }

#ifdef _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

	template <class X>
	vec(const VExpr<X>& x) : Array<T>(x.size()) { assign (*this, x); }

	template <class X>
	vec<T>& operator = (const VExpr<X>& x)
				{ logmodif(*this); return assign (*this, x); }
#endif // _RHEOLEF_HAVE_EXPRESSION_TEMPLATE
}; 
// io routines
template <class T> std::istream& operator >> (std::istream&, vec<T>&);
template <class T> std::ostream& operator << (std::ostream&, const vec<T>&);
//>vec:
}// namespace rheolef

#endif // _SKIT_VECTOR_H
