#ifndef _RHEO_PAR_INNER_PRODUCT_H
#define _RHEO_PAR_INNER_PRODUCT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/parallel.h"
#include <iterator>
namespace rheolef {

/*D:par_inner_product
NAME:  par_inner_product - parallel inner product (@PACKAGE@-@VERSION@)
DESCRIPTION:       
 STL-like inner product for distributed container on parallel
 environment.
SYNOPSIS:
  @example
  template<class InputIterator1, class InputIterator2>
  T par_inner_product (
      	InputIterator1 first1, InputIterator1 last1,
        InputIterator2 first2);
  @end example
EXAMPLE:
   A sample usage writes:
   @example
     # include "rheolef/array.h"
     # include "rheolef/par_inner_product.h"
     int main(int argc, char**argv) {
         parallel process(argc, argv);
         unsigned int n = 100;
         array<double> x(n, 2.0);
         double norme2_x = par_inner_product(x.begin(),x.end(),x.begin(),x.comm());
         pcout << "dot(x,x) = " << norme2_x << endl;
     }
   @end example
SEE ALSO: "array"(1)
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   24 november 1998
End:
*/
template <class InputIterator1, class InputIterator2>
inline
typename std::iterator_traits<InputIterator1>::value_type
par_inner_product (
    	InputIterator1 first1,
    	InputIterator1 last1,
    	InputIterator2 first2,
	const distributor::communicator_type& comm,
	sequential /* memory_model */)
{
    typedef typename std::iterator_traits<InputIterator1>::value_type T;
    return std::inner_product(first1, last1, first2, T(0));
}
#ifdef _RHEOLEF_HAVE_MPI
template <class InputIterator1, class InputIterator2>
inline
typename std::iterator_traits<InputIterator1>::value_type
par_inner_product (
    	InputIterator1 first1,
    	InputIterator1 last1,
    	InputIterator2 first2,
	const distributor::communicator_type& comm,
	distributed /* memory_model */)
{
    typedef typename std::iterator_traits<InputIterator1>::value_type T;
    T local_sum = std::inner_product(first1, last1, first2, T(0));
    return mpi::all_reduce (comm, local_sum, std::plus<T>());    
}
#endif // _RHEOLEF_HAVE_MPI

template <class InputIterator1, class InputIterator2>
inline
typename std::iterator_traits<InputIterator1>::value_type
par_inner_product (
    	InputIterator1 first1,
    	InputIterator1 last1,
    	InputIterator2 first2,
	const distributor::communicator_type& comm)
{
    return par_inner_product (first1, last1, first2, comm, rheo_default_memory_model());
}

} // namespace rheolef
#endif // _RHEO_PAR_INNER_PRODUCT_H
