///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"

#ifdef _RHEOLEF_HAVE_CLIMITS
#include <climits>   // PATH_MAX ?
#else
#include <limits.h>
#endif

#include <boost/iostreams/filter/gzip.hpp>


#ifdef _RHEOLEF_HAVE_UNISTD_H
#include<unistd.h>  // readink()
#endif

#ifdef _RHEOLEF_HAVE_SYMLINK_H
#include<symlink.h>  // readink() on hpux9*
#endif

#include<dirent.h>   // opendir()
#include<sys/stat.h> // stat()

#ifndef PATH_MAX
#define PATH_MAX 1023 // TODO: find it !
#endif // PATH_MAX

namespace rheolef {
using namespace std;
namespace io = boost::iostreams;

string
itos (string::size_type i)
{
    char buffer [100];
    sprintf (buffer, "%d", int(i));
    return buffer;
}
string
ftos (const Float& x)
{
    char buffer [100];
    sprintf (buffer, "%g", double(x));
    return buffer;
}
#ifdef TO_CLEAN
// read on a pipe with un-compression
static
string
make_i_pipe_name (const string& name, const string& suffix)
{
    string full_name = get_full_name_from_rheo_path (name, suffix);
    if (full_name.length() == 0) {
        return string();
    }
    bool verbose = iorheo::getverbose(clog);
    if (verbose) clog << "! load \"" << full_name << "\"\n";

    if (!has_suffix (full_name, "gz")) {
	
	// classical ifstream
	return "cat " + full_name;
    }
    return "gzip -dc < " + full_name;
}
// write on a pipe with compression
static
string
make_o_pipe_name (const string& name, const string& suffix)
{
    string full_name;

    // assume suffix.gz
    if (suffix=="")
        full_name = delete_suffix(name, "gz") + ".gz";
    else
        full_name = delete_suffix(delete_suffix(name, "gz"), suffix) 
		       + "." + suffix + ".gz";
    
    bool verbose = iorheo::getverbose(clog);
    if (verbose) clog << "! file \"" << full_name << "\" created.\n";

    return "gzip -c9 > " + full_name;
}
#endif // TO_CLEAN
// -----------------------------------------------------------------
// output
// -----------------------------------------------------------------
void
orheostream::open (const string& name, const string& suffix)
{
    using namespace io;

    // append the '.gz' suffix:
    string full_name;
    if (suffix=="")
        full_name = delete_suffix(name, "gz") + ".gz";
    else
        full_name = delete_suffix(delete_suffix(name, "gz"), suffix) 
		       + "." + suffix + ".gz";
    
    // open the file.gz:
    _ofs.open (full_name.c_str(), ios_base::out | ios_base::binary);
    bool verbose = iorheo::getverbose(clog);
    if (verbose) clog << "! file \"" << full_name << "\" created.\n";

    // create the output pipe with the gzip filter:
    filtering_stream<output>::push (gzip_compressor());    
    filtering_stream<output>::push (_ofs);    
}
void
orheostream::close ()
{
    using namespace io;
    while (! filtering_stream<io::output>::empty()) {
             filtering_stream<io::output>::pop();
    }
    _ofs.close();
}
orheostream::orheostream (const string& name, const string& suffix)
 : io::filtering_stream<io::output>()
{
    open (name, suffix);
}
orheostream::~orheostream ()
{
    orheostream::close();
}
// -----------------------------------------------------------------
// input
// -----------------------------------------------------------------
void
irheostream::open (const string& name, const string& suffix)
{
    warning_macro ("irheostream::open("<<name<<","<<suffix<<")...");
    using namespace io;

    // get full file path with the optional '.gz' suffix:
    string full_name = get_full_name_from_rheo_path (name, suffix);
    bool has_gz = has_suffix (full_name, "gz");

    // open the file[.gz]:
    if (!has_gz) {
      _ifs.open (full_name.c_str(), ios_base::in);
    } else {
      _ifs.open (full_name.c_str(), ios_base::in | ios_base::binary);
    }
    bool verbose = iorheo::getverbose(clog);
    if (verbose) clog << "! load \"" << full_name << "\"\n";

    // create the input pipe with the optional gzip filter:
    if (has_gz) {
      filtering_stream<input>::push (gzip_decompressor());    
    }
    filtering_stream<input>::push (_ifs);    
    warning_macro ("irheostream::open("<<name<<","<<suffix<<") done");
}
void
irheostream::close ()
{
    using namespace io;
    while (! filtering_stream<io::input>::empty()) {
             filtering_stream<io::input>::pop();
    }
    _ifs.close();
}
irheostream::irheostream (const string& name, const string& suffix)
 : io::filtering_stream<io::input>()
{
    open (name, suffix);
}
irheostream::~irheostream ()
{
    irheostream::close();
}

// has_suffix("toto.suffix", "suffix") -> true
bool
has_suffix (const string& name, const string& suffix)
{
    unsigned int ln = name.length();
    unsigned int ls = suffix.length();
    if (ln <= ls+1) return false;

    if (name[ln-ls-1] != '.') return false;
    for (unsigned int i = ln-ls, j = 0; i < ln; i++, j++)
        if (name [i] != suffix [j]) return false;

    return true;
}
// delete_suffix("toto.suffix", "suffix") --> "toto"
string
delete_suffix (const string& name, const string& suffix)
{
    if (!has_suffix(name, suffix)) return name;
    return string(name, 0, name.length() - suffix.length() - 1);
}
string
get_basename (const string& name)
{
    string::size_type l = name.length(); 
    string::size_type i = name.find_last_of ('/');
    if (i >= l) return name;
    string b = string(name, i+1, l-i-1);
    return b;
}
string
get_dirname (const string& name)
{
    string::size_type l = name.length(); 
    string::size_type i = name.find_last_of ('/');
    if (i >= l) return ".";
    string d = string(name, 0, i);
    return d;
}
bool 
scatch (istream& in, const string& ch)
{
    // null string case
    unsigned int l = ch.length();
    if (l == 0) return true;
  
    // check file
    char c = '\0';
    unsigned int state = 0;
    const char *p = ch.c_str();
    in.get(c);
    if (*p == '\n') {
	// begining of stream <==> begining of a line, e.g. 
	//   we look at "\nfield" while file starts 
	//   with string "field"; it's ok
	state++;
        p++;
    }
    do {
 
        if (*p == c) {
            // advance in the string
            state++;
            p++;
        } else if (state != 0 && ch[0] == c) {
            // come back to the second position
            state = 1;
            p = ch.c_str() + 1;
        } else if (state != 0) {
            // come back to the begining of the string
            state = 0;
            p = ch.c_str();
        }
    }
    while (state < l && in.get(c) && in.good());

    return (state == l);
}
//
// NOTE: path could have an iterator that points to
//       a directory name...
//
// NOTE 2: global cstors in shared libraries are not
// 	 handled by g++ and most compilers 
//	=> need char* instead of string here
//
static const char* rheo_path_name    = "RHEOPATH";
static const char* default_rheo_path = ".";
static char* rheo_path         = 0;

static
string
get_dir_from_path (const string& path, unsigned int& i_pos)
{
    // is search path finished ?
    unsigned int last = path.length();
    if (i_pos >= last) {
	return string();
    }
    // skip ':' separators
    while (i_pos < last && path [i_pos] == ':')
	i_pos++;

    // test end of path
    if (i_pos == last) {
	return string();
    }
    // path [i_pos] != ':' and i_pos < last, so we have a dir
    unsigned int i_last = i_pos;
    for (i_last = i_pos; i_last < last && path [i_last] != ':'; i_last++);
    string current_dir;
    if (i_last == last)
        current_dir = string(path, i_pos, i_last-i_pos+1);
    else
        current_dir = string(path, i_pos, i_last-i_pos);

    // i_last == last || path[i_last] == ':'
    i_pos = i_last;

    return current_dir;
}
static 
void
init_rheo_path ()
{
    // get directory path from environ
    if (rheo_path) {
        return;
    }
    const char *s1 = getenv (rheo_path_name);
    if (!s1) { 
        s1 = default_rheo_path;
    }
    rheo_path = new_tab_macro(char, strlen(s1)+1);
    strcpy (rheo_path, s1);
}
void
append_dir_to_rheo_path (const string& dir)
{
    init_rheo_path();
    string tmp = string(rheo_path) + ":" + dir;
    delete_tab_macro(rheo_path);
    rheo_path = new_tab_macro(char, strlen(tmp.c_str())+1);
    strcpy (rheo_path, tmp.c_str());
}
void
prepend_dir_to_rheo_path (const string& dir)
{
    init_rheo_path();
    string tmp = dir + ":" + string(rheo_path);
    delete_tab_macro(rheo_path);
    rheo_path = new_tab_macro(char, strlen(tmp.c_str())+1);
    strcpy (rheo_path, tmp.c_str());
}
bool 
file_exists (const string& filename)
{
    struct stat s;
    if (stat(filename.c_str(), &s) != 0) {
        return false;
    }
    return true;
}
static
bool
have_name_in_dir (const string& dir, const string& name, string& full_path)
{
    string prefix;
    if (dir != "") {
        // trace_macro ("scanning in \"" << dir.c_str() << "\"");
	prefix = dir + "/";
    }
    // try to open file like dir/rootname.suffix.gz
    string zip_full_name = prefix + name + ".gz";
    bool zip_status = file_exists (zip_full_name);

    // try to open file like dir/rootname.suffix
    string unzip_full_name = prefix + name;
    bool unzip_status = file_exists (unzip_full_name);

    if (unzip_status && zip_status) {
	    warning_macro ("both compressed and uncompressed files match:");
	    warning_macro ("    \"" << zip_full_name   << "\"");
	    warning_macro ("and \"" << unzip_full_name << "\"");
    }
    // prefer ziped than unziped version
    if (zip_status) {
	full_path = zip_full_name;
	return true;
    }
    if (unzip_status) {
	full_path = unzip_full_name;
	return true;
    }
    // 15 oct 2000: check that "dir" is a valid directory
    struct stat sd;
    if (stat(dir.c_str(), &sd) != 0) {
	warning_macro ("cannot not stat \"" << dir << "\"");
	warning_macro ("hint: check "<< rheo_path_name << " or -I options");
	return false;
    }
    if ((sd.st_mode & S_IFDIR) == 0) {
	warning_macro ("invalid directory \"" << dir << "\"");
	warning_macro ("hint: check "<< rheo_path_name << " or -I options");
	return false;
    }
    // scan subdirs
    DIR* d = opendir(dir.c_str());
    if (!d) {
	warning_macro ("cannot open directory \"" << dir << "\"");
	warning_macro ("hint: check "<< rheo_path_name << " or -I options");
	return false;
    }
    struct dirent* dp;
    while ((dp = readdir(d)) != 0) {

	string subdir = dir + "/" + (dp -> d_name);
	
	if (strcmp(dp -> d_name, ".") == 0 ||
	    strcmp(dp -> d_name, "..") == 0) continue;

	struct stat s;
	if (stat(subdir.c_str(), &s) != 0) {
	    warning_macro ("can not stat() for \"" << subdir << "\"");
	    continue;
	}
	if ((s.st_mode & S_IFLNK) == 0) {
	    // 16 january 1999: skip also symbolic links to "." and ".."
	    char linkname [PATH_MAX + 2];
            // extern "C" int readlink(const char *, char *, int);
	    char* subdir_cstr = (char*)(subdir.c_str());
	    int linksize = readlink (subdir_cstr, linkname, PATH_MAX + 1);
	    // PATH_MAX = max number of characters in a pathname
	    //  (not including terminating null)
	    // 
	    // from fileutils-3.14/src/ls.c (line 1724):
	    // "Some automounters give incorrect st_size for mount points.
	    //  I can't think of a good workaround for it, though."
	    //
	    if (linksize < 0) {
               // perhaps not a symklink ? 
	       // trace_macro ("can not read link name \"" << subdir << "\"");
	    } else {
	       linkname [linksize] = '\0';
	       if (strcmp(linkname, ".") == 0 ||
	          strcmp(linkname, "..") == 0) {
		    continue;
	       }
	    }
	}
	if ((s.st_mode & S_IFDIR) != 0) {
	    // recurse in subdir
            if (have_name_in_dir (subdir, name, full_path)) {
		return true;
	    }
        }
    }
    return false;
}
string
get_full_name_from_rheo_path (const string& rootname, const string& suffix)
{
    if (rootname == "") {
	return rootname;
    } 
    string name = delete_suffix(delete_suffix(rootname, "gz"), suffix);
	if (suffix != "") name += "." + suffix;
    string full_path;

    if (rootname [0] == '.' || rootname[0] == '/') {
        if (have_name_in_dir ("", name, full_path)) {
	  return full_path;
	}
	return string();
    }
    //
    // rootname has no explicit reference: use seach path
    //
    init_rheo_path();
    unsigned int i_dir = 0;
    string dir = get_dir_from_path (rheo_path, i_dir);
    while  (dir.length() != 0) {

        if (have_name_in_dir (dir, name, full_path)) {
	  return full_path;
	}
	dir = get_dir_from_path (rheo_path, i_dir);
    }
    return string();
}
bool
is_float (const string& s)
{
    // simple check for float argument
    // EXP     ([fFeEdD]([\-+])?([0-9]+))
    // ([\-])?[0-9]+ |
    // ([\-])?[0-9]+"."[0-9]*{EXP}?
    // ([\-])?[0-9]*"."[0-9]+{EXP}?
    // ([\-])?[0-9]+{EXP}
    unsigned int l = s.length();
    if (l < 1) return false;
    if (!isdigit(s[0]) && s[0] != '-' && s[0] != '.') return false;
    if (s[0] == '-') {
       if (l < 2) return false;
       if (!isdigit(s[1]) && s[1] != '.') return false;
    }
    return true;
}
Float
to_float (const string& s)
{
    return Float(atof(s.c_str()));
}
}// namespace rheolef
