//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
// SMESH SMESHGUI : GUI for SMESH component
// File   : SMESHGUI.cxx
// Author : Nicolas REJNERI, Open CASCADE S.A.S.
// SMESH includes

#include "SMESHGUI.h"
#include "SMESHGUI_NodesDlg.h"
#include "SMESHGUI_TransparencyDlg.h"
#include "SMESHGUI_ClippingDlg.h"
#include "SMESHGUI_GroupDlg.h"
#include "SMESHGUI_RemoveNodesDlg.h"
#include "SMESHGUI_RemoveElementsDlg.h"
#include "SMESHGUI_MeshInfosDlg.h"
#include "SMESHGUI_StandardMeshInfosDlg.h"
#include "SMESHGUI_WhatIsDlg.h"
#include "SMESHGUI_Preferences_ColorDlg.h"
#include "SMESHGUI_Preferences_ScalarBarDlg.h"
#include "SMESHGUI_Hypotheses.h"
#include "SMESHGUI_MoveNodesDlg.h"
#include "SMESHGUI_AddMeshElementDlg.h"
#include "SMESHGUI_AddQuadraticElementDlg.h"
#include "SMESHGUI_FilterDlg.h"
#include "SMESHGUI_FilterLibraryDlg.h"
#include "SMESHGUI_SingleEditDlg.h"
#include "SMESHGUI_MultiEditDlg.h"
#include "SMESHGUI_GroupOpDlg.h"
#include "SMESHGUI_GroupOnShapeDlg.h"
#include "SMESHGUI_DeleteGroupDlg.h"
#include "SMESHGUI_SmoothingDlg.h"
#include "SMESHGUI_RenumberingDlg.h"
#include "SMESHGUI_ExtrusionDlg.h"
#include "SMESHGUI_ExtrusionAlongPathDlg.h"
#include "SMESHGUI_RevolutionDlg.h"
#include "SMESHGUI_TranslationDlg.h"
#include "SMESHGUI_RotationDlg.h"
#include "SMESHGUI_SymmetryDlg.h"
#include "SMESHGUI_SewingDlg.h"
#include "SMESHGUI_EditMeshDlg.h"
#include "SMESHGUI_MeshPatternDlg.h"
#include "SMESHGUI_Selection.h"
#include "SMESHGUI_CreatePolyhedralVolumeDlg.h"
#include "SMESHGUI_ConvToQuadOp.h"
#include "SMESHGUI_MeshOp.h"
#include "SMESHGUI_Displayer.h"
#include "SMESHGUI_MakeNodeAtPointDlg.h"
#include "SMESHGUI_BuildCompoundDlg.h"
#include "SMESHGUI_ComputeDlg.h"
#include "SMESHGUI_FileInfoDlg.h"
#include "SMESHGUI_Make2DFrom3DOp.h"

#include "SMESHGUI_Utils.h"
#include "SMESHGUI_MeshUtils.h"
#include "SMESHGUI_GroupUtils.h"
#include "SMESHGUI_FilterUtils.h"
#include "SMESHGUI_PatternUtils.h"
#include "SMESHGUI_VTKUtils.h"
#include "SMESHGUI_HypothesesUtils.h"

#include <SMESH_Client.hxx>
#include <SMESH_Actor.h>
#include <SMESH_TypeFilter.hxx>

// SALOME GUI includes
#include <SalomeApp_Tools.h>
#include <SalomeApp_Study.h>
#include <SalomeApp_Application.h>
#include <SalomeApp_CheckFileDlg.h>

#include <LightApp_DataOwner.h>
#include <LightApp_Preferences.h>
#include <LightApp_SelectionMgr.h>
#include <LightApp_UpdateFlags.h>
#include <LightApp_NameDlg.h>

#include <SVTK_ViewWindow.h>
#include <SVTK_ViewModel.h>
#include <SVTK_ViewManager.h>

#include <VTKViewer_Algorithm.h>

#include <SUIT_MessageBox.h>
#include <SUIT_ResourceMgr.h>
#include <SUIT_FileDlg.h>
#include <SUIT_Desktop.h>
#include <SUIT_OverrideCursor.h>
#include <SUIT_Session.h>

#include <QtxPopupMgr.h>
#include <QtxFontEdit.h>

#include <SALOME_ListIO.hxx>
#include <SALOME_ListIteratorOfListIO.hxx>

// IDL includes
#include <SALOMEconfig.h>
#include CORBA_CLIENT_HEADER(SALOMEDS_Attributes)
#include CORBA_CLIENT_HEADER(SMESH_MeshEditor)

// Qt includes
// #define       INCLUDE_MENUITEM_DEF // VSR commented ????????
#include <QMenu>

// BOOST includes
#include <boost/shared_ptr.hpp>

// VTK includes
#include <vtkScalarBarActor.h>
#include <vtkCamera.h>
#include <vtkRenderer.h>
#include <vtkPlane.h>

// SALOME KERNEL includes
#include <SALOMEDS_Study.hxx>
#include <SALOMEDSClient_StudyBuilder.hxx>
#include <SALOMEDSClient_SComponent.hxx>
#include <SALOMEDSClient_ClientFactory.hxx>
#include <SALOMEDSClient_IParameters.hxx>

// OCCT includes
#include <Standard_ErrorHandler.hxx>
#include <NCollection_DataMap.hxx>

//namespace{
  // Declarations
  //=============================================================
  void ImportMeshesFromFile(SMESH::SMESH_Gen_ptr theComponentMesh,
                            int theCommandID);

  void ExportMeshToFile(int theCommandID);

  void SetDisplayMode(int theCommandID);

  void SetDisplayEntity(int theCommandID);

  void Control( int theCommandID );


  // Definitions
  //=============================================================
  void ImportMeshesFromFile( SMESH::SMESH_Gen_ptr theComponentMesh,
                             int theCommandID )
  {
    QStringList filter;
    std::string myExtension;

    if ( theCommandID == 113 ) {
      filter.append( QObject::tr( "MED files (*.med)" ) );
      filter.append( QObject::tr( "All files (*)" ) );
    }
    else if ( theCommandID == 112 ) {
      filter.append( QObject::tr( "IDEAS files (*.unv)" ) );
    }
    else if ( theCommandID == 111 ) {
      filter.append( QObject::tr( "DAT files (*.dat)" ) );
    }

    QString anInitialPath = "";
    if ( SUIT_FileDlg::getLastVisitedPath().isEmpty() )
      anInitialPath = QDir::currentPath();

    QStringList filenames = SUIT_FileDlg::getOpenFileNames( SMESHGUI::desktop(),
                                                            anInitialPath,
                                                            filter,
                                                            QObject::tr( "SMESH_IMPORT_MESH" ) );
    if ( filenames.count() > 0 ) {
      SUIT_OverrideCursor wc;
      _PTR(Study) aStudy = SMESH::GetActiveStudyDocument();

      QStringList errors;
      bool isEmpty = false;
      for ( QStringList::ConstIterator it = filenames.begin(); it != filenames.end(); ++it ) {
        QString filename = *it;
        SMESH::mesh_array_var aMeshes = new SMESH::mesh_array;
        try {
          switch ( theCommandID ) {
          case 111:
            {
              // DAT format (currently unsupported)
              errors.append( QString( "%1 :\n\t%2" ).arg( filename ).
                             arg( QObject::tr( "SMESH_ERR_NOT_SUPPORTED_FORMAT" ) ) );
              break;
            }
          case 112:
            {
              // UNV format
              aMeshes->length( 1 );
              aMeshes[0] = theComponentMesh->CreateMeshesFromUNV( filename.toLatin1().constData() );
              if ( aMeshes[0]->_is_nil() )
                errors.append( QString( "%1 :\n\t%2" ).arg( filename ).
                               arg( QObject::tr( "SMESH_ERR_UNKNOWN_IMPORT_ERROR" ) ) );
              break;
            }
          case 113:
            {
              // MED format
              SMESH::DriverMED_ReadStatus res;
              aMeshes = theComponentMesh->CreateMeshesFromMED( filename.toLatin1().constData(), res );
              if ( res != SMESH::DRS_OK ) {
                errors.append( QString( "%1 :\n\t%2" ).arg( filename ).
                               arg( QObject::tr( QString( "SMESH_DRS_%1" ).arg( res ).toLatin1().data() ) ) );
              }
              break;
            }
          }
        }
        catch ( const SALOME::SALOME_Exception& S_ex ) {
          errors.append( QString( "%1 :\n\t%2" ).arg( filename ).
                         arg( QObject::tr( "SMESH_ERR_UNKNOWN_IMPORT_ERROR" ) ) );
        }

        for ( int i = 0, iEnd = aMeshes->length(); i < iEnd; i++ ) {
          _PTR(SObject) aMeshSO = SMESH::FindSObject( aMeshes[i] );
          if ( aMeshSO ) {
            _PTR(StudyBuilder) aBuilder = aStudy->NewBuilder();
            _PTR(AttributePixMap) aPixmap = aBuilder->FindOrCreateAttribute( aMeshSO, "AttributePixMap" );
            aPixmap->SetPixMap( "ICON_SMESH_TREE_MESH_IMPORTED" );
            if ( theCommandID == 112 ) // mesh names aren't taken from the file for UNV import
              SMESH::SetName( aMeshSO, QFileInfo(filename).fileName() );
          }
          else {
            isEmpty = true;
          }
        }
      }

      // update Object browser
      SMESHGUI::GetSMESHGUI()->updateObjBrowser();

      // show Error message box if there were errors
      if ( errors.count() > 0 ) {
        SUIT_MessageBox::critical( SMESHGUI::desktop(),
                                   QObject::tr( "SMESH_ERROR" ),
                                   QObject::tr( "SMESH_IMPORT_ERRORS" ) + "\n" + errors.join( "\n" ) );
      }

      // show warning message box, if some imported mesh is empty
      if ( isEmpty ) {
          SUIT_MessageBox::warning( SMESHGUI::desktop(),
                                    QObject::tr( "SMESH_WRN_WARNING" ),
                                    QObject::tr( "SMESH_DRS_SOME_EMPTY" ) );
      }
    }
  }

  void ExportMeshToFile( int theCommandID )
  {
    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected;
    if( aSel )
      aSel->selectedObjects( selected );

    SMESH::SMESH_Mesh_var aMesh;
    if(selected.Extent() == 1)
      aMesh = SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(selected.First());
    if ( aMesh->_is_nil() ) {
      SUIT_MessageBox::warning( SMESHGUI::desktop(),
                                QObject::tr( "SMESH_WRN_WARNING" ),
                                QObject::tr( "SMESH_BAD_MESH_SELECTION" ));
      return;
    }

    Handle(SALOME_InteractiveObject) anIObject = selected.First();
    QString aFilter, aTitle = QObject::tr("Export mesh");
    QMap<QString, SMESH::MED_VERSION> aFilterMap;
    QMap<QString, int> aFilterMapSTL;
    switch ( theCommandID ) {
    case 125:
    case 122:
      {
        if (aMesh->HasDuplicatedGroupNamesMED()) {
          int aRet = SUIT_MessageBox::warning
            (SMESHGUI::desktop(),
             QObject::tr("SMESH_WRN_WARNING"),
             QObject::tr("SMESH_EXPORT_MED_DUPLICATED_GRP").arg(anIObject->getName()),
             SUIT_MessageBox::Yes | SUIT_MessageBox::No, SUIT_MessageBox::Yes);
          if (aRet != SUIT_MessageBox::Yes)
            return;
        }
        // PAL18696
        QString v21 (aMesh->GetVersionString(SMESH::MED_V2_1, 2));
        QString v22 (aMesh->GetVersionString(SMESH::MED_V2_2, 2));
        aFilterMap.insert( QString("MED ") +  v21 + " (*.med)", SMESH::MED_V2_1 );
        aFilterMap.insert( QString("MED ") +  v22 + " (*.med)", SMESH::MED_V2_2 );
      }
      break;
    case 124:
    case 121:
      aFilter = QObject::tr("DAT files (*.dat)");
      break;
    case 126:
    case 123:
      {
        if (aMesh->NbPyramids()) {
          int aRet = SUIT_MessageBox::warning
            (SMESHGUI::desktop(),
             QObject::tr("SMESH_WRN_WARNING"),
             QObject::tr("SMESH_EXPORT_UNV").arg(anIObject->getName()),
             SUIT_MessageBox::Yes | SUIT_MessageBox::No, SUIT_MessageBox::Yes);
          if (aRet != SUIT_MessageBox::Yes)
            return;
        }
        aFilter = QObject::tr("IDEAS files (*.unv)");
      }
      break;
    case 140:
    case 141:
      {
        // export STL
        /*
          there must be check on others mesh elements not equal triangles
        */
        if (aMesh->NbTriangles() < 1) {
          SUIT_MessageBox::warning
            (SMESHGUI::desktop(),
             QObject::tr("SMESH_WRN_WARNING"),
             QObject::tr("SMESH_EXPORT_STL1").arg(anIObject->getName()));
          return;
        }
        if (!(aMesh->NbElements() - aMesh->NbTriangles())) {
          int aRet = SUIT_MessageBox::warning
            (SMESHGUI::desktop(),
             QObject::tr("SMESH_WRN_WARNING"),
             QObject::tr("SMESH_EXPORT_STL2").arg(anIObject->getName()),
             SUIT_MessageBox::Yes | SUIT_MessageBox::No, SUIT_MessageBox::Yes);
          if (aRet != SUIT_MessageBox::Yes)
            return;
        }

        aFilterMapSTL.insert( QObject::tr("STL ASCII  (*.stl)"), 1 ); // 1 - ASCII mode
        aFilterMapSTL.insert( QObject::tr("STL Binary (*.stl)"), 0 ); // 0 - Binary mode
      }
      break;
    default:
      return;
    }

    QString aFilename;
    SMESH::MED_VERSION aFormat;
    // Init the parameter with the default value
    bool aIsASCII_STL = true;
    bool toCreateGroups = false;
    SUIT_ResourceMgr* resMgr = SUIT_Session::session()->resourceMgr();
    if ( resMgr )
      toCreateGroups = resMgr->booleanValue( "SMESH", "auto_groups", false );

    QString anInitialPath = "";
    if ( SUIT_FileDlg::getLastVisitedPath().isEmpty() )
      anInitialPath = QDir::currentPath();

    if ( theCommandID != 122 && theCommandID != 125 && theCommandID != 140 && theCommandID != 141) {
      if ( anInitialPath.isEmpty() ) anInitialPath = SUIT_FileDlg::getLastVisitedPath();
      aFilename = SUIT_FileDlg::getFileName(SMESHGUI::desktop(), anInitialPath + QString("/") + anIObject->getName(),
                                            aFilter, aTitle, false);
    }
    else if(theCommandID == 140 || theCommandID == 141) { // Export to STL
      QStringList filters;
      QMap<QString, int>::const_iterator it = aFilterMapSTL.begin();
      for ( ; it != aFilterMapSTL.end(); ++it )
        filters.push_back( it.key() );

      SUIT_FileDlg* fd = new SUIT_FileDlg( SMESHGUI::desktop(), false, true, true );
      fd->setWindowTitle( aTitle );
      fd->setFilters( filters );
      fd->selectFilter( QObject::tr("STL ASCII  (*.stl)") );
      if ( !anInitialPath.isEmpty() )
        fd->setDirectory( anInitialPath );
      fd->selectFile(anIObject->getName());
      bool is_ok = false;
      while (!is_ok) {
        if ( fd->exec() )
          aFilename = fd->selectedFile();
        aIsASCII_STL = (aFilterMapSTL[fd->selectedFilter()]) == 1 ? true: false;
        is_ok = true;
      }
      delete fd;
    }
    else { // Export to MED
      QStringList filters;
      QString aDefaultFilter;
      QMap<QString, SMESH::MED_VERSION>::const_iterator it = aFilterMap.begin();
      for ( ; it != aFilterMap.end(); ++it ) {
        filters.push_back( it.key() );
        if (it.value() == SMESH::MED_V2_2)
          aDefaultFilter = it.key();
      }

      //SUIT_FileDlg* fd = new SUIT_FileDlg( SMESHGUI::desktop(), false, true, true );
      SalomeApp_CheckFileDlg* fd = new SalomeApp_CheckFileDlg
        ( SMESHGUI::desktop(), false, QObject::tr("SMESH_AUTO_GROUPS"), true, true );
      fd->setWindowTitle( aTitle );
      fd->setFilters( filters );
      //fd->setSelectedFilter( QObject::tr("MED 2.2 (*.med)") );
      fd->selectFilter(aDefaultFilter);
      fd->SetChecked(toCreateGroups);
      if ( !anInitialPath.isEmpty() )
        fd->setDirectory( anInitialPath );
      fd->selectFile(anIObject->getName());
      bool is_ok = false;
      while (!is_ok) {
        if ( fd->exec() )
          aFilename = fd->selectedFile();
        aFormat = aFilterMap[fd->selectedFilter()];
        is_ok = true;
        if ( !aFilename.isEmpty()
             && (aMesh->NbPolygons()>0 || aMesh->NbPolyhedrons()>0)
             && aFormat==SMESH::MED_V2_1) {
          int aRet = SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                              QObject::tr("SMESH_WRN_WARNING"),
                                              QObject::tr("SMESH_EXPORT_MED_V2_1").arg(anIObject->getName()),
                                              SUIT_MessageBox::Yes | SUIT_MessageBox::No, SUIT_MessageBox::Yes);
          if (aRet != SUIT_MessageBox::Yes) {
            is_ok = false;
          }
        }
      }
      toCreateGroups = fd->IsChecked();
      delete fd;
    }
    if ( !aFilename.isEmpty() ) {
      // Check whether the file already exists and delete it if yes
      QFile aFile( aFilename );
      if ( aFile.exists() )
        aFile.remove();
      SUIT_OverrideCursor wc;

      try {
        bool Renumber = false;
        // PAL 14172  : Check of we have to renumber or not from the preferences before export
        if (resMgr)
          Renumber= resMgr->booleanValue("SMESH","renumbering");
        if (Renumber){
          SMESH::SMESH_MeshEditor_var aMeshEditor = aMesh->GetMeshEditor();
          aMeshEditor->RenumberNodes();
          aMeshEditor->RenumberElements();
          if ( SMESHGUI::automaticUpdate() )
            SMESH::UpdateView();
        }
        switch ( theCommandID ) {
        case 125:
        case 122:
          aMesh->ExportToMED( aFilename.toLatin1().data(), toCreateGroups, aFormat );
          break;
        case 124:
        case 121:
          aMesh->ExportDAT( aFilename.toLatin1().data() );
          break;
        case 126:
        case 123:
          aMesh->ExportUNV( aFilename.toLatin1().data() );
          break;
        case 140:
        case 141:
          aMesh->ExportSTL( aFilename.toLatin1().data(), aIsASCII_STL );
          break;
        default:
          break;
        }
      }
      catch (const SALOME::SALOME_Exception& S_ex){
        wc.suspend();
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 QObject::tr("SMESH_WRN_WARNING"),
                                 QObject::tr("SMESH_EXPORT_FAILED"));
        wc.resume();
      }
    }
  }

  inline void InverseEntityMode(unsigned int& theOutputMode,
                                unsigned int theMode)
  {
    bool anIsNotPresent = ~theOutputMode & theMode;
    if(anIsNotPresent)
      theOutputMode |= theMode;
    else
      theOutputMode &= ~theMode;
  }

  void SetDisplayEntity(int theCommandID){
    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected;
    if( aSel )
      aSel->selectedObjects( selected );

    if(selected.Extent() >= 1){
      SALOME_ListIteratorOfListIO It( selected );
      for( ; It.More(); It.Next()){
        Handle(SALOME_InteractiveObject) IObject = It.Value();
        if(IObject->hasEntry()){
          if(SMESH_Actor *anActor = SMESH::FindActorByEntry(IObject->getEntry())){
            unsigned int aMode = anActor->GetEntityMode();
            switch(theCommandID){
            case 216:
              InverseEntityMode(aMode,SMESH_Actor::e0DElements);
              break;
            case 217:
              InverseEntityMode(aMode,SMESH_Actor::eEdges);
              break;
            case 218:
              InverseEntityMode(aMode,SMESH_Actor::eFaces);
              break;
            case 219:
              InverseEntityMode(aMode,SMESH_Actor::eVolumes);
              break;
            case 220:
              aMode = SMESH_Actor::eAllEntity;
              break;
            }
            if(aMode)
              anActor->SetEntityMode(aMode);
          }
        }
      }
    }
  }

  void AutoColor(){
    SALOME_ListIO selected;
    SalomeApp_Application* app = dynamic_cast< SalomeApp_Application* >( SUIT_Session::session()->activeApplication() );
    if( !app )
      return;

    LightApp_SelectionMgr* aSel = app->selectionMgr();
    SalomeApp_Study* appStudy = dynamic_cast<SalomeApp_Study*>( app->activeStudy() );
    if( !aSel || !appStudy )
      return;

    aSel->selectedObjects( selected );
    if( selected.IsEmpty() )
      return;

    Handle(SALOME_InteractiveObject) anIObject = selected.First();

    _PTR(Study) aStudy = appStudy->studyDS();
    _PTR(SObject) aMainSObject( aStudy->FindObjectID( anIObject->getEntry() ) );
    SMESH::SMESH_Mesh_var aMainObject = SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(anIObject);
    if( aMainObject->_is_nil() )
      return;

    aMainObject->SetAutoColor( true );

    QList<SALOMEDS::Color> aReservedColors;

    SMESH::ListOfGroups aListOfGroups = *aMainObject->GetGroups();
    for( int i = 0, n = aListOfGroups.length(); i < n; i++ )
    {
      SMESH::SMESH_GroupBase_var aGroupObject = aListOfGroups[i];
      SALOMEDS::Color aCurrentColor = aGroupObject->GetColor();

      SALOMEDS::Color aColor = SMESHGUI::getUniqueColor( aReservedColors );
      aGroupObject->SetColor( aColor );
      aReservedColors.append( aColor );

      _PTR(SObject) aGroupSObject = SMESH::FindSObject(aGroupObject);
      if(SMESH_Actor *anActor = SMESH::FindActorByEntry(aGroupSObject->GetID().c_str()))
      {
        if( aGroupObject->GetType() == SMESH::NODE )
          anActor->SetNodeColor( aColor.R, aColor.G, aColor.B );
        else if( aGroupObject->GetType() == SMESH::EDGE )
          anActor->SetEdgeColor( aColor.R, aColor.G, aColor.B );
        else
          anActor->SetSufaceColor( aColor.R, aColor.G, aColor.B );
      }
    }

    SMESH::RepaintCurrentView();
  }

  void DisableAutoColor(){
    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected;
    if( aSel )
      aSel->selectedObjects( selected );

    if(selected.Extent()){
      Handle(SALOME_InteractiveObject) anIObject = selected.First();
      SMESH::SMESH_Mesh_var aMesh = SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(anIObject);
      if ( !aMesh->_is_nil() ) {
        aMesh->SetAutoColor( false );
      }
    }
  }

  void SetDisplayMode(int theCommandID){
    SALOME_ListIO selected;
    SalomeApp_Application* app = dynamic_cast< SalomeApp_Application* >( SUIT_Session::session()->activeApplication() );
    if( !app )
      return;

    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SalomeApp_Study* appStudy = dynamic_cast<SalomeApp_Study*>( app->activeStudy() );
    if( !aSel || !appStudy )
      return;

    _PTR(Study) aStudy = appStudy->studyDS();

    aSel->selectedObjects( selected );

    if(selected.Extent() >= 1){
      switch(theCommandID){
      case 1134:{
        SMESHGUI::GetSMESHGUI()->EmitSignalDeactivateDialog();
        (new SMESHGUI_ClippingDlg( SMESHGUI::GetSMESHGUI() ))->show();
        return;
      }
      case 1133:{
        SMESHGUI::GetSMESHGUI()->EmitSignalDeactivateDialog();
        (new SMESHGUI_TransparencyDlg( SMESHGUI::GetSMESHGUI() ))->show();
        return;
      }}
      SALOME_ListIteratorOfListIO It( selected );
      for( ; It.More(); It.Next()){
        Handle(SALOME_InteractiveObject) IObject = It.Value();
        if(IObject->hasEntry()){
          if(SMESH_Actor *anActor = SMESH::FindActorByEntry(IObject->getEntry())){
            switch(theCommandID){
            case 211:
              anActor->SetRepresentation(SMESH_Actor::eEdge);
              break;
            case 212:
              anActor->SetRepresentation(SMESH_Actor::eSurface);
              break;
            case 213:
              if(anActor->IsShrunk())
                anActor->UnShrink();
              else
                anActor->SetShrink();
              break;
            case 215:
              anActor->SetRepresentation(SMESH_Actor::ePoint);
              break;
            case 231:
              if(anActor->GetQuadratic2DRepresentation() != SMESH_Actor::eLines)
                anActor->SetQuadratic2DRepresentation(SMESH_Actor::eLines);
              break;
            case 232:
              if(anActor->GetQuadratic2DRepresentation() != SMESH_Actor::eArcs)
                anActor->SetQuadratic2DRepresentation(SMESH_Actor::eArcs);
              break;
            case 1132:{
              vtkFloatingPointType color[3];
              anActor->GetSufaceColor(color[0], color[1], color[2]);
              int c0 = int (color[0] * 255);
              int c1 = int (color[1] * 255);
              int c2 = int (color[2] * 255);
              QColor c(c0, c1, c2);

              vtkFloatingPointType edgecolor[3];
              anActor->GetEdgeColor(edgecolor[0], edgecolor[1], edgecolor[2]);
              c0 = int (edgecolor[0] * 255);
              c1 = int (edgecolor[1] * 255);
              c2 = int (edgecolor[2] * 255);
              QColor e(c0, c1, c2);

              vtkFloatingPointType backfacecolor[3];
              anActor->GetBackSufaceColor(backfacecolor[0], backfacecolor[1], backfacecolor[2]);
              c0 = int (backfacecolor[0] * 255);
              c1 = int (backfacecolor[1] * 255);
              c2 = int (backfacecolor[2] * 255);
              QColor b(c0, c1, c2);

              vtkFloatingPointType nodecolor[3];
              anActor->GetNodeColor(nodecolor[0], nodecolor[1], nodecolor[2]);
              c0 = int (nodecolor[0] * 255);
              c1 = int (nodecolor[1] * 255);
              c2 = int (nodecolor[2] * 255);
              QColor n(c0, c1, c2);

              int Edgewidth = (int)anActor->GetLineWidth();
              if(Edgewidth == 0)
                Edgewidth = 1;
              int intValue = int(anActor->GetNodeSize());
              vtkFloatingPointType Shrink = anActor->GetShrinkFactor();

              SMESHGUI_Preferences_ColorDlg *aDlg =
                new SMESHGUI_Preferences_ColorDlg( SMESHGUI::GetSMESHGUI() );
              aDlg->SetColor(1, c);
              aDlg->SetColor(2, e);
              aDlg->SetColor(3, n);
              aDlg->SetColor(4, b);
              aDlg->SetIntValue(1, Edgewidth);
              aDlg->SetIntValue(2, intValue);
              aDlg->SetIntValue(3, int(Shrink*100.));
              if(aDlg->exec()){
                QColor color = aDlg->GetColor(1);
                QColor edgecolor = aDlg->GetColor(2);
                QColor nodecolor = aDlg->GetColor(3);
                QColor backfacecolor = aDlg->GetColor(4);
                /* actor color and backface color */
                anActor->SetSufaceColor(vtkFloatingPointType (color.red()) / 255.,
                                        vtkFloatingPointType (color.green()) / 255.,
                                        vtkFloatingPointType (color.blue()) / 255.);
                anActor->SetBackSufaceColor(vtkFloatingPointType (backfacecolor.red()) / 255.,
                                            vtkFloatingPointType (backfacecolor.green()) / 255.,
                                            vtkFloatingPointType (backfacecolor.blue()) / 255.);

                /* edge color */
                anActor->SetEdgeColor(vtkFloatingPointType (edgecolor.red()) / 255.,
                                      vtkFloatingPointType (edgecolor.green()) / 255.,
                                      vtkFloatingPointType (edgecolor.blue()) / 255.);

                /* Shrink factor and size edges */
                anActor->SetShrinkFactor(aDlg->GetIntValue(3) / 100.);
                anActor->SetLineWidth(aDlg->GetIntValue(1));

                /* Nodes color and size */
                anActor->SetNodeColor(vtkFloatingPointType (nodecolor.red()) / 255.,
                                      vtkFloatingPointType (nodecolor.green()) / 255.,
                                      vtkFloatingPointType (nodecolor.blue()) / 255.);
                anActor->SetNodeSize(aDlg->GetIntValue(2));

                SMESH::SMESH_GroupBase_var aGroupObject = SMESH::IObjectToInterface<SMESH::SMESH_GroupBase>(IObject);
                if( !aGroupObject->_is_nil() )
                {
                  SMESH::ElementType anElementType = aGroupObject->GetType();
                  QColor aColor;
                  switch( anElementType )
                  {
                    case SMESH::NODE: aColor = nodecolor; break;
                    case SMESH::EDGE: aColor = edgecolor; break;
                    default: aColor = color; break;
                  }

                  SALOMEDS::Color aGroupColor;
                  aGroupColor.R = (float)aColor.red() / 255.0;
                  aGroupColor.G = (float)aColor.green() / 255.0;
                  aGroupColor.B = (float)aColor.blue() / 255.0;
                  aGroupObject->SetColor( aGroupColor );
                }

                delete aDlg;
              }
              break;
            }}
          }
        }
      }
      SMESH::RepaintCurrentView();
    }
  }

  void Control( int theCommandID )
  {
    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected;
    if( aSel )
      aSel->selectedObjects( selected );

    if( !selected.IsEmpty() ){
      Handle(SALOME_InteractiveObject) anIO = selected.First();
      if(!anIO.IsNull()){
        QString aTitle;
        SMESH_Actor::eControl aControl = SMESH_Actor::eNone;
        if(SMESH_Actor *anActor = SMESH::FindActorByEntry(anIO->getEntry())){
          switch ( theCommandID ){
          case 6001:
            aTitle = QObject::tr( "LENGTH_EDGES" );
            aControl = SMESH_Actor::eLength;
            break;
          case 6018:
            aTitle = QObject::tr( "LENGTH2D_EDGES" );
            aControl = SMESH_Actor::eLength2D;
            break;
          case 6002:
            aTitle = QObject::tr( "FREE_EDGES" );
            aControl = SMESH_Actor::eFreeEdges;
            break;
          case 6003:
            aTitle = QObject::tr( "FREE_BORDERS" );
            aControl = SMESH_Actor::eFreeBorders;
            break;
          case 6004:
            aTitle = QObject::tr( "MULTI_BORDERS" );
            aControl = SMESH_Actor::eMultiConnection;
            break;
          case 6005:
            aTitle = QObject::tr( "FREE_NODES" );
            aControl = SMESH_Actor::eFreeNodes;
            break;
          case 6019:
            aTitle = QObject::tr( "MULTI2D_BORDERS" );
            aControl = SMESH_Actor::eMultiConnection2D;
            break;
          case 6011:
            aTitle = QObject::tr( "AREA_ELEMENTS" );
            aControl = SMESH_Actor::eArea;
            break;
          case 6012:
            aTitle = QObject::tr( "TAPER_ELEMENTS" );
            aControl = SMESH_Actor::eTaper;
            break;
          case 6013:
            aTitle = QObject::tr( "ASPECTRATIO_ELEMENTS" );
            aControl = SMESH_Actor::eAspectRatio;
            break;
          case 6017:
            aTitle = QObject::tr( "ASPECTRATIO_3D_ELEMENTS" );
            aControl = SMESH_Actor::eAspectRatio3D;
            break;
          case 6014:
            aTitle = QObject::tr( "MINIMUMANGLE_ELEMENTS" );
            aControl = SMESH_Actor::eMinimumAngle;
            break;
          case 6015:
            aTitle = QObject::tr( "WARP_ELEMENTS" );
            aControl = SMESH_Actor::eWarping;
            break;
          case 6016:
            aTitle = QObject::tr( "SKEW_ELEMENTS" );
            aControl = SMESH_Actor::eSkew;
            break;
          case 6009:
            aTitle = QObject::tr( "SMESH_VOLUME" );
            aControl = SMESH_Actor::eVolume3D;
            break;
          case 6021:
            aTitle = QObject::tr( "FREE_FACES" );
            aControl = SMESH_Actor::eFreeFaces;
            break;
          }
          anActor->SetControlMode(aControl);
          anActor->GetScalarBarActor()->SetTitle(aTitle.toLatin1().data());
          SMESH::RepaintCurrentView();
        }
      }
    }
  }


  bool CheckOIType(const Handle(SALOME_InteractiveObject) & theIO,
                   MeshObjectType                           theType,
                   const QString                            theInTypeName,
                   QString &                                theOutTypeName)
  {
    SMESH_TypeFilter aTypeFilter( theType );
    QString entry;
    if( !theIO.IsNull() )
    {
      entry = theIO->getEntry();
      LightApp_DataOwner owner( entry );
      if ( aTypeFilter.isOk( &owner )) {
        theOutTypeName = theInTypeName;
        return true;
      }
    }
    return false;
  }


  QString CheckTypeObject(const Handle(SALOME_InteractiveObject) & theIO)
  {
    _PTR(Study) aStudy = SMESH::GetActiveStudyDocument();
    _PTR(SObject) aSObj = aStudy->FindObjectID(theIO->getEntry());
    if (aSObj) {
      _PTR(SComponent) aSComp = aSObj->GetFatherComponent();
      CORBA::String_var anID = aSComp->GetID().c_str();
      if (!strcmp(anID.in(),theIO->getEntry()))
        return "Component";
    }

    QString aTypeName;
    if (
        CheckOIType ( theIO, HYPOTHESIS,    "Hypothesis", aTypeName ) ||
        CheckOIType ( theIO, ALGORITHM,     "Algorithm",  aTypeName ) ||
        CheckOIType ( theIO, MESH,          "Mesh",       aTypeName ) ||
        CheckOIType ( theIO, SUBMESH,       "SubMesh",    aTypeName ) ||
        CheckOIType ( theIO, GROUP,         "Group",      aTypeName )
        )
      return aTypeName;

    return "NoType";
  }


  QString CheckHomogeneousSelection()
  {
    //SUIT_Study* aStudy = SMESH::GetActiveStudy();
    LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected;
    if( aSel )
      aSel->selectedObjects( selected );

    QString RefType = CheckTypeObject(selected.First());
    SALOME_ListIteratorOfListIO It(selected);
    for ( ; It.More(); It.Next())
      {
        Handle(SALOME_InteractiveObject) IObject = It.Value();
        QString Type = CheckTypeObject(IObject);
        if (Type.compare(RefType) != 0)
          return "Heterogeneous Selection";
      }

    return RefType;
  }


  void SMESHGUI::OnEditDelete()
  {
    // VSR 17/11/04: check if all objects selected belong to SMESH component --> start
    LightApp_SelectionMgr* aSel = SMESHGUI::selectionMgr();
    SALOME_ListIO selected; aSel->selectedObjects( selected, QString::null, false );

    _PTR(Study) aStudy = SMESH::GetActiveStudyDocument();
    _PTR(StudyBuilder) aStudyBuilder = aStudy->NewBuilder();
    _PTR(GenericAttribute) anAttr;
    _PTR(AttributeIOR) anIOR;

    int objectCount = 0;
    QString aNameList;
    QString aParentComponent = QString::null;
    Handle(SALOME_InteractiveObject) anIO;
    for( SALOME_ListIteratorOfListIO anIt( selected ); anIt.More(); anIt.Next() )
    {
      anIO = anIt.Value();
      QString cur = anIO->getComponentDataType();
      _PTR(SObject) aSO = aStudy->FindObjectID(anIO->getEntry());
      if (aSO) {
        // check if object is reference
        _PTR(SObject) aRefSObj;
        aNameList.append("\n    - ");
        if ( aSO->ReferencedObject( aRefSObj ) ) {
          QString aRefName = QString::fromStdString ( aRefSObj->GetName() );
          aNameList.append( aRefName );
          cur = QString::fromStdString ( aRefSObj->GetFatherComponent()->ComponentDataType() );
        }
        else
          aNameList.append(anIO->getName());
        objectCount++;
      }

      if( aParentComponent.isNull() )
        aParentComponent = cur;
      else if( !aParentComponent.isEmpty() && aParentComponent!=cur )
        aParentComponent = "";
    }

    if ( objectCount == 0 )
      return; // No Valid Objects Selected

    if ( aParentComponent != SMESHGUI::GetSMESHGUI()->name() )  {
      SUIT_MessageBox::warning( SMESHGUI::desktop(),
                                QObject::tr("ERR_ERROR"),
                                QObject::tr("NON_SMESH_OBJECTS_SELECTED").arg( SMESHGUI::GetSMESHGUI()->moduleName() ) );
      return;
    }
    // VSR 17/11/04: check if all objects selected belong to SMESH component <-- finish
    if (SUIT_MessageBox::warning
        (SMESHGUI::desktop(),
         QObject::tr("SMESH_WRN_WARNING"),
         QObject::tr("SMESH_REALLY_DELETE").arg( objectCount ).arg( aNameList ),
         SUIT_MessageBox::Yes | SUIT_MessageBox::No,
         SUIT_MessageBox::Yes) != SUIT_MessageBox::Yes)
      return;

    SalomeApp_Application* anApp = dynamic_cast<SalomeApp_Application*>( SUIT_Session::session()->activeApplication() );
    SUIT_ViewManager* vm = anApp->activeViewManager();
    int nbSf = vm->getViewsCount();

    SALOME_ListIteratorOfListIO It(selected);

    aStudyBuilder->NewCommand();  // There is a transaction
    for( ; It.More(); It.Next()){ // loop on selected IO's
      Handle(SALOME_InteractiveObject) IObject = It.Value();
      if(IObject->hasEntry()) {
        _PTR(SObject) aSO = aStudy->FindObjectID(IObject->getEntry());

        // disable removal of "SMESH" component object
        if(aSO->FindAttribute(anAttr, "AttributeIOR")){
          anIOR = anAttr;
          if ( engineIOR() == anIOR->Value().c_str() )
            continue;
        }
        //Check the referenced object
        _PTR(SObject) aRefSObject;
        if ( aSO && aSO->ReferencedObject( aRefSObject ) )
          aSO = aRefSObject; // Delete main Object instead of reference

        // put the whole hierarchy of sub-objects of the selected SO into a list and
        // then treat them all starting from the deepest objects (at list back)

        std::list< _PTR(SObject) > listSO;
        listSO.push_back( aSO );
        std::list< _PTR(SObject) >::iterator itSO = listSO.begin();
        for ( ; itSO != listSO.end(); ++itSO ) {
          _PTR(ChildIterator) it = aStudy->NewChildIterator( *itSO );
          for (it->InitEx(false); it->More(); it->Next())
            listSO.push_back( it->Value() );
        }

        // treat SO's in the list starting from the back

        std::list< _PTR(SObject) >::reverse_iterator ritSO = listSO.rbegin();
        for ( ; ritSO != listSO.rend(); ++ritSO ) {
          _PTR(SObject) SO = *ritSO;
          if ( !SO ) continue;
          std::string anEntry = SO->GetID();

          /** Erase graphical object **/
          if(SO->FindAttribute(anAttr, "AttributeIOR")){
            QVector<SUIT_ViewWindow*> aViews = vm->getViews();
            for(int i = 0; i < nbSf; i++){
              SUIT_ViewWindow *sf = aViews[i];
              if(SMESH_Actor* anActor = SMESH::FindActorByEntry(sf,anEntry.c_str())){
                SMESH::RemoveActor(sf,anActor);
              }
            }
          }

          /** Remove an object from data structures **/
          SMESH::SMESH_GroupBase_var aGroup = SMESH::SMESH_GroupBase::_narrow( SMESH::SObjectToObject( SO ));
          SMESH::SMESH_subMesh_var   aSubMesh = SMESH::SMESH_subMesh::_narrow( SMESH::SObjectToObject( SO ));
          if ( !aGroup->_is_nil() ) {                          // DELETE GROUP
            SMESH::SMESH_Mesh_var aMesh = aGroup->GetMesh();
            aMesh->RemoveGroup( aGroup );
          }
          else if ( !aSubMesh->_is_nil() ) {                   // DELETE SUBMESH
            SMESH::SMESH_Mesh_var aMesh = aSubMesh->GetFather();
            aMesh->RemoveSubMesh( aSubMesh );

            _PTR(SObject) aMeshSO = SMESH::FindSObject(aMesh);
            if (aMeshSO)
              SMESH::ModifiedMesh(aMeshSO, false, aMesh->NbNodes()==0);
          }
          else {
            IObject = new SALOME_InteractiveObject
              ( anEntry.c_str(), engineIOR().toLatin1().data(), SO->GetName().c_str() );
            QString objType = CheckTypeObject(IObject);
            if ( objType == "Hypothesis" || objType == "Algorithm" ) {// DELETE HYPOTHESIS
              SMESH::RemoveHypothesisOrAlgorithmOnMesh(IObject);
              aStudyBuilder->RemoveObjectWithChildren( SO );
            }
            else {// default action: remove SObject from the study
              // san - it's no use opening a transaction here until UNDO/REDO is provided in SMESH
              //SUIT_Operation *op = new SALOMEGUI_ImportOperation(myActiveStudy);
              //op->start();
              aStudyBuilder->RemoveObjectWithChildren( SO );
              //op->finish();
            }
          }
        } /* listSO back loop */
      } /* IObject->hasEntry() */
    } /* more/next */
    aStudyBuilder->CommitCommand();

    /* Clear any previous selection */
    SALOME_ListIO l1;
    aSel->setSelectedObjects( l1 );

    SMESHGUI::GetSMESHGUI()->updateObjBrowser();
  }
//}

extern "C" {
  SMESHGUI_EXPORT CAM_Module* createModule()
  {
    return new SMESHGUI();
  }
}

SMESH::SMESH_Gen_var SMESHGUI::myComponentSMESH = SMESH::SMESH_Gen::_nil();

//=============================================================================
/*!
 *
 */
//=============================================================================
SMESHGUI::SMESHGUI() :
SalomeApp_Module( "SMESH" ),
LightApp_Module( "SMESH" )
{
  if ( CORBA::is_nil( myComponentSMESH ) )
  {
    CORBA::Boolean anIsEmbeddedMode;
    myComponentSMESH = SMESH_Client::GetSMESHGen(getApp()->orb(),anIsEmbeddedMode);

    //  0019923: EDF 765 SMESH : default values of hypothesis
    SUIT_ResourceMgr* aResourceMgr = SMESH::GetResourceMgr(this);
    int nbSeg = aResourceMgr->integerValue( "SMESH", "segmentation", 10 );
    myComponentSMESH->SetBoundaryBoxSegmentation( nbSeg );
    nbSeg = aResourceMgr->integerValue( "SMESH", "nb_segments_per_edge", 15 );
    myComponentSMESH->SetDefaultNbSegments( nbSeg );
  }

  myActiveDialogBox = 0;
  myFilterLibraryDlg = 0;
  myState = -1;
  myDisplayer = 0;

  SMESH::GetFilterManager();
  SMESH::GetPattern();

  /* load resources for all available meshers */
  SMESH::InitAvailableHypotheses();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
SMESHGUI::~SMESHGUI()
{
  SMESH::GetFilterManager()->Destroy();
  SMESH::GetFilterManager() = SMESH::FilterManager::_nil();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
LightApp_SelectionMgr* SMESHGUI::selectionMgr()
{
  SalomeApp_Application* anApp = dynamic_cast<SalomeApp_Application*>( SUIT_Session::session()->activeApplication() );
  if( anApp )
    return dynamic_cast<LightApp_SelectionMgr*>( anApp->selectionMgr() );
  else
    return 0;
}

bool SMESHGUI::automaticUpdate()
{
  SUIT_ResourceMgr* resMgr = SUIT_Session::session()->resourceMgr();
  if ( !resMgr )
    return false;

  return resMgr->booleanValue( "SMESH", "auto_update", false );
}

//=============================================================================
/*!
 *
 */
//=============================================================================
SUIT_ResourceMgr* SMESHGUI::resourceMgr()
{
  return dynamic_cast<SUIT_ResourceMgr*>( SUIT_Session::session()->resourceMgr() );
}

//=============================================================================
/*!
 *
 */
//=============================================================================
SMESHGUI* SMESHGUI::GetSMESHGUI()
{
  SMESHGUI* smeshMod = 0;
  SalomeApp_Application* app = dynamic_cast<SalomeApp_Application*>(SUIT_Session::session()->activeApplication());
  if ( app )
  {
    CAM_Module* module = app->module( "Mesh" );
    smeshMod = dynamic_cast<SMESHGUI*>( module );
  }

  if ( smeshMod && smeshMod->application() && smeshMod->application()->activeStudy() )
  {
    SalomeApp_Study* study = dynamic_cast<SalomeApp_Study*>( smeshMod->application()->activeStudy() );
    if ( study )
    {
      _PTR(Study) aStudy = study->studyDS();
      if ( aStudy )
        GetSMESHGen()->SetCurrentStudy( _CAST(Study,aStudy)->GetStudy() );
    }
  }

  return smeshMod;
}

extern "C"
{
  Standard_EXPORT SMESHGUI* GetComponentGUI()
  {
    return SMESHGUI::GetSMESHGUI();
  }
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::SetState(int aState)
{
  myState = aState;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::ResetState()
{
  myState = -1;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::EmitSignalDeactivateDialog()
{
  emit SignalDeactivateActiveDialog();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::EmitSignalStudyFrameChanged()
{
  emit SignalStudyFrameChanged();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::EmitSignalCloseAllDialogs()
{
  emit SignalCloseAllDialogs();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::EmitSignalVisibilityChanged()
{
  emit SignalVisibilityChanged();
}

//=============================================================================
/*!
 *
 */
//=============================================================================
QDialog *SMESHGUI::GetActiveDialogBox()
{
  return myActiveDialogBox;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
void SMESHGUI::SetActiveDialogBox(QDialog * aDlg)
{
  myActiveDialogBox = (QDialog *) aDlg;
  return;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
SUIT_Desktop* SMESHGUI::desktop()
{
  SalomeApp_Application* app = dynamic_cast<SalomeApp_Application*>( SUIT_Session::session()->activeApplication() );
  if( app )
    return app->desktop();
  else
    return 0;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
SalomeApp_Study* SMESHGUI::activeStudy()
{
  SUIT_Application* app = SUIT_Session::session()->activeApplication();
  if( app )
    return dynamic_cast<SalomeApp_Study*>( app->activeStudy() );
  else
    return NULL;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
bool SMESHGUI::DefineDlgPosition(QWidget * aDlg, int &x, int &y)
{
  /* Here the position is on the bottom right corner - 10 */
  // aDlg->resize(QSize().expandedTo(aDlg->minimumSizeHint()));
  aDlg->adjustSize();
  SUIT_Desktop *PP = desktop();
  x = abs(PP->x() + PP->size().width() - aDlg->size().width() - 10);
  y = abs(PP->y() + PP->size().height() - aDlg->size().height() - 10);
  return true;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
static int isStudyLocked(_PTR(Study) theStudy){
  return theStudy->GetProperties()->IsLocked();
}

static bool checkLock(_PTR(Study) theStudy) {
  if (isStudyLocked(theStudy)) {
    SUIT_MessageBox::warning( SMESHGUI::desktop(),
                              QObject::tr("WRN_WARNING"),
                              QObject::tr("WRN_STUDY_LOCKED") );
    return true;
  }
  return false;
}

//=======================================================================
//function : CheckActiveStudyLocked
//purpose  :
//=======================================================================

bool SMESHGUI::isActiveStudyLocked()
{
  _PTR(Study) aStudy = activeStudy()->studyDS();
  return checkLock( aStudy );
}

//=============================================================================
/*!
 *
 */
//=============================================================================
bool SMESHGUI::OnGUIEvent( int theCommandID )
{
  SalomeApp_Application* anApp = dynamic_cast<SalomeApp_Application*>( application() );
  if( !anApp )
    return false;

  _PTR(Study) aStudy = SMESH::GetActiveStudyDocument(); //Document OCAF de l'etude active
  SUIT_ResourceMgr* mgr = resourceMgr();
  if( !mgr )
    return false;

  if (CORBA::is_nil(GetSMESHGen()->GetCurrentStudy())) {
    GetSMESHGen()->SetCurrentStudy(_CAST(Study,aStudy)->GetStudy());
  }

  SUIT_ViewWindow* view = application()->desktop()->activeWindow();
  SVTK_ViewWindow* vtkwnd = dynamic_cast<SVTK_ViewWindow*>( view );

  //QAction* act = action( theCommandID );

  switch (theCommandID) {
  case 33:                                      // DELETE
    if(checkLock(aStudy)) break;
    OnEditDelete();
    break;

  case 113:                                     // IMPORT
  case 112:
  case 111:
    {
      if(checkLock(aStudy)) break;
      ::ImportMeshesFromFile(GetSMESHGen(),theCommandID);
      break;
    }

  case 150:    //MED FILE INFORMATION
    {
      SALOME_ListIO selected;
      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      if( aSel )
        aSel->selectedObjects( selected );
      if( selected.Extent() )
      {
        Handle(SALOME_InteractiveObject) anIObject = selected.First();
        SMESH::SMESH_Mesh_var aMesh = SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(anIObject);
        if ( !aMesh->_is_nil() )
        {
          SMESHGUI_FileInfoDlg dlg( desktop(), aMesh->GetMEDFileInfo() );
          dlg.exec();
        }
      }
      break;
    }

  case 122:                                     // EXPORT MED
  case 121:
  case 123:
  case 124:
  case 125:
  case 126:
  case 140:
  case 141:
    {
      ::ExportMeshToFile(theCommandID);
      break;
    }

  case 200:                                     // SCALAR BAR
    {
      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      if( selected.Extent() ) {
        Handle(SALOME_InteractiveObject) anIO = selected.First();
        if( anIO->hasEntry() ) {
          if( SMESH_Actor* anActor = SMESH::FindActorByEntry( anIO->getEntry() ) ) {
            anActor->SetControlMode( SMESH_Actor::eNone );
          }
        }
      }
      break;
    }
  case 201:
    {
      SMESHGUI_Preferences_ScalarBarDlg::ScalarBarProperties( this );
      break;
    }

    // Auto-color
  case 1136:
    ::AutoColor();
  break;

  case 1137:
    ::DisableAutoColor();
  break;

  case 1134: // Clipping
  case 1133: // Tranparency
  case 1132: // Colors / Size

    // Display Mode
  case 215: // Nodes
  case 213: // Nodes
  case 212: // Nodes
  case 211: // Nodes
    ::SetDisplayMode(theCommandID);
  break;

  //2D quadratic representation
  case 231:
  case 232:
    ::SetDisplayMode(theCommandID);
  break;
  
  // Display Entity
  case 216: // 0D elements
  case 217: // Edges
  case 218: // Faces
  case 219: // Volumes
  case 220: // All Entity
    ::SetDisplayEntity(theCommandID);
  break;

  case 221: // Orientation of faces
    {
      LightApp_SelectionMgr* mgr = selectionMgr();
      SALOME_ListIO selected; mgr->selectedObjects( selected );

      SALOME_ListIteratorOfListIO it(selected);
      for( ; it.More(); it.Next()) {
        Handle(SALOME_InteractiveObject) anIObject = it.Value();
        if(anIObject->hasEntry()) {
          if(SMESH_Actor *anActor = SMESH::FindActorByEntry(anIObject->getEntry())){
            anActor->SetFacesOriented( !anActor->GetFacesOriented() );
          }
        }
      }
      break;
    }

  case 214:                                     // UPDATE
    {
      if(checkLock(aStudy)) break;
      try {
#if (OCC_VERSION_MAJOR << 16 | OCC_VERSION_MINOR << 8 | OCC_VERSION_MAINTENANCE) > 0x060100
        OCC_CATCH_SIGNALS;
#endif
        SMESH::UpdateView();
      }
      catch (std::bad_alloc) { // PAL16774 (Crash after display of many groups)
        SMESH::OnVisuException();
      }
      catch (...) { // PAL16774 (Crash after display of many groups)
        SMESH::OnVisuException();
      }

      SALOME_ListIO l;
      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      aSel->selectedObjects( l );
      aSel->setSelectedObjects( l );
      break;
    }

  case 300:                                     // ERASE
  case 301:                                     // DISPLAY
  case 302:                                     // DISPLAY ONLY
    {
      SMESH::EDisplaing anAction;
      switch (theCommandID) {
      case 300: anAction = SMESH::eErase; break;
      case 301: anAction = SMESH::eDisplay; break;
      case 302: anAction = SMESH::eDisplayOnly; break;
      }

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO sel_objects, to_process;
      if (aSel)
        aSel->selectedObjects( sel_objects );

      if( theCommandID==302 )
        startOperation( myEraseAll );

      extractContainers( sel_objects, to_process );

      try {
#if (OCC_VERSION_MAJOR << 16 | OCC_VERSION_MINOR << 8 | OCC_VERSION_MAINTENANCE) > 0x060100
        OCC_CATCH_SIGNALS;
#endif
        if (vtkwnd) {
          SALOME_ListIteratorOfListIO It( to_process );
          for ( ; It.More(); It.Next()) {
            Handle(SALOME_InteractiveObject) IOS = It.Value();
            if (IOS->hasEntry()) {
              if (!SMESH::UpdateView(anAction, IOS->getEntry())) {
                SMESHGUI::GetSMESHGUI()->EmitSignalVisibilityChanged();
                break; // PAL16774 (Crash after display of many groups)
              }
              if (anAction == SMESH::eDisplayOnly)
                anAction = SMESH::eDisplay;
            }
          }
        }

        // PAL13338 + PAL15161 -->
        if ( ( theCommandID==301 || theCommandID==302 ) && !checkLock(aStudy)) {
          SMESH::UpdateView();
          SMESHGUI::GetSMESHGUI()->EmitSignalVisibilityChanged();
        }
        // PAL13338 + PAL15161 <--
      }
      catch (...) { // PAL16774 (Crash after display of many groups)
        SMESH::OnVisuException();
      }

      if (anAction == SMESH::eErase) {
        SALOME_ListIO l1;
        aSel->setSelectedObjects( l1 );
      }
      else
        aSel->setSelectedObjects( to_process );
      
      break;
    }

  case 4000:                                    // NODES
    {
      if(checkLock(aStudy)) break;

      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();

        ( new SMESHGUI_NodesDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"),
                                 tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }

  case 2151:  // FILTER
  {
    if ( vtkwnd )
    {
      EmitSignalDeactivateDialog();
      ( new SMESHGUI_FilterDlg( this, SMESH::EDGE ) )->show();
    }
    break;
  }

  case 406:                                     // MOVE NODE
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if(checkLock(aStudy)) break;
      ( new SMESHGUI_MoveNodesDlg( this ) )->show();
      break;
    }

  case 701:                                     // COMPUTE MESH
  case 711:                                     // PRECOMPUTE MESH
  case 712:                                     // EVALUATE MESH
    {
      if (checkLock(aStudy)) break;
      startOperation( theCommandID );
    }
    break;

  case 702: // Create mesh
  case 703: // Create sub-mesh
  case 704: // Edit mesh/sub-mesh
    startOperation( theCommandID );
    break;
  case 710: // Build compound mesh
    {
      if (checkLock(aStudy)) break;
      EmitSignalDeactivateDialog();
      ( new SMESHGUI_BuildCompoundDlg( this ) )->show();
    }
    break;

  case 407: // DIAGONAL INVERSION
  case 408: // Delete diagonal
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if ( checkLock( aStudy ) )
        break;

      /*Standard_Boolean aRes;
      SMESH::SMESH_Mesh_var aMesh = SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(IObject);
      if ( aMesh->_is_nil() )
      {
        SUIT_MessageBox::warning(GetDesktop(), tr( "SMESH_WRN_WARNING" ),
          tr( "SMESH_BAD_SELECTION" ) );
        break;
      }
      */
      EmitSignalDeactivateDialog();
      if ( theCommandID == 407 )
        ( new SMESHGUI_TrianglesInversionDlg( this ) )->show();
      else
        ( new SMESHGUI_UnionOfTwoTrianglesDlg( this ) )->show();
      break;
    }
    case 409: // Change orientation
    case 410: // Union of triangles
    case 411: // Cutting of quadrangles
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if ( checkLock( aStudy ) )
        break;

      EmitSignalDeactivateDialog();
      SMESHGUI_MultiEditDlg* aDlg = NULL;
      if ( theCommandID == 409 )
        aDlg = new SMESHGUI_ChangeOrientationDlg(this);
      else if ( theCommandID == 410 )
        aDlg = new SMESHGUI_UnionOfTrianglesDlg(this);
      else
        aDlg = new SMESHGUI_CuttingOfQuadsDlg(this);

      aDlg->show();
      break;
    }
  case 412: // Smoothing
    {
      if(checkLock(aStudy)) break;
      if( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_SmoothingDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 413: // Extrusion
    {
      if (checkLock(aStudy)) break;
      if (vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_ExtrusionDlg ( this ) )->show();
      } else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 414: // Revolution
    {
      if(checkLock(aStudy)) break;
      if( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RevolutionDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 415: // Pattern mapping
    {
      if ( checkLock( aStudy ) )
        break;
      if ( vtkwnd )
      {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_MeshPatternDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 416: // Extrusion along a path
    {
      if (checkLock(aStudy)) break;
      if (vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_ExtrusionAlongPathDlg( this ) )->show();
      } else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 417: // Convert mesh to quadratic
    {
    startOperation( 417 );
      /*      if (checkLock(aStudy)) break;
      if (vtkwnd) {
        EmitSignalDeactivateDialog();
        new SMESHGUI_ConvToQuadDlg();
      } else {
        SUIT_MessageBox::warning(desktop(),
                               tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
                               }*/
      break;
    }
  case 418: // create 2D mesh from 3D
    {
      startOperation( 418 );
      break;
    }
  case 806:                                     // CREATE GEO GROUP
    {
      startOperation( 806 );
      break;
    }
  case 801:                                     // CREATE GROUP
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if(checkLock(aStudy)) break;
      EmitSignalDeactivateDialog();
      SMESH::SMESH_Mesh_var aMesh = SMESH::SMESH_Mesh::_nil();

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      int nbSel = selected.Extent();
      if (nbSel == 1) {
        // check if mesh is selected
        aMesh = SMESH::GetMeshByIO( selected.First() );
      }
      SMESHGUI_GroupDlg *aDlg = new SMESHGUI_GroupDlg( this, aMesh);
      aDlg->show();
      break;
    }

  case 802:                                     // CONSTRUCT GROUP
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if(checkLock(aStudy)) break;
      EmitSignalDeactivateDialog();

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      int nbSel = selected.Extent();
      if (nbSel == 1) {
        // check if submesh is selected
        Handle(SALOME_InteractiveObject) IObject = selected.First();
        if (IObject->hasEntry()) {
          _PTR(SObject) aSObj = aStudy->FindObjectID(IObject->getEntry());
          if( aSObj ) {
            SMESH::SMESH_subMesh_var aSubMesh = SMESH::SMESH_subMesh::_narrow( SMESH::SObjectToObject( aSObj ) );
            if (!aSubMesh->_is_nil()) {
              try {
                SMESH::SMESH_Mesh_var aMesh = aSubMesh->GetFather();
                // get submesh elements list by types
                SMESH::long_array_var aNodes = aSubMesh->GetElementsByType(SMESH::NODE);
                SMESH::long_array_var aEdges = aSubMesh->GetElementsByType(SMESH::EDGE);
                SMESH::long_array_var aFaces = aSubMesh->GetElementsByType(SMESH::FACE);
                SMESH::long_array_var aVolumes = aSubMesh->GetElementsByType(SMESH::VOLUME);
                // create group for each type o elements
                QString aName = IObject->getName();
                if (aNodes->length() > 0) {
                  SMESH::SMESH_Group_var aGroup = SMESH::AddGroup(aMesh, SMESH::NODE, aName + "_Nodes");
                  aGroup->Add(aNodes.inout());
                }
                if (aEdges->length() > 0) {
                  SMESH::SMESH_Group_var aGroup = SMESH::AddGroup(aMesh, SMESH::EDGE, aName + "_Edges");
                  aGroup->Add(aEdges.inout());
                }
                if (aFaces->length() > 0) {
                  SMESH::SMESH_Group_var aGroup = SMESH::AddGroup(aMesh, SMESH::FACE, aName + "_Faces");
                  aGroup->Add(aFaces.inout());
                }
                if (aVolumes->length() > 0) {
                  SMESH::SMESH_Group_var aGroup = SMESH::AddGroup(aMesh, SMESH::VOLUME, aName + "_Volumes");
                  aGroup->Add(aVolumes.inout());
                }
                updateObjBrowser();

              }
              catch(const SALOME::SALOME_Exception & S_ex){
                SalomeApp_Tools::QtCatchCorbaException(S_ex);
              }
            }
          }
        }
      }
      else if(nbSel==0) {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"),
                                 tr("SMESH_WRN_NO_AVAILABLE_DATA"));
      }
      break;
    }

  case 803:                                     // EDIT GROUP
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if(checkLock(aStudy)) break;
      EmitSignalDeactivateDialog();

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      SALOME_ListIteratorOfListIO It (selected);
      int nbSelectedGroups = 0;
      for ( ; It.More(); It.Next() )
      {
        SMESH::SMESH_GroupBase_var aGroup =
          SMESH::IObjectToInterface<SMESH::SMESH_GroupBase>(It.Value());
        if (!aGroup->_is_nil()) {
          nbSelectedGroups++;
          SMESHGUI_GroupDlg *aDlg = new SMESHGUI_GroupDlg( this, aGroup);
          aDlg->show();
        }
      }
      if (nbSelectedGroups == 0)
        {
          SMESHGUI_GroupDlg *aDlg = new SMESHGUI_GroupDlg( this, SMESH::SMESH_GroupBase::_nil());
          aDlg->show();
        }
      break;
    }

  case 804:                                     // Add elements to group
    {
      if(checkLock(aStudy)) break;
      if (myState == 800) {
        SMESHGUI_GroupDlg *aDlg = (SMESHGUI_GroupDlg*) myActiveDialogBox;
        if (aDlg) aDlg->onAdd();
      }
      break;
    }

  case 805:                                     // Remove elements from group
    {
      if(checkLock(aStudy)) break;
      if (myState == 800) {
        SMESHGUI_GroupDlg *aDlg = (SMESHGUI_GroupDlg*) myActiveDialogBox;
        if (aDlg) aDlg->onRemove();
      }
      break;
    }

  case 815:                                     // Edit GEOM GROUP as standalone
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if(checkLock(aStudy)) break;
      EmitSignalDeactivateDialog();

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      SALOME_ListIteratorOfListIO It (selected);
      for ( ; It.More(); It.Next() )
      {
        SMESH::SMESH_GroupOnGeom_var aGroup =
          SMESH::IObjectToInterface<SMESH::SMESH_GroupOnGeom>(It.Value());
        if (!aGroup->_is_nil()) {
          SMESHGUI_GroupDlg *aDlg = new SMESHGUI_GroupDlg( this, aGroup, true );
          aDlg->show();
        }
      }
      break;
    }

    case 810: // Union Groups
    case 811: // Intersect groups
    case 812: // Cut groups
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if ( checkLock( aStudy ) )
        break;

      EmitSignalDeactivateDialog();

      SMESHGUI_GroupOpDlg* aDlg = 0;
      if ( theCommandID == 810 )
        aDlg = new SMESHGUI_UnionGroupsDlg( this );
      else if ( theCommandID == 811 )
        aDlg = new SMESHGUI_IntersectGroupsDlg( this );
      else
        aDlg = new SMESHGUI_CutGroupsDlg( this );

      aDlg->show();

      break;
    }

    case 814: // Create groups of entities from existing groups of superior dimensions
    {
      if ( checkLock( aStudy ) )
        break;

      EmitSignalDeactivateDialog();
      SMESHGUI_GroupOpDlg* aDlg = new SMESHGUI_DimGroupDlg( this );
      aDlg->show();

      break;
    }

    case 813: // Delete groups with their contents
    {
      if ( !vtkwnd )
      {
        SUIT_MessageBox::warning( desktop(), tr( "SMESH_WRN_WARNING" ),
                                  tr( "NOT_A_VTK_VIEWER" ) );
        break;
      }

      if ( checkLock( aStudy ) )
        break;

      EmitSignalDeactivateDialog();

      ( new SMESHGUI_DeleteGroupDlg( this ) )->show();
      break;
    }

  case 900:                                     // MESH INFOS
    {
      EmitSignalDeactivateDialog();
      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      if ( selected.Extent() > 1 ) { // a dlg for each IO
        SALOME_ListIO IOs;
        SALOME_ListIteratorOfListIO It (selected);
        for ( ; It.More(); It.Next() ) {
          IOs.Clear(); IOs.Append( It.Value() );
          aSel->setSelectedObjects( IOs );
          ( new SMESHGUI_MeshInfosDlg( this ) )->show();
        }
        // restore selection
        aSel->setSelectedObjects( selected );
      }
      else
        ( new SMESHGUI_MeshInfosDlg( this ) )->show();
      break;
    }

  case 902:                                     // STANDARD MESH INFOS
    {
      EmitSignalDeactivateDialog();
      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      if ( selected.Extent() > 1 ) { // a dlg for each IO
        SALOME_ListIO IOs;
        SALOME_ListIteratorOfListIO It (selected);
        for ( ; It.More(); It.Next() ) {
          IOs.Clear();
          IOs.Append( It.Value() );
          aSel->setSelectedObjects( IOs );
          ( new SMESHGUI_StandardMeshInfosDlg( this ) )->show();
        }
        // restore selection
        aSel->setSelectedObjects( selected );
      }
      else
        ( new SMESHGUI_StandardMeshInfosDlg( this ) )->show();
      break;
    }

  case 903:                                     // WHAT IS
    {
      EmitSignalDeactivateDialog();
      ( new SMESHGUI_WhatIsDlg( this ) )->show();
      break;
    }

  case 1100:                                    // EDIT HYPOTHESIS
    {
      if(checkLock(aStudy)) break;

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      int nbSel = selected.Extent();

      if (nbSel == 1) {
        Handle(SALOME_InteractiveObject) anIObject = selected.First();
        SMESH::SMESH_Hypothesis_var aHypothesis = SMESH::IObjectToInterface<SMESH::SMESH_Hypothesis>(anIObject);

        /* Look for all mesh objects that have this hypothesis affected in order to flag as ModifiedMesh */
        /* At end below '...->updateObjBrowser(true)' will change icon of mesh objects                   */
        /* Warning : however by internal mechanism all subMeshes icons are changed !                     */
        if ( !aHypothesis->_is_nil() )
        {
          // BUG 0020378
          //SMESHGUI_GenericHypothesisCreator* aCreator = SMESH::GetHypothesisCreator(aHypothesis->GetName());
          SMESHGUI_GenericHypothesisCreator* aCreator = SMESH::GetHypothesisCreator(aHypothesis->GetName());
          if (aCreator) {
            aCreator->edit( aHypothesis.in(), anIObject->getName(), desktop(), this, SLOT( onHypothesisEdit( int ) ) );
          }
          else
          {
            // report error
          }
        }
      }
      break;
    }

  case 1101:                                    // RENAME
    {
      if ( checkLock( aStudy ) )
        break;

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected );

      bool isAny = false; // is there any appropriate object selected

      SALOME_ListIteratorOfListIO It( selected );
      for ( ; It.More(); It.Next() )
      {
        Handle(SALOME_InteractiveObject) IObject = It.Value();
        _PTR(SObject) obj = aStudy->FindObjectID( IObject->getEntry() );
        _PTR(GenericAttribute) anAttr;
        _PTR(AttributeName) aName;
        if ( obj )
        {
          if ( obj->FindAttribute(anAttr, "AttributeName") )
          {
            aName = anAttr;
            QString newName = QString(aName->Value().c_str());

            // check type to prevent renaming of inappropriate objects
            int aType = SMESHGUI_Selection::type(IObject->getEntry(), aStudy);
            if (aType == MESH || aType == GROUP ||
                aType == SUBMESH || aType == SUBMESH_COMPOUND ||
                aType == SUBMESH_SOLID || aType == SUBMESH_FACE ||
                aType == SUBMESH_EDGE || aType == SUBMESH_VERTEX ||
                aType == HYPOTHESIS || aType == ALGORITHM)
            {
              isAny = true;
              newName = LightApp_NameDlg::getName(desktop(), newName);
              if ( !newName.isEmpty() )
              {
                SMESHGUI::GetSMESHGen()->SetName(obj->GetIOR().c_str(), newName.toLatin1().data());

                // update name of group object and its actor
                SMESH::SMESH_GroupBase_var aGroupObject = SMESH::IObjectToInterface<SMESH::SMESH_GroupBase>(IObject);
                if( !aGroupObject->_is_nil() )
                {
                  aGroupObject->SetName( newName.toLatin1().data() );
                  if ( SMESH_Actor *anActor = SMESH::FindActorByEntry( IObject->getEntry() ) )
                    anActor->setName( newName.toLatin1().data() );
                }

                updateObjBrowser();
              }
            }
          }
        }
      } // for

      if (!isAny) {
        SUIT_MessageBox::warning(desktop(),
                                 QObject::tr("SMESH_WRN_WARNING"),
                                 QObject::tr("SMESH_WRN_NO_APPROPRIATE_SELECTION"));
      }
      break;
    }

  case 1102:                                    // REMOVE HYPOTHESIS / ALGORITHMS
    {
      if(checkLock(aStudy)) break;
      SUIT_OverrideCursor wc;

      LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr();
      SALOME_ListIO selected;
      if( aSel )
        aSel->selectedObjects( selected, QString::null, false );

      SALOME_ListIteratorOfListIO It(selected);
      for (int i = 0; It.More(); It.Next(), i++) {
        Handle(SALOME_InteractiveObject) IObject = It.Value();
        SMESH::RemoveHypothesisOrAlgorithmOnMesh(IObject);
      }
      SALOME_ListIO l1;
      aSel->setSelectedObjects( l1 );
      updateObjBrowser();
      break;
    }

  case 4009:                                    // ELEM0D
  case 4010:                                    // GEOM::EDGE
  case 4021:                                    // TRIANGLE
  case 4022:                                    // QUAD
  case 4023:                                    // POLYGON
  case 4031:                                    // TETRA
  case 4032:                                    // HEXA
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        SMDSAbs_ElementType type    = SMDSAbs_Edge;
        int                 nbNodes = 2;
        switch (theCommandID) {
        case 4009:                                      // ELEM0D
          type = SMDSAbs_0DElement; nbNodes = 1; break;
        case 4021:                                      // TRIANGLE
          type = SMDSAbs_Face; nbNodes = 3; break;
        case 4022:                                      // QUAD
          type = SMDSAbs_Face; nbNodes = 4; break;
        case 4031:                                      // TETRA
          type = SMDSAbs_Volume; nbNodes = 4; break;
        case 4023:                                      // POLYGON
          type = SMDSAbs_Face; nbNodes = 5; break;     // 5 - identificator for POLYGON
        case 4032:                                      // HEXA
          type = SMDSAbs_Volume; nbNodes = 8; break;
        case 4033:                                      // POLYHEDRE
          type = SMDSAbs_Volume; nbNodes = 9; break; // 9 - identificator for POLYHEDRE
        default:;
        }
        ( new SMESHGUI_AddMeshElementDlg( this, type, nbNodes ) )->show();
      }
      else {
        SUIT_MessageBox::warning(desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4033:                                    // POLYHEDRON
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_CreatePolyhedralVolumeDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4034:     // QUADRATIC EDGE
  case 4035:     // QUADRATIC TRIANGLE
  case 4036:     // QUADRATIC QUADRANGLE
  case 4037:     // QUADRATIC TETRAHEDRON
  case 4038:     // QUADRATIC PYRAMID
  case 4039:     // QUADRATIC PENTAHEDRON
  case 4040:     // QUADRATIC HEXAHEDRON
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        int type;

        switch (theCommandID) {
        case 4034:
          type = QUAD_EDGE; break;
        case 4035:
          type = QUAD_TRIANGLE; break;
        case 4036:
          type = QUAD_QUADRANGLE; break;
        case 4037:
          type = QUAD_TETRAHEDRON; break;
        case 4038:
          type = QUAD_PYRAMID; break;
        case 4039:
          type = QUAD_PENTAHEDRON; break;
        case 4040:
          type = QUAD_HEXAHEDRON;
          break;
        default:;
        }
         ( new SMESHGUI_AddQuadraticElementDlg( this, type ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4041:                                    // REMOVES NODES
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RemoveNodesDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4042:                                    // REMOVES ELEMENTS
    {
      if(checkLock(aStudy)) break;
      if( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RemoveElementsDlg( this ) )->show();
      }
      else
        {
          SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                   tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
        }
      break;
    }
  case 4043: {                                // CLEAR_MESH

    if(checkLock(aStudy)) break;

    SALOME_ListIO selected;
    if( LightApp_SelectionMgr *aSel = SMESHGUI::selectionMgr() )
      aSel->selectedObjects( selected );

    SUIT_OverrideCursor wc;
    SALOME_ListIteratorOfListIO It (selected);
    for ( ; It.More(); It.Next() )
    {
      Handle(SALOME_InteractiveObject) IOS = It.Value();
      SMESH::SMESH_Mesh_var aMesh =
        SMESH::IObjectToInterface<SMESH::SMESH_Mesh>(IOS);
      if ( aMesh->_is_nil()) continue;
      try {
        SMESH::UpdateView(SMESH::eErase, IOS->getEntry());
        aMesh->Clear();
        _PTR(SObject) aMeshSObj = SMESH::FindSObject(aMesh);
        SMESH::ModifiedMesh( aMeshSObj, false, true);
        // hide groups and submeshes
        _PTR(ChildIterator) anIter =
          SMESH::GetActiveStudyDocument()->NewChildIterator( aMeshSObj );
        for ( anIter->InitEx(true); anIter->More(); anIter->Next() )
        {
          _PTR(SObject) so = anIter->Value();
          SMESH::UpdateView(SMESH::eErase, so->GetID().c_str());
        }
      }
      catch (const SALOME::SALOME_Exception& S_ex){
        wc.suspend();
        SalomeApp_Tools::QtCatchCorbaException(S_ex);
        wc.resume();
      }
    }
    SMESH::UpdateView();
    updateObjBrowser();
    break;
  }
  case 4051:                                    // RENUMBERING NODES
    {
      if(checkLock(aStudy)) break;
      if( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RenumberingDlg( this, 0 ) )->show();
      }
      else
        {
          SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                   tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
        }
      break;
    }
  case 4052:                                    // RENUMBERING ELEMENTS
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RenumberingDlg( this, 1 ) )->show();
      }
      else
        {
          SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                   tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
        }
      break;
    }
  case 4061:                                   // TRANSLATION
    {
      if(checkLock(aStudy)) break;
      if ( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_TranslationDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4062:                                   // ROTATION
    {
      if(checkLock(aStudy)) break;
      if( vtkwnd ) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_RotationDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4063:                                   // SYMMETRY
    {
      if(checkLock(aStudy)) break;
      if(vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_SymmetryDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4064:                                   // SEWING
    {
      if(checkLock(aStudy)) break;
      if(vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_SewingDlg( this ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4065:                                   // MERGE NODES
    {
      if(checkLock(aStudy)) break;
      if(vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_EditMeshDlg( this, 0 ) )->show();
      }
      else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }
  case 4066:                                   // MERGE EQUAL ELEMENTS
    {
      if (checkLock(aStudy)) break;
      if (vtkwnd) {
        EmitSignalDeactivateDialog();
        ( new SMESHGUI_EditMeshDlg( this, 1 ) )->show();
      } else {
        SUIT_MessageBox::warning(SMESHGUI::desktop(),
                                 tr("SMESH_WRN_WARNING"), tr("SMESH_WRN_VIEWER_VTK"));
      }
      break;
    }

  case 4067: // MAKE MESH PASS THROUGH POINT
    startOperation( 4067 );
    break;

  case 5105: // Library of selection filters
  {
    static QList<int> aTypes;
    if ( aTypes.isEmpty() )
    {
      aTypes.append( SMESH::NODE );
      aTypes.append( SMESH::EDGE );
      aTypes.append( SMESH::FACE );
      aTypes.append( SMESH::VOLUME );
    }
    if (!myFilterLibraryDlg)
      myFilterLibraryDlg = new SMESHGUI_FilterLibraryDlg( this, SMESH::GetDesktop( this ), aTypes, SMESHGUI_FilterLibraryDlg::EDIT );
    else if (myFilterLibraryDlg->isHidden())
      myFilterLibraryDlg->Init( aTypes, SMESHGUI_FilterLibraryDlg::EDIT );
    myFilterLibraryDlg->raise();
  }
  break;

  case 6017:                                    // CONTROLS
  case 6016:
  case 6015:
  case 6014:
  case 6013:
  case 6012:
  case 6011:
  case 6001:
  case 6018:
  case 6019:
  case 6002:
  case 6003:
  case 6004:
  case 6005:
  case 6009:
  case 6021:
    if ( vtkwnd ) {

      LightApp_SelectionMgr* mgr = selectionMgr();
      SALOME_ListIO selected; mgr->selectedObjects( selected );

      if ( selected.Extent() == 1 && selected.First()->hasEntry() ) {
        _PTR(SObject) SO = aStudy->FindObjectID( selected.First()->getEntry() );
        if ( SO ) {
          CORBA::Object_var aObject = SMESH::SObjectToObject( SO );
          SMESH::SMESH_Mesh_var      aMesh    = SMESH::SMESH_Mesh::_narrow( aObject );
          SMESH::SMESH_subMesh_var   aSubMesh = SMESH::SMESH_subMesh::_narrow( aObject );
          SMESH::SMESH_GroupBase_var aGroup   = SMESH::SMESH_GroupBase::_narrow( aObject );
          if ( !aMesh->_is_nil() || !aSubMesh->_is_nil() || !aGroup->_is_nil() ) {
            ::Control( theCommandID );
            break;
          }
        }
      }
      SUIT_MessageBox::warning(desktop(),
                               tr( "SMESH_WRN_WARNING" ),
                               tr( "SMESH_BAD_SELECTION" ) );
      break;
    }
    else {
      SUIT_MessageBox::warning(desktop(),
                               tr( "SMESH_WRN_WARNING" ),
                               tr( "NOT_A_VTK_VIEWER" ) );
    }
    break;
  case 9010:
    {
      LightApp_SelectionMgr* mgr = selectionMgr();
      SALOME_ListIO selected; mgr->selectedObjects( selected );

      SALOME_ListIteratorOfListIO it(selected);
      for( ; it.More(); it.Next()) {
        Handle(SALOME_InteractiveObject) anIObject = it.Value();
        if(anIObject->hasEntry()) {
          if(SMESH_Actor *anActor = SMESH::FindActorByEntry(anIObject->getEntry())){
            anActor->SetPointsLabeled( !anActor->GetPointsLabeled() );
          }
        }
      }
      break;
    }
  case 9011:
    {
      LightApp_SelectionMgr* mgr = selectionMgr();
      SALOME_ListIO selected; mgr->selectedObjects( selected );

      if (selected.Extent() == 1)       {
        Handle(SALOME_InteractiveObject) anIObject = selected.First();
        if(anIObject->hasEntry())
          if(SMESH_Actor *anActor = SMESH::FindActorByEntry(anIObject->getEntry())){
            anActor->SetCellsLabeled( !anActor->GetCellsLabeled() );
          }
      }
      break;
    }
  }

  anApp->updateActions(); //SRN: To update a Save button in the toolbar
  //updateObjBrowser();
  return true;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
bool SMESHGUI::OnMousePress( QMouseEvent * pe, SUIT_ViewWindow * wnd )
{
  return false;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
bool SMESHGUI::OnMouseMove( QMouseEvent * pe, SUIT_ViewWindow * wnd )
{
  return true;
}

//=============================================================================
/*!
 *
 */
//=============================================================================
bool SMESHGUI::OnKeyPress( QKeyEvent * pe, SUIT_ViewWindow * wnd )
{
  return true;
}

//=============================================================================
/*! Method:  BuildPresentation(const Handle(SALOME_InteractiveObject)& theIO)
 *  Purpose: ensures that the actor for the given <theIO> exists in the active VTK view
 */
//=============================================================================
void SMESHGUI::BuildPresentation( const Handle(SALOME_InteractiveObject) & theIO,
                                  SUIT_ViewWindow* wnd )
{
  if(theIO->hasEntry()){
    //SUIT_ViewWindow* wnd = SMESH::GetActiveWindow();
    SMESH::UpdateView(wnd,SMESH::eDisplay,theIO->getEntry());
  }
}

//=======================================================================
// function : createSMESHAction
// purpose  :
//=======================================================================
void SMESHGUI::createSMESHAction( const int id, const QString& po_id, const QString& icon_id, const int key, const bool toggle  )
{
  QIcon icon;
  QWidget* parent = application()->desktop();
  SUIT_ResourceMgr* resMgr = resourceMgr();
  QPixmap pix;
  if ( !icon_id.isEmpty() )
    pix = resMgr->loadPixmap( "SMESH", tr( icon_id.toLatin1().data() ) );
  else
    pix = resMgr->loadPixmap( "SMESH", tr( QString( "ICO_%1" ).arg( po_id ).toLatin1().data() ), false );
  if ( !pix.isNull() )
    icon = QIcon( pix );

  QString tooltip    = tr( QString( "TOP_%1" ).arg( po_id ).toLatin1().data() ),
          menu       = tr( QString( "MEN_%1" ).arg( po_id ).toLatin1().data() ),
          status_bar = tr( QString( "STB_%1" ).arg( po_id ).toLatin1().data() );

  createAction( id, tooltip, icon, menu, status_bar, key, parent, toggle, this, SLOT( OnGUIEvent() )  );
}

//=======================================================================
// function : createPopupItem
// purpose  :
//=======================================================================
void SMESHGUI::createPopupItem( const int id,
                                const QString& clients,
                                const QString& types,
                                const QString& theRule,
                                const int pId )
{
  int parentId = pId;
  if( pId!=-1 )
    parentId = popupMgr()->actionId( action( pId ) );

  if( !popupMgr()->contains( popupMgr()->actionId( action( id ) ) ) )
    popupMgr()->insert( action( id ), parentId, 0 );

  QString lc = "$";        // VSR : instead of QtxPopupSelection::defEquality();
  QString dc = "selcount"; // VSR : insetad of QtxPopupSelection::defSelCountParam()
  QString rule = "(%1) and (%2) and (%3)";
  rule = rule.arg( QString( "%1>0" ).arg( dc ) );
  if( clients.isEmpty() )
    rule = rule.arg( QString( "true" ) );
  else
    rule = rule.arg( QString( "%1client in {%2}" ).arg( lc ).arg( clients ) );
  rule = rule.arg( QString( "%1type in {%2}" ).arg( lc ).arg( types ) );
  rule += theRule;

  bool cont = myRules.contains( id );
  if( cont )
    rule = QString( "%1 or (%2)" ).arg( myRules[ id ] ).arg( rule );

  popupMgr()->setRule( action( id ), rule, QtxPopupMgr::VisibleRule );
  myRules[ id ] = QString( cont ? "%1" : "(%1)" ).arg( rule );
}

//=======================================================================
// function : initialize
// purpose  :
//=======================================================================
void SMESHGUI::initialize( CAM_Application* app )
{
  SalomeApp_Module::initialize( app );

//   SUIT_ResourceMgr* mgr = app->resourceMgr();
//   if ( mgr )
  /* Automatic Update flag */
//     myAutomaticUpdate = mgr->booleanValue( "SMESH", "AutomaticUpdate", myAutomaticUpdate );

  // ----- create actions --------------

  createSMESHAction(  111, "DAT", "", (Qt::CTRL+Qt::Key_B) );
  createSMESHAction(  112, "UNV", "", (Qt::CTRL+Qt::Key_U) );
  createSMESHAction(  113, "MED", "", (Qt::CTRL+Qt::Key_M) );
  createSMESHAction(  114, "NUM" );
  createSMESHAction(  121, "DAT" );
  createSMESHAction(  122, "MED" );
  createSMESHAction(  123, "UNV" );
  createSMESHAction(  140, "STL" );
  createSMESHAction(  124, "EXPORT_DAT" );
  createSMESHAction(  125, "EXPORT_MED" );
  createSMESHAction(  126, "EXPORT_UNV" );
  createSMESHAction(  141, "EXPORT_STL" );
  createSMESHAction(  150, "FILE_INFO" );
  createSMESHAction(   33, "DELETE",          "ICON_DELETE", Qt::Key_Delete );
  createSMESHAction( 5105, "SEL_FILTER_LIB" );
  createSMESHAction(  701, "COMPUTE",         "ICON_COMPUTE" );
  createSMESHAction(  702, "CREATE_MESH",     "ICON_DLG_INIT_MESH" );
  createSMESHAction(  703, "CREATE_SUBMESH",  "ICON_DLG_ADD_SUBMESH" );
  createSMESHAction(  704, "EDIT_MESHSUBMESH","ICON_DLG_EDIT_MESH" );
  createSMESHAction(  710, "BUILD_COMPOUND",  "ICON_BUILD_COMPOUND" );
  createSMESHAction(  711, "PRECOMPUTE",      "ICON_PRECOMPUTE" );
  createSMESHAction(  712, "EVALUATE",        "ICON_COMPUTE" );
  createSMESHAction(  806, "CREATE_GEO_GROUP","ICON_CREATE_GEO_GROUP" );
  createSMESHAction(  801, "CREATE_GROUP",    "ICON_CREATE_GROUP" );
  createSMESHAction(  802, "CONSTRUCT_GROUP", "ICON_CONSTRUCT_GROUP" );
  createSMESHAction(  803, "EDIT_GROUP",      "ICON_EDIT_GROUP" );
  createSMESHAction(  815, "EDIT_GEOMGROUP_AS_GROUP", "ICON_EDIT_GROUP" );
  createSMESHAction(  804, "ADD" );
  createSMESHAction(  805, "REMOVE" );
  createSMESHAction(  810, "UN_GROUP",        "ICON_UNION" );
  createSMESHAction(  811, "INT_GROUP",       "ICON_INTERSECT" );
  createSMESHAction(  812, "CUT_GROUP",       "ICON_CUT" );
  createSMESHAction(  814, "UNDERLYING_ELEMS","ICON_UNDERLYING_ELEMS" );
  createSMESHAction(  813, "DEL_GROUP",       "ICON_DEL_GROUP" );
  createSMESHAction(  900, "ADV_INFO",        "ICON_ADV_INFO" );
  createSMESHAction(  902, "STD_INFO",        "ICON_STD_INFO" );
  createSMESHAction(  903, "WHAT_IS",         "ICON_WHAT_IS" );
  createSMESHAction( 6001, "LENGTH",          "ICON_LENGTH",        0, true );
  createSMESHAction( 6002, "FREE_EDGE",       "ICON_FREE_EDGE",     0, true );
  createSMESHAction( 6021, "FREE_FACES",      "ICON_FREE_FACES",    0, true );
  createSMESHAction( 6003, "FREE_BORDER",     "ICON_FREE_EDGE_2D",  0, true );
  createSMESHAction( 6004, "CONNECTION",      "ICON_CONNECTION",    0, true );
  createSMESHAction( 6005, "FREE_NODE",       "ICON_FREE_NODE",     0, true );
  createSMESHAction( 6011, "AREA",            "ICON_AREA",          0, true );
  createSMESHAction( 6012, "TAPER",           "ICON_TAPER",         0, true );
  createSMESHAction( 6013, "ASPECT",          "ICON_ASPECT",        0, true );
  createSMESHAction( 6014, "MIN_ANG",         "ICON_ANGLE",         0, true );
  createSMESHAction( 6015, "WARP",            "ICON_WARP",          0, true );
  createSMESHAction( 6016, "SKEW",            "ICON_SKEW",          0, true );
  createSMESHAction( 6017, "ASPECT_3D",       "ICON_ASPECT_3D",     0, true );
  createSMESHAction( 6018, "LENGTH_2D",       "ICON_LENGTH_2D",     0, true );
  createSMESHAction( 6019, "CONNECTION_2D",   "ICON_CONNECTION_2D", 0, true );
  createSMESHAction( 6009, "VOLUME_3D",       "ICON_VOLUME_3D",     0, true );
  createSMESHAction( 4000, "NODE",            "ICON_DLG_NODE" );
  createSMESHAction( 4009, "ELEM0D",          "ICON_DLG_ELEM0D" );
  createSMESHAction( 4010, "EDGE",            "ICON_DLG_EDGE" );
  createSMESHAction( 4021, "TRIANGLE",        "ICON_DLG_TRIANGLE" );
  createSMESHAction( 4022, "QUAD",            "ICON_DLG_QUADRANGLE" );
  createSMESHAction( 4023, "POLYGON",         "ICON_DLG_POLYGON" );
  createSMESHAction( 4031, "TETRA",           "ICON_DLG_TETRAS" );
  createSMESHAction( 4032, "HEXA",            "ICON_DLG_HEXAS" );
  createSMESHAction( 4041, "REMOVE_NODES",    "ICON_DLG_REM_NODE" );
  createSMESHAction( 4042, "REMOVE_ELEMENTS", "ICON_DLG_REM_ELEMENT" );
  createSMESHAction( 4043, "CLEAR_MESH"    ,  "ICON_CLEAR_MESH" );
  createSMESHAction( 4051, "RENUM_NODES",     "ICON_DLG_RENUMBERING_NODES" );
  createSMESHAction( 4052, "RENUM_ELEMENTS",  "ICON_DLG_RENUMBERING_ELEMENTS" );
  createSMESHAction( 4061, "TRANS",           "ICON_SMESH_TRANSLATION_VECTOR" );
  createSMESHAction( 4062, "ROT",             "ICON_DLG_MESH_ROTATION" );
  createSMESHAction( 4063, "SYM",             "ICON_SMESH_SYMMETRY_PLANE" );
  createSMESHAction( 4064, "SEW",             "ICON_SMESH_SEWING_FREEBORDERS" );
  createSMESHAction( 4065, "MERGE",           "ICON_SMESH_MERGE_NODES" );
  createSMESHAction( 4066, "MERGE_ELEMENTS",  "ICON_DLG_MERGE_ELEMENTS" );
  createSMESHAction( 4067, "MESH_THROU_POINT","ICON_DLG_MESH_THROU_POINT" );
  createSMESHAction(  406, "MOVE",            "ICON_DLG_MOVE_NODE" );
  createSMESHAction(  407, "INV",             "ICON_DLG_MESH_DIAGONAL" );
  createSMESHAction(  408, "UNION2",          "ICON_UNION2TRI" );
  createSMESHAction(  409, "ORIENT",          "ICON_DLG_MESH_ORIENTATION" );
  createSMESHAction(  410, "UNION",           "ICON_UNIONTRI" );
  createSMESHAction(  411, "CUT",             "ICON_CUTQUAD" );
  createSMESHAction(  412, "SMOOTH",          "ICON_DLG_SMOOTHING" );
  createSMESHAction(  413, "EXTRUSION",       "ICON_EXTRUSION" );
  createSMESHAction(  414, "REVOLUTION",      "ICON_REVOLUTION" );
  createSMESHAction(  415, "MAP",             "ICON_MAP" );
  createSMESHAction(  416, "EXTRUSION_ALONG", "ICON_EXTRUSION_ALONG" );
  createSMESHAction(  417, "CONV_TO_QUAD",    "ICON_CONV_TO_QUAD" );
  createSMESHAction(  418, "2D_FROM_3D",      "ICON_2D_FROM_3D" );
  createSMESHAction(  200, "RESET" );
  createSMESHAction(  201, "SCALAR_BAR_PROP" );
  createSMESHAction(  211, "WIRE",           "ICON_WIRE", 0, true );
  createSMESHAction(  212, "SHADE",          "ICON_SHADE", 0, true );
  createSMESHAction(  213, "SHRINK",         "ICON_SHRINK", 0, true );
  createSMESHAction(  214, "UPDATE",         "ICON_UPDATE" );
  createSMESHAction(  215, "NODES",          "ICON_POINTS", 0, true );
  createSMESHAction(  216, "ELEMS0D",        "ICON_DLG_ELEM0D", 0, true );
  createSMESHAction(  217, "EDGES",          "ICON_DLG_EDGE", 0, true );
  createSMESHAction(  218, "FACES",          "ICON_DLG_TRIANGLE", 0, true );
  createSMESHAction(  219, "VOLUMES",        "ICON_DLG_TETRAS", 0, true );
  createSMESHAction(  220, "ALL" );
  createSMESHAction(  221, "FACE_ORIENTATION", "", 0, true );

  createSMESHAction(  231, "LINE_REPRESENTATION", "", 0, true );
  createSMESHAction(  232, "ARC_REPRESENTATION", "", 0, true );

  createSMESHAction( 1100, "EDIT_HYPO" );
  createSMESHAction( 1101, "RENAME", "", Qt::Key_F2 );
  createSMESHAction( 1102, "UNASSIGN" );
  createSMESHAction( 9010, "NUM_NODES", "", 0, true );
  createSMESHAction( 9011, "NUM_ELEMENTS", "", 0, true );
  createSMESHAction( 1131, "DISPMODE" );
  createSMESHAction( 1132, "COLORS" );
  createSMESHAction( 1133, "TRANSP" );
  createSMESHAction( 1134, "CLIP" );
  createSMESHAction( 1135, "DISP_ENT" );
  createSMESHAction( 1136, "AUTO_COLOR" );
  createSMESHAction( 1137, "DISABLE_AUTO_COLOR" );
  createSMESHAction( 2000, "CTRL" );

  createSMESHAction( 300, "ERASE" );
  createSMESHAction( 301, "DISPLAY" );
  createSMESHAction( 302, "DISPLAY_ONLY" );
  createSMESHAction( 4033, "POLYHEDRON", "ICON_DLG_POLYHEDRON" );
  createSMESHAction( 4034, "QUADRATIC_EDGE", "ICON_DLG_QUADRATIC_EDGE" );
  createSMESHAction( 4035, "QUADRATIC_TRIANGLE", "ICON_DLG_QUADRATIC_TRIANGLE" );
  createSMESHAction( 4036, "QUADRATIC_QUADRANGLE", "ICON_DLG_QUADRATIC_QUADRANGLE" );
  createSMESHAction( 4037, "QUADRATIC_TETRAHEDRON", "ICON_DLG_QUADRATIC_TETRAHEDRON" );
  createSMESHAction( 4038, "QUADRATIC_PYRAMID", "ICON_DLG_QUADRATIC_PYRAMID" );
  createSMESHAction( 4039, "QUADRATIC_PENTAHEDRON", "ICON_DLG_QUADRATIC_PENTAHEDRON" );
  createSMESHAction( 4040, "QUADRATIC_HEXAHEDRON", "ICON_DLG_QUADRATIC_HEXAHEDRON" );

  // ----- create menu --------------
  int fileId   = createMenu( tr( "MEN_FILE" ),   -1,  1 ),
      editId   = createMenu( tr( "MEN_EDIT" ),   -1,  3 ),
      toolsId  = createMenu( tr( "MEN_TOOLS" ),  -1,  5, 50 ),
      meshId   = createMenu( tr( "MEN_MESH" ),   -1, 70, 10 ),
      ctrlId   = createMenu( tr( "MEN_CTRL" ),   -1, 60, 10 ),
      modifyId = createMenu( tr( "MEN_MODIFY" ), -1, 40, 10 ),
      viewId   = createMenu( tr( "MEN_VIEW" ),   -1,  2 );

  createMenu( separator(), fileId );

  int importId = createMenu( tr( "MEN_IMPORT" ), fileId, -1, 10 ),
      exportId = createMenu( tr( "MEN_EXPORT" ), fileId, -1, 10 ),
      addId    = createMenu( tr( "MEN_ADD" ),    modifyId, 402 ),
      removeId = createMenu( tr( "MEN_REMOVE" ), modifyId, 403 ),
      renumId  = createMenu( tr( "MEN_RENUM" ),  modifyId, 404 ),
      transfId = createMenu( tr( "MEN_TRANSF" ), modifyId, 405 );

  createMenu( 111, importId, -1 );
  createMenu( 112, importId, -1 );
  createMenu( 113, importId, -1 );

  createMenu( 121, exportId, -1 );
  createMenu( 122, exportId, -1 );
  createMenu( 123, exportId, -1 );
  createMenu( 140, exportId, -1 ); // export to stl STL

  createMenu( separator(), fileId, 10 );

  createMenu( 33, editId, -1 );

  createMenu( 5105, toolsId, -1 );

  createMenu( 702, meshId, -1 ); // "Mesh" menu
  createMenu( 703, meshId, -1 );
  createMenu( 704, meshId, -1 );
  createMenu( 710, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 701, meshId, -1 );
  createMenu( 711, meshId, -1 );
  createMenu( 712, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 801, meshId, -1 );
  createMenu( 806, meshId, -1 );
  createMenu( 802, meshId, -1 );
  createMenu( 803, meshId, -1 );
  createMenu( 815, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 810, meshId, -1 );
  createMenu( 811, meshId, -1 );
  createMenu( 812, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 814, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 813, meshId, -1 );
  createMenu( separator(), meshId, -1 );
  createMenu( 900, meshId, -1 );
  createMenu( 902, meshId, -1 );
  createMenu( 903, meshId, -1 );
  createMenu( separator(), meshId, -1 );

  createMenu( 6003, ctrlId, -1 );
  createMenu( 6001, ctrlId, -1 );
  createMenu( 6004, ctrlId, -1 );
  createMenu( separator(), ctrlId, -1 );
  createMenu( 6005, ctrlId, -1 );
  createMenu( 6002, ctrlId, -1 );
  createMenu( 6018, ctrlId, -1 );
  createMenu( 6019, ctrlId, -1 );
  createMenu( 6011, ctrlId, -1 );
  createMenu( 6012, ctrlId, -1 );
  createMenu( 6013, ctrlId, -1 );
  createMenu( 6014, ctrlId, -1 );
  createMenu( 6015, ctrlId, -1 );
  createMenu( 6016, ctrlId, -1 );
  createMenu( separator(), ctrlId, -1 );
  createMenu( 6017, ctrlId, -1 );
  createMenu( 6009, ctrlId, -1 );
  createMenu( 6021, ctrlId, -1 );
  createMenu( separator(), ctrlId, -1 );

  createMenu( 4000, addId, -1 );
  createMenu( 4009, addId, -1 );
  createMenu( 4010, addId, -1 );
  createMenu( 4021, addId, -1 );
  createMenu( 4022, addId, -1 );
  createMenu( 4023, addId, -1 );
  createMenu( 4031, addId, -1 );
  createMenu( 4032, addId, -1 );
  createMenu( 4033, addId, -1 );
  createMenu( separator(), addId, -1 );
  createMenu( 4034, addId, -1 );
  createMenu( 4035, addId, -1 );
  createMenu( 4036, addId, -1 );
  createMenu( 4037, addId, -1 );
  createMenu( 4038, addId, -1 );
  createMenu( 4039, addId, -1 );
  createMenu( 4040, addId, -1 );

  createMenu( 4041, removeId, -1 );
  createMenu( 4042, removeId, -1 );
  createMenu( 4043, removeId, -1 );

  createMenu( 4051, renumId, -1 );
  createMenu( 4052, renumId, -1 );

  createMenu( 4061, transfId, -1 );
  createMenu( 4062, transfId, -1 );
  createMenu( 4063, transfId, -1 );
  createMenu( 4064, transfId, -1 );
  createMenu( 4065, transfId, -1 );
  createMenu( 4066, transfId, -1 );

  createMenu( 406, modifyId, -1 );
  createMenu( 4067,modifyId, -1 );
  createMenu( 407, modifyId, -1 );
  createMenu( 408, modifyId, -1 );
  createMenu( 409, modifyId, -1 );
  createMenu( 410, modifyId, -1 );
  createMenu( 411, modifyId, -1 );
  createMenu( 412, modifyId, -1 );
  createMenu( 413, modifyId, -1 );
  createMenu( 416, modifyId, -1 );
  createMenu( 414, modifyId, -1 );
  createMenu( 415, modifyId, -1 );
  createMenu( 417, modifyId, -1 );
  createMenu( 418, modifyId, -1 );

  createMenu( 214, viewId, -1 );

  // ----- create toolbars --------------
  int meshTb     = createTool( tr( "TB_MESH" ) ),
      ctrlTb     = createTool( tr( "TB_CTRL" ) ),
      addRemTb   = createTool( tr( "TB_ADD_REMOVE" ) ),
      modifyTb   = createTool( tr( "TB_MODIFY" ) ),
      dispModeTb = createTool( tr( "TB_DISP_MODE" ) );

  createTool( 702, meshTb );
  createTool( 703, meshTb );
  createTool( 704, meshTb );
  createTool( 710, meshTb );
  createTool( separator(), meshTb );
  createTool( 701, meshTb );
  createTool( 711, meshTb );
  createTool( 712, meshTb );
  createTool( separator(), meshTb );
  createTool( 801, meshTb );
  createTool( 806, meshTb );
  createTool( 802, meshTb );
  createTool( 803, meshTb );
  //createTool( 815, meshTb );
  createTool( separator(), meshTb );
  createTool( 900, meshTb );
  createTool( 902, meshTb );
  createTool( 903, meshTb );
  createTool( separator(), meshTb );

  createTool( 6001, ctrlTb );
  createTool( 6003, ctrlTb );
  createTool( 6004, ctrlTb );
  createTool( separator(), ctrlTb );
  createTool( 6005, ctrlTb );
  createTool( 6002, ctrlTb );
  createTool( 6018, ctrlTb );
  createTool( 6019, ctrlTb );
  createTool( 6011, ctrlTb );
  createTool( 6012, ctrlTb );
  createTool( 6013, ctrlTb );
  createTool( 6014, ctrlTb );
  createTool( 6015, ctrlTb );
  createTool( 6016, ctrlTb );
  createTool( separator(), ctrlTb );
  createTool( 6017, ctrlTb );
  createTool( 6009, ctrlTb );
  createTool( 6021, ctrlTb );
  createTool( separator(), ctrlTb );

  createTool( 4000, addRemTb );
  createTool( 4009, addRemTb );
  createTool( 4010, addRemTb );
  createTool( 4021, addRemTb );
  createTool( 4022, addRemTb );
  createTool( 4023, addRemTb );
  createTool( 4031, addRemTb );
  createTool( 4032, addRemTb );
  createTool( 4033, addRemTb );
  createTool( separator(), addRemTb );
  createTool( 4034, addRemTb );
  createTool( 4035, addRemTb );
  createTool( 4036, addRemTb );
  createTool( 4037, addRemTb );
  createTool( 4038, addRemTb );
  createTool( 4039, addRemTb );
  createTool( 4040, addRemTb );
  createTool( separator(), addRemTb );
  createTool( 4041, addRemTb );
  createTool( 4042, addRemTb );
  createTool( 4043, addRemTb );
  createTool( separator(), addRemTb );
  createTool( 4051, addRemTb );
  createTool( 4052, addRemTb );
  createTool( separator(), addRemTb );
  createTool( 4061, addRemTb );
  createTool( 4062, addRemTb );
  createTool( 4063, addRemTb );
  createTool( 4064, addRemTb );
  createTool( 4065, addRemTb );
  createTool( 4066, addRemTb );
  createTool( separator(), addRemTb );

  createTool( 406, modifyTb );
  createTool( 4067,modifyTb );
  createTool( 407, modifyTb );
  createTool( 408, modifyTb );
  createTool( 409, modifyTb );
  createTool( 410, modifyTb );
  createTool( 411, modifyTb );
  createTool( 412, modifyTb );
  createTool( 413, modifyTb );
  createTool( 416, modifyTb );
  createTool( 414, modifyTb );
  createTool( 415, modifyTb );
  createTool( 417, modifyTb );
  createTool( 418, modifyTb );

  createTool( 214, dispModeTb );

  QString lc = "$";        // VSR : instead of QtxPopupSelection::defEquality();
  QString dc = "selcount"; // VSR : instead of QtxPopupSelection::defSelCountParam()

  myRules.clear();
  QString OB = "'ObjectBrowser'",
          View = "'" + SVTK_Viewer::Type() + "'",
          pat = "'%1'",
          mesh    = pat.arg( SMESHGUI_Selection::typeName( MESH ) ),
          group   = pat.arg( SMESHGUI_Selection::typeName( GROUP ) ),
          hypo    = pat.arg( SMESHGUI_Selection::typeName( HYPOTHESIS ) ),
          algo    = pat.arg( SMESHGUI_Selection::typeName( ALGORITHM ) ),
          elems   = QString( "'%1' '%2' '%3' '%4' '%5' '%6'" ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH_VERTEX ) ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH_EDGE ) ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH_FACE ) ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH_SOLID ) ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH_COMPOUND ) ).
                       arg( SMESHGUI_Selection::typeName( SUBMESH ) ),
          subMesh = elems,
          mesh_group = mesh + " " + subMesh + " " + group,
          hyp_alg = hypo + " " + algo;

  // popup for object browser
  QString
    isInvisible("not( isVisible )"),
    isEmpty("numberOfNodes = 0"),
    isNotEmpty("numberOfNodes <> 0"),

    // has nodes, edges, etc in VISIBLE! actor
    hasNodes("(numberOfNodes > 0 )"),//&& isVisible)"),
    hasElems("(count( elemTypes ) > 0)"),
    hasDifferentElems("(count( elemTypes ) > 1)"),
    hasElems0d("({'Elem0d'} in elemTypes)"),
    hasEdges("({'Edge'} in elemTypes)"),
    hasFaces("({'Face'} in elemTypes)"),
    hasVolumes("({'Volume'} in elemTypes)");

  createPopupItem( 150, OB, mesh, "&& selcount=1 && isImported" );      // FILE INFORMATION
  createPopupItem( 703, OB, mesh, "&& isComputable");      // CREATE_SUBMESH
  //createPopupItem( 703, OB, subMesh, "&& isComputable" );  // CREATE_SUBMESH
  createPopupItem( 704, OB, mesh, "&& isComputable");      // EDIT_MESHSUBMESH
  createPopupItem( 704, OB, subMesh, "&& isComputable" );  // EDIT_MESHSUBMESH
  createPopupItem( 803, OB, group );                       // EDIT_GROUP
  createPopupItem( 815, OB, group, "&& groupType = 'GroupOnGeom'" ); // EDIT_GROUP

  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 701, OB, mesh, "&& isComputable" );     // COMPUTE
  createPopupItem( 711, OB, mesh, "&& isComputable && isPreComputable" ); // PRECOMPUTE
  createPopupItem( 712, OB, mesh, "&& isComputable" );     // COMPUTE
  createPopupItem( 214, OB, mesh_group );                  // UPDATE
  createPopupItem( 900, OB, mesh_group );                  // ADV_INFO
  createPopupItem( 902, OB, mesh );                        // STD_INFO
  createPopupItem( 903, OB, mesh_group );                  // WHAT_IS
  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 801, OB, mesh );                        // CREATE_GROUP
  createPopupItem( 806, OB, mesh );                        // CREATE_GEO_GROUP
  createPopupItem( 802, OB, subMesh );                     // CONSTRUCT_GROUP
  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 1100, OB, hypo);                        // EDIT HYPOTHESIS
  createPopupItem( 1102, OB, hyp_alg ); // REMOVE HYPOTHESIS / ALGORITHMS
  createPopupItem( 1101, OB, mesh_group + " " + hyp_alg ); // RENAME
  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 4043, OB, mesh );                       // CLEAR_MESH
  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 417, OB, mesh/*, "&& " + hasElems*/);       // convert to quadratic
  createPopupItem( 418, OB, mesh/*, "&& " + hasVolumes*/);     // create 2D mesh on 3D
  popupMgr()->insert( separator(), -1, 0 );

  QString only_one_non_empty = QString( " && %1=1 && numberOfNodes>0" ).arg( dc );

  createPopupItem( 125, OB, mesh, only_one_non_empty );   // EXPORT_MED
  createPopupItem( 126, OB, mesh, only_one_non_empty );   // EXPORT_UNV
  createPopupItem( 141, OB, mesh, only_one_non_empty );   // EXPORT_STL
  //createPopupItem( 33, OB, subMesh + " " + group );       // DELETE
  createPopupItem(  33, OB, mesh_group + " " + hyp_alg ); // DELETE
  popupMgr()->insert( separator(), -1, 0 );

  // popup for viewer
  createPopupItem( 803, View, group ); // EDIT_GROUP
  createPopupItem( 804, View, elems ); // ADD
  createPopupItem( 805, View, elems ); // REMOVE

  popupMgr()->insert( separator(), -1, 0 );
  createPopupItem( 214, View, mesh_group ); // UPDATE
  createPopupItem( 900, View, mesh_group ); // ADV_INFO
  createPopupItem( 902, View, mesh );       // STD_INFO
  createPopupItem( 903, View, mesh_group ); // WHAT_IS
  popupMgr()->insert( separator(), -1, 0 );

  createPopupItem( 1136, OB + " " + View, mesh, "&& (not isAutoColor)" ); // AUTO_COLOR
  createPopupItem( 1137, OB + " " + View, mesh, "&& isAutoColor" );       // DISABLE_AUTO_COLOR
  popupMgr()->insert( separator(), -1, 0 );

  int anId;
  QString aClient = QString( "%1client in {%2}" ).arg( lc ).arg( "'VTKViewer'" );
  QString aType = QString( "%1type in {%2}" ).arg( lc );
  aType = aType.arg( mesh_group );
  QString aMeshInVTK = aClient + "&&" + aType;

  aClient = "($client in {'VTKViewer' 'ObjectBrowser'})";
  QString anActiveVTK = QString("activeView = '%1'").arg(SVTK_Viewer::Type());
  QString aSelCount = QString( "%1 > 0" ).arg( dc );

  //-------------------------------------------------
  // Numbering
  //-------------------------------------------------
  anId = popupMgr()->insert( tr( "MEN_NUM" ), -1, -1 );

  popupMgr()->insert( action( 9010 ), anId, -1 );
  popupMgr()->setRule( action( 9010 ), aMeshInVTK + "&& isVisible &&" + hasNodes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 9010 ), "{'Point'} in labeledTypes", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 9011 ), anId, -1 );
  popupMgr()->setRule( action( 9011 ), aMeshInVTK + "&& isVisible &&" + hasElems, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 9011 ), "{'Cell'} in labeledTypes", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), -1, -1 );

  //-------------------------------------------------
  // Display Mode
  //-------------------------------------------------
  anId = popupMgr()->insert( tr( "MEN_DISPMODE" ), -1, -1 );

  popupMgr()->insert( action( 211 ), anId, -1 ); // WIRE
  popupMgr()->setRule( action( 211 ), aMeshInVTK + "&&" + hasElems, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 211 ), "displayMode = 'eEdge'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 212 ), anId, -1 ); // SHADE
  popupMgr()->setRule( action( 212 ),aMeshInVTK+ "&& (" + hasFaces + "||" + hasVolumes + ")", QtxPopupMgr::VisibleRule);
  popupMgr()->setRule( action( 212 ), "displayMode = 'eSurface'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 215 ), anId, -1 ); // POINTS
  popupMgr()->setRule( action( 215 ), aMeshInVTK + "&&" + hasNodes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 215 ), "displayMode = 'ePoint'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 213 ), anId, -1 ); // SHRINK
  popupMgr()->setRule( action( 213 ), aMeshInVTK + "&& shrinkMode <> 'IsNotShrinkable' && displayMode <> 'ePoint'", QtxPopupMgr::VisibleRule);
  popupMgr()->setRule( action( 213 ), "shrinkMode = 'IsShrunk'", QtxPopupMgr::ToggleRule );

  //-------------------------------------------------
  // Display Entity
  //-------------------------------------------------
  QString aDiffElemsInVTK = aMeshInVTK + "&&" + hasDifferentElems;

  anId = popupMgr()->insert( tr( "MEN_DISP_ENT" ), -1, -1 );

  popupMgr()->insert( action(216), anId, -1 ); // ELEMS 0D
  popupMgr()->setRule(action(216), aDiffElemsInVTK + "&& isVisible &&" + hasElems0d, QtxPopupMgr::VisibleRule);
  popupMgr()->setRule(action(216), "{'Elem0d'} in entityMode", QtxPopupMgr::ToggleRule);

  popupMgr()->insert( action( 217 ), anId, -1 ); // EDGES
  popupMgr()->setRule( action( 217 ), aDiffElemsInVTK + "&& isVisible &&" + hasEdges, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 217 ), "{'Edge'} in entityMode", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 218 ), anId, -1 ); // FACES
  popupMgr()->setRule( action( 218 ), aDiffElemsInVTK + "&& isVisible &&" + hasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 218 ), "{'Face'} in entityMode", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 219 ), anId, -1 ); // VOLUMES
  popupMgr()->setRule( action( 219 ), aDiffElemsInVTK + "&& isVisible &&" + hasVolumes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 219 ), "{'Volume'} in entityMode", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 220 ), anId, -1 ); // ALL
  popupMgr()->setRule( action( 220 ), aDiffElemsInVTK + "&& isVisible && not( elemTypes in entityMode )", QtxPopupMgr::VisibleRule );


  //-------------------------------------------------
  // Representation of the 2D Quadratic elements
  //-------------------------------------------------  
  anId = popupMgr()->insert( tr( "MEN_QUADRATIC_REPRESENT" ), -1, -1 );
  popupMgr()->insert( action( 231 ), anId, -1 ); // LINE REPRESENTATION
  popupMgr()->setRule( action( 231 ), aMeshInVTK + "and isVisible",QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 231 ), "quadratic2DMode = 'eLines'", QtxPopupMgr::ToggleRule );
  
  popupMgr()->insert( action( 232 ), anId, -1 ); // ARC REPRESENTATION
  popupMgr()->setRule( action( 232 ), aMeshInVTK + "and isVisible", QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 232 ), "quadratic2DMode = 'eArcs'", QtxPopupMgr::ToggleRule );

  //-------------------------------------------------
  // Orientation of faces
  //-------------------------------------------------
  popupMgr()->insert( action( 221 ), -1, -1 );
  popupMgr()->setRule( action( 221 ), aMeshInVTK + "&& isVisible", QtxPopupMgr::VisibleRule);
  popupMgr()->setRule( action( 221 ), "facesOrientationMode = 'IsOriented'", QtxPopupMgr::ToggleRule );

  //-------------------------------------------------
  // Color / Size
  //-------------------------------------------------
  popupMgr()->insert( action( 1132 ), -1, -1 );
  popupMgr()->setRule( action( 1132 ), aMeshInVTK + "&& isVisible", QtxPopupMgr::VisibleRule );

  //-------------------------------------------------
  // Transparency
  //-------------------------------------------------
  popupMgr()->insert( action( 1133 ), -1, -1 );
  popupMgr()->setRule( action( 1133 ), aMeshInVTK + "&& isVisible", QtxPopupMgr::VisibleRule );

  //-------------------------------------------------
  // Clipping
  //-------------------------------------------------
  popupMgr()->insert( action( 1134 ), -1, -1 );
  popupMgr()->setRule( action( 1134 ), aMeshInVTK + "&& selcount=1 && isVisible", QtxPopupMgr::VisibleRule );

  popupMgr()->insert( separator(), -1, -1 );

  //-------------------------------------------------
  // Controls
  //-------------------------------------------------
  QString
    aMeshInVtkHasNodes = aMeshInVTK + "&&" + hasNodes,
    aMeshInVtkHasEdges = aMeshInVTK + "&&" + hasEdges,
    aMeshInVtkHasFaces = aMeshInVTK + "&&" + hasFaces,
    aMeshInVtkHasVolumes = aMeshInVTK + "&&" + hasVolumes;

  anId = popupMgr()->insert( tr( "MEN_CTRL" ), -1, -1 );
  
  popupMgr()->insert( action( 200 ), anId, -1 ); // RESET
  popupMgr()->setRule( action( 200 ), aMeshInVTK + "&& controlMode <> 'eNone'", QtxPopupMgr::VisibleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 6003 ), anId, -1 ); // FREE_BORDER
  popupMgr()->setRule( action( 6003 ), aMeshInVtkHasEdges, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6003 ), "controlMode = 'eFreeBorders'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6001 ), anId, -1 ); // LENGTH
  popupMgr()->setRule( action( 6001 ), aMeshInVtkHasEdges, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6001 ), "controlMode = 'eLength'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6004 ), anId, -1 ); // CONNECTION
  popupMgr()->setRule( action( 6004 ), aMeshInVtkHasEdges, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6004 ), "controlMode = 'eMultiConnection'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 6005 ), anId, -1 ); // FREE_NODE
  popupMgr()->setRule( action( 6005 ), aMeshInVtkHasNodes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6005 ), "controlMode = 'eFreeNodes'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6002 ), anId, -1 ); // FREE_EDGE
  popupMgr()->setRule( action( 6002 ), aMeshInVtkHasEdges, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6002 ), "controlMode = 'eFreeEdges'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6018 ), anId, -1 ); // LENGTH_2D
  popupMgr()->setRule( action( 6018 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6018 ), "controlMode = 'eLength2D'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6019 ), anId, -1 ); // CONNECTION_2D
  popupMgr()->setRule( action( 6019 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6019 ), "controlMode = 'eMultiConnection2D'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6011 ), anId, -1 ); // AREA
  popupMgr()->setRule( action( 6011 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6011 ), "controlMode = 'eArea'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6012 ), anId, -1 ); // TAPER
  popupMgr()->setRule( action( 6012 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6012 ), "controlMode = 'eTaper'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6013 ), anId, -1 ); // ASPECT
  popupMgr()->setRule( action( 6013 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6013 ), "controlMode = 'eAspectRatio'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6014 ), anId, -1 ); // MIN_ANG
  popupMgr()->setRule( action( 6014 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6014 ), "controlMode = 'eMinimumAngle'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6015 ), anId, -1 ); // WARP
  popupMgr()->setRule( action( 6015 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6015 ), "controlMode = 'eWarping'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6016 ), anId, -1 ); // SKEW
  popupMgr()->setRule( action( 6016 ), aMeshInVtkHasFaces, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6016 ), "controlMode = 'eSkew'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 6017 ), anId, -1 ); // ASPECT_3D
  popupMgr()->setRule( action( 6017 ), aMeshInVtkHasVolumes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6017 ), "controlMode = 'eAspectRatio3D'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert ( action( 6009 ), anId, -1 ); // VOLUME_3D
  popupMgr()->setRule( action( 6009 ), aMeshInVtkHasVolumes, QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6009 ), "controlMode = 'eVolume3D'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( action( 6021 ), anId, -1 ); // FREE_FACE
  popupMgr()->setRule( action( 6021 ), aMeshInVtkHasFaces /*aMeshInVtkHasVolumes*/,
                                       QtxPopupMgr::VisibleRule );
  popupMgr()->setRule( action( 6021 ), "controlMode = 'eFreeFaces'", QtxPopupMgr::ToggleRule );

  popupMgr()->insert( separator(), anId, -1 );

  popupMgr()->insert( action( 201 ), anId, -1 ); // SCALAR_BAR_PROP
  popupMgr()->setRule( action( 201 ), aMeshInVTK + "&& controlMode <> 'eNone'", QtxPopupMgr::VisibleRule );

  popupMgr()->insert( separator(), -1, -1 );
  
  //-------------------------------------------------
  // Display / Erase
  //-------------------------------------------------
  QString aRule = "$component={'SMESH'} and ( type='Component' or (" + aClient + " and " +
    aType + " and " + aSelCount + " and " + anActiveVTK + " and " + isNotEmpty + " %1 ) )";
  popupMgr()->insert( action( 301 ), -1, -1 ); // DISPLAY
  popupMgr()->setRule( action( 301 ), aRule.arg( "and (not isVisible)" ), QtxPopupMgr::VisibleRule );

  popupMgr()->insert( action( 300 ), -1, -1 ); // ERASE
  popupMgr()->setRule( action( 300 ), aRule.arg( "and isVisible" ), QtxPopupMgr::VisibleRule );

  popupMgr()->insert( action( 302 ), -1, -1 ); // DISPLAY_ONLY
  popupMgr()->setRule( action( 302 ), aRule.arg( "" ), QtxPopupMgr::VisibleRule );

  popupMgr()->insert( separator(), -1, -1 );

  connect( application(), SIGNAL( viewManagerActivated( SUIT_ViewManager* ) ),
           this, SLOT( onViewManagerActivated( SUIT_ViewManager* ) ) );
}

//================================================================================
/*!
 * \brief Return true if SMESH or GEOM objects are selected.
 * Is called form LightApp_Module::activateModule() which clear selection if
 * not isSelectionCompatible()
 */
//================================================================================

bool SMESHGUI::isSelectionCompatible()
{
  bool isCompatible = true;
  SALOME_ListIO selected;
  if ( LightApp_SelectionMgr *Sel = selectionMgr() )
    Sel->selectedObjects( selected );

  SALOME_ListIteratorOfListIO It( selected );
  for ( ; isCompatible && It.More(); It.Next())
    isCompatible =
      ( strcmp("GEOM", It.Value()->getComponentDataType()) == 0 ) ||
      ( strcmp("SMESH", It.Value()->getComponentDataType()) == 0 );

  return isCompatible;
}

bool SMESHGUI::activateModule( SUIT_Study* study )
{
  bool res = SalomeApp_Module::activateModule( study );

  setMenuShown( true );
  setToolShown( true );

  // Reset actions accelerator keys
  action(111)->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_B)); // Import DAT
  action(112)->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_U)); // Import UNV
  action(113)->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_M)); // Import MED

  action(  33)->setEnabled(true); // Delete: Key_Delete
  action(1101)->setEnabled(true); // Rename: Key_F2

  //  0020210. Make SMESH_Gen update meshes at switching GEOM->SMESH
  GetSMESHGen()->SetCurrentStudy(SALOMEDS::Study::_nil());
  if ( SalomeApp_Study* s = dynamic_cast<SalomeApp_Study*>( study ))
    if ( _PTR(Study) aStudy = s->studyDS()) {
      GetSMESHGen()->SetCurrentStudy( _CAST(Study,aStudy)->GetStudy() );
      updateObjBrowser(); // objects can be removed
    }

  return res;
}

bool SMESHGUI::deactivateModule( SUIT_Study* study )
{
  setMenuShown( false );
  setToolShown( false );

  EmitSignalCloseAllDialogs();

  // Unset actions accelerator keys
  action(111)->setShortcut(QKeySequence()); // Import DAT
  action(112)->setShortcut(QKeySequence()); // Import UNV
  action(113)->setShortcut(QKeySequence()); // Import MED

  action(  33)->setEnabled(false); // Delete: Key_Delete
  action(1101)->setEnabled(false); // Rename: Key_F2

  return SalomeApp_Module::deactivateModule( study );
}

void SMESHGUI::studyClosed( SUIT_Study* s )
{
  SMESH::RemoveVisuData( s->id() );
  SalomeApp_Module::studyClosed( s );
}

void SMESHGUI::OnGUIEvent()
{
  const QObject* obj = sender();
  if ( !obj || !obj->inherits( "QAction" ) )
    return;
  int id = actionId((QAction*)obj);
  if ( id != -1 )
    OnGUIEvent( id );
}

SMESH::SMESH_Gen_var SMESHGUI::GetSMESHGen()
{
  _PTR(Study) aStudy = SMESH::GetActiveStudyDocument(); //Document OCAF de l'etude active
  if ( CORBA::is_nil( myComponentSMESH ) )
    {
      SMESHGUI aGUI; //SRN BugID: IPAL9186: Create an instance of SMESHGUI to initialize myComponentSMESH
      if ( aStudy )
        aGUI.myComponentSMESH->SetCurrentStudy(_CAST(Study,aStudy)->GetStudy());
      return aGUI.myComponentSMESH;
    }
  if ( aStudy )
    myComponentSMESH->SetCurrentStudy(_CAST(Study,aStudy)->GetStudy());
  return myComponentSMESH;
}

QString SMESHGUI::engineIOR() const
{
  CORBA::ORB_var anORB = getApp()->orb();
  CORBA::String_var anIOR = anORB->object_to_string(GetSMESHGen());
  return QString( anIOR.in() );
}

void SMESHGUI::contextMenuPopup( const QString& client, QMenu* menu, QString& title )
{
  SalomeApp_Module::contextMenuPopup( client, menu, title );
  SALOME_ListIO lst;
  selectionMgr()->selectedObjects( lst );
  if ( ( client == "OCCViewer" || client == "VTKViewer" ) && lst.Extent() == 1 ) {
    Handle(SALOME_InteractiveObject) io = lst.First();
    SalomeApp_Study* appStudy = dynamic_cast<SalomeApp_Study*>( application()->activeStudy() );
    _PTR(Study) study = appStudy->studyDS();
    _PTR(SObject) obj = study->FindObjectID( io->getEntry() );
    if ( obj ) {
      QString aName = QString( obj->GetName().c_str() );
      while ( aName.at( aName.length() - 1 ) == ' ' ) // Remove extraspaces in Name of Popup
          aName.remove( (aName.length() - 1), 1 );
      title = aName;
    }
  }
}

LightApp_Selection* SMESHGUI::createSelection() const
{
  return new SMESHGUI_Selection();
}

void SMESHGUI::windows( QMap<int, int>& aMap ) const
{
  aMap.insert( SalomeApp_Application::WT_ObjectBrowser, Qt::LeftDockWidgetArea );
  aMap.insert( SalomeApp_Application::WT_PyConsole, Qt::BottomDockWidgetArea );
}

void SMESHGUI::viewManagers( QStringList& list ) const
{
  list.append( SVTK_Viewer::Type() );
}

void SMESHGUI::onViewManagerActivated( SUIT_ViewManager* mgr )
{
  if ( dynamic_cast<SVTK_ViewManager*>( mgr ) )
    SMESH::UpdateSelectionProp( this );
}

void SMESHGUI::createPreferences()
{
  // General tab ------------------------------------------------------------------------
  int genTab = addPreference( tr( "PREF_TAB_GENERAL" ) );

  int updateGroup = addPreference( tr( "PREF_GROUP_UPDATE" ), genTab );
  addPreference( tr( "PREF_AUTO_UPDATE" ), updateGroup, LightApp_Preferences::Bool, "SMESH", "auto_update" );

  int qaGroup = addPreference( tr( "PREF_GROUP_QUALITY" ), genTab );
  setPreferenceProperty( qaGroup, "columns", 2 );
  addPreference( tr( "PREF_DISPLAY_ENTITY" ), qaGroup, LightApp_Preferences::Bool, "SMESH", "display_entity" );
  addPreference( tr( "PREF_PRECISION_USE" ), qaGroup, LightApp_Preferences::Bool, "SMESH", "use_precision" );
  int prec = addPreference( tr( "PREF_PRECISION_VALUE" ), qaGroup, LightApp_Preferences::IntSpin, "SMESH", "controls_precision" );
  setPreferenceProperty( prec, "min", 0 );
  setPreferenceProperty( prec, "max", 16 );

  int dispgroup = addPreference( tr( "PREF_DISPLAY_MODE" ), genTab );
  setPreferenceProperty( dispgroup, "columns", 2 );
  int dispmode = addPreference( tr( "PREF_DISPLAY_MODE" ), dispgroup, LightApp_Preferences::Selector, "SMESH", "display_mode" );
  QStringList modes;
  modes.append( "Wireframe" );
  modes.append( "Shading" );
  modes.append( "Nodes" );
  modes.append( "Shrink" );
  QList<QVariant> indices;
  indices.append( 0 );
  indices.append( 1 );
  indices.append( 2 );
  indices.append( 3 );
  setPreferenceProperty( dispmode, "strings", modes );
  setPreferenceProperty( dispmode, "indexes", indices );

  int arcgroup = addPreference( tr( "QUADRATIC_REPRESENT_MODE" ), genTab );
  setPreferenceProperty( arcgroup, "columns", 2 );
  int quadraticmode = addPreference( tr( "QUADRATIC_REPRESENT_MODE" ), arcgroup, LightApp_Preferences::Selector, "SMESH", "quadratic_mode" );
  QStringList quadraticModes;
  quadraticModes.append("Lines");
  quadraticModes.append("Arcs");
  indices.clear();
  indices.append( 0 );
  indices.append( 1 );
  setPreferenceProperty( quadraticmode, "strings", quadraticModes );
  setPreferenceProperty( quadraticmode, "indexes", indices );

  int maxAngle = addPreference( tr( "MAX_ARC_ANGLE" ), arcgroup, LightApp_Preferences::IntSpin,
                              "SMESH", "max_angle" );
  setPreferenceProperty( maxAngle, "min", 1 );
  setPreferenceProperty( maxAngle, "max", 90 );
  
  

  int exportgroup = addPreference( tr( "PREF_GROUP_EXPORT" ), genTab );
  setPreferenceProperty( exportgroup, "columns", 2 );
  addPreference( tr( "PREF_AUTO_GROUPS" ), exportgroup, LightApp_Preferences::Bool, "SMESH", "auto_groups" );
  addPreference( tr( "PREF_RENUMBER" ), exportgroup, LightApp_Preferences::Bool, "SMESH", "renumbering" );

  int computeGroup = addPreference( tr( "PREF_GROUP_COMPUTE" ), genTab );
  setPreferenceProperty( computeGroup, "columns", 2 );
  int notifyMode = addPreference( tr( "PREF_NOTIFY_MODE" ), computeGroup, LightApp_Preferences::Selector, "SMESH", "show_result_notification" );
  modes.clear();
  modes.append( "Never" );
  modes.append( "Errors only" );
  modes.append( "Always" );
  indices.clear();
  indices.append( 0 );
  indices.append( 1 );
  indices.append( 2 );
  setPreferenceProperty( notifyMode, "strings", modes );
  setPreferenceProperty( notifyMode, "indexes", indices );

  int segGroup = addPreference( tr( "PREF_GROUP_SEGMENT_LENGTH" ), genTab );
  setPreferenceProperty( segGroup, "columns", 2 );
  int segLen = addPreference( tr( "PREF_SEGMENT_LENGTH" ), segGroup, LightApp_Preferences::IntSpin,
                              "SMESH", "segmentation" );
  setPreferenceProperty( segLen, "min", 1 );
  setPreferenceProperty( segLen, "max", 10000000 );
  int nbSeg = addPreference( tr( "PREF_NB_SEGMENTS" ), segGroup, LightApp_Preferences::IntSpin,
                             "SMESH", "nb_segments_per_edge" );
  setPreferenceProperty( nbSeg, "min", 1 );
  setPreferenceProperty( nbSeg, "max", 10000000 );

  // Mesh tab ------------------------------------------------------------------------
  int meshTab = addPreference( tr( "PREF_TAB_MESH" ) );
  int nodeGroup = addPreference( tr( "PREF_GROUP_NODES" ), meshTab );
  setPreferenceProperty( nodeGroup, "columns", 2 );

  addPreference( tr( "PREF_COLOR" ), nodeGroup, LightApp_Preferences::Color, "SMESH", "node_color" );
  int nodeSz = addPreference( tr( "PREF_SIZE" ), nodeGroup, LightApp_Preferences::IntSpin, "SMESH", "node_size" );

  setPreferenceProperty( nodeSz, "min", 1 );
  setPreferenceProperty( nodeSz, "max", 5 );

  int elemGroup = addPreference( tr( "PREF_GROUP_ELEMENTS" ), meshTab );
  setPreferenceProperty( elemGroup, "columns", 2 );

  addPreference( tr( "PREF_FILL"     ), elemGroup, LightApp_Preferences::Color, "SMESH", "fill_color" );
  addPreference( tr( "PREF_OUTLINE"  ), elemGroup, LightApp_Preferences::Color, "SMESH", "outline_color" );
  addPreference( tr( "PREF_BACKFACE" ), elemGroup, LightApp_Preferences::Color, "SMESH", "backface_color" );
  addPreference( tr( "PREF_COLOR_0D" ), elemGroup, LightApp_Preferences::Color, "SMESH", "elem0d_color" );

  //int sp = addPreference( "", elemGroup, LightApp_Preferences::Space );
  //setPreferenceProperty( sp, "hstretch", 0 );
  //setPreferenceProperty( sp, "vstretch", 0 );

  int size0d = addPreference(tr("PREF_SIZE_0D"), elemGroup,
                             LightApp_Preferences::IntSpin, "SMESH", "elem0d_size");
  int sp = addPreference( "", elemGroup, LightApp_Preferences::Space );
  int elemW  = addPreference(tr("PREF_WIDTH"), elemGroup,
                             LightApp_Preferences::IntSpin, "SMESH", "element_width");
  int shrink = addPreference(tr("PREF_SHRINK_COEFF"), elemGroup,
                             LightApp_Preferences::IntSpin, "SMESH", "shrink_coeff");

  setPreferenceProperty( size0d, "min", 1 );
  setPreferenceProperty( size0d, "max", 10 );

  setPreferenceProperty( sp, "hstretch", 0 );
  setPreferenceProperty( sp, "vstretch", 0 );

  setPreferenceProperty( elemW, "min", 1 );
  setPreferenceProperty( elemW, "max", 5 );

  setPreferenceProperty( shrink, "min", 0 );
  setPreferenceProperty( shrink, "max", 100 );

  int orientGroup = addPreference( tr( "PREF_GROUP_FACES_ORIENTATION" ), meshTab );
  setPreferenceProperty( orientGroup, "columns", 1 );

  addPreference( tr( "PREF_ORIENTATION_COLOR" ), orientGroup, LightApp_Preferences::Color, "SMESH", "orientation_color" );
  int orientScale = addPreference( tr( "PREF_ORIENTATION_SCALE" ), orientGroup, LightApp_Preferences::DblSpin, "SMESH", "orientation_scale" );

  setPreferenceProperty( orientScale, "min", 0.05 );
  setPreferenceProperty( orientScale, "max", 0.5 );
  setPreferenceProperty( orientScale, "step", 0.05 );

  addPreference( tr( "PREF_ORIENTATION_3D_VECTORS" ), orientGroup, LightApp_Preferences::Bool, "SMESH", "orientation_3d_vectors" );

  // Selection tab ------------------------------------------------------------------------
  int selTab = addPreference( tr( "PREF_TAB_SELECTION" ) );

  int selGroup = addPreference( tr( "PREF_GROUP_SELECTION" ), selTab );
  setPreferenceProperty( selGroup, "columns", 2 );

  addPreference( tr( "PREF_OBJECT_COLOR" ), selGroup, LightApp_Preferences::Color, "SMESH", "selection_object_color" );
  addPreference( tr( "PREF_ELEMENT_COLOR" ), selGroup, LightApp_Preferences::Color, "SMESH", "selection_element_color" );
  int selW = addPreference( tr( "PREF_WIDTH" ), selGroup, LightApp_Preferences::IntSpin, "SMESH", "selection_width" );

  setPreferenceProperty( selW, "min", 1 );
  setPreferenceProperty( selW, "max", 5 );

  int preGroup = addPreference( tr( "PREF_GROUP_PRESELECTION" ), selTab );
  setPreferenceProperty( preGroup, "columns", 2 );

  addPreference( tr( "PREF_HIGHLIGHT_COLOR" ), preGroup, LightApp_Preferences::Color, "SMESH", "highlight_color" );
  int preW = addPreference( tr( "PREF_WIDTH" ), preGroup, LightApp_Preferences::IntSpin, "SMESH", "highlight_width" );

  setPreferenceProperty( preW, "min", 1 );
  setPreferenceProperty( preW, "max", 5 );

  int precSelGroup = addPreference( tr( "PREF_GROUP_PRECISION" ), selTab );
  setPreferenceProperty( precSelGroup, "columns", 2 );

  addPreference( tr( "PREF_NODES" ), precSelGroup, LightApp_Preferences::Double, "SMESH", "selection_precision_node" );
  addPreference( tr( "PREF_ELEMENTS" ), precSelGroup, LightApp_Preferences::Double, "SMESH", "selection_precision_element" );
  addPreference( tr( "PREF_OBJECTS" ), precSelGroup, LightApp_Preferences::Double, "SMESH", "selection_precision_object" );

  // Scalar Bar tab ------------------------------------------------------------------------
  int sbarTab = addPreference( tr( "SMESH_SCALARBAR" ) );
  int fontGr = addPreference( tr( "SMESH_FONT_SCALARBAR" ), sbarTab );
  setPreferenceProperty( fontGr, "columns", 2 );

  addVtkFontPref( tr( "SMESH_TITLE" ), fontGr, "scalar_bar_title_font" );
  addPreference( tr( "PREF_TITLE_COLOR" ), fontGr, LightApp_Preferences::Color, "SMESH", "scalar_bar_title_color" );

  addVtkFontPref( tr( "SMESH_LABELS" ), fontGr, "scalar_bar_label_font" );
  addPreference( tr( "PREF_LABELS_COLOR" ), fontGr, LightApp_Preferences::Color, "SMESH", "scalar_bar_label_color" );

  int colorsLabelsGr = addPreference( tr( "SMESH_LABELS_COLORS_SCALARBAR" ), sbarTab );
  setPreferenceProperty( colorsLabelsGr, "columns", 2 );

  int numcol = addPreference( tr( "SMESH_NUMBEROFCOLORS" ), colorsLabelsGr, LightApp_Preferences::IntSpin, "SMESH", "scalar_bar_num_colors" );
  setPreferenceProperty( numcol, "min", 2 );
  setPreferenceProperty( numcol, "max", 256 );

  int numlab = addPreference( tr( "SMESH_NUMBEROFLABELS" ), colorsLabelsGr, LightApp_Preferences::IntSpin, "SMESH", "scalar_bar_num_labels" );
  setPreferenceProperty( numlab, "min", 2 );
  setPreferenceProperty( numlab, "max", 65 );

  int orientGr = addPreference( tr( "SMESH_ORIENTATION" ), sbarTab );
  setPreferenceProperty( orientGr, "columns", 2 );
  int orient = addPreference( tr( "SMESH_ORIENTATION" ), orientGr, LightApp_Preferences::Selector, "SMESH", "scalar_bar_orientation" );
  QStringList orients;
  orients.append( tr( "SMESH_VERTICAL" ) );
  orients.append( tr( "SMESH_HORIZONTAL" ) );
  indices.clear(); indices.append( 0 ); indices.append( 1 );
  setPreferenceProperty( orient, "strings", orients );
  setPreferenceProperty( orient, "indexes", indices );

  int posVSizeGr = addPreference( tr( "SMESH_POSITION_SIZE_SCALARBAR" ) + " " + tr( "SMESH_VERTICAL" ), sbarTab );
  setPreferenceProperty( posVSizeGr, "columns", 2 );
  int xv = addPreference( tr( "SMESH_X_SCALARBAR" ), posVSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_vertical_x" );
  int yv = addPreference( tr( "SMESH_Y_SCALARBAR" ), posVSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_vertical_y" );
  int wv = addPreference( tr( "SMESH_WIDTH" ), posVSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_vertical_width" );
  int hv = addPreference( tr( "SMESH_HEIGHT" ), posVSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_vertical_height" );
  setPreferenceProperty( xv, "step", 0.1 );
  setPreferenceProperty( xv, "min", 0.0 );
  setPreferenceProperty( xv, "max", 1.0 );
  setPreferenceProperty( yv, "step", 0.1 );
  setPreferenceProperty( yv, "min", 0.0 );
  setPreferenceProperty( yv, "max", 1.0 );
  setPreferenceProperty( wv, "step", 0.1 );
  setPreferenceProperty( wv, "min", 0.0 );
  setPreferenceProperty( wv, "max", 1.0 );
  setPreferenceProperty( hv, "min", 0.0 );
  setPreferenceProperty( hv, "max", 1.0 );
  setPreferenceProperty( hv, "step", 0.1 );

  int posHSizeGr = addPreference( tr( "SMESH_POSITION_SIZE_SCALARBAR" ) + " " + tr( "SMESH_HORIZONTAL" ), sbarTab );
  setPreferenceProperty( posHSizeGr, "columns", 2 );
  int xh = addPreference( tr( "SMESH_X_SCALARBAR" ), posHSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_horizontal_x" );
  int yh = addPreference( tr( "SMESH_Y_SCALARBAR" ), posHSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_horizontal_y" );
  int wh = addPreference( tr( "SMESH_WIDTH" ), posHSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_horizontal_width" );
  int hh = addPreference( tr( "SMESH_HEIGHT" ), posHSizeGr, LightApp_Preferences::DblSpin, "SMESH", "scalar_bar_horizontal_height" );
  setPreferenceProperty( xv, "min", 0.0 );
  setPreferenceProperty( xv, "max", 1.0 );
  setPreferenceProperty( xv, "step", 0.1 );
  setPreferenceProperty( xh, "min", 0.0 );
  setPreferenceProperty( xh, "max", 1.0 );
  setPreferenceProperty( xh, "step", 0.1 );
  setPreferenceProperty( yh, "min", 0.0 );
  setPreferenceProperty( yh, "max", 1.0 );
  setPreferenceProperty( yh, "step", 0.1 );
  setPreferenceProperty( wh, "min", 0.0 );
  setPreferenceProperty( wh, "max", 1.0 );
  setPreferenceProperty( wh, "step", 0.1 );
  setPreferenceProperty( hh, "min", 0.0 );
  setPreferenceProperty( hh, "max", 1.0 );
  setPreferenceProperty( hh, "step", 0.1 );
}

void SMESHGUI::preferencesChanged( const QString& sect, const QString& name )
{
  if( sect=="SMESH" ) {
    float sbX1,sbY1,sbW,sbH;
    float aTol = 1.00000009999999;
    std::string aWarning;
    SUIT_ResourceMgr* aResourceMgr = SMESH::GetResourceMgr(this);
    if( name=="selection_object_color" || name=="selection_element_color" ||
        name=="selection_width" || name=="highlight_color" || name=="highlight_width" ||
        name=="selection_precision_node" || name=="selection_precision_element" ||
        name=="selection_precision_object")
      SMESH::UpdateSelectionProp( this );
    else if (name == QString("scalar_bar_vertical_x") || name == QString("scalar_bar_vertical_width")){
      sbX1 = aResourceMgr->doubleValue("SMESH", "scalar_bar_vertical_x", sbX1);
      sbW = aResourceMgr->doubleValue("SMESH", "scalar_bar_vertical_width", sbW);
      if(sbX1+sbW > aTol){
        aWarning = "Origin and Size Vertical: X+Width > 1\n";
        sbX1=0.01;
        sbW=0.08;
        aResourceMgr->setValue("SMESH", "scalar_bar_vertical_x", sbX1);
        aResourceMgr->setValue("SMESH", "scalar_bar_vertical_width", sbW);
      }
    }
    else if(name == QString("scalar_bar_vertical_y") || name == QString("scalar_bar_vertical_height")){
      sbY1 = aResourceMgr->doubleValue("SMESH", "scalar_bar_vertical_y", sbY1);
      sbH = aResourceMgr->doubleValue("SMESH", "scalar_bar_vertical_height",sbH);
      if(sbY1+sbH > aTol){
        aWarning = "Origin and Size Vertical: Y+Height > 1\n";
        aResourceMgr->setValue("SMESH", "scalar_bar_vertical_y", sbY1);
        aResourceMgr->setValue("SMESH", "scalar_bar_vertical_height",sbH);
      }
    }
    else if(name ==  QString("scalar_bar_horizontal_x") || name ==  QString("scalar_bar_horizontal_width")){
      sbX1 = aResourceMgr->doubleValue("SMESH", "scalar_bar_horizontal_x", sbX1);
      sbW = aResourceMgr->doubleValue("SMESH", "scalar_bar_horizontal_width", sbW);
      if(sbX1+sbW > aTol){
        aWarning = "Origin and Size Horizontal: X+Width > 1\n";
        sbX1=0.1;
        sbW=0.08;
        aResourceMgr->setValue("SMESH", "scalar_bar_horizontal_x", sbX1);
        aResourceMgr->setValue("SMESH", "scalar_bar_horizontal_width", sbW);
      }
    }
    else if(name ==  QString("scalar_bar_horizontal_y") || name ==  QString("scalar_bar_horizontal_height")){
      sbY1 = aResourceMgr->doubleValue("SMESH", "scalar_bar_horizontal_y", sbY1);
      sbH = aResourceMgr->doubleValue("SMESH", "scalar_bar_horizontal_height",sbH);
      if(sbY1+sbH > aTol){
        aWarning = "Origin and Size Horizontal: Y+Height > 1\n";
        sbY1=0.01;
        sbH=0.08;
        aResourceMgr->setValue("SMESH", "scalar_bar_horizontal_y", sbY1);
        aResourceMgr->setValue("SMESH", "scalar_bar_horizontal_height",sbH);
      }
    }
    else if ( name == "segmentation" ) {
      int nbSeg = aResourceMgr->integerValue( "SMESH", "segmentation", 10 );
      myComponentSMESH->SetBoundaryBoxSegmentation( nbSeg );
    }
    else if ( name == "nb_segments_per_edge" ) {
      int nbSeg = aResourceMgr->integerValue( "SMESH", "nb_segments_per_edge", 15 );
      myComponentSMESH->SetDefaultNbSegments( nbSeg );
    }

    if(aWarning.size() != 0){
      aWarning += "The default values are applied instead.";
      SUIT_MessageBox::warning(SMESHGUI::desktop(),
                               QObject::tr("SMESH_ERR_SCALARBAR_PARAMS"),
                               QObject::tr(aWarning.c_str()));
    }
  }
}

//================================================================================
/*!
 * \brief Update something in accordance with update flags
  * \param theFlags - update flags
*
* Update viewer or/and object browser etc. in accordance with update flags ( see
* LightApp_UpdateFlags enumeration ).
*/
//================================================================================
void SMESHGUI::update( const int flags )
{
  if ( flags & UF_Viewer | flags & UF_Forced )
    SMESH::UpdateView();
  else
    SalomeApp_Module::update( flags );
}

//================================================================================
/*!
 * \brief Set default selection mode
*
* SLOT called when operation commited. Sets default selection mode
*/
//================================================================================
void SMESHGUI::onOperationCommited( SUIT_Operation* )
{
  SVTK_ViewWindow* vtkWnd =
    dynamic_cast<SVTK_ViewWindow*>( application()->desktop()->activeWindow() );
  if ( vtkWnd )
    vtkWnd->SetSelectionMode( ActorSelection );
}

//================================================================================
/*!
 * \brief Set default selection mode
*
* SLOT called when operation aborted. Sets default selection mode
*/
//================================================================================
void SMESHGUI::onOperationAborted( SUIT_Operation* )
{
  SVTK_ViewWindow* vtkWnd =
    dynamic_cast<SVTK_ViewWindow*>( application()->desktop()->activeWindow() );
  if ( vtkWnd )
    vtkWnd->SetSelectionMode( ActorSelection );
}

//================================================================================
/*!
 * \brief Creates operation with given identifier
  * \param id - identifier of operation to be started
  * \return Pointer on created operation or NULL if operation is not created
*
* Virtual method redefined from the base class creates operation with given id.
* It is called called automatically from startOperation method of base class.
*/
//================================================================================
LightApp_Operation* SMESHGUI::createOperation( const int id ) const
{
  LightApp_Operation* op = 0;
  // to do : create operation here
  switch( id )
  {
    case 701: // Compute mesh
      op = new SMESHGUI_ComputeOp();
    break;
    case 702: // Create mesh
      op = new SMESHGUI_MeshOp( true, true );
    break;
    case 703: // Create sub-mesh
      op = new SMESHGUI_MeshOp( true, false );
    break;
    case 704: // Edit mesh/sub-mesh
      op = new SMESHGUI_MeshOp( false );
    break;
    case 711: // Precompute mesh
      op = new SMESHGUI_PrecomputeOp();
    break;
    case 712: // Evaluate mesh
      op = new SMESHGUI_EvaluateOp();
    break;
    case 806: // Create group on geom
      op = new SMESHGUI_GroupOnShapeOp();
      break;
    case 417: //convert to quadratic
      op = new SMESHGUI_ConvToQuadOp();
    break;
    case 418: // create 2D mesh as boundary on 3D
      op = new SMESHGUI_Make2DFrom3DOp();
    break;
    case 4067: // make mesh pass through point
      op = new SMESHGUI_MakeNodeAtPointOp();
      break;
    default:
    break;
  }

  if( !op )
    op = SalomeApp_Module::createOperation( id );
  return op;
}

//================================================================================
/*!
 * \brief Stops current operations and starts a given one
  * \param id - The id of the operation to start
 */
//================================================================================

void SMESHGUI::switchToOperation(int id)
{
  if ( _PTR(Study) aStudy = SMESH::GetActiveStudyDocument() )
    activeStudy()->abortAllOperations();
  startOperation( id );
}

LightApp_Displayer* SMESHGUI::displayer()
{
  if( !myDisplayer )
    myDisplayer = new SMESHGUI_Displayer( getApp() );
  return myDisplayer;
}

SALOMEDS::Color SMESHGUI::getUniqueColor( const QList<SALOMEDS::Color>& theReservedColors )
{
  int aHue = -1;
  int aTolerance = 64;
  int anIterations = 0;
  int aPeriod = 5;

  while( 1 )
  {
    anIterations++;
    if( anIterations % aPeriod == 0 )
    {
      aTolerance /= 2;
      if( aTolerance < 1 )
        break;
    }
    //cout << "Iteration N" << anIterations << " (tolerance=" << aTolerance << ")"<< endl;

    aHue = (int)( 360.0 * rand() / RAND_MAX );
    //cout << "Hue = " << aHue << endl;

    //cout << "Auto colors : ";
    bool ok = true;
    QList<SALOMEDS::Color>::const_iterator it = theReservedColors.constBegin();
    QList<SALOMEDS::Color>::const_iterator itEnd = theReservedColors.constEnd();
    for( ; it != itEnd; ++it )
    {
      SALOMEDS::Color anAutoColor = *it;
      QColor aQColor( (int)( anAutoColor.R * 255.0 ), (int)( anAutoColor.G * 255.0 ), (int)( anAutoColor.B * 255.0 ) );

      int h, s, v;
      aQColor.getHsv( &h, &s, &v );
      //cout << h << " ";
      if( abs( h - aHue ) < aTolerance )
      {
        ok = false;
        //cout << "break (diff = " << abs( h - aHue ) << ")";
        break;
      }
    }
    //cout << endl;

    if( ok )
      break;
  }

  //cout << "Hue of the returned color = " << aHue << endl;
  QColor aColor;
  aColor.setHsv( aHue, 255, 255 );

  SALOMEDS::Color aSColor;
  aSColor.R = (double)aColor.red() / 255.0;
  aSColor.G = (double)aColor.green() / 255.0;
  aSColor.B = (double)aColor.blue() / 255.0;

  return aSColor;
}

const char gSeparator = '_'; // character used to separate parameter names
const char gDigitsSep = ':'; // character used to separate numeric parameter values (color = r:g:b)

/*!
 * \brief Store visual parameters
 *
 * This method is called just before the study document is saved.
 * Store visual parameters in AttributeParameter attribue(s)
 */
void SMESHGUI::storeVisualParameters (int savePoint)
{
  SalomeApp_Study* appStudy = dynamic_cast<SalomeApp_Study*>(application()->activeStudy());
  if (!appStudy || !appStudy->studyDS())
    return;
  _PTR(Study) studyDS = appStudy->studyDS();

  // componentName is used for encoding of entries when storing them in IParameters
  std::string componentName = myComponentSMESH->ComponentDataType();
  //_PTR(SComponent) aSComponent = studyDS->FindComponent("SMESH");
  //if (!aSComponent) return;

  // IParameters
  _PTR(AttributeParameter) ap = studyDS->GetModuleParameters("Interface Applicative",
                                                             componentName.c_str(),
                                                             savePoint);
  _PTR(IParameters) ip = ClientFactory::getIParameters(ap);

  // viewers counters are used for storing view_numbers in IParameters
  int vtkViewers = 0;

  // main cycle to store parameters of displayed objects
  QList<SUIT_ViewManager*> lst;
  QList<SUIT_ViewManager*>::Iterator it;
  getApp()->viewManagers(lst);
  for (it = lst.begin(); it != lst.end(); it++)
  {
    SUIT_ViewManager* vman = *it;
    QString vType = vman->getType();

    // saving VTK actors properties
    if (vType == SVTK_Viewer::Type())
    {
      QVector<SUIT_ViewWindow*> views = vman->getViews();
      for (int i = 0, iEnd = vman->getViewsCount(); i < iEnd; i++)
      {
        if (SVTK_ViewWindow* vtkView = dynamic_cast<SVTK_ViewWindow*>(views[i]))
        {
          VTK::ActorCollectionCopy aCopy(vtkView->getRenderer()->GetActors());
          vtkActorCollection* allActors = aCopy.GetActors();
          allActors->InitTraversal();
          while (vtkActor* actor = allActors->GetNextActor())
          {
            if (actor->GetVisibility()) // store only visible actors
            {
              SMESH_Actor* aSmeshActor = 0;
              if (actor->IsA("SMESH_Actor"))
                aSmeshActor = SMESH_Actor::SafeDownCast(actor);
              if (aSmeshActor && aSmeshActor->hasIO())
              {
                Handle(SALOME_InteractiveObject) io = aSmeshActor->getIO();
                if (io->hasEntry())
                {
                  // entry is "encoded" = it does NOT contain component adress,
                  // since it is a subject to change on next component loading
                  std::string entry = ip->encodeEntry(io->getEntry(), componentName);

                  std::string param, vtkParam = vType.toLatin1().data();
                  vtkParam += gSeparator;
                  vtkParam += QString::number(vtkViewers).toLatin1().data();
                  vtkParam += gSeparator;

                  // Visibility
                  param = vtkParam + "Visibility";
                  ip->setParameter(entry, param, "On");

                  // Representation
                  param = vtkParam + "Representation";
                  ip->setParameter(entry, param, QString::number
                                   ((int)aSmeshActor->GetRepresentation()).toLatin1().data());

                  // IsShrunk
                  param = vtkParam + "IsShrunk";
                  ip->setParameter(entry, param, QString::number
                                   ((int)aSmeshActor->IsShrunk()).toLatin1().data());

                  // Displayed entities
                  unsigned int aMode = aSmeshActor->GetEntityMode();
                  bool isE = aMode & SMESH_Actor::eEdges;
                  bool isF = aMode & SMESH_Actor::eFaces;
                  bool isV = aMode & SMESH_Actor::eVolumes;

                  QString modeStr ("e");
                  modeStr += gDigitsSep; modeStr += QString::number(isE);
                  modeStr += gDigitsSep; modeStr += "f";
                  modeStr += gDigitsSep; modeStr += QString::number(isF);
                  modeStr += gDigitsSep; modeStr += "v";
                  modeStr += gDigitsSep; modeStr += QString::number(isV);

                  param = vtkParam + "Entities";
                  ip->setParameter(entry, param, modeStr.toLatin1().data());

                  // Colors (surface:edge:)
                  vtkFloatingPointType r, g, b;

                  aSmeshActor->GetSufaceColor(r, g, b);
                  QString colorStr ("surface");
                  colorStr += gDigitsSep; colorStr += QString::number(r);
                  colorStr += gDigitsSep; colorStr += QString::number(g);
                  colorStr += gDigitsSep; colorStr += QString::number(b);

                  aSmeshActor->GetBackSufaceColor(r, g, b);
                  colorStr += gDigitsSep; colorStr += "backsurface";
                  colorStr += gDigitsSep; colorStr += QString::number(r);
                  colorStr += gDigitsSep; colorStr += QString::number(g);
                  colorStr += gDigitsSep; colorStr += QString::number(b);

                  aSmeshActor->GetEdgeColor(r, g, b);
                  colorStr += gDigitsSep; colorStr += "edge";
                  colorStr += gDigitsSep; colorStr += QString::number(r);
                  colorStr += gDigitsSep; colorStr += QString::number(g);
                  colorStr += gDigitsSep; colorStr += QString::number(b);

                  aSmeshActor->GetNodeColor(r, g, b);
                  colorStr += gDigitsSep; colorStr += "node";
                  colorStr += gDigitsSep; colorStr += QString::number(r);
                  colorStr += gDigitsSep; colorStr += QString::number(g);
                  colorStr += gDigitsSep; colorStr += QString::number(b);

                  param = vtkParam + "Colors";
                  ip->setParameter(entry, param, colorStr.toLatin1().data());

                  // Sizes of lines and points
                  QString sizeStr ("line");
                  sizeStr += gDigitsSep; sizeStr += QString::number((int)aSmeshActor->GetLineWidth());
                  sizeStr += gDigitsSep; sizeStr += "node";
                  sizeStr += gDigitsSep; sizeStr += QString::number((int)aSmeshActor->GetNodeSize());
                  sizeStr += gDigitsSep; sizeStr += "shrink";
                  sizeStr += gDigitsSep; sizeStr += QString::number(aSmeshActor->GetShrinkFactor());

                  param = vtkParam + "Sizes";
                  ip->setParameter(entry, param, sizeStr.toLatin1().data());

                  // Opacity
                  param = vtkParam + "Opacity";
                  ip->setParameter(entry, param,
                                   QString::number(aSmeshActor->GetOpacity()).toLatin1().data());

                  // Clipping
                  param = vtkParam + "ClippingPlane";
                  int nPlanes = aSmeshActor->GetNumberOfClippingPlanes();
                  if (!nPlanes)
                    ip->setParameter(entry, param, "Off");
                  for (int ipl = 0; ipl < nPlanes; ipl++) {
                    //vtkPlane* plane = aSmeshActor->GetClippingPlane(ipl);
                    SMESH::Orientation anOrientation;
                    double aDistance;
                    vtkFloatingPointType anAngle[2];
                    SMESHGUI_ClippingDlg::GetPlaneParam(aSmeshActor, ipl, anOrientation, aDistance, anAngle);
                    std::string planeValue = QString::number((int)anOrientation).toLatin1().data();
                    planeValue += gDigitsSep; planeValue += QString::number(aDistance).toLatin1().data();
                    planeValue += gDigitsSep; planeValue += QString::number(anAngle[0]).toLatin1().data();
                    planeValue += gDigitsSep; planeValue += QString::number(anAngle[1]).toLatin1().data();

                    ip->setParameter(entry, param + QString::number(ipl+1).toLatin1().data(), planeValue);
                  }
                } // if (io->hasEntry())
              } // SMESH_Actor && hasIO
            } // isVisible
          } // while.. actors traversal
        } // if (vtkView)
      } // for (views)
      vtkViewers++;
    } // if (SVTK view model)
  } // for (viewManagers)
}

/*!
 * \brief Restore visual parameters
 *
 * This method is called after the study document is opened.
 * Restore visual parameters from AttributeParameter attribue(s)
 */
void SMESHGUI::restoreVisualParameters (int savePoint)
{
  SalomeApp_Study* appStudy = dynamic_cast<SalomeApp_Study*>(application()->activeStudy());
  if (!appStudy || !appStudy->studyDS())
    return;
  _PTR(Study) studyDS = appStudy->studyDS();

  // componentName is used for encoding of entries when storing them in IParameters
  std::string componentName = myComponentSMESH->ComponentDataType();
  //_PTR(SComponent) aSComponent = studyDS->FindComponent("GEOM");
  //if (!aSComponent) return;

  // IParameters
  _PTR(AttributeParameter) ap = studyDS->GetModuleParameters("Interface Applicative",
                                                             componentName.c_str(),
                                                             savePoint);
  _PTR(IParameters) ip = ClientFactory::getIParameters(ap);

  std::vector<std::string> entries = ip->getEntries();

  for (std::vector<std::string>::iterator entIt = entries.begin(); entIt != entries.end(); ++entIt)
  {
    // entry is a normal entry - it should be "decoded" (setting base adress of component)
    QString entry (ip->decodeEntry(*entIt).c_str());

    // Check that the entry corresponds to a real object in the Study
    // as the object may be deleted or modified after the visual state is saved.
    _PTR(SObject) so = studyDS->FindObjectID(entry.toLatin1().data());
    if (!so) continue; //Skip the not existent entry

    std::vector<std::string> paramNames = ip->getAllParameterNames( *entIt );
    std::vector<std::string> paramValues = ip->getAllParameterValues( *entIt );

    std::vector<std::string>::iterator namesIt = paramNames.begin();
    std::vector<std::string>::iterator valuesIt = paramValues.begin();

    // actors are stored in a map after displaying of them for
    // quicker access in the future: map < viewID to actor >
    NCollection_DataMap<int, SMESH_Actor*> vtkActors;

    for (; namesIt != paramNames.end(); ++namesIt, ++valuesIt)
    {
      // visual parameters are stored in strings as follows: ViewerType_ViewIndex_ParamName.
      // '_' is used as separator and should not be used in viewer type or parameter names.
      QStringList lst = QString((*namesIt).c_str()).split(gSeparator, QString::SkipEmptyParts);
      if (lst.size() != 3)
        continue;

      QString viewerTypStr = lst[0];
      QString viewIndexStr = lst[1];
      QString paramNameStr = lst[2];

      bool ok;
      int viewIndex = viewIndexStr.toUInt(&ok);
      if (!ok) // bad conversion of view index to integer
        continue;

      // viewers
      if (viewerTypStr == SVTK_Viewer::Type())
      {
        SMESH_Actor* aSmeshActor = 0;
        if (vtkActors.IsBound(viewIndex))
          aSmeshActor = vtkActors.Find(viewIndex);

        if (paramNameStr == "Visibility")
        {
          if (!aSmeshActor && displayer())
          {
            QList<SUIT_ViewManager*> lst;
            getApp()->viewManagers(viewerTypStr, lst);

            // SVTK ViewManager always has 1 ViewWindow, so view index is index of view manager
            if (viewIndex >= 0 && viewIndex < lst.count()) {
              SUIT_ViewManager* vman = lst.at(viewIndex);
              SUIT_ViewModel* vmodel = vman->getViewModel();
              // SVTK view model can be casted to SALOME_View
              displayer()->Display(entry, true, dynamic_cast<SALOME_View*>(vmodel));

              // store displayed actor in a temporary map for quicker
              // access later when restoring other parameters
              SVTK_ViewWindow* vtkView = (SVTK_ViewWindow*) vman->getActiveView();
              vtkRenderer* Renderer = vtkView->getRenderer();
              VTK::ActorCollectionCopy aCopy(Renderer->GetActors());
              vtkActorCollection* theActors = aCopy.GetActors();
              theActors->InitTraversal();
              bool isFound = false;
              vtkActor *ac = theActors->GetNextActor();
              for (; ac != NULL && !isFound; ac = theActors->GetNextActor()) {
                if (ac->IsA("SMESH_Actor")) {
                  SMESH_Actor* aGeomAc = SMESH_Actor::SafeDownCast(ac);
                  if (aGeomAc->hasIO()) {
                    Handle(SALOME_InteractiveObject) io =
                      Handle(SALOME_InteractiveObject)::DownCast(aGeomAc->getIO());
                    if (io->hasEntry() && strcmp(io->getEntry(), entry.toLatin1().data()) == 0) {
                      isFound = true;
                      vtkActors.Bind(viewIndex, aGeomAc);
                    }
                  }
                }
              }
            }
          }
        } // if (paramNameStr == "Visibility")
        else
        {
          // the rest properties "work" with SMESH_Actor
          if (aSmeshActor)
          {
            QString val ((*valuesIt).c_str());

            // Representation
            if (paramNameStr == "Representation") {
              aSmeshActor->SetRepresentation((SMESH_Actor::EReperesent)val.toInt());
            }
            // IsShrunk
            else if (paramNameStr == "IsShrunk") {
              if (val.toInt()) {
                if (!aSmeshActor->IsShrunk())
                  aSmeshActor->SetShrink();
              }
              else {
                if (aSmeshActor->IsShrunk())
                  aSmeshActor->UnShrink();
              }
            }
            // Displayed entities
            else if (paramNameStr == "Entities") {
              QStringList mode = val.split(gDigitsSep, QString::SkipEmptyParts);
              if (mode.count() == 6) {
                if (mode[0] != "e" || mode[2]  != "f" || mode[4] != "v") {
                  MESSAGE("Invalid order of data in Entities, must be: "
                          "e:0/1:f:0/1:v:0/1");
                }
                else {
                  unsigned int aMode = aSmeshActor->GetEntityMode();
                  unsigned int aNewMode =
                    (int(SMESH_Actor::eEdges  ) * mode[1].toInt()) |
                    (int(SMESH_Actor::eFaces  ) * mode[3].toInt()) |
                    (int(SMESH_Actor::eVolumes) * mode[5].toInt());
                  if (aNewMode != aMode)
                    aSmeshActor->SetEntityMode(aNewMode);
                }
              }
            }
            // Colors
            else if (paramNameStr == "Colors") {
              QStringList colors = val.split(gDigitsSep, QString::SkipEmptyParts);
              if (colors.count() == 16) {
                if (colors[0] != "surface" || colors[4]  != "backsurface" ||
                    colors[8] != "edge"    || colors[12] != "node") {
                  MESSAGE("Invalid order of data in Colors, must be: "
                          "surface:r:g:b:backsurface:r:g:b:edge:r:g:b:node:r:g:b");
                }
                else {
                  aSmeshActor->SetSufaceColor(colors[1].toFloat(), colors[2].toFloat(), colors[3].toFloat());
                  aSmeshActor->SetBackSufaceColor(colors[5].toFloat(), colors[6].toFloat(), colors[7].toFloat());
                  aSmeshActor->SetEdgeColor(colors[9].toFloat(), colors[10].toFloat(), colors[11].toFloat());
                  aSmeshActor->SetNodeColor(colors[13].toFloat(), colors[14].toFloat(), colors[15].toFloat());
                }
              }
            }
            // Sizes of lines and points
            else if (paramNameStr == "Sizes") {
              QStringList sizes = val.split(gDigitsSep, QString::SkipEmptyParts);
              if (sizes.count() == 6) {
                if (sizes[0] != "line" || sizes[2]  != "node" || sizes[4] != "shrink") {
                  MESSAGE("Invalid order of data in Sizes, must be: "
                          "line:int:node:int:shrink:float");
                }
                else {
                  aSmeshActor->SetLineWidth(sizes[1].toInt());
                  aSmeshActor->SetNodeSize(sizes[3].toInt());
                  aSmeshActor->SetShrinkFactor(sizes[5].toFloat());
                }
              }
            }
            // Opacity
            else if (paramNameStr == "Opacity") {
              aSmeshActor->SetOpacity(val.toFloat());
            }
            // Clipping
            else if (paramNameStr.startsWith("ClippingPlane")) {
              cout << "$$$ ClippingPlane 1" << endl;
              if (paramNameStr == "ClippingPlane1" || val == "Off")
                aSmeshActor->RemoveAllClippingPlanes();
              if (val != "Off") {
                cout << "$$$ ClippingPlane 2" << endl;
                QStringList vals = val.split(gDigitsSep, QString::SkipEmptyParts);
                if (vals.count() == 4) { // format check: 4 values
                  cout << "$$$ ClippingPlane 3" << endl;
                  SMESH::Orientation anOrientation = (SMESH::Orientation)vals[0].toInt();
                  double aDistance = vals[1].toFloat();
                  vtkFloatingPointType anAngle[2];
                  anAngle[0] = vals[2].toFloat();
                  anAngle[1] = vals[3].toFloat();

                  QList<SUIT_ViewManager*> lst;
                  getApp()->viewManagers(viewerTypStr, lst);
                  // SVTK ViewManager always has 1 ViewWindow, so view index is index of view manager
                  if (viewIndex >= 0 && viewIndex < lst.count()) {
                    SUIT_ViewManager* vman = lst.at(viewIndex);
                    SVTK_ViewWindow* vtkView = (SVTK_ViewWindow*) vman->getActiveView();
                    SMESHGUI_ClippingDlg::AddPlane(aSmeshActor, vtkView,
                                                   anOrientation, aDistance, anAngle);
                  }
                }
              }
            }
          } // if (aSmeshActor)
        } // other parameters than Visibility
      }
    } // for names/parameters iterator
  } // for entries iterator

  // update all VTK views
  QList<SUIT_ViewManager*> lst;
  getApp()->viewManagers(lst);
  for (QList<SUIT_ViewManager*>::Iterator it = lst.begin(); it != lst.end(); it++) {
    SUIT_ViewModel* vmodel = (*it)->getViewModel();
    if (vmodel && vmodel->getType() == SVTK_Viewer::Type()) {
      SVTK_ViewWindow* vtkView = (SVTK_ViewWindow*) (*it)->getActiveView();
      vtkView->getRenderer()->ResetCameraClippingRange();
      vtkView->Repaint();
    }
  }
}

/*!
  \brief Adds preferences for dfont of VTK viewer
  \param label label
  \param pIf group identifier
  \param param parameter
  \return identifier of preferences
*/
int SMESHGUI::addVtkFontPref( const QString& label, const int pId, const QString& param )
{
  int tfont = addPreference( label, pId, LightApp_Preferences::Font, "VISU", param );
  
  setPreferenceProperty( tfont, "mode", QtxFontEdit::Custom );

  QStringList fam;
  fam.append( tr( "SMESH_FONT_ARIAL" ) );
  fam.append( tr( "SMESH_FONT_COURIER" ) );
  fam.append( tr( "SMESH_FONT_TIMES" ) );

  setPreferenceProperty( tfont, "fonts", fam );

  int f = QtxFontEdit::Family | QtxFontEdit::Bold | QtxFontEdit::Italic | QtxFontEdit::Shadow;
  setPreferenceProperty( tfont, "features", f );

  return tfont;
}

/*!
  \brief Actions after hypothesis edition
  Updates object browser after hypothesis edition
*/
void SMESHGUI::onHypothesisEdit( int result )
{
  updateObjBrowser( true );
}
