# Copyright (c) 2013 Mirantis Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
import unittest2

from savanna.utils import files as f
from savanna.utils.openstack import heat as h


class TestHeat(unittest2.TestCase):
    def test_gets(self):
        inst_name = "cluster-worker-1"
        self.assertEqual(h._get_inst_name("cluster", "worker", 0), inst_name)
        self.assertEqual(h._get_port_name(inst_name), "cluster-worker-1-port")
        self.assertEqual(h._get_floating_name(inst_name),
                         "cluster-worker-1-floating")
        self.assertEqual(h._get_volume_name(inst_name, 1),
                         "cluster-worker-1-volume-1")
        self.assertEqual(h._get_volume_attach_name(inst_name, 1),
                         "cluster-worker-1-volume-attachment-1")

    def test_prepare_user_data(self):
        userdata = "line1\nline2"
        self.assertEqual(h._prepare_userdata(userdata), '"line1",\n"line2"')


class TestClusterTemplate(unittest2.TestCase):
    """This test checks valid structure of Resources
       section in Heat templates generated by Savanna:
       1. It checks templates generation with different OpenStack
       network installations: Neutron, NovaNetwork with floating Ip auto
       assignment set to True or False.
       2. Cinder volume attachments.
       3. Basic instances creations with multi line user data provided
    """
    def setUp(self):
        self.cluster_template = h.ClusterTemplate("cluster", "private_net",
                                                  'user_key')
        self.cluster_template.add_node_group('master', 1, '42', '1',
                                             'line1\nline2', 'floating', 0, 0)
        self.cluster_template.add_node_group('worker', 1, '42', '1',
                                             'line2\nline3', 'floating', 2, 10)

    def test_load_template_use_neutron(self):
        """This test checks Heat cluster template with Neutron enabled.
           Two NodeGroups used: 'master' with Ephemeral drive attached and
           'worker' with 2 attached volumes 10GB size each
        """
        h.CONF.use_neutron = True

        main_template = h._load_template(
            'main.heat', {'resources':
                          self.cluster_template._serialize_resources()})

        self.assertEqual(
            json.loads(main_template),
            json.loads(f.get_file_text(
                "tests/unit/resources/"
                "test_serialize_resources_use_neutron.heat")))
