#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from zope.component import queryUtility
from zope.location import location
from zope.interface import implements
from zope.publisher.interfaces import IPublishTraverse, NotFound
from zope.security.checker import canAccess
from zope.security.interfaces import Unauthorized
from zope.security.proxy import removeSecurityProxy

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.intervention import interfaces
from schooltool.intervention.intervention import getInterventionStudent
from schooltool.person.interfaces import IPerson
from schooltool.term.interfaces import IDateManager
from schooltool.traverser.traverser import NameTraverserPlugin


class InterventionTabProxy(object):
    """Proxy used as context of the Intervention tab views."""
    implements(interfaces.IInterventionTabProxy)

    def __init__(self, user):
        self.user = user


class InterventionTabTraverserPlugin(NameTraverserPlugin):
    """Traverses from a user's BasicPerson object to the Intervention tab
       proxy which in turn will serve as context to the Intervention tab views.

    i.e. if you go to localhost/persons/someuser/intervention_tab this traverser
    will get called instead of standard traverser.  It returns an
    InterventionTabProxy for the user.
    """

    traversalName = "intervention_tab"
    def _traverse(self, request, name):
        person = IPerson(request.principal, None)
        if person is None or person.username != self.context.username:
            raise NotFound(self.context, name, request)
        proxyInterventionTab = InterventionTabProxy(self.context)
        return location.LocationProxy(proxyInterventionTab, self.context,
                                      self.traversalName)


class StudentSchoolYearsProxy(object):
    """Proxy used for traversing to a student message or goal."""

    def __init__(self, student):
        self.student = student


class StudentSchoolYearsTraverserPlugin(NameTraverserPlugin):
    """Represents the first step in traversing from a student to a goal or
       message object.

    i.e. if you go to localhost/persons/somestudent/schoolyears this traverser
    will get called instead of standard traverser.  It returns the an
    StudentSchoolYearsProxy for the student.
    """

    traversalName = "schoolyears"
    def _traverse(self, request, name):
        proxySchoolYears = StudentSchoolYearsProxy(self.context)
        return location.LocationProxy(proxySchoolYears, self.context,
                                      self.traversalName)


class StudentSchoolYearProxy(object):
    """Proxy used for traversing to a student message or goal."""
    implements(interfaces.IStudentSchoolYearProxy)

    def __init__(self, year):
        self.year = removeSecurityProxy(year)


class StudentSchoolYearTraverser(object):
    """Allows traversal to a student's schoolyear proxy.

    i.e. if you go to localhost/persons/somestudent/schoolyears/someyear this
    traverser will get called instead of standard traverser.  It also wraps the
    message in a location proxy so the url looks like what is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """
    implements(IPublishTraverse)

    def __init__(self, context, request):
        self.context = context
        self.request = request

    def publishTraverse(self, request, name):
        app = ISchoolToolApplication(None)
        interventionRoot = app[u'schooltool.interventions']
        if name not in interventionRoot:
            raise NotFound(self.context, name, request)
        proxySchoolYear = StudentSchoolYearProxy(interventionRoot[name])
        return location.LocationProxy(proxySchoolYear, self.context,
                                      name)


class StudentMessagesGoalsProxy(object):
    """Proxy used for traversing to a student message or goal."""

    def __init__(self, type):
        self.type = type


class StudentMessagesGoalsTraverser(object):
    """Allows traversal to a student's schoolyear proxy.

    i.e. go to localhost/persons/somestudent/schoolyears/someyear/messages or
    goals and this traverser will get called instead of standard traverser.
    It also wraps the message in a location proxy so the url looks like what
    is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """
    implements(IPublishTraverse)

    def __init__(self, context, request):
        self.context = context
        self.request = request

    def publishTraverse(self, request, name):
        proxyMessagesGoals = StudentMessagesGoalsProxy(name)
        return location.LocationProxy(proxyMessagesGoals, self.context,
                                      name)


class StudentMessageGoalTraverser(object):
    """Allows traversal to a student's schoolyear proxy.

    i.e. localhost/persons/somestudent/schoolyears/someyear/goals/somegoal or
    message and this traverser will get called instead of standard traverser.
    It also wraps the message in a location proxy so the url looks like what
    is above.

    The whole reason for having this traversal adaptation is to allow
    persons responsible to see messages or goals for students that they
    are not teaching, advising or administrating.
    """
    implements(IPublishTraverse)

    def __init__(self, context, request):
        self.context = context
        self.request = request

    def publishTraverse(self, request, name):
        context = removeSecurityProxy(self.context)
        yearproxy = removeSecurityProxy(context.__parent__)
        yearsproxy = removeSecurityProxy(yearproxy.__parent__)
        student = removeSecurityProxy(yearsproxy.student)
        interventionStudent = removeSecurityProxy(
            yearproxy.year[student.username])
        messagesGoals = interventionStudent[context.type]
        if name not in messagesGoals:
            raise NotFound(self.context, name, request)
        messageGoal = messagesGoals[name]
        if not canAccess(messageGoal, 'created'):
            raise Unauthorized
        return location.LocationProxy(messageGoal, self.context,
                                      name)


class  InterventionStudentTraverserPlugin(NameTraverserPlugin):
    """Allows traversal to a student's intervention object.

    i.e. if you go to localhost/persons/somestudent/intervention this traverser
    will get called instead of standard traverser.  It returns the student's
    intervention object for the current school year.
    """

    traversalName = "intervention"
    def _traverse(self, request, name):
        if queryUtility(IDateManager).current_term is None:
            return self.context
        interventionStudent = getInterventionStudent(self.context.__name__)
        return interventionStudent


class SectionMessagesProxy(object):
    """Proxy used for traversing to a section's student messages."""

    def __init__(self, section):
        self.section = section


class SectionMessagesTraverserPlugin(NameTraverserPlugin):
    """Allows traversal from a section object to a SectionMessagesProxy
    object, which in turn will be used to traverse to the messages
    container for a given student.

    i.e. if you go to localhost/sections/somesec/messages this traverser
    will get called instead of standard traverser.
    """

    traversalName = "messages"
    def _traverse(self, request, name):
        proxyMessages = SectionMessagesProxy(self.context)
        return location.LocationProxy(proxyMessages, self.context,
                                      self.traversalName)

class SectionMessagesTraverser(object):
    """Allows traversal to a student's messages container from the
    section messages proxy.

    i.e. if you go to localhost/sections/somesec/messages/someone this
    traverser will get called instead of standard traverser.  It also wraps the
    message in a location proxy so the url looks like what is above.

    The whole reason for having this traversal adaptation is to allow
    the message add view to know what section the message is for.
    """
    implements(IPublishTraverse)

    def __init__(self, context, request):
        self.context = context
        self.request = request

    def publishTraverse(self, request, name):
        interventionStudent = getInterventionStudent(name)
        messages = interventionStudent['messages']
        messages = location.LocationProxy(messages, self.context, name)
        return messages

