/* 
 * $Id: my_con.c,v 1.7.2.1 2008/02/11 15:50:41 janakj Exp $
 *
 * Copyright (C) 2001-2004 iptel.org
 *
 * This file is part of ser, a free SIP server.
 *
 * ser is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * For a license to use the ser software under conditions
 * other than those described here, or to purchase support for this
 * software, please contact iptel.org by e-mail at the following addresses:
 *    info@iptel.org
 *
 * ser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "my_con.h"
#include "db_mod.h"
#include "../../mem/mem.h"
#include "../../dprint.h"
#include "../../ut.h"
#include "utils.h"
#include <string.h>
#include <time.h>


/*
 * Create a new connection structure,
 * open the MySQL connection and set reference count to 1
 */
struct my_con* new_connection(struct db_id* id)
{
	struct my_con* ptr;
	my_bool reconnect;

	if (!id) {
		LOG(L_ERR, "new_connection: Invalid parameter value\n");
		return 0;
	}

	my_client_version = mysql_get_client_version();

	ptr = (struct my_con*)pkg_malloc(sizeof(struct my_con));
	if (!ptr) {
		LOG(L_ERR, "new_connection: No memory left\n");
		return 0;
	}

	memset(ptr, 0, sizeof(struct my_con));
	ptr->ref = 1;
	
	ptr->con = (MYSQL*)pkg_malloc(sizeof(MYSQL));
	if (!ptr->con) {
		LOG(L_ERR, "new_connection: No enough memory\n");
		goto err;
	}

	mysql_init(ptr->con);

	if (id->port) {
		DBG("new_connection: Opening MySQL connection: %s://%s:%s@%s:%d/%s\n",
		    ZSW(id->scheme),
		    ZSW(id->username),
		    ZSW(id->password),
		    ZSW(id->host),
		    id->port,
		    ZSW(id->database)
		    );
	} else {
		DBG("new_connection: Opening MySQL connection: %s://%s:%s@%s/%s\n",
		    ZSW(id->scheme),
		    ZSW(id->username),
		    ZSW(id->password),
		    ZSW(id->host),
		    ZSW(id->database)
		    );
	}

	/* The reconnection magic comes here. The objective is to make mysql
	 * client reconnect automatically when the connection to the server gets
	 * terminated. In mysql client versions up to 5.0.3 reconnection was enabled
	 * implicitly so there was no need to enable it manually. Starting from 5.0.3
	 * mysql decided to disable implicit reconnection, therefore it is necessary
	 * to enable automatic reconnection before calling mysql_real_connect.
	 *
	 * There are two ways of enabling automatic reconnects, the user can either
	 * set the 'reconnect' variable in MYSQL data structure to one, or he/she
	 * can use mysql_options with option MYSQL_OPT_RECONNECT. The option was
	 * first introduced in 5.0.13 so we have to sure that we are using newer
	 * version of mysql client than 5.0.13. We use both methods (setting the
	 * variable and calling mysql_options) to be on the safe side.
	 *
	 */

	ptr->con->reconnect = 1;

	/* Compile the code only if MYSQL_OPT_RECONNECT is defined which was
	 * introduced in 5.0.13
	 */
#if MYSQL_VERSION_ID >= 50013
	if (my_client_version >= 50013) {
		reconnect = 1;
		if (mysql_options(ptr->con, MYSQL_OPT_RECONNECT, &reconnect)) {
			WARN("mysql: Failed to set MYSQL_OPT_RECONNECT\n");
		}
	}
#endif

	if (!mysql_real_connect(ptr->con, id->host, id->username, id->password, id->database, id->port, 0, 0)) {
		LOG(L_ERR, "new_connection: %s\n", mysql_error(ptr->con));
		mysql_close(ptr->con);
		goto err;
	}

	/* Reconnection magic, part two. To make it even more complicated, there is a bug
	 * in mysql_real_connect which causes the MYSQL_OPT_RECONNECT flag to be reset
	 * to false after the connection is established. This bug is present in versions
	 * < 5.0.19 and in 5.1 in versions 5.1.0 to 5.1.6. To get around this problem
	 * we simply repeat the reconnectio magic here, after calling mysql_real_connect.
	 */

	ptr->con->reconnect = 1;

	/* Compile the code only if MYSQL_OPT_RECONNECT is defined which was
	 * introduced in 5.0.13
	 */
#if MYSQL_VERSION_ID >= 50013
	if (my_client_version >= 50013) {
		reconnect = 1;
		if (mysql_options(ptr->con, MYSQL_OPT_RECONNECT, &reconnect)) {
			WARN("mysql: Failed to set MYSQL_OPT_RECONNECT\n");
		}
	}
#endif

	DBG("new_connection: Connection type is %s\n", mysql_get_host_info(ptr->con));
	DBG("new_connection: Protocol version is %d\n", mysql_get_proto_info(ptr->con));
	DBG("new_connection: Server version is %s\n", mysql_get_server_info(ptr->con));


	ptr->timestamp = time(0);
	ptr->id = id;
	return ptr;

 err:
	if (ptr && ptr->con) pkg_free(ptr->con);
	if (ptr) pkg_free(ptr);
	return 0;
}


/*
 * Close the connection and release memory
 */
void free_connection(struct my_con* con)
{
	if (!con) return;
	if (con->id) free_db_id(con->id);
	if (con->con) {
		mysql_close(con->con);
		pkg_free(con->con);
	}
	pkg_free(con);
}
