# Authors: John Dennis <jdennis@redhat.com>
#
# Copyright (C) 2006,2007,2008 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

from setroubleshoot.signature import *
from setroubleshoot.util import *
from setroubleshoot.log import *

#import sys
#import os.path
import re

#------------------------------------------------------------------------------
class Plugin(object):
    """
    Each plugin object recognizes one or more access denials and
    presents a description of the denial to the user. Optionally,
    the plugin can provide a suggestion for allowing the access
    to the user.

    There are four user visible strings that are part of each Plugin
    subclass (some or all of these can be changed by the plugin author):
     * summary: summary of the denial
     * problem_description: detailed description of the denial
     * fix_description: description of how to allow the denied access
     * fix_cmd: command that can be used to allow the access

    All of the strings will have a standard set of substitutions performed.
    Each keyword (proceeded by a '$' will be replace by a string) - see
    http://docs.python.org/lib/node109.html for more information. The
    keywords are:
     * $SOURCE_TYPE - type for the source of the avc (usually the
       process performing the operation).
     * $TARGET_TYPE - type for the target of the avc (the type of
       the object).
     * $SOURCE_PATH - source of the executable (from the exe or comm
       field of the exe). A full path is not always available.
     * $TARGET_PATH - path for the target object. A full path is not
       always available.
     * $TARGET_DIR - path of the containing directory for TARGET_PATH.
       Essentially os.path.dirname($TARGET_PATH)
     * $TARGET_CLASS - the object class for the target.
     * $PERMS - the permissions denied.
     * $SOURCE_PACKAGE - name of the package which contains the
       executable (from $SOURCE_PATH).
     * $PORT_NUMBER - the port number for the connection denied.
    Additional subtitutions can be added with set_template_substitutions.

    You can also optional pass the name for a single boolean which will be
    used to set the $BOOLEAN subtitution into Plugin.__init__.

    You can also set the level, of the alert, if the plugin believes discovers 
    a signature of an attack, the level should be set to red
    * level:  Defines the level of the alert, two supported (red, yellow(default)
        """
    summary = ""
    problem_description = ""
    fix_description = ""
    fix_cmd = ""
        
    def __init__(self, name):
        self.analysis_id = re.sub(r'^plugins\.', '', name)
        self.priority = 50
        
    def report(self, avc, category, summary_template, problem_description_template,
               fix_description_template, fix_cmd_template, level="yellow"):
        """
        Report a denial and solution to the fault server.
        """
        
        environment = self.get_environment(avc.query_environment)
        avc.validate_template_substitutions()

        for k,v in avc.template_substitutions.items():
            if not isinstance(v, str):
                log_plugin.warn("template substitution (%s) is not a string, %s", k, v)

        from string import Template
        summary = Template(summary_template).\
                  safe_substitute(avc.template_substitutions)
        problem_description = Template(problem_description_template).\
                              safe_substitute(avc.template_substitutions)
        fix_description = Template(fix_description_template).\
                          safe_substitute(avc.template_substitutions)
        fix_cmd = Template(fix_cmd_template).\
                  safe_substitute(avc.template_substitutions)
        
        solution = SEFaultSolution(summary, problem_description, fix_description, fix_cmd)

        if avc.audit_event.line_numbers is not None:
            avc.audit_event.line_numbers.sort()
            
        siginfo = SEFaultSignatureInfo(
            analysis_id    = self.analysis_id,
            audit_event    = avc.audit_event,
            source         = avc.source,
            spath          = avc.spath,
            tpath          = avc.tpath,
            src_rpm_list   = avc.src_rpms,
            tgt_rpm_list   = avc.tgt_rpms,
            scontext       = avc.scontext,
            tcontext       = avc.tcontext,
            tclass         = avc.tclass,
            port           = avc.port,
            host           = avc.host,

            sig            = self.get_signature(avc, environment),
            solution       = solution,
            category       = category,
            environment    = environment,
            line_numbers   = avc.audit_event.line_numbers,
            last_seen_date = TimeStamp(avc.audit_event.timestamp),
            level          = level
            )

        return siginfo
        
    def get_environment(self, query_environment):
        environment = SEEnvironment()
        if query_environment:
            environment.update()
        return environment
    
    def get_signature(self, avc, environment):
        sig = SEFaultSignature(
            analysis_id = self.analysis_id,
            host        = avc.host,
            access      = avc.access,
            scontext    = avc.scontext,
            tcontext    = avc.tcontext,
            tclass      = avc.tclass,
            tpath       = avc.tpath)
        return sig

    def analyze(self, avc):
        return False

    def set_priority(self, priority):
        self.priority = priority
        
    def get_priority(self):
        return self.priority 
        
        
