/*
 * seapplet.c
 *
 * Authors: John Dennis <jdennis@redhat.com>
 * Authors: Dan Walsh <dwalsh@redhat.com>
 *
 * Copyright (C) 2009 Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *       compile command
 *       gcc -g sealerttrayicon.c -o sealerttrayicon `pkg-config --cflags --libs gtk+-2.0` -lnotify
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <libnotify/notify.h>

#include <dbus/dbus-glib-lowlevel.h>
#include "sedbus.h"

#ifdef USE_NLS
#include <locale.h>		/* for setlocale() */
#include <libintl.h>		/* for gettext() */
#define _(msgid) gettext (msgid)
#else
#define _(msgid) (msgid)
#endif
#ifndef PACKAGE
#define PACKAGE "setroubleshoot"	/* the name of this package lang translation */
#endif

typedef struct {
	GtkStatusIcon *trayIcon;
	NotifyNotification *notify;
        guint32 need_bubble : 1;
	gchar *redFile;
	gchar *yellowFile;
} sealert;

#define TIMEOUT (10 * 1000)
static gchar *icon_file = "/usr/share/setroubleshoot/gui/setroubleshoot_icon.png";
static gchar *redicon_file = "/usr/share/setroubleshoot/gui/setroubleshoot_red_icon.png";


static void show_notification_now(sealert *alert) {
        GError *err = NULL;
	notify_notification_attach_to_status_icon (alert->notify, alert->trayIcon);
		
	notify_notification_show (alert->notify, &err);
	if (err) {
		g_warning ("Error showing notification: %s", err->message);
		g_error_free (err);
	}
        alert->need_bubble = FALSE;
}

static void on_notify_embedded_changed (sealert *alert) {
        if (gtk_status_icon_is_embedded (alert->trayIcon) && alert->need_bubble) {
		show_notification_now (alert);
        }
}

static void show_notification(sealert *alert) {

	if (gtk_status_icon_is_embedded (alert->trayIcon)) {
		show_notification_now (alert);
	} else {
		g_signal_connect_swapped (alert->trayIcon, "notify::embedded",
				          G_CALLBACK (on_notify_embedded_changed),
                                          alert);
                alert->need_bubble = TRUE;
	}

}

static void trayIconActivated(GObject *notused, gpointer ptr)
{
    sealert *alert = (sealert*) ptr;
    gtk_status_icon_set_visible(alert->trayIcon, FALSE);
    alert->need_bubble = FALSE;
    notify_notification_close (alert->notify, NULL);
    if (system("/usr/bin/sealert -b") == -1 ) {
	    perror("/usr/bin/sealert failed");
    }
}


static void on_activate(NotifyNotification *notification, 
			const char *action, 
			sealert *alert) {
	if (strcmp(action, "dismiss") == 0)  {
		gtk_status_icon_set_visible(alert->trayIcon, FALSE);
		alert->need_bubble = FALSE;
		notify_notification_close (alert->notify, NULL);
	} else {
		trayIconActivated(NULL, alert);
	}
}

static void show_star(gpointer ptr, int red) {
	sealert *alert = (sealert *) ptr;
	gchar *file = NULL;
	if (red) {
		gtk_status_icon_set_from_file(alert->trayIcon, redicon_file);
		file = alert->redFile;

	} else {
		gtk_status_icon_set_from_file(alert->trayIcon, icon_file);
		file = alert->yellowFile;
	}
	if ((! gtk_status_icon_get_visible (alert->trayIcon) || red ) &&
	    alert->need_bubble == FALSE) {
		gtk_status_icon_set_visible(alert->trayIcon, TRUE);
		alert->notify = notify_notification_new_with_status_icon(_("New SELinux security alert"),
									 _("AVC denial, click icon to view"),
									 red ? file : GTK_STOCK_DIALOG_WARNING,
								       alert->trayIcon);
		if (!red) {
			notify_notification_set_timeout (alert->notify, NOTIFY_EXPIRES_DEFAULT);

			notify_notification_add_action(alert->notify, 
						       "dismiss", 
						       _("Dismiss"),
						       (NotifyActionCallback) on_activate,
						       alert,
						       NULL);
		} else {
			notify_notification_set_timeout (alert->notify, NOTIFY_EXPIRES_NEVER);
		}

		notify_notification_add_action(alert->notify, 
					       "show", 
					       _("Show"),
					       (NotifyActionCallback) on_activate,
					       alert,
					       NULL);

		show_notification (alert);
	}
}

int main(int argc, char *argv[])
{
	sealert alert;

	gtk_init (&argc, &argv);
	
	notify_init ("Sealert notification");

	GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (window, 200, -1);
	
	alert.yellowFile = g_filename_to_uri (icon_file, NULL, NULL);
	alert.redFile = g_filename_to_uri (redicon_file, NULL, NULL);
	alert.trayIcon  = gtk_status_icon_new_from_file (icon_file);

	//set tooltip
	gtk_status_icon_set_tooltip (alert.trayIcon, "SELinux AVC denial, click to view");
	g_signal_connect(alert.trayIcon, "activate", GTK_SIGNAL_FUNC (trayIconActivated), &alert);
	gtk_status_icon_set_visible(alert.trayIcon, FALSE); //set icon initially invisible
        alert.need_bubble = FALSE;
	
	DBusConnection *conn = sedbus_receive(show_star, (void *) &alert);
	dbus_connection_setup_with_g_main(conn, NULL);
	gtk_main ();

	return 0;
}
