/* $Id: eth.c,v 1.5 2003/05/30 00:42:40 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "libc.h"
#include "eth.h"
#include "misc.h"
#include "packet.h"
#include "global.h"

/*
 * Ethernet routines
 */


/*
 * Data structures
 */

typedef struct {
	uint8_t    dest_addr[6];
	uint8_t    src_addr[6];
	uint8_t    type[2];
} mac_header_t;


typedef struct packet_handler_node_struct {
	int (*handler)(mbuf_t *mbuf, void *data, uint16_t frame_type);
	void *data;
	uint16_t frame_type;
	struct packet_handler_node_struct *next;
} packet_handler_node_t;


/* 
 * Static data 
 */

static packet_handler_node_t *handler_chain = 0;
static uint64_t timeout;
static uint64_t start_time;


/* 
 * Exported interface.  Semantics of these functions are described in the header
 */

void eth_send_packet(mbuf_t *mbuf, uint64_t dest_mac, uint64_t src_mac, uint16_t type)
{
	mac_header_t header;
	set_uint48_field(header.dest_addr, dest_mac);
	set_uint48_field(header.src_addr, src_mac);
	set_uint16_field(header.type, type);
	if (mbuf_prepend(mbuf, &header, sizeof(header)) != MBUF_SUCCESS) {
		lib_die("Out of memory");
	}
	packet_send(mbuf->front_ptr, mbuf->size + 4);
}



/* This can be called from handlers to keep things sane */
void eth_reset_timeout(uint64_t new_timeout)
{
	start_time = lib_get_time();
	timeout = new_timeout;
}

/* 
 * Loop through, dispatching packets to handlers, <timeout> microseconds
 * have passed, or flag becomes true 
 */
void eth_recv_loop(unsigned int timeout, unsigned int *flag)
{
	int length;
	mbuf_t *mbuf;
	
	mbuf = lib_malloc(sizeof(mbuf_t));
	
	if (!mbuf) {
		lib_die("Out of memory");
	}

	eth_reset_timeout(timeout);
	
	while (!(*flag) && ((lib_get_time() - start_time) < timeout)) {
		if ((length = packet_read(mbuf->buf, MBUF_BUFSIZE)) != 0) {
			uint64_t recv_mac_addr;
			mbuf_init(mbuf, length);
			mac_header_t *header = (mac_header_t *)mbuf->buf;
			recv_mac_addr = get_uint48_field(header->dest_addr);
			if ((recv_mac_addr == net_cfg.mac_addr) || (recv_mac_addr == ETH_ADDR_BROADCAST)) {
				packet_handler_node_t *tmp = handler_chain;
				uint16_t frame_type = get_uint16_field(header->type);
				mbuf_trim_head(mbuf, sizeof(mac_header_t));
				while (tmp) {
					if ((tmp->frame_type == frame_type) 
					    && (*(tmp->handler))(mbuf, tmp->data, frame_type)) {
						break;
					}
					tmp = tmp->next;
				}
			} 
		}  else {
		}
	}
	lib_free(mbuf);
}



void eth_add_handler(int (*handler)(mbuf_t *mbuf, void *data, uint16_t frame_type), 
		     void *data, uint16_t frame_type)
{
	packet_handler_node_t *tmp = lib_malloc(sizeof(packet_handler_node_t));
	if (!tmp) {
		lib_die("Out of memory");
	}
	tmp->next = handler_chain;
	handler_chain = tmp;
	tmp->handler = handler;
	tmp->frame_type = frame_type;
	tmp->data = data;
}

void eth_del_handler(int (*handler)(mbuf_t *mbuf, void *data, uint16_t frame_type), 
		     void *data, uint16_t frame_type)
{
	packet_handler_node_t *prev = 0;
	packet_handler_node_t *tmp;
	
	for (tmp = handler_chain; tmp != 0; tmp = tmp->next) {
		if ((tmp->handler == handler) 
		    && (tmp->data == data)
		    && (tmp->frame_type == frame_type)) {
			if (prev) {
				prev->next = tmp->next;
			} else {
				handler_chain = tmp->next;
			}
			lib_free(tmp);
			return;
		}
		prev = tmp;
	}
	lib_die("eth_del_handler didn't find a matching handler\n");
}
