/////////////////////////////////////////////////////////////////////////////
// File:        basetype.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_BASETYPE_H
#define SPCORE_BASETYPE_H

#include "spcore/baseobj.h"
#include "spcore/iterator.h"
#include "spcore/coreruntime.h"

namespace spcore {

class IOutputPin;

enum EBasicID {
  TYPE_INVALID = -1,
  TYPE_ANY = 0
};

//Base type for all instances that traverses the graph.
//It supports reference counting
class CTypeAny : public IBaseObject {
public:
	virtual int GetTypeID() const { return m_typeID; }
	
	// Composition methods
	// Return 0 when OK
	virtual int AddChild(SmartPtr<CTypeAny> component) {
		return -1;
	}

    virtual SmartPtr<IIterator<CTypeAny*> > QueryChildren() const {
		return SmartPtr<IIterator<CTypeAny*> >(NULL);
	}

	static const char* getTypeName() { return "any"; }

	/**
		Clone one instance. 

		\param dst: if not NULL tries to perform the copy to the given
			instance if it is of the same type. If types mismatch
			new copy will be created on dynamic memory

		\param recurse: is passed untouched to CopyTo. if it is true
			deep copies are performed on composite types, if false only
			copies the instance.

		\return an smart pointer to the new instance which can be the same 
			to dst, a new one or NULL if the instance doesn't support copies
	*/
	virtual SmartPtr<CTypeAny> Clone (CTypeAny * dst, bool recurse) const {
		if (this== dst) return SmartPtr<CTypeAny>(dst);

		if (dst && this->GetTypeID()== dst->GetTypeID()) {
			if (CopyTo (*dst, recurse)) 
				// Copy performed
				return SmartPtr<CTypeAny>(dst);
			else 
				// Same type but copy cannot be performed
				return SmartPtr<CTypeAny>();
		}
		else {
			// Diferent types or dst NULL
			ICoreRuntime* cr= getSpCoreRuntime(); assert (cr);
			SmartPtr<CTypeAny> newInstance= cr->CreateTypeInstance(m_typeID);
			if (newInstance.get()== NULL) {
				// Something went really wrong
				assert (false);
				return newInstance;
			}		
						
			if (CopyTo(*newInstance, recurse)) {
				// Old instance is not needed anymore
				//if (dst) dst->Release();
				return newInstance;
			}
			else
				// Copy failed
				return SmartPtr<CTypeAny>();
		}
	}

	static SmartPtr<IOutputPin> CreateOutputPinAny(const char* name) {
		return getSpCoreRuntime()->CreateOutputPin(CTypeAny::getTypeName(), name, false);
	}
	static SmartPtr<IOutputPin> CreateOutputPinLockAny(const char* name) {
		return getSpCoreRuntime()->CreateOutputPin(CTypeAny::getTypeName(), name, true);
	}

  protected:
	CTypeAny(int id) { assert (id>= 0); m_typeID= id; }
	virtual ~CTypeAny() {}

	/**
		Copies this into dst

		\param dst copy destination, is allways an already created instance
		\param recurse, intended for allowing to perform deep copies of
			composites when true
	*/
	virtual bool CopyTo ( CTypeAny& , bool ) const {
		// Calls are expected to be performed on derived classes
		assert (false);
		return false;
	}

	// Assignement operator. Available to derived classe to allow to use 
	// the compiler generated copy constructor. Note that copying between 
	// instances with different m_typeID it is not allowed and derived
	// classed must ensure this. Use with care.
	CTypeAny& operator=( const CTypeAny& that )	{
		assert (this->m_typeID== that.m_typeID);
		return *this;
	}

  private:
	// disable copy constructor
	CTypeAny ( const CTypeAny& );      
	

    int m_typeID;
};


class ITypeFactory : public IBaseObject {
  protected:
	virtual ~ITypeFactory() {}


  public:
    virtual const char* GetName() const = 0;

    virtual SmartPtr<CTypeAny> CreateInstance(int id) = 0;

};



//
// Template functions for downcasting (smart pointed) CTypeAny instances
//
template <typename T>
T* sptype_static_cast (CTypeAny* v){
	return static_cast<T*>(v);
}

template <typename T>
T const * sptype_static_cast (CTypeAny const * v){
	return static_cast<T const *>(v);
}

template <typename T>
T* sptype_dynamic_cast (CTypeAny* v){
	if (T::getTypeID()!= v->GetTypeID()) return NULL;
	return static_cast<T*>(v);
}

template <typename T>
T const * sptype_dynamic_cast (CTypeAny const * v){
	if (T::getTypeID()!= v->GetTypeID()) return NULL;
	return static_cast<T const *>(v);
}

template <typename T>
SmartPtr<T> sptype_static_cast (const SmartPtr<const CTypeAny> &v){
	return smartptr_static_cast<T,const CTypeAny>(v);
}

template <typename T>
SmartPtr<T> sptype_static_cast (const SmartPtr<CTypeAny> &v){
	return smartptr_static_cast<T,CTypeAny>(v);
}

template <typename T>
SmartPtr<T> sptype_dynamic_cast (const SmartPtr<const CTypeAny> &v){
	if (T::getTypeID()!= v->GetTypeID()) return SmartPtr<T>(NULL);
	return smartptr_static_cast<T,const CTypeAny>(v);
}

template <typename T>
SmartPtr<T> sptype_dynamic_cast (const SmartPtr<CTypeAny> &v){
	if (T::getTypeID()!= v->GetTypeID()) return SmartPtr<T>(NULL);
	return smartptr_static_cast<T,CTypeAny>(v);
}


} // namespace spcore
#endif
