/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to calculate.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgcalculatorpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgobjectmodel.h"
#include "skginterestobject.h"
#include "skgaccountobject.h"
#include "skgtransactionmng.h"
#include "skgunitobject.h"

#include <kstandarddirs.h>
#include <klocale.h>

#include <QDomDocument>
#include <QDir>
#include <QKeyEvent>

#include <math.h>

SKGCalculatorPluginWidget::SKGCalculatorPluginWidget ( SKGDocumentBank* iDocument )
        :SKGTabPage ( iDocument )
{
    SKGTRACEIN ( 1, "SKGCalculatorPluginWidget::SKGCalculatorPluginWidget" );

    timer = new QTimer ( this );
    timer->setSingleShot ( true );
    connect ( timer, SIGNAL ( timeout() ), this, SLOT ( onAmortizationComputation() ) );

    ui.setupUi ( this );
    ui.kUnitEdit->setDocument(iDocument);
    ui.kUnitEdit->setWhereClauseCondition("t_type IN ('1','2','C')");

    ui.kUpdate->setIcon ( KIcon ( "dialog-ok-apply" ) );
    ui.kAdd->setIcon ( KIcon ( "list-add" ) );

    ui.kLoadSummaryTitle->setPixmap(KIcon("configure").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
    ui.kPaymentSummaryTitle->setPixmap(KIcon("rating").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
    ui.kAmortizationTableTitle->setPixmap(KIcon("view-pim-calendar").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

    ui.kInterestResultsTitle->setPixmap(KIcon("rating").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
    ui.kInterestValuesTitle->setPixmap(KIcon("configure").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
    ui.kDefineValueTitle->setPixmap(KIcon("configure").pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

    ui.kAmortizationBtn->setIcon(KIcon("accessories-calculator"));
    ui.kInterestBtn->setIcon(KIcon("skrooge_more"));

    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Fifteen" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +0" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +1" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +2" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +3" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +4" ) );
    ui.kCreditValueDate->addItem ( i18nc ( "Noun",  "Day +5" ) );

    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Fifteen" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -0" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -1" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -2" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -3" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -4" ) );
    ui.kDebitValueDate->addItem ( i18nc ( "Noun",  "Day -5" ) );

    ui.kMode->addItem ( i18nc ( "Noun",  "24 fifteen" ) );
    ui.kMode->addItem ( i18nc ( "Noun",  "360 days" ) );
    ui.kMode->addItem ( i18nc ( "Noun",  "365 days" ) );

    ui.kAmortizationTable->verticalHeader()->setDefaultSectionSize ( ui.kAmortizationTable->verticalHeader()->minimumSectionSize() );

    //Bind account creation view
    objectModel = new SKGObjectModel ( static_cast<SKGDocumentBank*>(getDocument()), "v_interest", "", this, "", false );
    ui.kInterestView->setModel ( objectModel );

    SKGObjectModel* resultModel = new SKGObjectModel ( static_cast<SKGDocumentBank*>(getDocument()), "interest_result", "", this, "", false );
    ui.kInterestResultTable->setModel ( resultModel );

    connect ( ui.kInterestView, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );
    connect ( objectModel, SIGNAL ( beforeReset() ), ui.kInterestView, SLOT ( saveSelection() ) );
    connect ( objectModel, SIGNAL ( afterReset() ), ui.kInterestView, SLOT ( resetSelection() ) );

    //Build you panel here
    setCurrentMode ( -1 );

    //Refresh
    connect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );
    connect ( ui.KYearEdit, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ) , Qt::QueuedConnection );

    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
    dataModified ( "", 0 );

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter ( this );

    //Synchronize zooms of both tables
    connect ( ui.kInterestResultTable, SIGNAL ( zoomChanged(int)), ui.kInterestView, SLOT ( setZoomPosition(int)) );
    connect ( ui.kInterestView, SIGNAL ( zoomChanged(int)), ui.kInterestResultTable, SLOT ( setZoomPosition(int)) );
}

SKGCalculatorPluginWidget::~SKGCalculatorPluginWidget()
{
    SKGTRACEIN ( 1, "SKGCalculatorPluginWidget::~SKGCalculatorPluginWidget" );
    objectModel=NULL;
    timer=NULL;
}

bool SKGCalculatorPluginWidget::eventFilter ( QObject *object, QEvent *event )
{
    if ( event && event->type() == QEvent::KeyPress )
    {
        QKeyEvent *keyEvent = ( QKeyEvent * ) event;
        if ( ( keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter ) && object==this )
        {
            if ( QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAdd->isEnabled() )
            {
                ui.kAdd->click();
            }
            else if ( QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kUpdate->isEnabled() )
            {
                ui.kUpdate->click();
            }
        }
    }

    return false;
}

int SKGCalculatorPluginWidget::getCurrentMode()
{
    return mode;
}

void SKGCalculatorPluginWidget::onBtnModeClicked()
{
    QWidget* sender=static_cast<QWidget*> ( this->sender() );

    int currentMode=getCurrentMode();
    int newMode=0;

    if ( sender==ui.kInterestBtn ) newMode=0;
    else if ( sender==ui.kAmortizationBtn ) newMode=1;

    if ( currentMode==newMode ) newMode=-1;
    setCurrentMode ( newMode );
}

void SKGCalculatorPluginWidget::setCurrentMode ( int iMode )
{
    //Save zoom position of previous page
    int zoom=zoomPosition();

    mode=iMode;
    if ( mode==-1 ) mode=0;

    //Save zoom position on new page
    setZoomPosition(zoom);

    ui.kInterestFrm->hide();
    ui.kAmortizationFrm->hide();
    ui.kBtnFrm->hide();

    ui.kInterestBtn->setChecked ( false );
    ui.kAmortizationBtn->setChecked ( false );

    if ( mode==0 )
    {
        ui.kInterestFrm->show();
        ui.kBtnFrm->show();
        ui.kInterestBtn->setChecked ( true );
    }
    else if ( mode==1 )
    {
        ui.kAmortizationFrm->show();
        ui.kAmortizationBtn->setChecked ( true );
    }

    Q_EMIT selectionChanged();
}

QString SKGCalculatorPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGCalculatorPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "currentPage", SKGServices::intToString ( getCurrentMode() ) );
    root.setAttribute ( "account", ui.kDisplayAccountCombo->currentText() );
    root.setAttribute ( "year", ui.KYearEdit->text() );

    root.setAttribute ( "amortizationLoan", ui.kLoanEdit->text() );
    root.setAttribute ( "amortizationUnit", ui.kUnitEdit->text() );
    root.setAttribute ( "amortizationRate", SKGServices::doubleToString(ui.kAnnualRateEdit->value()) );
    root.setAttribute ( "amortizationLenght", SKGServices::intToString(ui.kLenghtEdit->value()) );
    root.setAttribute ( "amortizationInsuranceRate", SKGServices::doubleToString(ui.kInsuranceRateEdit->value()) );


    //Memorize table settings
    root.setAttribute ( "view", ui.kInterestView->getState() );
    root.setAttribute ( "viewResult", ui.kInterestResultTable->getState() );

    return doc.toString();
}

void SKGCalculatorPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGCalculatorPluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString account=root.attribute ( "account" );
    QString currentPage=root.attribute ( "currentPage" );
    QString year=root.attribute ( "year" );
    QString amortizationLoan=root.attribute ( "amortizationLoan" );
    QString amortizationUnit=root.attribute ( "amortizationUnit" );
    QString amortizationRate=root.attribute ( "amortizationRate" );
    QString amortizationInsuranceRate=root.attribute ( "amortizationInsuranceRate" );
    QString amortizationLenght=root.attribute ( "amortizationLenght" );

    if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );
    if ( !account.isEmpty() )
    {
        bool previous=ui.kDisplayAccountCombo->blockSignals ( true );
        ui.kDisplayAccountCombo->setText ( account );
        ui.kDisplayAccountCombo->blockSignals ( previous );
    }
    if ( !amortizationLoan.isEmpty() ) ui.kLoanEdit->setText(amortizationLoan);
    if ( !amortizationUnit.isEmpty() ) ui.kUnitEdit->setText(amortizationUnit);
    if ( !amortizationRate.isEmpty() ) ui.kAnnualRateEdit->setValue(SKGServices::stringToDouble(amortizationRate));
    if ( !amortizationInsuranceRate.isEmpty() ) ui.kInsuranceRateEdit->setValue(SKGServices::stringToDouble(amortizationInsuranceRate));
    if ( !amortizationLenght.isEmpty() ) ui.kLenghtEdit->setValue(SKGServices::stringToInt(amortizationLenght));

    if ( !year.isEmpty() ) ui.KYearEdit->setText ( year );

    //Update model
    if ( objectModel )
    {
        bool previous=objectModel->blockRefresh ( true );
        onFilterChanged();
        objectModel->blockRefresh ( previous );
    }

    //!!! Must be done here after onFilterChanged
    ui.kInterestView->setState ( root.attribute ( "view" ) );
    ui.kInterestResultTable->setState ( root.attribute ( "viewResult" ) );
}

QString SKGCalculatorPluginWidget::getDefaultStateAttribute()
{
    return "SKGCALCULATOR_DEFAULT_PARAMETERS";
}

void SKGCalculatorPluginWidget::onSelectionChanged()
{
    SKGTRACEIN ( 10, "SKGCalculatorPluginWidget::onSelectionChanged" );
    //Mapping
    QItemSelectionModel *selModel=ui.kInterestView->selectionModel();
    if ( selModel )
    {
        QModelIndexList indexes=selModel->selectedRows();
        if ( indexes.count() && objectModel )
        {
            QModelIndex idx=indexes[indexes.count()-1];

            SKGInterestObject interest ( objectModel->getObject ( idx ) );

            ui.kDateEdit->setDate ( interest.getDate() );
            ui.kRateEdit->setValue ( interest.getRate() );
            ui.kCreditValueDate->setCurrentIndex ( ( int ) interest.getIncomeValueDateMode() );
            ui.kDebitValueDate->setCurrentIndex ( ( int ) interest.getExpenditueValueDateMode() );
            ui.kMode->setCurrentIndex ( ( int ) interest.getInterestComputationMode() );
        }

        Q_EMIT selectionChanged();
    }
}

void SKGCalculatorPluginWidget::dataModified ( const QString& iTableName, int iIdTransaction )

{
    SKGTRACEIN ( 10, "SKGCalculatorPluginWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Refresh widgets
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled ( db!=NULL );
    if ( db!=NULL && ( iTableName=="interest" || iTableName.isEmpty() ) )
    {
        //Correction bug 2299394 vvv
        if ( ui.kInterestView->isAutoResized() ) ui.kInterestView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^

        //Correction 215658: vvv because the table is modified, the selection is modified
        onSelectionChanged();
        //Correction 215658: ^^^
    }

    //Refresh interest parameter view
    QString current=ui.kDisplayAccountCombo->currentText();
    if ( db!=NULL && ( iTableName=="account" || iTableName.isEmpty() ) )
    {
        //Clear
        ui.kDisplayAccountCombo->clear();

        SKGStringListList listAccount;
        SKGServices::executeSelectSqliteOrder ( getDocument(), QString ( "SELECT t_ICON, t_name from v_account_display where t_close='N' order by t_name" ), listAccount );

        int nbAccounts=listAccount.count();
        if ( nbAccounts>=0 )
        {
            QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );
            for ( int i=1; i<nbAccounts; ++i ) //Ignore header
            {
                QIcon icon ( dirLogo.absoluteFilePath ( listAccount.at ( i ).at ( 0 ) ) );
                QString text=listAccount.at ( i ).at ( 1 );
                ui.kDisplayAccountCombo->addItem ( icon, text );
            }
        }

        int pos=ui.kDisplayAccountCombo->findText ( current );
        if ( pos==-1 ) pos=0;
        ui.kDisplayAccountCombo->setCurrentIndex ( pos );

        ui.kAdd->setEnabled ( ui.kDisplayAccountCombo->count() >0 );
    }

    //Fill years
    if ( db!=NULL && ( iTableName=="operation" || iTableName.isEmpty() ) )
    {
        disconnect ( ui.KYearEdit, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ) );

        int lastYear=QDate::currentDate().year();
        int firstYear=lastYear;

        QStringList list;
        SKGServices::getDistinctValues ( getDocument(), "v_operation_display", "d_DATEYEAR", "", list );
        if ( list.count() ) firstYear=SKGServices::stringToInt ( list.at ( 0 ) );

        QString y=ui.KYearEdit->text();
        ui.KYearEdit->clear();
        for ( int i=lastYear; i>=firstYear; --i )
        {
            QString v=SKGServices::intToString ( i );
            if ( y.isEmpty() ) y=v;
            ui.KYearEdit->addItem ( v );
        }

        ui.KYearEdit->setText ( y );
        connect ( ui.KYearEdit, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ) );
    }
    //Refresh computation
    if ( db!=NULL && ( iTableName=="interest" ||iTableName=="account" || iTableName.isEmpty() ) )
    {
        computeInterest();
    }
    QApplication::restoreOverrideCursor();
}

void SKGCalculatorPluginWidget::computeInterest()
{
    //Compute
    SKGAccountObject account ( getDocument() );
    SKGError err=account.setName ( ui.kDisplayAccountCombo->currentText() );
    if ( err.isSucceeded() ) err=account.load();
    QList<SKGAccountObject::SKGInterestItem> oInterestList;
    double oInterests=0;
    if ( err.isSucceeded() ) err=account.getInterestItems ( oInterestList, oInterests, SKGServices::stringToInt ( ui.KYearEdit->text() ) );
    if ( err.isSucceeded() )
    {
        //Refresh table
        ui.kInterestResultTable->setState ( ui.kInterestResultTable->getState() );

        //Set text
        SKGDocumentBank* doc= static_cast<SKGDocumentBank*>(getDocument());
        if ( doc )
        {
            SKGServices::SKGUnitInfo unit1=doc->getPrimaryUnit();
            SKGServices::SKGUnitInfo unit2=doc->getSecondaryUnit();

            QString s1=doc->formatMoney ( oInterests, unit1 );
            ui.kInterestLbl->setText ( i18nc ( "Noun",  "Annual interest=%1", s1 ) );
            if ( !unit2.Symbol.isEmpty() && unit2.Value )
            {
                s1=doc->formatMoney ( oInterests, unit2 );
                ui.kInterestLbl->setToolTip ( i18nc ( "Noun",  "Annual interest=%1", s1 ) );
            }
        }
    }
}

void SKGCalculatorPluginWidget::onFilterChanged()
{
    SKGTRACEIN ( 1, "SKGCalculatorPluginWidget::onFilterChanged" );
    if ( !isEnabled() ) return;
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Compute where clause
    QString account=ui.kDisplayAccountCombo->currentText();
    QString filter2="t_ACCOUNT='"+SKGServices::stringToSqlString ( account ) +'\'';

    //Update model
    if ( objectModel )
    {
        objectModel->setFilter ( filter2 );
        ui.kInterestView->setState ( ui.kInterestView->getState() );

        //Compute result
        computeInterest();
    }
    QApplication::restoreOverrideCursor();
}

SKGObjectBase::SKGListSKGObjectBase SKGCalculatorPluginWidget::getSelectedObjects()
{
    SKGObjectBase::SKGListSKGObjectBase output;
    if (getCurrentMode()==0) output=ui.kInterestView->getSelectedObjects();
    return output;
}

int SKGCalculatorPluginWidget::getNbSelectedObjects()
{
    int output=0;
    if (getCurrentMode()==0) output=ui.kInterestView->getNbSelectedObjects();
    return output;
}

// void SKGCalculatorPluginWidget::onCalculatorReturnPressed(const QString& iText)
// {
// //         ui.kCalculatorList->addItem(iText);
// }

// /*void SKGCalculatorPluginWidget::onCalculatorListClicked(const QString& iText)
// {
// //         ui.kCalculatorEdit->setText(iText);
// //         ui.kCalculatorEdit->setFocus();
// }*/

void SKGCalculatorPluginWidget::onSelectedInterestChanged()
{
    int nbSel=getNbSelectedObjects();
    ui.kDateEdit->setEnabled ( nbSel<=1 );
    ui.kRateEdit->setEnabled ( nbSel<=1 );
    ui.kAdd->setEnabled ( nbSel<=1 );
}

void SKGCalculatorPluginWidget::onAdd()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGCalculatorPluginWidget::onAdd",err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    {
        SKGInterestObject interestObj;
        {
            QString accountname=ui.kDisplayAccountCombo->currentText();
            SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Interest parameter creation for account '%1'", accountname ), err );

            //Get Parent account
            SKGAccountObject accountObj ( getDocument() );
            if ( err.isSucceeded() ) err=accountObj.setName ( accountname );
            if ( err.isSucceeded() ) err=accountObj.load();

            //Create interest object
            if ( err.isSucceeded() ) err=accountObj.addInterest ( interestObj );
            if ( err.isSucceeded() ) err=interestObj.setDate ( ui.kDateEdit->date() );
            if ( err.isSucceeded() ) err=interestObj.setRate ( ui.kRateEdit->value() );
            if ( err.isSucceeded() ) err=interestObj.setIncomeValueDateMode ( ( SKGInterestObject::ValueDateMode ) ui.kCreditValueDate->currentIndex() );
            if ( err.isSucceeded() ) err=interestObj.setExpenditueValueDateMode ( ( SKGInterestObject::ValueDateMode ) ui.kDebitValueDate->currentIndex() );
            if ( err.isSucceeded() ) err=interestObj.setInterestComputationMode ( ( SKGInterestObject::InterestMode ) ui.kMode->currentIndex() );
            if ( err.isSucceeded() ) err=interestObj.save();
        }

        //status bar
        if ( err.isSucceeded() )
        {
            err=SKGError ( 0, i18nc ( "Successful message after an user action", "Interest parameter created" ) );
            ui.kInterestView->selectObject ( interestObj.getUniqueID() );
        }
        else err.addError ( ERR_FAIL, i18nc ( "Error message",  "Interest parameter creation failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGCalculatorPluginWidget::onUpdate()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGCalculatorPluginWidget::onUpdate",err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Get Parent account
    QString accountname=ui.kDisplayAccountCombo->currentText();
    SKGAccountObject accountObj ( getDocument() );
    if ( err.isSucceeded() ) err=accountObj.setName ( accountname );
    if ( err.isSucceeded() ) err=accountObj.load();

    SKGObjectBase::SKGListSKGObjectBase updatedInterest=getSelectedObjects();
    int nb=updatedInterest.count();
    {
        {
            SKGInterestObject interestObj;
            SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Interest parameter update for account '%1'", accountname ), err );

            for ( int i=0; err.isSucceeded() && i<nb; ++i )
            {
                interestObj=updatedInterest.at ( i );
                if ( nb==1 )
                {
                    if ( err.isSucceeded() ) err=interestObj.setDate ( ui.kDateEdit->date() );
                    if ( err.isSucceeded() ) err=interestObj.setRate ( ui.kRateEdit->value() );
                }
                if ( err.isSucceeded() ) err=interestObj.setIncomeValueDateMode ( ( SKGInterestObject::ValueDateMode ) ui.kCreditValueDate->currentIndex() );
                if ( err.isSucceeded() ) err=interestObj.setExpenditueValueDateMode ( ( SKGInterestObject::ValueDateMode ) ui.kDebitValueDate->currentIndex() );
                if ( err.isSucceeded() ) err=interestObj.setInterestComputationMode ( ( SKGInterestObject::InterestMode ) ui.kMode->currentIndex() );
                if ( err.isSucceeded() ) err=interestObj.save();
            }
        }

        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18nc ( "Successful message after an user action", "Interest parameter updated" ) );
        else err.addError ( ERR_FAIL, i18nc ( "Error message",  "Interest parameter update failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGCalculatorPluginWidget::onAmortizationComputationDelayed()
{
    if ( timer ) timer->start ( 300 );
}

void SKGCalculatorPluginWidget::onAmortizationComputation()
{
    SKGTRACEIN ( 10, "SKGCalculatorPluginWidget::onAmortizationComputation" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    KLocale* locale=KGlobal::locale();
    if ( locale )
    {
        //Get unit
        SKGServices::SKGUnitInfo unitInfo;
        unitInfo.Symbol=ui.kUnitEdit->text();

        SKGUnitObject unit ( getDocument() );
        unit.setSymbol ( unitInfo.Symbol );
        unit.load();
        unitInfo.NbDecimal=unit.getNumberDecimal();

        int p=pow ( 10,unitInfo.NbDecimal );

        //Computation
        int periodInMonths=12;
        int numberPayments=ui.kLenghtEdit->value() *periodInMonths;
        double periodicRate=ui.kAnnualRateEdit->value() /100/periodInMonths;
        double insurance=round ( ui.kLoanEdit->value()*ui.kInsuranceRateEdit->value() /100/periodInMonths*p ) /p;
        double periodicAmount=round ( ui.kLoanEdit->value() *periodicRate/ ( 1-pow ( 1+periodicRate,-numberPayments ) ) *p ) /p;

        //Fill table
        double dueAmount=ui.kLoanEdit->value();
        double sumInterest=0;
        ui.kAmortizationTable->setRowCount ( numberPayments );
        for ( int i=0; i<numberPayments; ++i )
        {
            //Compute
            double roundedInterest=round ( dueAmount*periodicRate*p ) /p;
            double principal=round ( ( i==numberPayments-1 ? dueAmount : periodicAmount-roundedInterest ) *p ) /p;
            double amount=roundedInterest+principal;
            dueAmount-=principal;
            sumInterest+=roundedInterest;

            //Add items
            ui.kAmortizationTable->setItem ( i, 0, new QTableWidgetItem ( locale->formatMoney ( amount, unitInfo.Symbol, unitInfo.NbDecimal ) ) );
            ui.kAmortizationTable->setItem ( i, 1, new QTableWidgetItem ( locale->formatMoney ( principal, unitInfo.Symbol, unitInfo.NbDecimal ) ) );
            ui.kAmortizationTable->setItem ( i, 2, new QTableWidgetItem ( locale->formatMoney ( roundedInterest, unitInfo.Symbol, unitInfo.NbDecimal ) ) );
            ui.kAmortizationTable->setItem ( i, 3, new QTableWidgetItem ( locale->formatMoney ( insurance, unitInfo.Symbol, unitInfo.NbDecimal ) ) );
            ui.kAmortizationTable->setItem ( i, 4, new QTableWidgetItem ( locale->formatMoney ( dueAmount, unitInfo.Symbol, unitInfo.NbDecimal ) ) );
        }

        ui.kAmortizationResult->setText ( i18n ( "Number of payments:	%1\n"
                                          "Monthly payment: 	%2\n"
                                          "Monthly insurance: 	%3\n"
                                          "Total principal paid: 	%4\n"
                                          "Total interest paid: 	%5\n"
                                          "Total insurance paid: 	%6\n"
                                          "Total paid: 		%7" ,
                                          SKGServices::intToString(numberPayments),
                                          locale->formatMoney ( periodicAmount, unitInfo.Symbol, unitInfo.NbDecimal ),
                                          locale->formatMoney ( insurance, unitInfo.Symbol, unitInfo.NbDecimal ),
                                          locale->formatMoney ( ui.kLoanEdit->value(), unitInfo.Symbol, unitInfo.NbDecimal ),
                                          locale->formatMoney ( sumInterest, unitInfo.Symbol, unitInfo.NbDecimal ),
                                          locale->formatMoney ( numberPayments*insurance, unitInfo.Symbol, unitInfo.NbDecimal ),
                                          locale->formatMoney ( ui.kLoanEdit->value()+sumInterest+numberPayments*insurance, unitInfo.Symbol, unitInfo.NbDecimal )
                                               ));
    }

    QApplication::restoreOverrideCursor();
}

QWidget* SKGCalculatorPluginWidget::zoomableWidget() const
{
    if (mode==0) return ui.kInterestView;
    else return ui.kAmortizationTable;
}

#include "skgcalculatorpluginwidget.moc"


