/*
 * smil2wav.cc -- Extracts a Wav Stream from a kino smil project
 * Copyright (C) 2002 Raphael Amaury Jacquot <sxpert@esitcom.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <iostream>
#include <Diagnostics.h>
#include <extensions.h>
#include <PlayListDVProvider.h>
#include <Threader.h>

class SMIL2Wav : public Threader
{
	private:
		ExtendedPlayList playlist;
		int offset;
		int frames;
		string audio_file;

	public: 
		SMIL2Wav( ) : offset( 0 ), frames( -1 ), audio_file( "-" )
		{
		}

		virtual ~SMIL2Wav( )
		{
		}

		virtual string LogId( )
		{
			return "SMIL2Wav";
		}

		void SetOffset( int _offset )
		{
			offset = _offset;
		}

		void SetFrames( int _frames )
		{
			frames = _frames - 1;
		}

		void SetAudioFile( string _audio_file )
		{
			audio_file = _audio_file;
		}

		bool Append( string file_name )
		{
			return playlist.Append( file_name );
		}

		bool IsRunnable( )
		{
			return playlist.GetNumFrames( ) != 0;
		}

	protected:
		void Thread( )
		{
			bool first = true;
			bool running = true;

			PlayListDVProvider input( playlist );
			input.SetPumpSize( 50 );

			PlayList copy;
			int end = frames == -1 ? playlist.GetNumFrames( ) - offset - 1 : offset + frames;
			playlist.GetPlayList( offset, end, copy );
			input.SetPlayList( copy );

			input.ThreadStart( );

			AudioExtractor *audio = NULL;

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );

					if ( first )
					{
						audio = AudioExtractor::GetExtractor( audio_file );
						audio->Initialise( frame );
						first = false;
					}
					audio->Output( frame );
					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}

			}

			audio->Flush( );

			delete audio;

			input.ThreadStop( );
		}
};

void Usage( )
{
	std::cerr << "Usage: smil2wav [ -a audio-file ] file [ file ... ]" << std::endl;
	std::cerr << "Where: file is smil, dv avi (1 or 2) or raw dv" << std::endl;
	std::cerr << "       -o offset - frame offset (default: 0)" << std::endl;
	std::cerr << "       -f count  - frame count (default: all)" << std::endl;
	std::cerr << "       -a file   - wav or mp2 to write to (more formats soon)" << std::endl;
	std::cerr << "                   (default is a non-fixated wav on stdout)" << std::endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	bool result = true;
	int index = 1;
	SMIL2Wav output;

	Diagnostics::SetApp( "smil2wav" );

	try
	{
		for ( index = 1; result && index < argc; index ++ )
		{
			if ( !strcmp( argv[ index ], "--help" ) )
				Usage( );
			else if ( !strcmp( argv[ index ], "-o" ) )
				output.SetOffset( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-f" ) )
				output.SetFrames( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-a" ) )
				output.SetAudioFile( argv[ ++ index ] );
			else if ( !strcmp( argv[ index ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ index ] ) );
			else
				result = output.Append( argv[ index ] );
		}

		if ( result && output.IsRunnable( ) )
		{
			output.ThreadExecute( );
		}
		else if ( argc > 1 )
		{
			std::cerr << "Failed to load " << argv[ index - 1 ] << std::endl;
			Usage( );
		}
		else
		{
			Usage( );
		}
	}
	catch ( string exc )
	{
		std::cerr << "Exception caught: " << exc << std::endl;
	}

	return !result;
}
