/*
 * DVEncoder.cc -- DV Encoder Class
 * Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <algorithm>
#include <iostream>
using std::cerr;
using std::endl;

#include "DVEncoder.h"
#include <frame.h>

/** Constructor for the DVEncoder class.
*/

DVEncoder::DVEncoder( ) : 
	encoder( NULL ), 
	frame_count( 0 ),
	audio_count( 0 ),
	import( NULL ),
	fixed_audio_samples( false ),
	temp_image( NULL )
{
	for ( int n = 0; n < 4; n++ )
	{
		audio_buffers[ n ] = new int16_t [2 * DV_AUDIO_MAX_SAMPLES];
		memset( audio_buffers[ n ], 0, 2 * DV_AUDIO_MAX_SAMPLES * sizeof( int16_t ) );
	}
}

DVEncoder::DVEncoder( DVEncoderParams &_params ) : 
	DVEncoderParams( _params ), 
	encoder( NULL ), 
	frame_count( 0 ),
	audio_count( 0 ),
	import( NULL ),
	fixed_audio_samples( false ),
	temp_image( NULL )
{
	for ( int n = 0; n < 4; n++ )
	{
		audio_buffers[ n ] = new int16_t [2 * DV_AUDIO_MAX_SAMPLES];
		memset( audio_buffers[ n ], 0, 2 * DV_AUDIO_MAX_SAMPLES * sizeof( int16_t ) );
	}
	
	if ( _params.GetAudioFile( ) != "" && import == NULL )
	{
		import = AudioImporter::GetImporter( _params.GetAudioFile( ) );
		if ( import != NULL )
		{
			SetFrequency( import->GetFrequency( ) );
			SetChannels( import->GetChannels( ) );
			SetBitsPerSample( import->GetBytesPerSample( ) * 8 );
		}
	}
}


/** Destructor for the DVEncoder class.
*/

DVEncoder::~DVEncoder( )
{
	delete temp_image;
	for ( int n = 0; n < 4; n++ )
		delete audio_buffers[ n ];
	if ( encoder != NULL )
		dv_encoder_free( encoder );
	delete import;
}

/** Encode the image.
*/

void DVEncoder::EncodeRGB( uint8_t *data, uint8_t *image )
{
	if ( GetTwoPassEncoding( ) )
	{
		if ( temp_image == NULL )
			temp_image = new uint8_t[ 720 * 576 * 3 ];

		Frame *frame = GetFramePool( )->GetFrame( );
		dv_encode_full_frame( GetEncoder( ), &image, e_dv_color_rgb, frame->data );
		frame->decoder->quality = DV_QUALITY_BEST;
		frame->ExtractHeader( );
		frame->ExtractRGB( temp_image );
		int size = frame->GetWidth( ) * frame->GetHeight( ) * 3;
		for ( int i = 0; i < size; i ++ )
			image[ i ] = std::min(std::max((int)image[i] - ( (int)temp_image[i] - (int)image[i] ), 0), 255);
		dv_encode_full_frame( GetEncoder( ), &image, e_dv_color_rgb, data );		
		GetFramePool( )->DoneWithFrame( frame );
	}
	else
	{
		dv_encode_full_frame( GetEncoder( ), &image, e_dv_color_rgb, data );
	}
	dv_encode_metadata( data, encoder->isPAL, encoder->is16x9, &datetime, frame_count );
	dv_encode_timecode( data, encoder->isPAL, frame_count ++ );
}

void DVEncoder::EncodeMetaData( Frame &frame )
{
	GetEncoder( );
	dv_encode_metadata( frame.data, frame.IsPAL( ), frame.IsWide( ), &datetime, frame_count );
	dv_encode_timecode( frame.data, frame.IsPAL( ), frame_count ++ );
}

void DVEncoder::SetNewRecording( )
{
	if ( encoder != NULL )
	{
		dv_encoder_free( encoder );
		encoder = NULL;
	}
}

/** Encode the image to the Frame object.
*/

void DVEncoder::EncodeRGB( Frame &frame, uint8_t *image )
{
	EncodeRGB( frame.data, image );
	frame.ExtractHeader( );
}

/** Encode the audio from the source (only silence is permitted right now).
*/

void DVEncoder::EncodeAudio( Frame &frame )
{
	// Kludge - pick up audio file info
	if ( GetAudioFile( ) != "" && import == NULL )
	{
		import = AudioImporter::GetImporter( GetAudioFile( ) );
		if ( import != NULL )
		{
			SetFrequency( import->GetFrequency( ) );
			SetChannels( import->GetChannels( ) );
			SetBitsPerSample( import->GetBytesPerSample( ) * 8 );
		}
		else
		{
			SetAudioFile( "" );
		}
	}

	// If we're importing audio, then pick it up now
	if ( import != NULL )
	{
		int samples = GetAudioSamplesPerFrame( frame );
		int val = import->Get( audio_buffers, samples );
		
		if ( ! val )
		{
			if ( GetAudioLoop( ) )
			{
				delete import;
				import = NULL;
			}
			for ( int index = 0; index < 4; index ++ )
				memset( audio_buffers[ index ], 0, sizeof( int16_t ) * 2 * DV_AUDIO_MAX_SAMPLES );
		}
	}

	dv_encode_full_audio( GetEncoder( ), audio_buffers, GetChannels( ), GetFrequency( ), frame.data );
	frame.ExtractHeader( );
}

void DVEncoder::SetFixedAudioSamples( bool fixed_audio_samples )
{
	this->fixed_audio_samples = fixed_audio_samples;
}

int DVEncoder::GetAudioSamplesPerFrame( Frame &frame )
{
	GetEncoder( )->isPAL = frame.IsPAL( );
#ifdef LIBDV_HAS_SAMPLE_CALCULATOR
	if ( fixed_audio_samples )
		return GetFrequency( ) / ( frame.IsPAL( ) ? 25 : 30 );
	else
		return dv_calculate_samples( GetEncoder( ), GetFrequency( ), audio_count ++ );
#else
	return GetFrequency( ) / ( frame.IsPAL( ) ? 25 : 30 );
#endif
}

/** Obtain the encoder.
*/

dv_encoder_t *DVEncoder::GetEncoder( )
{
	if ( encoder == NULL )
	{
		encoder = dv_encoder_new( 0, !GetPAL( ), !GetPAL( ) );
		frame_count = 0;
		datetime = time( NULL );
	}

	encoder->isPAL = GetPAL( );
	encoder->is16x9 = GetWide( );
	encoder->vlc_encode_passes = GetEncodePasses( );
	encoder->static_qno = GetStaticQno( );
	encoder->force_dct = DV_DCT_AUTO;

	return encoder;
}

/** Destroy the encoder.
*/

void DVEncoder::DestroyEncoder( )
{
	if ( encoder != NULL )
	{
		dv_encoder_free( encoder );
		encoder = NULL;
	}
}

