<?php
/**
 * Forwards_Driver_ldap:: implements the Forwards_Driver API for LDAP driven
 * mail servers.
 *
 * $Horde: forwards/lib/Driver/ldap.php,v 1.10.2.4 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @author  Ben Klang <ben@alkaloid.net>
 * @author  Jan Schneider <jan@horde.org>
 * @package Forwards
 */
class Forwards_Driver_ldap extends Forwards_Driver {

    /**
     * Pointer to the ldap connection.
     *
     * @var resource
     */
    var $_connection;

    /**
     * Boolean true if connected, false otherwise
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * The current user's corresponding distinguished name
     *
     * @var string
     */
    var $_dn;

    /**
     * The current user's password
     *
     * @var string
     */
    var $_password;

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        $res = $this->_connect($password);
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        // $keeplocal is one of "on" or "off"
        $keeplocal = $keeplocal == 'on';

        list($forwardattr, $keeplocalattr) = $this->_getSchemaAttrs();

        // Change the user's forwards.
        switch($this->_params[$this->_realm]['schema']) {
        case 'sunone':
            $newDetails[$forwardattr] = array($target);
            $newDetails[$keeplocalattr] = array('forward');
            if ($keeplocal) {
                $newDetails[$keeplocalattr][] = 'mailbox';
            }
            break;

        case 'qmail-ldap':
            $newDetails[$forwardattr] = array($target);
            if ($keeplocal) {
                // If the record previously had data, we have to send an
                // empty array to remove the attribute.  However sending an
                // an empty array when the attribute is not already populated
                // causes PHP to return with "Protocol Error". 
                $res = $this->_getForwardInfo();
                if (is_a($res, 'PEAR_Error')) {
                    return $res;
                }
                if (!$res['keeplocal']) {
                    $newDetails[$keeplocalattr] = array();
                }
            } else {
                $newDetails[$keeplocalattr] = array('nolocal');
            }
            break;

        case 'exim':
            $newDetails[$forwardattr] = array($target);
            break;

        case 'custom':
            // The "custom" schema has no way to track $keeplocal because even
            // if we knew the attribute name there would be no way to know what
            // value(s) are expected.
            $newDetails[$forwardattr] = array($target);
            break;
        }

        $res = ldap_mod_replace($this->_connection, $this->_dn, $newDetails);
        if ($res === false) {
            Horde::logMessage(sprintf("Error while updating LDAP object: %s",
                                      ldap_error($this->_connection)),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError(_("An internal error has occurred.  Details have been logged for the administrator."));
        }
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        $res = $this->_connect($password);
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        $res = $this->_getForwardInfo();
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }
        $forwards = $res['forwards'];
        $keeplocal = $res['keeplocal'];
        list($forwardattr, $keeplocalattr) = $this->_getSchemaAttrs();

        if (empty($forwards) && !$keeplocal) {
            // Nothing to delete, so treat as success.
            return;
        }

        switch($this->_params[$this->_realm]['schema']) {
        case 'sunone':
            if (!empty($forwards)) {
                $newDetails[$forwardattr] = array();
            }
            $newDetails[$keeplocalattr] = array('mailbox');
            break;

        case 'qmail-ldap':
            if (!empty($forwards)) {
                $newDetails[$forwardattr] = array();
            }
            if (!$keeplocal) {
                // FIXME: Should this be set to the default behavior (empty)
                // or set explicitly ('localonly')?  I think the default
                // behavior is most appropriate.
                $newDetails[$keeplocalattr] = array();
            }
            break;

        case 'exim':
            if (!empty($forwards)) {
                $newDetails[$forwardattr] = array();
            }
            break;

        case 'custom':
            if (!empty($forwards)) {
                $newDetails[$forwardattr] = array();
            }
            break;
        }
        $res = ldap_modify($this->_connection, $this->_dn, $newDetails);
        if ($res === false) {
            Horde::logMessage(sprintf('Error while removing forwarding information from LDAP: %s',
                                      ldap_error($this->_connection)),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError(_("An internal error has occurred.  Details have been logged for the administrator."));
        }
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        $res = $this->_connect($password);
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }
 
        $res = $this->_getForwardInfo();
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        if ($res['forwards'] != null) {
            return 'Y';
        } else {
            return 'N';
        }
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        $res = $this->_connect($password);
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }
 
        $res = $this->_getForwardInfo();
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        return $res['keeplocal'];
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        $res = $this->_connect($password);
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        $res = $this->_getForwardInfo();
        if (is_a($res, 'PEAR_Error')) {
            return $res;
        }

        return $res['forwards'];
    }

    /**
     * Get the forwarding information for the requested user
     *
     * @return mixed  Array of forwarding data on succes, false on failure
     */
    function _getForwardInfo()
    {
        $error = _("Internal error while searching LDAP.  Details have been logged for the administrator.");
        list($forwardattr, $keeplocalattr) = $this->_getSchemaAttrs();
        $attribs = array();
        if (!is_null($forwardattr)) {
            $attribs[] = $forwardattr;
        }
        if (!is_null($keeplocalattr)) {
            $attribs[] = $keeplocalattr;
        }
        $sr = ldap_read($this->_connection, $this->_dn, 'objectClass=*', $attribs);

        if ($sr === false) {
            Horde::logMessage(sprintf('Error while searching LDAP: %s',
                                      ldap_error($this->_connection)),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }

        $res = ldap_get_entries($this->_connection, $sr);
        if ($res === false) {
            Horde::logMessage(sprintf('Error while getting LDAP results: %s',
                                      ldap_error($this->_connection)),
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }

        // LDAP results always return attribute names in lower case
        $forwardattr = String::lower($forwardattr);
        $keeplocalattr = String::lower($keeplocalattr);

        // Note: we only process the first result.

        // This block is sufficient for determining the configured forwards for
        // all the various supported schemas.
        if (isset($res[0][$forwardattr])) {
            $forwards = $res[0][$forwardattr];
            // Prune unnecessary 'count' field from forward array
            unset($forwards['count']);
        } else {
            $forwards = array();
        }

        // Determining the keeplocal state is a bit more tricky.
        switch($this->_params[$this->_realm]['schema']) {
        case 'sunone':
            if (in_array('mailbox', $res[0][$keeplocalattr])) {
                $keeplocal = true;
            } else {
                $keeplocal = false;
            }
            break;

        case 'qmail-ldap':
            if (!isset($res[0][$keeplocalattr][0])) {
                // No entry defaults to local delivery enabled
                $keeplocal = true;
                break;
            }
            switch($res[0][$keeplocalattr][0]) {
            // FIXME: Handle other valid attribute values
            case 'nolocal':
                $keeplocal = false;
                break;
            case 'localonly':
                // Technically this means forwarding is disabled but there
                // is no way (currently) to handle that in the Horde Vaction
                // UI so we just treat it as if local delivery is enabled.
            default:
                $keeplocal = true;
                break;
            }
            break;

        case 'exim':
            // FIXME: There is no way to indicate keeplocal on Exim
            // configurations because at the time of this writing I don't
            // know what (if any) attributes or values Exim uses to store 
            // this information.
            // If forward addresses are configured we'll guess keeplocal is
            // false.  Without forward addresses we'll guess it's true.
            if (isset($res[0][$forwardattr]) &&
                count($res[0][$forwardattr]) != 0) {
                $keeplocal = true;
            } else {
                $keeplocal = false;
            }
            break;
       
        case 'custom':
            // FIXME: There is no way to indicate keeplocal on custom
            // configurations because we don't know what values to look for
            // even if we knew the attributes.
            // If forward addresses are configured we'll guess keeplocal is
            // false.  Without forward addresses we'll guess it's true.
            if (count($forwards) != 0) {
                $keeplocal = false;
            } else {
                $keeplocal = true;
            }
            break;
        }

        // FIXME:
        // This application only allows a single configured forward address.
        if (count($forwards) > 0) {
            $forwards = $forwards[0];
        } else {
            $forwards = null;
        }
        
        return array('forwards' => $forwards, 'keeplocal' => $keeplocal);
    }

    /**
     * Get the attributes for storing forward addresses and local delivery
     * option based on the configured schema.
     *
     * @return array  List of attributes (forwardsattr, keeplocalattr)
     */
    function _getSchemaAttrs()
    {
        switch($this->_params[$this->_realm]['schema']) {
        case 'sunone':
            $attribs = array('mailForwardingAddress', 'mailDeliveryOption');
            break;
        case 'qmail-ldap':
            $attribs = array('mailForwardingAddress', 'deliveryMode');
            break;
        case 'exim':
            $attribs = array('mailForward', null);
        case 'custom':
            $attribs = array($this->_params[$this->_realm]['attribute'], null);
            break;
        }

        return $attribs;
    }

    /**
     * Opens a connection to the LDAP server.
     *
     * @return mixed  True on success or a PEAR_Error object on failure.
     */
    function _connect($password)
    {
        if ($this->_connected) {
            return true;
        }

        if (!Util::extensionExists('ldap')) {
            return PEAR::raiseError("Forwards_Driver_ldap: Required LDAP extension not found.");
        }

        if (is_a($checked = $this->_checkConfig($this->_realm), 'PEAR_Error')) {
            return $checked;
        }

        $this->_password = $password;
        $error = _("Internal LDAP error.  Details have been logged for the administrator.");

        /* Connect to the LDAP server anonymously. */
        $conn = ldap_connect($this->_params[$this->_realm]['hostspec'],
                             $this->_params[$this->_realm]['port']);
        if (!$conn) {
            Horde::logMessage(
                sprintf('Failed to open an LDAP connection to %s.',
                        $this->_params[$this->_realm]['hostspec']),
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError();
        }

        /* Set the LDAP protocol version. */
        if (isset($this->_params[$this->_realm]['version'])) {
            $result = @ldap_set_option($conn, LDAP_OPT_PROTOCOL_VERSION,
                                       $this->_params[$this->_realm]['version']);
            if ($result === false) {
                Horde::logMessage(
                    sprintf('Set LDAP protocol version to %d failed: [%d] %s',
                            $this->_params[$this->_realm]['version'],
                            @ldap_errno($conn),
                            @ldap_error($conn)),
                    __FILE__, __LINE__, PEAR_LOG_WARNING);
                return PEAR::raiseError($error);
            }
        }

        /* If necessary, bind to the LDAP server as the user with search
         * permissions. */
        if (!empty($this->_params[$this->_realm]['searchdn'])) {
            $bind = @ldap_bind($conn, $this->_params[$this->_realm]['searchdn'],
                               $this->_params[$this->_realm]['searchpw']);
            if ($bind === false) {
                Horde::logMessage(
                    sprintf('Bind to server %s:%d with DN %s failed: [%d] %s',
                            $this->_params[$this->_realm]['hostspec'],
                            $this->_params[$this->_realm]['port'],
                            $this->_params[$this->_realm]['searchdn'],
                            @ldap_errno($conn),
                            @ldap_error($conn)),
                    __FILE__, __LINE__, PEAR_LOG_ERR);
                return PEAR::raiseError($error);
            }
        }

        /* Register our callback function to handle referrals. */
        if (function_exists('ldap_set_rebind_proc')) {
            $result = @ldap_set_rebind_proc($conn, array($this, '_rebindProc'));
            if ($result === false) {
                Horde::logMessage(
                    sprintf('Setting referral callback failed: [%d] %s',
                            @ldap_errno($conn),
                            @ldap_error($conn)),
                    __FILE__, __LINE__, PEAR_LOG_WARNING);
                return PEAR::raiseError($error);
            }
        }

        /* Store the connection handle at the instance level. */
        $this->_connection = $conn;

        /* Search for the user's full DN. */
        $search = @ldap_search($this->_connection,
                               $this->_params[$this->_realm]['basedn'],
                               $this->_params[$this->_realm]['uid'] . '=' . $this->_user,
                               array('dn'));

        if ($search === false) {
            Horde::logMessage(
                sprintf('Error while searching the directory for the user\'s DN: [%d]: %s %s',
                        @ldap_errno($this->_connection),
                        @ldap_error($this->_connection)),
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }

        $result = @ldap_get_entries($this->_connection, $search);
        if ($result === false) {
            Horde::logMessage(
                sprintf('Error while retrieving LDAP search results for the user\'s DN: [%d]: %s',
                        @ldap_errno($this->_connection),
                        @ldap_error($this->_connection)),
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }

        if ($result['count'] != 1) {
            Horde::logMessage(
                'Zero or more than one DN returned from search; unable to determine user\'s correct DN.',
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }
        $this->_dn = $result[0]['dn'];

        // Now we should have the user's DN.  Re-bind as appropriate with write
        // permissions to be able to store preferences.
        switch($this->_params[$this->_realm]['writedn']) {
        case 'user':
            $result = @ldap_bind($this->_connection,
                                 $this->_dn, $password);
            break;
        case 'admin':
            $result = @ldap_bind($this->_connection,
                                 $this->_params[$this->_realm]['admindn'],
                                 $this->_params[$this->_realm]['adminpw']);
            break;
        case 'searchdn':
            // Since we've already bound as the search DN above, no rebinding
            // is necessary.
            $result = true;
            break;
        }

        if ($result === false) {
            Horde::logMessage(
                sprintf('Error rebinding for forwards writing: [%d]: %s',
                        @ldap_errno($this->_connection),
                        @ldap_error($this->_connection)),
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return PEAR::raiseError($error);
        }

        // The connection is now fully initialized and usable.
        $this->_connected = true;
        return true;
    }

    /**
     * Callback function for LDAP referrals.
     *
     * This function is called when an LDAP operation returns a referral to an
     * alternate server.
     *
     * @return integer  1 on error, 0 on success.
     */
    function _rebindProc($conn, $who)
    {
        /* Strip out the hostname we're being redirected to. */
        $who = preg_replace(array('|^.*://|', '|:\d*$|'), '', $who);

        /* Make sure the server we're being redirected to is in our list of
         * valid servers. */
        if (strpos($this->_params[$this->_realm]['hostspec'], $who) === false) {
            Horde::logMessage(
                sprintf('Referral target %s for DN %s is not in the authorized server list.',
                        $who, $bind_dn),
                __FILE__, __LINE__, PEAR_LOG_ERR);
            return 1;
        }

        /* Figure out the DN of the authenticating user. */
        switch($this->_params[$this->_realm]['writedn']) {
        case 'user':
            $bind_dn = $this->_dn;
            $bind_pw = $this->_password;
            break;
        case 'admin':
            $bind_dn = $this->_params[$this->_realm]['admindn'];
            $bind_pw = $this->_params[$this->_realm]['adminpw'];
            break;
        case 'searchdn':
            $bind_dn = $this->_params[$this->_realm]['searchdn'];
            $bind_dn = $this->_params[$this->_realm]['searchpw'];
            break;
        }

        /* Bind to the new server. */
        $bind = @ldap_bind($conn, $bind_dn, $bind_pw);
        if ($bind === false) {
            Horde::logMessage(
                sprintf('Rebind to server %s:%d with DN %s failed: [%d] %s',
                        $this->_params[$this->_realm]['hostspec'],
                        $this->_params[$this->_realm]['port'],
                        $bind_dn,
                        @ldap_errno($this->_connection),
                        @ldap_error($this->_connection)),
                __FILE__, __LINE__, PEAR_LOG_ERR);
        }

        return 0;
    }

    /**
     * Check if the realm has a specific configuration.
     *
     * If not, try to fall back on the default configuration.  If
     * still not a valid configuration then exit with an error.
     */
    function _checkConfig()
    {
        // If no host config for the realm, then we fall back to the default
        // realm.
        if (!isset($this->_params[$this->_realm])) {
            $this->_realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['schema']) ||
            empty($this->_params[$this->_realm]['hostspec']) ||
            empty($this->_params[$this->_realm]['port']) ||
            empty($this->_params[$this->_realm]['version']) ||
            empty($this->_params[$this->_realm]['basedn']) ||
            empty($this->_params[$this->_realm]['uid'])) {
            return PEAR::raiseError(_("The module is not properly configured!"));
        }
    }

}
