/* ====================================================================
 * Copyright (c) 2006-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "Bookmark.h"
#include "Project.h"
#include "svn/Client.h"


const long NoItemItemId = -1111;

Bookmark::Bookmark( Project* prj )
: _id(Id::next()), _idParent(0), _prj(prj), _prjItemId(NoItemItemId)
{
}

Bookmark::Bookmark( Project* prj, long prjItemId, ID parentId )
: _id(Id::next()), _idParent(parentId), _prj(prj), _prjItemId(prjItemId)
{
}

ID Bookmark::getId() const
{
  return _id;
}

ID Bookmark::getParentId() const
{
  return _idParent;
}

Project* Bookmark::getProject()
{
  return _prj;
}

long Bookmark::getPrjItemId()
{
  return _prjItemId;
}

const sc::String& Bookmark::getName() const
{
  if( isProject() )
  {
    return _prj->getName();
  }
  else
  {
    return _prj->getItem(_prjItemId).getName();
  }
}

void Bookmark::setName( const sc::String& name )
{
  if( isProject() )
  {
    _prj->setName(name);
  }
  else
  {
    _prj->setItemName(_prjItemId,name);
  }
}

const sc::String& Bookmark::getSource() const
{
  if( isProject() )
  {
    return sc::NullString;
  }
  else
  {
    return _prj->getItemSource(_prjItemId);
  }
}

const sc::String& Bookmark::getUrl() const
{
  if( isProject() )
    return sc::NullString;

  if( _url.isEmpty() )
    svn::Client::getUrlFromPath( getSource(), _url );

  return _url;
}

void Bookmark::setSource( const sc::String& source )
{
  if( isProject() )
  {
    // nop
    assert(false);
  }
  else
  {
    _prj->setItemSource(_prjItemId,source);
  }
}

bool Bookmark::isProject() const
{
  return _prjItemId == NoItemItemId;
}

long Bookmark::getSortPos() const
{
  if( isProject() )
  {
    return _prj->getSortPos();
  }
  else
  {
    return _prj->getItem(_prjItemId).getSortPos();
  }
}

bool Bookmark::isWorkingCopy() const
{
  return _prj->getItem(_prjItemId).isWorkingCopy();
}

bool Bookmark::isRepository() const
{
  return _prj->getItem(_prjItemId).isRepository();
}

bool Bookmark::isTrunk() const
{
  return _prj->getItem(_prjItemId).isTrunk();
}

bool Bookmark::isBranches() const
{
  return _prj->getItem(_prjItemId).isBranches();
}

bool Bookmark::isTags() const
{
  return _prj->getItem(_prjItemId).isTags();
}

bool Bookmark::isRemote() const
{
  return isRepository() || isTrunk() || isBranches() || isTags();
}

bool Bookmark::isVisible() const
{
  return ! (getSource().isEmpty() && (isTrunk() || isBranches() || isTags()));
}

bool Bookmark::isCurrent() const
{
  return _prj->getCurWorkingCopyId() == _prjItemId;
}

void Bookmark::setCurrent()
{
  _prj->setCurWorkingCopy(_prjItemId);
}

bool Bookmark::getRecursive() const
{
  return _prj->getItem(_prjItemId).getRecursive();
}

void Bookmark::setRecursive( bool on )
{
  Project::Item item = _prj->getItem(_prjItemId);
  item.setRecursive(on);
  _prj->setItem(item);
}

bool Bookmark::getUpdate() const
{
  return _prj->getItem(_prjItemId).getUpdate();
}

void Bookmark::setUpdate( bool on )
{
  Project::Item item = _prj->getItem(_prjItemId);
  item.setUpdate(on);
  _prj->setItem(item);
}

bool Bookmark::getAutoUpdate() const
{
  return _prj->getItem(_prjItemId).getAutoUpdate();
}

void Bookmark::setAutoUpdate( bool on )
{
  Project::Item item = _prj->getItem(_prjItemId);
  item.setAutoUpdate(on);
  _prj->setItem(item);
}

bool Bookmark::getAutoRefresh()
{
  return _prj->getItem(_prjItemId).getAutoRefresh();
}

void Bookmark::setAutoRefresh( bool on )
{
  Project::Item item = _prj->getItem(_prjItemId);
  item.setAutoRefresh(on);
  _prj->setItem(item);
}
