/* ====================================================================
 * Copyright (c) 2006-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

#ifndef _SC_BOOKMARKVIEWMODEL_H
#define _SC_BOOKMARKVIEWMODEL_H

// sc
#include "util/Id.h"
namespace sc {
  class String;
}
class Bookmark;
class Project;
class ScModel;

// qt
#include <QtCore/QObject>
#include <QtCore/QStringList>

// sys
#include <map>

typedef std::map< ID, Bookmark* > Bookmarks;


class BookmarkViewModel : public QObject
{
  typedef QObject super;
  Q_OBJECT

public:
  BookmarkViewModel( ScModel* model );
  ~BookmarkViewModel();

  /** initialize all bookmarks from model */
  void initBookmarks();
  /** update all bookmarks from model */
  void updateBookmarks();
  /** update bookmarks of given project */
  void updateBookmarks( Project* );
  Bookmark* createProject();
  Bookmark* getBookmark( ID id );

  /**
   * return the currently selected bookmark.
   */
  Bookmark* getBookmark();

  /**
   * return the bookmark selected by a right mouse click.
   */
  Bookmark* getBookmarkMenu();

  void renameBookmark( Bookmark*, const sc::String& newName );

  /**
   * Move the \a src bookmark to the position of \a dst bookmark. Both
   * bookmarks must be of the same time, either project or project item.
   */
  void moveBookmark( Bookmark* src, Bookmark* dst );

  /**
   * the first 3 use only the first item in the given list.
   */
  void setTrunkUrl( Bookmark*, const QStringList& );
  void setBranchesUrl( Bookmark*, const QStringList& );
  void setTagsUrl( Bookmark*, const QStringList& );
  void addRepositories( Bookmark*, const QStringList& );
  void addWorkingCopies( Bookmark*, const QStringList& );

signals:
  void bookmarkAdded( Bookmark* );
  void bookmarkModified( Bookmark* );
  void bookmarkRemoved( Bookmark* );
  void bookmarkSelected( Bookmark* );
  void bookmarkCurrent( Bookmark* );
  void bookmarksModified();

  void showEditBookmark( Bookmark*, bool& );
  void showRenameBookmark( Bookmark* );
  void showNewProjectBookmark( Project*, bool& );
  void showRemoveBookmark( Bookmark*, bool& );

public slots:
  void setBookmark( Bookmark* );      ///< left click
  void setBookmarkMenu( Bookmark* );  ///< right click

  void createNewProject();
  void createNewProjectSimple();
  void addBookmarkRepository();
  void addBookmarkWorkingCopy();

  void setCurrent();
  void editBookmark();
  void editBookmarkTrunk();
  void editBookmarkBranches();
  void editBookmarkTags();
  void removeBookmark();
  void toggleRecursive( bool on );
  void toggleUpdate( bool on );
  void toggleAutoUpdate( bool on );
  void toggleAutoRefresh( bool on );

private:
  Bookmark* createBookmarks( Project* );
  void updateChildBookmark( Bookmark* bmPrj, long itemId );
  void editBookmark( Bookmark* bm );
  enum EditBookmarkType { Trunk, Branches, Tags };
  void editBookmark( EditBookmarkType type );
  void removeBookmarks( Bookmark* bmPrj );

private:
  ScModel*  _model;
  Bookmarks _bookmarks;
  
  ID        _current;         ///< last bookmark selected by left click
  ID        _currentMenu;     ///< last bookmark selected by right click
};


#endif // _SC_BOOKMARKVIEWMODEL_H
