/* ====================================================================
 * Copyright (c) 2007-2008  Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "WcViewTreeProxyModel.h"
#include "WcViewTreeItemModel.h"
#include "WcViewStatus.h"
#include "svn/WcStatus.h"
#include "svn/Path.h"
#include "util/Compare.h"


WcViewTreeProxyModel::WcViewTreeProxyModel()
: _flat(false), _all(false), _ignored(false), _outofdate(false), _modifiedFolder(false)
{
}

WcViewTreeProxyModel::~WcViewTreeProxyModel()
{
}

QVariant WcViewTreeProxyModel::data( const QModelIndex& index, int role ) const
{
  QVariant value = sourceModel()->data( mapToSource(index), role );

  if( index.column() == 0 && role == Qt::DisplayRole || role == Qt::EditRole )
  {
    QString temp = value.toString();

    if( temp == (const char*)_current )
    {
      temp = ".";
    }
    else
    {
      if(_flat)
        temp.remove( 0, _current.getCharCnt()+1 );
      else
        temp = temp.section('/', -1);
    }

    value = temp;
  }

  return value;
}

Qt::ItemFlags WcViewTreeProxyModel::flags( const QModelIndex &index ) const
{
  Qt::ItemFlags flags = sourceModel()->flags( mapToSource(index) );
  sc::String target = index.data(WcViewTreeItemModel::NameRole).value<sc::String>();

  if( target == _current )
    flags &= ~(Qt::ItemIsDragEnabled | Qt::ItemIsEditable);

  return flags;
}

bool WcViewTreeProxyModel::filterAcceptsRow( int sourceRow, const QModelIndex& sourceParent ) const
{
  QModelIndex index = sourceModel()->index( sourceRow, 0, sourceParent );

  const WcViewItem* item = index.data(WcViewTreeItemModel::WcViewItemRole).value<const WcViewItem*>();
  svn::WcStatusPtr status = static_cast<const WcViewStatus*>(item)->status();

  sc::String target = status->getName();

  if( target == _current )
    return true;

  // filter siblings of current root index
  sc::String targetDir  = svn::Path::getDirName(target);
  sc::String currentDir = svn::Path::getDirName(_current);
  if( targetDir == currentDir )
    return false;

  if( _all && ! status->isIgnored() && ! status->isReposChanged() )
    return true;

  if( _outofdate && status->isReposChanged() )
    return true;

  if( _ignored && status->isIgnored() )
    return true;

  if( status->isChanged() )
    return true;

  if( status->isDir() )
  {
    bool deep = index.data(WcViewTreeItemModel::DeepRole).toBool();

    if( _flat )
      return deep;

    else if( _modifiedFolder )
      return deep;

    else
      return true;
  }
  return false;
}

bool WcViewTreeProxyModel::lessThan(const QModelIndex &left, const QModelIndex &right) const
{
  if( left.column() == 0 && left.isValid() && right.isValid() )
  {
    const WcViewItem* litem = left.data(WcViewTreeItemModel::WcViewItemRole).value<const WcViewItem*>();
    svn::WcStatusPtr leftItem = static_cast<const WcViewStatus*>(litem)->status();

    const WcViewItem* ritem = right.data(WcViewTreeItemModel::WcViewItemRole).value<const WcViewItem*>();
    svn::WcStatusPtr rightItem = static_cast<const WcViewStatus*>(ritem)->status();

    int result = compare3( 
      leftItem->getName(), leftItem->isDir(),
      rightItem->getName(), rightItem->isDir() );

    return result < 0;
  }
  else
  {
    return super::lessThan(left,right);
  }

  return false;
}

void WcViewTreeProxyModel::setCurrentPath( const sc::String& current )
{
  _current = current;
}

void WcViewTreeProxyModel::setFilterFlat( bool on )
{
  _flat = on;
}

void WcViewTreeProxyModel::setFilterAll( bool on )
{
  _all = on;
}

void WcViewTreeProxyModel::setFilterIgnored( bool on )
{
  _ignored = on;
}

void WcViewTreeProxyModel::setFilterOutOfDate( bool on )
{
  _outofdate = on;
}

void WcViewTreeProxyModel::setFilterModifiedFolder( bool on )
{
  _modifiedFolder = on;
}
