"""
 Elements - 2D rigid body physics for python (supporting box2d and chipmunk)

 Copyright (C) 2008, Chris Hager and Joshua Minor

       Home:  http://wiki.laptop.org/go/Elements
        IRC:  #elements on irc.freenode.org
       Code:  http://www.assembla.com/wiki/show/elements
              svn co http://svn2.assembla.com/svn/elements

 This file is part of the Elements Project

     Elements is free software: you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     Elements is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with Elements.  If not, see <http://www.gnu.org/licenses/>.
"""

import pygame
from pygame.locals import *
from pygame.color import *

from Box2D import *

class elements:
    element_count = 0
    def __init__(self):
        # Init Boundaries
        self.worldAABB=b2AABB()
        self.worldAABB.lowerBound.Set(-100.0, -100.0);
        self.worldAABB.upperBound.Set(100.0, 100.0);
        
        # Gravity + Bodies will sleep on outside
        self.gravity = b2Vec2(0.0, -10.0);
        self.doSleep = True
    
        # Create the World
        self.world = b2World(self.worldAABB, self.gravity, self.doSleep)
        
        self.autoset_screen_size()
        
        # Pixels per Meter
        self.ppm = self.display_width / 6.;

    def autoset_screen_size(self, size=None):
        """ Get the current PyGame Screen Size, or sets it manually
            Optional: size == (int(width), int(height)) 
        """
        if size != None:
            self.display_width, self.display_height = size
            return
            
        try:
            x,y,w,h = pygame.display.get_surface().get_rect()
            self.display_width = w
            self.display_height = h
        except:
            print "Elements Error: Please start pygame.init() before loading Elements Physics"
            exit(0)

    def update(self, fps=50.0, iterations=10):
        timeStep = 1.0 / fps
        self.world.Step(timeStep, iterations);
    
    def draw(self, surface):
        body = self.world.GetBodyList()
        
        while body:
            xform = body.GetXForm()
            shape = body.GetShapeList()
            
            while shape:
                type = shape.GetType()
                if type == e_circleShape:
                    circle = shape.asCircle()
                    position = b2Mul(xform, circle.GetLocalPosition())
                    rect = pygame.Rect((position.x-circle.m_radius)*self.ppm,
                      self.display_height - (position.y+circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm,
                      (2*circle.m_radius)*self.ppm)
                    pygame.draw.ellipse(surface, (0,0,0), rect, 1)
                    
                elif type == e_polygonShape:
                    poly = shape.asPolygon()
                    points = []
                    for i in range(0,poly.GetVertexCount()):
                        pt = b2Mul(xform, poly.getVertex(i))
                        pt.x = pt.x * self.ppm
                        pt.y = self.display_height - (pt.y * self.ppm)
                        points.append([pt.x, pt.y])
                    pygame.draw.polygon(surface, (0,0,0), points, 1)
    
                else:
                    print "  unknown shape type:%d" % shape.GetType()
    
                shape = shape.GetNext()            
            body = body.GetNext()
        
    
    def add_ground(self):
        # Define the ground body.
        BodyDef = b2BodyDef();
        BodyDef.position.Set(0.0, -10.0);
    
        # Create the Ground Body
        Body = self.world.CreateStaticBody(BodyDef);
    
        # Add a shape to the Ground Body
        ShapeDef = b2PolygonDef()
        ShapeDef.SetAsBox(50.0, 10.0);
        Body.CreateShape(ShapeDef);
        Body.SetMassFromShapes();
    
    def add_wall(self, a, b):
        BodyDef = b2BodyDef();
        BodyDef.position.Set(a[0], a[1]);
        Body = self.world.CreateStaticBody(BodyDef);
        ShapeDef = b2PolygonDef()
        ShapeDef.SetAsBox(max(0.1,b[0]-a[0]), max(0.1,b[1]-a[1]));
        Body.CreateShape(ShapeDef);
        Body.SetMassFromShapes();

    def add_ball(self, pos):
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        bodyDef = b2BodyDef()
        bodyDef.position.Set(x, y)
#       bodyDef.rotation = b2_pi/10.0 * i
        body = self.world.CreateDynamicBody(bodyDef)
        circleDef = b2CircleDef()
        circleDef.density = 1.0
        circleDef.radius = 0.1
        circleDef.restitution = 0.3
        circleDef.friction = 0.5
        body.CreateShape(circleDef)
        body.SetMassFromShapes();

        self.element_count += 1

    def add_rect(self, pos, a=0.2, b=0.4):
        x, y = pos
        x /= self.ppm
        y = self.display_height - y
        y /= self.ppm
        
        bodyDef = b2BodyDef()
        bodyDef.position.Set(x, y)
#       bodyDef.rotation = b2_pi/10.0 * i
        body = self.world.CreateDynamicBody(bodyDef)
        boxDef = b2PolygonDef()
        boxDef.SetAsBox(a, b)
        boxDef.density = 1.0
        boxDef.restitution = 0.15
        boxDef.friction = 0.5
        body.CreateShape(boxDef)
        body.SetMassFromShapes();
        
        self.element_count += 1

