# Physics.activity
#
# Copyright (C) 2007  Joshua Minor
# This file is part of Physics.activity
# 
#     Physics.activity is free software: you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation, either version 3 of the License, or
#     (at your option) any later version.
# 
#     Foobar is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
# 
#     You should have received a copy of the GNU General Public License
#     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.


import sys
import os
import time

import pygame
import olpcgames

import olpcgames.pausescreen as pausescreen
import olpcgames.mesh as mesh
from olpcgames.util import get_bundle_path
from sugar.presence import presenceservice

from Box2D import *

bundlepath = get_bundle_path()
presenceService = presenceservice.get_instance()

class PhysicsGame:
    """Physics game controller.
    This class handles all of the game logic, event loop, mulitplayer, etc."""

    def __init__(self, screen):
        xoOwner = presenceService.get_owner()
        self.screen = screen
        self.font = pygame.font.Font(None, 30)
        
        self.running = True
        self.frame = 0
        self.pause = False
        self.timeStep = 1.0 / 30.0;
        self.iterations = 10;
        self.mousePos = b2Vec2()
        self.prevMousePos = b2Vec2()
        self.mouseJoint = None
        self.selectedShape = None
        
        # support arrow keys, game pad arrows and game pad buttons
        self.arrowkeys = {
        # real key:     ideal key
        pygame.K_UP:    pygame.K_UP,
        pygame.K_DOWN:  pygame.K_DOWN,
        pygame.K_LEFT:  pygame.K_LEFT,
        pygame.K_RIGHT: pygame.K_RIGHT,
        pygame.K_KP8:   pygame.K_UP,
        pygame.K_KP2:   pygame.K_DOWN,
        pygame.K_KP4:   pygame.K_LEFT,
        pygame.K_KP6:   pygame.K_RIGHT,
        pygame.K_KP9:   pygame.K_UP,
        pygame.K_KP3:   pygame.K_DOWN,
        pygame.K_KP7:   pygame.K_LEFT,
        pygame.K_KP1:   pygame.K_RIGHT
        }
        
        self.render = True
        self.ppm = 1
        self.world = None
        try:
            from scenes.default import *
            self.scene = Scene(self, screen)
        except:
            print "ERROR: Problem while reading scene"
 
    def processEvent(self, event):
        """Process a single pygame event.  This includes keystrokes
        as well as multiplayer events from the mesh."""
        screen_w, screen_h = self.screen.get_size()
        
        if event.type == pygame.QUIT:
            self.running = False
        elif event.type == pygame.KEYDOWN:
            if event.key == pygame.K_ESCAPE:
                self.running = False
            elif event.key == pygame.K_d:
                self.render = not self.render
                print "Render:", self.render
            elif self.arrowkeys.has_key(event.key):
                direction = self.arrowkeys[event.key]
                
                # if direction == pygame.K_UP:
                #     player.direction=(0,-1)
                # elif direction == pygame.K_DOWN:
                #     player.direction=(0,1)
                # elif direction == pygame.K_LEFT:
                #     player.direction=(-1,0)
                # elif direction == pygame.K_RIGHT:
                #     player.direction=(1,0)
                
        elif event.type == pygame.KEYUP:
            pass
        elif event.type == pygame.MOUSEMOTION:
            self.prevMousePos.Set(self.mousePos.x, self.mousePos.y)
            self.mousePos.Set(event.pos[0] / self.ppm, (screen_h - event.pos[1]) / self.ppm)
            if self.mouseJoint:
                self.mouseJoint.SetTarget(self.mousePos)
                body = self.mouseJoint.GetBody2()
                if self.pause or body.IsStatic():
                    p = b2Vec2()
                    p.x = body.GetPosition().x + self.mousePos.x - self.prevMousePos.x
                    p.y = body.GetPosition().y + self.mousePos.y - self.prevMousePos.y
                    body.SetXForm(p, body.GetAngle())
        elif event.type == pygame.MOUSEBUTTONDOWN:
            self.prevMousePos.Set(self.mousePos.x, self.mousePos.y)
            self.mousePos.Set(event.pos[0] / self.ppm, (screen_h - event.pos[1]) / self.ppm)
            self.selectedShape = self.world.shapeAtPoint(self.mousePos)
            if self.selectedShape:
                body = self.selectedShape.GetBody()
                md = b2MouseJointDef()
                md.body1 = self.world.GetGroundBody()
                md.body2 = body
                md.target = self.mousePos
                md.maxForce = 1000.0 * body.m_mass
                self.mouseJoint = self.world.CreateJoint(md).asMouseJoint()
                body.WakeUp()
        elif event.type == pygame.MOUSEBUTTONUP:
            self.prevMousePos.Set(self.mousePos.x, self.mousePos.y)
            self.mousePos.Set(event.pos[0] / self.ppm, (screen_h - event.pos[1]) / self.ppm)
            if self.mouseJoint:
                self.world.DestroyJoint(self.mouseJoint)
                self.mouseJoint = None
        elif event.type == mesh.CONNECT:
            print "Connected to the mesh."
        elif event.type == mesh.PARTICIPANT_ADD:
            buddy = mesh.get_buddy(event.handle)
            if event.handle == mesh.my_handle():
                print "Me:", buddy.props.nick, buddy.props.color
            else:
                print "Join:", buddy.props.nick, buddy.props.color
        elif event.type == mesh.PARTICIPANT_REMOVE:
            pass
        elif event.type == mesh.MESSAGE_UNI or event.type == mesh.MESSAGE_MULTI:
            buddy = mesh.get_buddy(event.handle)
            print "Message from %s / %s: %s" % (buddy.props.nick, event.handle, event.content)
        elif event.type == pygame.USEREVENT:
            if event.action == 'scene-changed':
                scenename = event.scene
                try:
                    exec("from scenes.%s import Scene" % scenename)
                    self.scene = Scene(self, self.screen)
                except:
                    print "ERROR: Problem while reading scene"
                    raise
            else:
                print "Unknown user event action:", event.action
        else:
            print "Unknown event:", event

    def run(self):
        """Run the main loop of the game."""
        # lets draw once before we enter the event loop
        self.clock = pygame.time.Clock()
        self.draw()
        pygame.display.flip()
        
        while self.running:
            self.frame += 1
            # process all queued events
            for event in pausescreen.get_events(sleep_timeout=30):
                self.processEvent(event)
            
            self.animate()
            self.draw()
            
            pygame.display.flip()
            # don't animate faster than about 30 frames per second
            # this keeps the speed reasonable and limits cpu usage
            self.clock.tick(30)

    def animate(self):
        """Animate one frame of action."""
        if not self.pause and self.world is not None:
            # Instruct the world to perform a single step of simulation. It is
            # generally best to keep the time step and iterations fixed.
            self.world.Step(self.timeStep, self.iterations)
    
    def draw(self):
        # compute the size of the tiles given the screen size, etc.
        screen_w, screen_h = self.screen.get_size()
        rect = pygame.Rect(0, 0, screen_w, screen_h)
        pygame.draw.rect(self.screen, (0, 0, 0), rect, 0)
        
        if self.world is None:
            text = "Please choose a scene.  D = toggle drawing on/off"
            textimg = self.font.render(text, 1, (255,255,255))
            textwidth, textheight = self.font.size(text)
            rect = pygame.Rect(10, 10, textwidth, textheight)
            self.screen.blit(textimg, rect)
            return
        
        if self.render:
            body = self.world.GetBodyList()
            while body:
                xform = body.GetXForm()
                shape = body.GetShapeList()
                while shape:
            
                    if shape.GetType() == e_circleShape:
                        circle = shape.asCircle()
                        position = b2Mul(xform, circle.GetLocalPosition())
                        rect = pygame.Rect((position.x-circle.m_radius)*self.ppm,
                         screen_h - (position.y+circle.m_radius)*self.ppm,
                          (2*circle.m_radius)*self.ppm,
                           (2*circle.m_radius)*self.ppm)
                        pygame.draw.ellipse(self.screen, (255,255,255), rect, 1)
                    elif shape.GetType() == e_polygonShape:
                        poly = shape.asPolygon()
                        #position = position + poly.GetCentroid()
                        points = []
                        #print shape, poly
                        for i in range(0,poly.GetVertexCount()):
                            pt = b2Mul(xform, poly.getVertex(i))
                            pt.x = pt.x * self.ppm
                            pt.y = screen_h - (pt.y * self.ppm)
                            points.append([pt.x, pt.y])
                        pygame.draw.polygon(self.screen, (255,255,255), points, 1)
                    else:
                        print "  unknown shape type:%d" % shape.GetType()
            
                    shape = shape.GetNext()
        
                body = body.GetNext()
        
            # joint = self.world.GetJointList()
            # while joint:
            #     
            #     if joint.GetType() == e_mouseJoint:
            #         mouseJoint = joint.asMouseJoint()
            #         a1 = mouseJoint.GetAnchor1()
            #         a2 = mouseJoint.GetAnchor2()
            #         pygame.draw.line(self.screen, (255,255,255), (a1.x*self.ppm, screen_h - a1.y*self.ppm), (a2.x*self.ppm, screen_h - a2.y*self.ppm), 1)
            #     elif joint.GetType() == e_revoluteJoint:
            #         revJoint = joint.asRevoluteJoint()
            #         a1 = revJoint.GetAnchor1()
            #         a2 = revJoint.GetAnchor2()
            #         pygame.draw.line(self.screen, (255,255,255), (a1.x*self.ppm, screen_h - a1.y*self.ppm), (a2.x*self.ppm, screen_h - a2.y*self.ppm), 1)
            #         pass
            #     else:
            #         print "Unknown joint type: %d" % joint.GetType()
            #     joint = joint.GetNext()
            # 
            # contact = self.world.GetContactList()
            # while contact:
            #     if contact.GetManifoldCount() > 0:
            #         manifold = contact.GetManifolds()
            #         normal = manifold.normal
            #         t = b2Vec2()
            #         t.Set(normal.y, -normal.x)
            #         for i in range(0, manifold.pointCount):
            #             p = b2Manifold_getPoint(manifold, i)
            #             forceVector = b2Vec2()
            #             forceVector.x = p.position.x + p.normalImpulse * normal.x + p.tangentImpulse * t.x
            #             forceVector.y = p.position.y + p.normalImpulse * normal.y + p.tangentImpulse * t.y
            #             pygame.draw.line(self.screen, (255,255,255), (p.position.x*self.ppm, screen_h - p.position.y*self.ppm), (forceVector.x*self.ppm, screen_h - forceVector.y*self.ppm), 1)
            #             pass
            #     contact = contact.GetNext()
            
        text = "%3.2f frames/second" % self.clock.get_fps()
        textimg = self.font.render(text, 1, (255,255,255))
        textwidth, textheight = self.font.size(text)
        rect = pygame.Rect(10, 10, textwidth, textheight)
        self.screen.blit(textimg, rect)

def main():
    """Run a game of Physics."""
    #canvas_size = 1024,768-75
    #screen = pygame.display.set_mode(canvas_size)

    # ask pygame how big the screen is, leaving a little room for the toolbar
    toolbarheight = 75
    pygame.display.init()
    maxX,maxY = pygame.display.list_modes()[0]
    screen = pygame.display.set_mode( ( maxX, maxY-toolbarheight ) )

    game = PhysicsGame(screen)
    game.run()

if __name__ == '__main__':
    main()

