#!/usr/bin/python
"""
This file is part of the 'Elements' Project
Elements is a 2D Physics API for Python (supporting Box2D2)

Copyright (C) 2008, The Elements Team, <elements@linuxuser.at>

Home:  http://wiki.laptop.org/go/Elements
IRC:   #elements on irc.freenode.org

Code:  http://www.assembla.com/wiki/show/elements
       svn co http://svn2.assembla.com/svn/elements                     

License:  GPLv3 | See LICENSE for the full text
This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.              
"""
import pygame
from pygame.locals import *
from pygame.color import *

from time import ctime
from time import sleep

from sys import argv
from sys import exit

from threading import Thread
from elements import *


# Threaded, delayed function
class add_things(Thread):
    running = True
    def __init__(self, delay, rect, callback):
        Thread.__init__(self)
        self.delay = delay
        self.callback = callback
        self.x = 20
        self.y = 600
        
        if rect:
            self.adder = self.add_rect
        else:
            self.adder = self.add_ball
        
    def add_rect(self):
        self.callback((self.x, self.y), 0.2, 0.1)
        self.callback((self.x, self.y), 0.2, 0.1)
        self.x += 30
                            
    def add_ball(self):
        self.callback((self.x, self.y), 0.15)
        self.callback((self.x+20, self.y), 0.15)
        self.x += 40
        
    def run(self):
        while self.running:
            for i in range(10):
                if self.running:
                    sleep(self.delay/10.0)

            if self.running:
                self.adder()
            
class capture_stats(Thread):
    running = True
    def __init__(self, delay, clock, count, logfile):
        Thread.__init__(self)
        self.delay = delay
        self.clock = clock
        self.count = count
        
        self.logf = open(logfile, "a")
        self.logf.write("----- Test started [%s]-----\n" % ctime())
        self.logf.write("Elements| FPS\t| Userdata\n")
    
    def log(self, s):
        self.logf.write("\t|\t|%s \n" % s)
        
    def run(self):
        while self.running:
            sleep(self.delay)
            self.logf.write(" %i\t| %i\t|\n" % (self.count(), int(self.clock.get_fps())))

        self.logf.write("----- Test stopped [%s]-----\n\n" % ctime())
        self.logf.close()
        
def main():
    print "Box2D & Elements Speed Test Skeleton"
    print "  [1] ....... add 20 balls"
    print "  [2] ....... add 20 rectangles"
    print "  [space] ... toggles drawing on/off"
    print
    
    if len(argv) == 1:
        print "> please start with 'python %s [logfile]'" % argv[0]
        exit(1)
        
    # PyGame Init
    pygame.init()
    screen = pygame.display.set_mode((1200, 900))
    clock = pygame.time.Clock()

    # Create the Physical Space Class
    world = Elements((1200, 900))
    world.set_drawingMethod("pygame")
        
    # Add A Ground
    world.add_ground()

    # Start Capture
    capt = capture_stats(1, clock, world.get_elementCount, argv[1])
    capt.start()
        
    # Default Settings
    running = True
    drawing = True
    draw_poly = False
    points = []
    adder = None
    
    # Main Loop
    while running:
        for event in pygame.event.get():
            if event.type == QUIT or (event.type == KEYDOWN and event.key == K_ESCAPE):
                # Bye Bye
                running = False
                capt.running = False
                if adder != None: 
                    adder.running = False
                    
            elif event.type == KEYDOWN:
                if event.key == K_SPACE:    
                    # Pause with SPACE
                    drawing = not drawing
                    if drawing:
                        capt.log("Drawing set to [ON]")
                    else:
                        capt.log("Drawing set to [OFF]")

                elif event.unicode == "1":
                    if adder == None:
                        adder = add_things(6, False, world.add_ball)
                        adder.start()
                        
                elif event.unicode == "2":
                    if adder == None:
                        adder = add_things(6, True, world.add_rect)
                        adder.start()
                                  
            elif event.type == MOUSEBUTTONDOWN and event.button == 1:
                # Add Ball
                world.add_ball(event.pos, 0.15)
                
            elif event.type == MOUSEBUTTONDOWN and event.button == 3:
                # Add Square
                world.add_rect(event.pos, 0.2, 0.1)

            
        # Clear Display
        screen.fill((255,255,255))

        # Update & Draw World
        world.update()
        if drawing: 
            world.draw(screen)

        # Flip Display
        pygame.display.flip()
        
        # Try to stay at 50 FPS
        clock.tick(50)

if __name__ == "__main__":
    main()
