#!/usr/bin/env python

# Copyright (C) 2010-2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import re
import sys
import shutil
import tarfile
import zipfile
import gettext
import subprocess
from fnmatch import fnmatch
from ConfigParser import ConfigParser
from os.path import dirname, join, exists, isdir, islink, basename
from os.path import abspath, relpath


_IGNORE_FILES = [
        '.git', '.gitignore', '.tmp', '*.pyc', '*.pyo', '*.~',
        '*.bak', '*.la', '*.tar.*', '.sweets', '/debian/**', '.osc',
        '.svn',
        ]


def enforce(condition, message=None):
    if condition:
        return
    if not message:
        # pylint: disable-msg=W0212
        frame = sys._getframe(1)
        message = 'Runtime assertion failed at %s:%s' % \
                (frame.f_globals['__file__'], frame.f_lineno - 1)
    raise RuntimeError(message)


def glob(path, include=None, exclude=None):
    """Search files using include/exclude patterns.

    Search patterns are extended variants of Unix shell-style wildcards:
    http://wiki.sugarlabs.org/go/Platform_Team/Guide/Packaging#Glob_patterns

    :param path:
        the root path to start searching
    :param include:
        list of string patterns to include files
    :param exclude:
        list of string patterns to exclude files
    :returns:
        list of file paths that are relative to the `path`

    """
    result = []
    include = include or []
    exclude = (exclude or []) + _IGNORE_FILES

    def is_path_filter(filter_value):
        return '/' in filter_value or '**' in filter_value

    for root, dirs, files in os.walk(path):
        rel_path = root[len(path):].lstrip(os.sep)

        for filename in dirs[:]:

            def name_match(filters):
                for i in filters:
                    if not is_path_filter(i) and fnmatch(filename, i):
                        return True

            if name_match(exclude):
                dirs.remove(filename)

        links_to_dirs = []
        for filename in os.listdir(root):
            if islink(join(root, filename)) and isdir(join(root, filename)):
                links_to_dirs.append(filename)

        for filename in files + links_to_dirs:
            file_path = join(rel_path, filename)

            def match(filters):
                for filter_value in filters:
                    if not is_path_filter(filter_value):
                        matched = fnmatch(filename, filter_value)
                    else:
                        if filter_value.startswith('/'):
                            filter_value = filter_value[1:]
                        regexp = ''
                        for i in filter_value.replace('**', '\0'):
                            if i == '?':
                                regexp += '[^/]'
                            elif i == '*':
                                regexp += '[^/]*'
                            elif i == '\0':
                                regexp += '.*'
                            else:
                                regexp += re.escape(i)
                        regexp += '$'
                        matched = re.match(regexp, file_path)
                    if matched:
                        return True
                return False

            if (not include or match(include)) and not match(exclude):
                result.append(file_path)

    return result


def move(pms, srcdir, destdir, prefix, include, exclude):
    """Move files from temporary DESTDIR location to the final one.

    :param pms:
        the type of PMS, `deb` or `rpm`
    :param srcdir:
        path to the directory with files to move
    :param destdir:
        final DESTDIR location
    :param prefix:
        the --prefix that was used in configure phase
    :param include:
        list of `glob` patterns for files to include
    :param exclude:
        list of `glob` patterns for files to exclude
    :returns:
        list of moved files to include to `pms` file lists, e.g.,
        to `%files` section in rpm spec

    """
    prefix = prefix.lstrip(os.sep).strip()
    result = []

    if pms == 'rpm':
        result.append('%defattr(-,root,root,-)')
        entry_mask = '/%(path)s'
    else:
        sweet = basename(destdir)
        entry_mask = 'debian/%s/%%(path)s /%%(dir)s' % sweet

    for path in glob(srcdir, include=include, exclude=exclude):
        src_abs = join(srcdir, path)
        if prefix and prefix.startswith('opt') and not path.startswith(prefix):
            path = join(prefix, path)
        dst_abs = join(destdir, path)

        if not exists(dirname(dst_abs)):
            os.makedirs(dirname(dst_abs))
        shutil.move(src_abs, dst_abs)

        if exists(join(srcdir, 'opt', 'sweets')):
            _fix_install_paths(join(destdir, prefix), dst_abs)

        escaped_path = ''
        for ch in path.strip(os.sep):
            if not ch.isalnum() and ch not in ['_', '-', '.', os.sep]:
                escaped_path += '?'
            else:
                escaped_path += ch

        result.append(entry_mask % \
                {'path': escaped_path,
                 'dir': dirname(escaped_path),
                 })

    return result


def activity(srcdir, destdir):
    """Install activity sources.

    :param srcdir:
        path to activity sources
    :param destdir:
        path to install activity

    """
    link(srcdir, destdir, '', '')

    if not exists(join(srcdir, 'po')):
        return

    info_file = join(srcdir, 'activity', 'activity.info')
    enforce(exists(info_file), 'Cannot find activity.info file')
    info = ConfigParser()
    info.read(info_file)

    enforce(info.has_option('Activity', 'bundle_id'),
            'Cannot find bundle_id option in activity.info')
    bundle_id = info.get('Activity', 'bundle_id')

    enforce(info.has_option('Activity', 'name'),
            'Cannot find name option in activity.info')
    name = info.get('Activity', 'name')

    localedir = join(destdir, 'locale')
    shutil.rmtree(localedir, ignore_errors=True)

    for f in os.listdir(join(srcdir, 'po')):
        if not f.endswith('.po') or f == 'pseudo.po':
            continue

        file_name = join(srcdir, 'po', f)
        lang = f[:-3]

        mo_path = join(localedir, lang, 'LC_MESSAGES')
        if not isdir(mo_path):
            os.makedirs(mo_path)

        mo_file = join(mo_path, '%s.mo' % bundle_id)
        args = ['msgfmt', '--output-file=%s' % mo_file, file_name]
        if subprocess.call(args):
            print 'msgfmt failed for %s' % file_name
            continue

        cat = gettext.GNUTranslations(open(mo_file, 'r'))
        translated_name = cat.gettext(name)
        linfo_file = join(localedir, lang, 'activity.linfo')
        f = open(linfo_file, 'w')
        f.write('[Activity]\nname = %s\n' % translated_name)
        f.close()


def link(srcdir, destdir, include, exclude):
    """Default sweet installation.

    All sources files according to include/except patterns will be hard linked
    to the final destination.

    :param srcdir:
        path sweet sources
    :param destdir:
        path to install sweet sources to
    :param include:
        list of `glob` patterns for files to include
    :param exclude:
        list of `glob` patterns for files to exclude

    """
    if abspath(srcdir) == abspath(destdir):
        return

    for src in glob('.', include=include, exclude=exclude):
        dst = join(destdir, src)
        if exists(dst):
            continue
        if not exists(dirname(dst)):
            os.makedirs(dirname(dst))
        try:
            os.link(src, dst)
        except OSError:
            shutil.copy(src, dst)


def uncpio(cpio, destdir):
    """Deep extract cpio files with bundles.

    If cpio content is tarballs or zipped files, it will be extracted as well.

    :param cpio:
        path to .cpio file
    :param destdir:
        path to extract files to

    """
    if not exists(destdir):
        os.makedirs(destdir)

    subprocess.check_call('cpio -mid --quiet < %s' % abspath(cpio),
            shell=True, cwd=destdir)

    for i in os.listdir(destdir):
        i = join(destdir, i)
        if tarfile.is_tarfile(i):
            tarfile.open(i).extractall(destdir)
        elif zipfile.is_zipfile(i):
            zipfile.ZipFile(i).extractall(destdir)
        else:
            print 'Unsupported bundle type for %s file' % i
            exit(1)
        os.unlink(i)


def _fix_install_paths(destdir_and_prefix, path):
    content = ''
    modified = False

    if path.endswith('.pc'):
        for line in file(path).readlines():
            if re.match('prefix\s*=', line, re.IGNORECASE) is not None:
                pc_relpath = relpath(path, destdir_and_prefix)
                pc_prefix = join('${pcfiledir}',
                        *(['..'] * (len(pc_relpath.split(os.sep)) - 1)))
                line = 'prefix=%s\n' % pc_prefix
                modified = True
            content += line
    elif path.endswith('.service'):
        for line in file(path).readlines():
            if re.match('Exec\s*=', line, re.IGNORECASE) is not None:
                cmd = basename(line.split('=', 1)[-1].strip())
                line = 'Exec=/usr/bin/env %s\n' % cmd
                modified = True
            content += line
    else:
        return

    if modified:
        f = file(path, 'w')
        f.write(content)
        f.close()


def _cmd_move(args):
    pms, srcdir, destdir, prefix, include, exclude = args
    include = include.split(';') if include else []
    exclude = exclude.split(';') if exclude else []
    for i in move(pms, srcdir, destdir, prefix, include, exclude):
        print i


def _cmd_activity(args):
    srcdir, destdir = args
    activity(srcdir, destdir)


def _cmd_link(args):
    destdir, include, exclude = args
    include = include.split(';') if include else []
    exclude = exclude.split(';') if exclude else []
    link('.', destdir, include, exclude)


def _cmd_uncpio(args):
    cpio, destdir = args
    uncpio(cpio, destdir)


def main():
    args = sys.argv[1:]
    cmd = args.pop(0)

    if '_cmd_' + cmd not in globals():
        print 'Wrong command %s' % cmd
        exit(1)

    globals()['_cmd_' + cmd](args)


if __name__ == '__main__':
    main()
