# Copyright (C) 2010 Collabora Ltd. <http://www.collabora.co.uk/>
# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from functools import partial

import dbus
import gobject
from telepathy import client

from telepathy.interfaces import ACCOUNT, CHANNEL, \
        CHANNEL_INTERFACE_GROUP, CHANNEL_TYPE_CONTACT_LIST, \
        CHANNEL_TYPE_FILE_TRANSFER, CLIENT, CONNECTION, \
        CONNECTION_INTERFACE_ALIASING, CONNECTION_INTERFACE_CONTACTS, \
        CONNECTION_INTERFACE_CONTACT_CAPABILITIES, \
        CONNECTION_INTERFACE_REQUESTS, CONNECTION_INTERFACE_SIMPLE_PRESENCE

from telepathy.constants import HANDLE_TYPE_CONTACT, HANDLE_TYPE_LIST, \
        CONNECTION_PRESENCE_TYPE_OFFLINE, CONNECTION_PRESENCE_TYPE_AVAILABLE, \
        CONNECTION_STATUS_CONNECTED, CONNECTION_STATUS_DISCONNECTED

from sugar.graphics.xocolor import XoColor

from jarabe.model import shell, bundleregistry
from .buddy import BuddyModel, get_owner_instance
from .activity import ActivityModel

ACCOUNT_MANAGER_SERVICE = 'org.freedesktop.Telepathy.AccountManager'

CONNECTION_INTERFACE_BUDDY_INFO = 'org.laptop.Telepathy.BuddyInfo'
CONNECTION_INTERFACE_ACTIVITY_PROPERTIES = \
        'org.laptop.Telepathy.ActivityProperties'

# Time in seconds to wait when querying contact properties. Some jabber servers
# will be very slow in returning these queries, so just be patient.
_QUERY_DBUS_TIMEOUT = 200

_logger = logging.getLogger('connection')


class Connection(object):

    on_connect = None
    on_disconnect = None
    on_buddy_add = None
    on_buddy_remove = None
    on_activity_add = None
    on_activity_remove = None
    on_current_activity_change = None

    def __init__(self, object_path):
        self.object_path = object_path
        self.buddies = {}
        self.activities = {}

        self._connection = None
        self._self_handle = None
        self._last_room_handle = 0
        self._signals = []

        account_manager = dbus.Bus().get_object(
                ACCOUNT_MANAGER_SERVICE, object_path)
        account_manager.Get(ACCOUNT, 'Connection',
                reply_handler=self._update_connection,
                error_handler=partial(self.__error_cb, 'Connection'))
        account_manager.connect_to_signal('AccountPropertyChanged',
                lambda props: self._update_connection(props.get('Connection')))

        shell.get_model().connect('active-activity-changed',
                self.__active_activity_changed_cb)

    @property
    def ready(self):
        return bool(self.buddies)

    def enable(self):
        self._set_enabled(True)

    def disable(self):
        self._set_enabled(False)

    @property
    def _self_buddy(self):
        return self.buddies.get(self._self_handle)

    def _set_enabled(self, new):
        account_manager = dbus.Bus().get_object(
                ACCOUNT_MANAGER_SERVICE, self.object_path)

        old = account_manager.Get(ACCOUNT, 'Enabled')
        if old == new:
            return

        if new:
            _logger.info('Enabling %s', self.object_path)
        else:
            _logger.info('Disabling %s', self.object_path)

        def reply_handler():
            _logger.debug('Set Enabled=%s for %s', new, self.object_path)

        account_manager.Set(ACCOUNT, 'Enabled', new,
                reply_handler=reply_handler,
                error_handler=partial(self.__error_cb, 'Enabled'),
                dbus_interface=dbus.PROPERTIES_IFACE)

    def _iface(self, iface):
        if self._connection is not None and iface in self._connection:
            return self._connection[iface]
        else:
            return None

    def _connect_to_signal(self, iface, *args, **kwargs):
        if iface is None:
            iface = self._connection
        else:
            iface = self._iface(iface)
        sid = iface.connect_to_signal(*args, **kwargs)
        self._signals.append(sid)

    def _update_connection(self, object_path):
        if object_path == '/':
            account_manager = dbus.Bus().get_object(
                    ACCOUNT_MANAGER_SERVICE, self.object_path)
            error = account_manager.Get(ACCOUNT, 'ConnectionError')

            if error == 'org.freedesktop.Telepathy.Error.RegistrationExists':
                _logger.info('Registration already exisits for %s, unregister',
                        self.object_path)
                account_manager.UpdateParameters(
                        {'register': False}, [], dbus_interface=ACCOUNT)

            if error and self._connection is not None:
                _logger.info('Close connection for %s: %s',
                        self.object_path, error)
                self._connection = None

        elif object_path and self._connection is None:
            _logger.info('Prepare connection for %s', self.object_path)

            name = object_path.replace('/', '.')[1:]
            self._connection = client.Connection(name, object_path)

            self._iface(dbus.PROPERTIES_IFACE).Get(CONNECTION, 'Status',
                    reply_handler=lambda status: self._update_status(status),
                    error_handler=partial(self.__error_cb, 'Status'))
            self._connect_to_signal(None, 'StatusChanged', self._update_status)

    def _update_status(self, status, reason=None):
        _logger.debug('Connection status changed to %r for %s: %s',
                status, self.object_path, reason)

        if status == CONNECTION_STATUS_CONNECTED:
            self._iface(dbus.PROPERTIES_IFACE).Get(CONNECTION, 'SelfHandle',
                    reply_handler=self.__SelfHandle_cb,
                    error_handler=partial(self.__error_cb, 'SelfHandle'))
            self.on_connect(self)
        elif status == CONNECTION_STATUS_DISCONNECTED:
            for buddy in self.buddies.values():
                self._popin_buddy(buddy)
            for room_handle in self.activities.keys():
                self._checkout_activity(room_handle)
            self.buddies.clear()
            while self._signals:
                self._signals.pop().remove()
            self._self_handle = None
            self.on_disconnect(self)

    def _checkin_owner(self):
        buddy = get_owner_instance()
        buddy.handle = self._self_handle
        self.buddies[self._self_handle] = buddy

        properties = {
                'color': buddy.color.to_string(),
                'key': dbus.ByteArray(buddy.pubkey),
                }

        def reply_handler():
            _logger.debug('Published owner properties: %r', properties)

        iface = self._iface(CONNECTION_INTERFACE_BUDDY_INFO)
        iface.SetProperties(properties,
                reply_handler=reply_handler,
                error_handler=partial(self.__error_cb, 'Owner.SetProperties'))

    def _checkin_buddy(self, handle):
        buddy = self.buddies.get(handle)
        if buddy is not None:
            return buddy

        iface = self._iface(CONNECTION_INTERFACE_CONTACTS)
        attributes = iface.GetContactAttributes([handle], [], False)
        for __, props in attributes.items():
            contact_id = props.get(CONNECTION + '/contact-id')
            if contact_id:
                break
        else:
            _logger.debug('Ignore buddy %r w/o contact_id', handle)
            return None

        if '/' in contact_id.split('@')[-1]:
            _logger.debug('Ignore non-roster buddy: handle=%r contact_id=%r',
                    handle, contact_id)
            return None

        _logger.debug('Check-in buddy: handle=%r contact_id=%r',
                handle, contact_id)

        buddy = BuddyModel(account=self.object_path, handle=handle,
                contact_id=contact_id)
        self.buddies[handle] = buddy

        iface = self._iface(CONNECTION_INTERFACE_BUDDY_INFO)
        iface.GetProperties(handle,
                reply_handler=partial(self._update_buddy, handle),
                error_handler=partial(self.__error_cb, 'Buddy.Properties'),
                timeout=_QUERY_DBUS_TIMEOUT, byte_arrays=True)
        iface.GetActivities(handle,
                reply_handler=partial(self._update_buddy_activities, handle),
                error_handler=partial(self.__error_cb, 'GetActivities'),
                timeout=_QUERY_DBUS_TIMEOUT)
        iface.GetCurrentActivity(handle,
                reply_handler=partial(self._current_activity_updated, handle),
                error_handler=partial(self.__error_cb, 'GetCurrentActivity'),
                timeout=_QUERY_DBUS_TIMEOUT)

        return buddy

    def _popup_buddy(self, buddy):
        if buddy.online or not buddy.nick:
            return

        _logger.debug('Pop-up buddy: handle=%r', buddy.handle)

        buddy.online = True
        self.on_buddy_add(buddy)

    def _popin_buddy(self, buddy):
        if not buddy.online or buddy.is_owner():
            return

        _logger.debug('Pop-in buddy: handle=%r', buddy.handle)

        buddy.online = False
        self.on_buddy_remove(buddy)

    def _update_buddy(self, handle, props):
        buddy = self._checkin_buddy(handle)
        if buddy is None or buddy.is_owner():
            return

        _logger.debug('Update buddy: handle=%r props=%r', handle, props)

        color = props.get('color')
        if color:
            buddy.color = XoColor(color)

        key = props.get('key')
        if key:
            buddy.pubkey = key

        nick = props.get('nick') or \
                props.get(CONNECTION_INTERFACE_ALIASING + '/alias')
        if nick:
            buddy.nick = nick

        self._popup_buddy(buddy)

    def _checkin_activity(self, room_handle):
        activity = self.activities.get(room_handle)
        if activity is not None:
            return activity

        _logger.debug('Check-in activity: room=%r', room_handle)

        activity = ActivityModel(None, room_handle)
        self.activities[room_handle] = activity

        # XXX Using telepathy-salut-0.6.0 and telepathy-glib-0.14.10,
        # Salut doesn't send ActivitiesChanged signal, i.e., no way to know
        # what is the `activity_id` of newly appeared activity.
        #
        # Usecase (1):
        #   - start new shell session
        #   - make sure that salut is being used
        #   - share Chat instance
        #   - no Chat icon in F1
        #
        # Enabling this workaround for Gabble, breaks the logic
        #
        if 'salut' in self.object_path:
            iface = self._iface(CONNECTION_INTERFACE_BUDDY_INFO)
            iface.GetActivities(self._self_handle,
                    reply_handler=partial(self._update_buddy_activities,
                        self._self_handle),
                    error_handler=partial(self.__error_cb, 'GetActivities'))

        iface = self._iface(CONNECTION_INTERFACE_ACTIVITY_PROPERTIES)
        iface.GetProperties(room_handle,
                reply_handler=partial(self._update_activity, room_handle),
                error_handler=partial(self.__error_cb, 'Activity.Properties'))

        return activity

    def _checkout_activity(self, room_handle):
        if room_handle not in self.activities:
            return

        activity = self.activities.pop(room_handle)
        self._popin_activity(activity)

        _logger.debug('Check-out activity: activity_id=%r room=%r',
                activity.activity_id, room_handle)

    def _popup_activity(self, activity):
        if activity.shared or not activity.activity_id or \
                activity.bundle is None:
            return

        _logger.debug('Pop-up activity: activity_id=%r room=%r',
                activity.activity_id, activity.room_handle)

        activity.shared = True
        shell.get_model().add_shared_activity(
                activity.activity_id, activity.color)
        self.on_activity_add(activity)

        for buddy in activity.current_buddies:
            self.on_current_activity_change(buddy)

    def _popin_activity(self, activity):
        if not activity.shared:
            return

        _logger.debug('Pop-in activity: activity_id=%r room=%r',
                activity.activity_id, activity.room_handle)

        activity.shared = False
        shell.get_model().remove_shared_activity(activity.activity_id)
        self.on_activity_remove(activity)

    def _update_activity(self, room_handle, props):
        activity = self._checkin_activity(room_handle)

        _logger.debug('Update activity: activity_id=%r room=%s props=%r',
                activity.activity_id, room_handle, props)

        activity_id = props.get('activity_id')
        if activity_id:
            activity.activity_id = activity_id

        bundle_id = props.get('type')
        if bundle_id and activity.bundle is None:
            bundle = bundleregistry.get_registry().get_bundle(bundle_id)
            if bundle is None:
                _logger.warning('Ignore unknown %r shared activity', bundle_id)
                return
            activity.bundle = bundle

        name = props.get('name')
        if name:
            activity.name = name

        color = props.get('color')
        if color:
            activity.color = XoColor(color)

        private = props.get('color')
        if private:
            activity.private = private

        self._popup_activity(activity)

    def _current_activity_updated(self, buddy_handle, activity_id,
            room_handle):
        _logger.debug('Current activity: buddy=%r activity_id=%r room=%r',
                buddy_handle, activity_id, room_handle)

        buddy = self._checkin_buddy(buddy_handle)
        activity = self.activities.get(room_handle)
        if buddy is None or buddy.current_activity is activity:
            return

        if buddy.current_activity is None:
            activity.add_current_buddy(buddy)
            buddy.current_activity = activity
            bundle = activity.bundle
        else:
            bundle = buddy.current_activity.bundle
            buddy.current_activity.remove_current_buddy(buddy)
            buddy.current_activity = None

        if bundle is not None:
            self.on_current_activity_change(buddy)

    def _update_buddy_activities(self, buddy_handle, activities):
        buddy = self._checkin_buddy(buddy_handle)
        if buddy is None:
            return

        _logger.debug('Update buddy in activities: buddy=%r activities=%r',
                buddy_handle, activities)

        buddy_activities = []

        for activity_id, room_handle in activities:
            buddy_activities.append(activity_id)

            activity = self._checkin_activity(room_handle)
            self._update_activity(room_handle, {'activity_id': activity_id})

            buddy.activities.add(activity)
            activity.add_buddy(buddy)

            if buddy_handle == self._self_handle:
                self._set_current_activity(activity)
            else:
                # Sometimes we'll get CurrentActivityChanged before we get
                # to know about the activity so we miss the event. In that
                # case, request again the current activity for this buddy.
                iface = self._iface(CONNECTION_INTERFACE_BUDDY_INFO)
                iface.GetCurrentActivity(buddy_handle,
                        reply_handler=partial(self._current_activity_updated,
                            buddy_handle),
                        error_handler=partial(self.__error_cb,
                            'GetCurrentActivity'))

        for activity in buddy.activities.copy():
            if activity.activity_id in buddy_activities:
                continue

            buddy.activities.remove(activity)
            activity.remove_buddy(buddy)

            if not activity.buddies:
                self._checkout_activity(activity.room_handle)

        if not buddy.activities:
            self._current_activity_updated(buddy_handle, '', 0)

    def _set_current_activity(self, activity):
        if activity is None or self._self_buddy not in activity.buddies:
            room_handle = 0
            activity_id = ''
        else:
            room_handle = activity.room_handle
            activity_id = activity.activity_id
        if room_handle == self._last_room_handle:
            return

        _logger.debug('Set current activity: buddy=%r, activity_id=%r ' \
                'room=%r', self._self_handle, activity_id, room_handle)

        def reply_handler():
            self._last_room_handle = room_handle

        iface = self._iface(CONNECTION_INTERFACE_BUDDY_INFO)
        iface.SetCurrentActivity(activity_id, room_handle,
                reply_handler=reply_handler,
                error_handler=partial(self.__error_cb, 'SetCurrentActivity'))

    def __error_cb(self, funtion_name, error):
        _logger.warning('Telepathy error when calling %s: %s',
                funtion_name, error)

    def __SelfHandle_cb(self, self_handle):
        self._self_handle = self_handle
        _logger.debug('Self buddy: handle=%r', self_handle)
        self._checkin_owner()

        iface = self._iface(CONNECTION_INTERFACE_CONTACT_CAPABILITIES)
        if iface is not None:
            client_name = CLIENT + '.Sugar.FileTransfer'
            file_transfer_channel_class = {
                    CHANNEL + '.ChannelType': CHANNEL_TYPE_FILE_TRANSFER,
                    CHANNEL + '.TargetHandleType': HANDLE_TYPE_CONTACT}
            capabilities = []
            iface.UpdateCapabilities(
                    [(client_name, [file_transfer_channel_class],
                        capabilities)],
                    reply_handler=lambda: None,
                    error_handler=partial(self.__error_cb, 'Capabilities'))

        self._connect_to_signal(CONNECTION_INTERFACE_ALIASING,
                'AliasesChanged', self.__AliasesChanged_cb)

        self._connect_to_signal(CONNECTION_INTERFACE_SIMPLE_PRESENCE,
                'PresencesChanged', self.__PresencesChanged_cb)

        self._connect_to_signal(CONNECTION_INTERFACE_BUDDY_INFO,
                'PropertiesChanged', self._update_buddy, byte_arrays=True)
        self._connect_to_signal(CONNECTION_INTERFACE_BUDDY_INFO,
                'ActivitiesChanged', self._update_buddy_activities)
        self._connect_to_signal(CONNECTION_INTERFACE_BUDDY_INFO,
                'CurrentActivityChanged', self._current_activity_updated)

        self._connect_to_signal(CONNECTION_INTERFACE_ACTIVITY_PROPERTIES,
                'ActivityPropertiesChanged', self._update_activity)

        properties = {
                CHANNEL + '.ChannelType': CHANNEL_TYPE_CONTACT_LIST,
                CHANNEL + '.TargetHandleType': HANDLE_TYPE_LIST,
                CHANNEL + '.TargetID': 'subscribe',
                }
        iface = self._iface(CONNECTION_INTERFACE_REQUESTS)
        __, channel_path, __ = iface.EnsureChannel(
                dbus.Dictionary(properties, signature='sv'))

        channel = client.Channel(self._connection.service_name, channel_path)
        channel[dbus.PROPERTIES_IFACE].Get(CHANNEL_INTERFACE_GROUP, 'Members',
                reply_handler=self.__Members_cb,
                error_handler=partial(self.__error_cb, 'Members'))

    def __AliasesChanged_cb(self, aliases):
        _logger.debug('Got aliases: %r', aliases)

        for handle, alias in aliases:
            self._update_buddy(handle, {'nick': alias})

    def __PresencesChanged_cb(self, presences):
        for handle, (presence, status, __) in presences.iteritems():
            _logger.debug('Got presence: buddy=%r status=%r',
                    handle, status)
            buddy = self.buddies.get(handle)
            if presence == CONNECTION_PRESENCE_TYPE_OFFLINE:
                if buddy is not None:
                    self._popin_buddy(buddy)
            elif presence == CONNECTION_PRESENCE_TYPE_AVAILABLE:
                if buddy is not None:
                    self._popup_buddy(buddy)
                else:
                    self._checkin_buddy(handle)

    def __active_activity_changed_cb(self, model, home_activity):
        if self._self_handle is None:
            return

        activity_id = home_activity.get_activity_id()
        for activity in self.activities.values():
            if activity.activity_id == activity_id:
                break
        else:
            activity = None

        self._set_current_activity(activity)

    def __Members_cb(self, handles):
        self._iface(CONNECTION_INTERFACE_SIMPLE_PRESENCE).GetPresences(
                handles,
                reply_handler=self.__PresencesChanged_cb,
                error_handler=partial(self.__error_cb, 'GetPresences'))
