/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
static gchar	*error_msg;

#endif

/* are these values ok? */
#define HTTP_REQUEST_MAXLEN		(8 * 1024 - 1)
#define HTTP_HEADER_FIELD_MAXLEN	(2 * 1024 - 1)
#define RECV_CHUNK_LEN			(16 * 1024 - 1)
#define CONNECT_TIMEOUT			5
#define SEND_RECV_TIMEOUT		1
#define MAX_HTTP_REDIRECT		8
#define DEFAULT_HTTP_PORT_STR		"80"
#define DEFAULT_HTTPS_PORT_STR		"443"
#define	PORT_STR_MAXLEN			PROXY_PORT_MAXLEN
#define TAB				'\x09'

#ifndef G_OS_WIN32
# define CLOSE_SOCK	close(sock)
#else
# define CLOSE_SOCK	closesocket(sock)
#endif
/* S_MOD -> true in multiple selection mode / false otherwise */
#define S_MOD		(get_ticker_env()->selection_mode == MULTIPLE)

static void remove_chunk_info(char **response)
{
	char	*response2;
	int	i, j;

	response2 = l_str_new(*response);
	response2[0] = '\0';
	i = go_after_next_empty_line(*response);
	while ((j = (int)get_http_chunk_size(*response + i)) > 0) {
		i += go_after_next_cr_lf(*response + i);
		str_n_cat(response2, *response + i, j);
		i += j;
		i += go_after_next_cr_lf(*response + i);
	}
	free2(*response);
	*response = response2;
}

/* 'quickly check feed format' (xml/rss2.0/atom), get shift to beginnig
 * and cut off trailing part */
static int format_quick_check(char *response, int *shift)
{
	int	i;

	/* this may return false positives so disabled
	if (strcmp(get_http_header_value("Content-Type", response), "text/xml") != 0)
		return FEED_FORMAT_ERROR;*/
	i = 0;
	while (strncmp(response + i, "<rss", 4) != 0 && response[i] != '\0')	/* rss 2.0 */
		i++;
	if (response[i] != '\0') {
		*shift = i;
		while (strncmp(response + i, "/rss>", 5) != 0 && response[i] != '\0')
			i++;
		if (response[i] != '\0')
			response[i + 5] = '\0';
	 	else
			return FEED_FORMAT_ERROR;
	} else {
		i = 0;
		while (strncmp(response + i, "<feed", 5) != 0 && response[i] != '\0')	/* atom */
			i++;
		if (response[i] != '\0') {
			*shift = i;
			while (strncmp(response + i, "/feed>", 6) != 0 && response[i] != '\0')
				i++;
			if (response[i] != '\0')
				response[i + 6] = '\0';
			else
				return FEED_FORMAT_ERROR;
		} else
			return FEED_FORMAT_ERROR;
	}
	i = 0;
	while (strncmp(response + i, "<?xml", 5) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
	*shift = i;
	return OK;
}

/* get host from url but if connecting through proxy */
#ifndef G_OS_WIN32
int connect_with_url(int *sock, char *url)
#else
int connect_with_url(SOCKET *sock, char *url)
#endif
{
	char		host[FILE_NAME_MAXLEN + 1];
	char		port_num[PORT_STR_MAXLEN + 1];
	char		tmp[256];
	static int	connect_fail_count = 0;

	if (!get_use_proxy()) {
		str_n_cpy(host, get_host_from_url(url), FILE_NAME_MAXLEN);
		if (strcmp(get_scheme_from_url(url), "http") == 0)
			str_n_cpy(port_num, DEFAULT_HTTP_PORT_STR, PORT_STR_MAXLEN);
		/* https is not supported so far */
		else if (strcmp(get_scheme_from_url(url), "https") == 0) {
			/*str_n_cpy(port_num, DEFAULT_HTTPS_PORT_STR, PORT_STR_MAXLEN);*/
			warning(url, ":\nHTTPS protocol not currently supported",
				"", "", S_MOD);
			return HTTP_UNSUPPORTED_SCHEME;
		} else if (strcmp(get_scheme_from_url(url), "file") == 0)
			;/* do nothing here */
		else if (get_scheme_from_url(url)[0] == '\0') {
			warning("No scheme found in URL:", url, "", "", S_MOD);
			return HTTP_UNSUPPORTED_SCHEME;
		} else {
			warning("Unsupported or unknown scheme in URL:",
				get_scheme_from_url(url), "", "", S_MOD);
			return HTTP_UNSUPPORTED_SCHEME;
		}
	} else {
		/* connect via proxy */
		str_n_cpy(host, get_proxy_host(), FILE_NAME_MAXLEN);
		str_n_cpy(port_num, get_proxy_port(), PORT_STR_MAXLEN);
	}
#ifndef G_OS_WIN32
	if ((*sock = connect_to_host(host, port_num)) > -1) {

#else
	if ((*sock = connect_to_host(host, port_num)) != INVALID_SOCKET) {
#endif
		connect_fail_count = 0;
		return OK;
	} else {
		if (get_use_proxy()) {
			snprintf(tmp, 256, "%s:%s", get_proxy_host(), get_proxy_port());
			warning("Can't connect to proxy:", tmp, "", "", FALSE);
			current_feed();
			connection_settings(PROXY_PAGE);
		} else if (get_ticker_env()->selection_mode == SINGLE)
			warning("Can't connect to host:", host, "", "", FALSE);
		else
			fprintf(STD_ERR, "Can't connect to host: %s\n", host);
		if (++connect_fail_count >= CONNECT_FAIL_MAX) {
			connect_fail_count = 0;
			return CONNECT_TOO_MANY_ERRORS;
		} else
			return SOCK_CANT_CONNECT;
	}
}

/* we replace resrc->id with modified url and file_name = downloaded resource
 * maxlen of file_name, url, new_url = FILE_NAME_MAXLEN */
int fetch_resource(const char *resrc_id, const char *file_name, char *url)
{
#ifndef G_OS_WIN32
	int	sock;
#else
	SOCKET	sock;
	gchar	*win32_filename;
#endif
	int	status, recv_status;
	char	*response;
	char	*new_url;
	char	header_field[HTTP_HEADER_FIELD_MAXLEN + 1];
	FILE	*fp;
	int	content_length;
	GError	*error = NULL;
	int	i, j;

	if (strncmp(resrc_id, "file://", strlen("file://")) == 0) {
		/* 'file' scheme */
		str_n_cpy(url, resrc_id + strlen("file://"), FILE_NAME_MAXLEN - strlen("file://"));
		if (strncmp(url, "localhost/", strlen("localhost/")) == 0)
			i = strlen("localhost/") - 1;
		else if (url[0] == '/')
			i = 0;
		else
			return RESOURCE_ERROR;

#ifndef G_OS_WIN32
		if (g_file_get_contents(url + i, &response, NULL, &error)) {
#else
		win32_filename = g_win32_locale_filename_from_utf8(url + i);
		if (g_file_get_contents(win32_filename, &response, NULL, &error)) {
			g_free(win32_filename);
#endif
			if ((j = format_quick_check(response, &i)) == OK) {
				if (g_file_set_contents(file_name, response + i, -1, &error)) {
					g_free(response);
					return OK;
				} else {
					g_free(response);
					if (error != NULL) {
						warning(error->message, "", "", "", FALSE);
						g_error_free(error);
					} else
						warning("Can't create file:", file_name, "", "", FALSE);
					return CREATE_FILE_ERROR;
				}
			} else {
				g_free(response);
				return j;
			}
		} else {
#ifdef G_OS_WIN32
			g_free(win32_filename);
#endif
			if (error != NULL) {
				warning(error->message, "", "", "", S_MOD);
				g_error_free(error);
			} else
				warning("Can't open file:", url + i, "", "", S_MOD);
			return OPEN_FILE_ERROR;
		}
	} else
		/* 'http' scheme ('https' not supported so far) */
		str_n_cpy(url, resrc_id, FILE_NAME_MAXLEN);
	if ((i = connect_with_url(&sock, url)) == OK) {
		if ((status = get_http_response(sock, "GET", "", url, &new_url,
				&response, &recv_status)) == SEND_ERROR || status == RECV_ERROR) {
			if (recv_status == CONNECTION_CLOSED_BY_SERVER || recv_status == SOCK_SHOULD_BE_CLOSED)
				CLOSE_SOCK;
			return i;
		}
	} else
		return i;
	while (TRUE) {
		/* status checked here are those returned by get_http_response()
		 * so we must make sure they all match */
		if (status == HTTP_CONTINUE) {
			free2(response);
			status = get_http_response(sock, "GET", "", url, &new_url, &response, &recv_status);
			continue;
		} else if (status == HTTP_SWITCH_PROTO) {
			if (strcmp(get_http_header_value("Upgrade", response), "HTTP") == 0)
				warning(url, ":\nSwitching to HTTP protocol requested for this connection",
					"-", "Request currently not supported", S_MOD);
			else if (strcmp(get_http_header_value("Upgrade", response), "HTTPS") == 0)
				warning(url, ":\nSwitching to HTTPS protocol requested for this connection",
					"-", "Request currently not supported", S_MOD);
			return status;
		} else if (status == OK) {
			if (strcmp(get_http_header_value("Transfer-Encoding", response), "chunked") == 0) {
				/* chunked transfer encoding */
				remove_chunk_info(&response);
			} else if ((content_length = atoi(get_http_header_value("Content-Length", response))) > 0) {
				/* 'Content-Length' = length of entity body is mandatory
				 * but if 'Transfer-Encoding: chunked' */
				/* do nothing */
			} else {
#ifdef VERBOSE_OUTPUT		/* only if verbose because notification only, nothing done */
				fprintf(STD_ERR, "No 'Transfer-Encoding' nor 'Content-Length' "
					"header field in HTTP response\n");
#endif
			}
			/* quickly 'check format' (xml and rss2.0/atom stuff),
			 * shift to beginnig and cut off trailing part */
			if ((j = format_quick_check(response, &i)) != OK) {
				free2(response);
				return j;
			}
			/* if ok, save to dump file */
			if ((fp = g_fopen(file_name, "wb+")) != NULL) {
				fprintf(fp, "%s", response + i);
				fclose(fp);
				free2(response);
				CLOSE_SOCK;
				return OK;
			} else {
				warning("Can't create file:", file_name,
					"", "", FALSE);
				fclose(fp);
				free2(response);
				CLOSE_SOCK;
				return CREATE_FILE_ERROR;
			}
		} else if (status == HTTP_MOVED) {
			i = 0;
			do {
				str_n_cpy(url, new_url, FILE_NAME_MAXLEN);
				if (strcmp(get_http_header_value("Connection", response), "close") == 0 ||\
						strcmp(get_scheme_from_url(url), "https") == 0) {
					free2(response);
					CLOSE_SOCK;
					if (strcmp(get_scheme_from_url(url), "https") == 0) {
						warning(resrc_id, ":\nHTTPS protocol requested for this connection",
							"(following HTTP moved) -", "Not currently supported", S_MOD);
						return HTTP_ERROR;	/* ???? */
					}
					if ((j = connect_with_url(&sock, url)) != OK)
						return j;
				} else
					free2(response);
			} while ((status = get_http_response(sock, "GET", "", url,
				&new_url, &response, &recv_status)) == HTTP_MOVED &&\
				++i < MAX_HTTP_REDIRECT);
			if (status != HTTP_MOVED && i <= MAX_HTTP_REDIRECT)
				continue;
			else {
				warning(resrc_id, ":\nToo many HTTP redirects", "", "", S_MOD);
				free2(response);
				CLOSE_SOCK;
				return HTTP_TOO_MANY_REDIRECTS;
			}
		} else if (status == HTTP_USE_PROXY) {
			warning("Resource: ", url, "\nmust be accessed through proxy.\nProxy host:",
				new_url, FALSE);
			str_n_cpy(get_proxy_host(), new_url, PROXY_HOST_MAXLEN);
			free2(response);
			CLOSE_SOCK;
			connection_settings(PROXY_PAGE);
			current_feed();
			return status;
		} else if (status == HTTP_PROXY_AUTH_REQUIRED) {
			/* proxy authentication */
			free2(response);
			CLOSE_SOCK;
			if (get_use_proxy_auth()) {
				if ((i = connect_with_url(&sock, url)) != OK)
					return i;
				snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
					"Proxy-Authorization: Basic %s\r\n\r\n",
					get_proxy_auth_str());
				if ((status = get_http_response(sock, "GET", header_field, url, &new_url,
						&response, &recv_status)) == HTTP_PROXY_AUTH_REQUIRED) {
					warning("Proxy authentication failed for:",
						get_proxy_host(), "", "", FALSE);
					free2(response);
					CLOSE_SOCK;
					if (connection_settings(PROXY_PAGE) == GTK_RESPONSE_OK) {
						if ((i = connect_with_url(&sock, url)) != OK)
							return i;
						status = get_http_response(sock, "GET", header_field, url,
							&new_url, &response, &recv_status);
					} else
						return HTTP_NO_PROXY_AUTH_CREDENTIALS;
				}
				continue;
			} else {
				warning("Proxy authentication required for:",
					get_proxy_host(), "", "", FALSE);
				if (connection_settings(PROXY_PAGE) == GTK_RESPONSE_OK &&\
					get_use_proxy_auth()) {
					if ((i = connect_with_url(&sock, url)) != OK)
						return i;
					snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
						"Proxy-Authorization: Basic %s\r\n\r\n",
						get_proxy_auth_str());
					status = get_http_response(sock, "GET", header_field, url,
						&new_url, &response, &recv_status);
					continue;
				} else
					return HTTP_NO_PROXY_AUTH_CREDENTIALS;
			}
		} else if (status == HTTP_UNAUTHORIZED) {
			/* http authentication - only basic so far */
			free2(response);
			CLOSE_SOCK;
			if (get_use_authentication()) {
				if ((i = connect_with_url(&sock, url)) != OK)
					return i;
				snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
					"Authorization: Basic %s\r\n\r\n",
					get_http_auth_str());
				if ((status = get_http_response(sock, "GET", header_field,
						url, &new_url, &response, &recv_status)) == HTTP_UNAUTHORIZED) {
					warning("HTTP authentication failed for:", url, "", "", FALSE);
					free2(response);
					CLOSE_SOCK;
					if (connection_settings(AUTH_PAGE) == GTK_RESPONSE_OK) {
						if ((i = connect_with_url(&sock, url)) != OK)
							return i;
						status = get_http_response(sock, "GET", header_field,
							url, &new_url, &response, &recv_status);
					} else
						return HTTP_NO_AUTH_CREDENTIALS;
				}
				continue;
			} else {
				warning("HTTP authentication required for:",
					url, "", "", FALSE);
				if (connection_settings(AUTH_PAGE) == GTK_RESPONSE_OK &&\
					get_use_authentication()) {
					if ((i = connect_with_url(&sock, url)) != OK)
						return i;
					snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
						"Authorization: Basic %s\r\n\r\n",
						get_http_auth_str());
					status = get_http_response(sock, "GET", header_field,
						url, &new_url, &response, &recv_status);
					continue;
				} else
					return HTTP_NO_AUTH_CREDENTIALS;
			}
		} else if (status == HTTP_BAD_REQUEST) {
			warning(url, ":\nBad Request", "", "", S_MOD);
			free2(response);
			CLOSE_SOCK;
			return status;
		} else if (status == HTTP_FORBIDDEN) {
			warning(url, ":\nForbidden", "", "", S_MOD);
			free2(response);
			CLOSE_SOCK;
			return status;
		} else if (status == HTTP_NOT_FOUND) {
			warning(url, ":\nNot found", "", "", S_MOD);
			free2(response);
			CLOSE_SOCK;
			return status;
		} else if (status == HTTP_INT_SERVER_ERROR) {
			warning(url, ":\nInternal server error", "", "",
				S_MOD);
			free2(response);
			CLOSE_SOCK;
			return status;
		} else if (status == HTTP_NO_STATUS_CODE) {
			warning(url, ":\nNo HTTP status code returned",
				"", "", S_MOD);
			free2(response);
			CLOSE_SOCK;
			return status;
		} else {
			warning(url, ":\nHTTP response status code: ",
				itoa2(status - 1000), "", S_MOD);
			if (response != NULL)
				free2(response);
			CLOSE_SOCK;
			return HTTP_ERROR;
		}
	}
}

/* rq_str may contain header fields(s) separated and ended by "\r\n" */
const char *build_http_request(const char *method, const char *path,\
		const char *host, const char* rq_str)
{
	static char	str[HTTP_REQUEST_MAXLEN + 1];

	snprintf(str, HTTP_REQUEST_MAXLEN,
		"%s %s HTTP/1.1\r\n"	/* start line with method and path */
		"Host: %s\r\n"		/* mandatory host header field */
		"User-Agent: "APP_NAME"-"APP_VERSION_NUMBER"\r\n"
		"%s\r\n",		/* optional extra header field(s) */
		method, path, host, rq_str);
	return (const char *)str;
}

/*
 * rq_str may contain header field(s) separated and ended by "\r\n"
 * must 'free2' response afterwards
 */
int get_http_response(int sock, const char *rq_method, const char *rq_str,
	const char *rq_url, char **new_rq_url, char **response, int *recv_status)
{
	char		*str;
	static char	location[FILE_NAME_MAXLEN + 1];
	int		bytes_sent, bytes_received, status_code;

	location[0] = '\0';
	*new_rq_url = (char *)location;
	*response = NULL;
	str = (char *)build_http_request(
		rq_method,
		(get_use_proxy() ? rq_url : get_path_from_url(rq_url)),	/* path or full (absolute) url if using proxy */
		/* is that correct when using proxy? */
		get_host_from_url(rq_url),
		rq_str);
	if ((bytes_sent = send_full(sock, (const char *)str)) >= 0) {
		if ((*response = recv_full(sock, &bytes_received, recv_status)) != NULL) {
			if ((status_code = get_http_status_code(*response)) == 100)
				return HTTP_CONTINUE;
			else if (status_code == 101)
				return HTTP_SWITCH_PROTO;
			else if (status_code == 200)
				return OK;
			else if (	/* 'multiple choices' */
					status_code == 300 ||\
					/* 'moved permanently' */
					status_code == 301 ||\
					/* 'found' */
					status_code == 302 ||\
					/* 'see other' */
					status_code == 303 ||\
					/* 'not modified' */
					status_code == 304 ||\
					/* 'moved temporarily' */
					status_code == 307) {
				str_n_cpy(location, get_http_header_value("Location", *response),
					FILE_NAME_MAXLEN);
				return HTTP_MOVED;	/* must use the new url in 'Location' */
			} else if (status_code == 305)
				return HTTP_USE_PROXY;
			else if (status_code == 400)
				return HTTP_BAD_REQUEST;
			else if (status_code == 401)
				return HTTP_UNAUTHORIZED;
			else if (status_code == 403)
				return HTTP_FORBIDDEN;
			else if (status_code ==  404)
				return HTTP_NOT_FOUND;
			else if (status_code == 407)
				return HTTP_PROXY_AUTH_REQUIRED;
			else if (status_code == 500)
				return HTTP_INT_SERVER_ERROR;
			else if (status_code == -1)
				return HTTP_NO_STATUS_CODE;
			else
				return status_code + 1000;
		} else
			return RECV_ERROR;
	} else
		return SEND_ERROR;
}

/* return -1 if error */
int get_http_status_code(const char *response)
{
	char	status_code[4];
	int	i = 0;

	while (response[i] != ' ' && response[i] != TAB) {
		if (response[i] == '\n' || response[i] == '\r' || response[i] == '\0')
			return -1;
		else
			i++;
	}
	while (response[i] == ' ' || response[i] == TAB)
		i++;
	if (response[i] == '\n' || response[i] == '\r' || response[i] == '\0')
		status_code[0] = '\0';
	else
		str_n_cpy(status_code, response + i, 3);
	return atoi(status_code);
}

/* get header_value as string (empty one if header name not found) */
const char *get_http_header_value(const char *header_name, const char *response)
{
	static char	header_value[1024];
	int		len = strlen(header_name), i = 0;

	while (strncasecmp(response + i, header_name, len) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0') {
		i += len;
		if (response[i] != ':') {
				header_value[0] = '\0';
				return (const char *)header_value;
		} else
			i++;
		while (response[i] == ' ' || response[i] == TAB) {
			if (response[i] == '\n' || response[i] == '\r' ||\
					response[i] == '\0') {
				header_value[0] = '\0';
				return (const char *)header_value;
			} else
				i++;
		}
		str_n_cpy(header_value, response + i, 1023);
		i = 0;
		while (header_value[i] != '\0' && header_value[i] != ' ' &&\
			header_value[i] != TAB && header_value[i] != '\n' &&
			header_value[i] != '\r' && i < 1023)
			i++;
		header_value[i] = '\0';
	} else
		header_value[0] = '\0';
	return (const char *)header_value;
}

/* return 0 if none found */
int go_after_next_cr_lf(const char *response)
{
	int	i = 0;

	while (strncmp(response + i, "\r\n", 2) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
		return i + 2;
	else
		return 0;
}

/* return 0 if none found */
int go_after_next_empty_line(const char *response)
{
	int	i = 0;

	while (strncmp(response + i, "\r\n\r\n", 4) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
		return i + 4;
	else
		return 0;
}

/* response must point to chunk size hexa str or preceeding space(s)
 * return -1 if invalid chunk size format (ie not an hexa str) */
int get_http_chunk_size(const char *response)
{
	char	size_str[32], *tailptr;
	int	size, i = 0;

	while (response[i] == ' ' || response[i] == TAB ||\
			response[i] == '\n' || response[i] == '\r')
		i++;
	str_n_cpy(size_str, response + i, 31);
	i = 0;
	while (size_str[i] != ' ' && size_str[i] != TAB &&\
			size_str[i] != '\n' && size_str[i] != '\r' &&\
			size_str[i] != ';' && size_str[i] != '\0' &&\
			i < 31)
		i++;
	size_str[i] = '\0';
	size = (int)strtoul(size_str, &tailptr, 16);
	if (tailptr == size_str) {
		fprintf(STD_ERR, "Invalid hexadecimal value in HTTP chunk size: %s\n",
			size_str);
		return -1;
	} else
		return size;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> 'http'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_scheme_from_url(const char* url)
{
	static char	str[16];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0')
		str_n_cpy(str, url, MIN(i, 15));
	else
		str[0] = '\0';
	return (const char *)str;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> 'www.sth1.org'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_host_from_url(const char* url)
{
	static char	str[FILE_NAME_MAXLEN + 1];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0') {
		str_n_cpy(str, url + i + 3, FILE_NAME_MAXLEN);
		i = 0;
		while (str[i] != '\0' && str[i] != '/' && i < FILE_NAME_MAXLEN)
			i++;
		str[i] = '\0';
	} else
		str[0] = '\0';
	return (const char *)str;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> '/sth2/sth2.xml'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_path_from_url(const char* url)
{
	static char	str[FILE_NAME_MAXLEN + 1];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0') {
		i += 3;
		while (url[i] != '\0' && url[i] != '/')
			i++;
		if (url[i] != '\0')
			str_n_cpy(str, url + i, FILE_NAME_MAXLEN);
		else
			str[0] = '\0';
	} else
		str[0] = '\0';
	return (const char *)str;
}

#ifndef G_OS_WIN32
/* can use IPv4 or IPv6 */
static void *get_in_addr(struct sockaddr *s_a)
{
	if (s_a->sa_family == AF_INET)
		return &(((struct sockaddr_in *)s_a)->sin_addr);
	else
		return &(((struct sockaddr_in6 *)s_a)->sin6_addr);
}
#endif

/* return socket fd (> 0) / -1 if error */
#ifndef G_OS_WIN32
int connect_to_host(const char *host, const char *portnum_str)
#else
SOCKET connect_to_host(const char *host, const char *portnum_str)
#endif
{
#ifndef G_OS_WIN32
	int			sock;
	char			ipa_str[INET6_ADDRSTRLEN];
#else
	SOCKET			sock;
	u_long			i_mode = 1;	/* != 0 to enable non-blocking mode */
#endif
	struct addrinfo		hints, *server_info, *ptr;
	fd_set			readset, writeset;
	struct timeval		timeout;
	int			s_opt_value;
	socklen_t		s_opt_len = sizeof(int);
	int			i;

	/* addrinfo stuff */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
/*#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Resolving %s ... ", host);
#endif*/
	if ((i = getaddrinfo(host, portnum_str, &hints, &server_info)) != 0) {
		if (get_use_proxy() || get_ticker_env()->selection_mode == SINGLE) {
#ifndef G_OS_WIN32
			warning("getaddrinfo() error:", host, ":", gai_strerror(i), FALSE);
#else
			error_msg = g_win32_error_message(WSAGetLastError());
			warning("getaddrinfo() error:", host, ":", error_msg, FALSE);
			g_free(error_msg);
#endif
		} else {
#ifndef G_OS_WIN32
			fprintf(STD_ERR, "getaddrinfo() error: %s: %s\n", host, gai_strerror(i));
#else
			error_msg = g_win32_error_message(WSAGetLastError());
			fprintf(STD_ERR, "getaddrinfo() error: %s: %s\n", host, error_msg);
			g_free(error_msg);
#endif
		}
		return -1;
	}
/*#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Done\n");
#endif*/
	/* we get a list */
	for (ptr = server_info; ptr != NULL; ptr = ptr->ai_next) {
		/* create socket */
#ifndef G_OS_WIN32
		if ((sock = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol)) == -1) {
			fprintf(STD_ERR, "Error: %s\n", strerror(errno));
#else
		if ((sock = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol)) == INVALID_SOCKET) {
			error_msg = g_win32_error_message(WSAGetLastError());
			fprintf(STD_ERR, "Error: %s\n", error_msg);
			g_free(error_msg);
#endif
			continue;
		}
		/* set socket in non-blocking mode */
#ifndef G_OS_WIN32
		if ((i = fcntl(sock, F_GETFL, 0)) == -1) {
			fprintf(STD_ERR, "fcntl() error: %s\n", strerror(errno));
			close(sock);
			break;
		} else if (fcntl(sock, F_SETFL, i | O_NONBLOCK) == -1) {
			fprintf(STD_ERR, "fcntl() error: %s\n", strerror(errno));
			close(sock);
			break;
		}
#else
		if (ioctlsocket(sock, FIONBIO, &i_mode) != NO_ERROR) {
			error_msg = g_win32_error_message(WSAGetLastError());
			fprintf(STD_ERR, "ioctlsocket() error %s\n", error_msg);
			g_free(error_msg);
			closesocket(sock);
			break;
		}
#endif
		/* get IP addr from server_info */
#ifdef VERBOSE_OUTPUT
#ifndef G_OS_WIN32
		inet_ntop(ptr->ai_family, get_in_addr((struct sockaddr *)ptr->ai_addr),
			ipa_str, sizeof(ipa_str));
		fprintf(STD_OUT, "Connecting to %s (%s) ... ", ipa_str, host);
#else
		/* available only on vista and above but we don't really need that (do we?) so disabled
		InetNtop(ptr->ai_family, get_in_addr((struct sockaddr *)ptr->ai_addr), s, sizeof(s));*/
		fprintf(STD_OUT, "Connecting to %s ... ", host);
#endif
		fflush(stdout);
#endif
		/* connect */
#ifndef G_OS_WIN32
		if ((i = connect(sock, ptr->ai_addr, ptr->ai_addrlen)) == -1 && errno == EINPROGRESS) {
#else
		if ((i = connect(sock, ptr->ai_addr, ptr->ai_addrlen)) == SOCKET_ERROR &&\
				WSAGetLastError() == WSAEWOULDBLOCK) {
#endif
			/* as socket is in non-blocking mode, we must use select() */
			FD_ZERO(&readset);
			FD_ZERO(&writeset);
			FD_SET(sock, &readset);
			FD_SET(sock, &writeset);
			timeout.tv_sec = CONNECT_TIMEOUT;
			timeout.tv_usec = 0;
#ifndef G_OS_WIN32
			if ((i = select(sock + 1, &readset, &writeset, NULL, &timeout)) == -1) {
				fprintf(STD_ERR, "select() error: %s\n", strerror(errno));
#else
			if ((i = select(sock + 1, &readset, &writeset, NULL, &timeout)) == SOCKET_ERROR) {
				error_msg = g_win32_error_message(WSAGetLastError());
				fprintf(STD_ERR, "select() error: %s\n", error_msg);
				g_free(error_msg);

#endif
			} else if (i == 0) {
				fprintf(STD_ERR, "Timed out\n");
			} else if (FD_ISSET(sock, &readset) || FD_ISSET(sock, &writeset)) {
#ifndef G_OS_WIN32
				if (getsockopt(sock, SOL_SOCKET, SO_ERROR,
						(void *)(&s_opt_value), &s_opt_len) == -1) {
					fprintf(STD_ERR, "getsockopt() error: %s\n", strerror(errno));
#else
				if (getsockopt(sock, SOL_SOCKET, SO_ERROR,
						(void *)(&s_opt_value), &s_opt_len) == SOCKET_ERROR) {
					error_msg = g_win32_error_message(WSAGetLastError());
					fprintf(STD_ERR, "getsockopt() error: %s\n", error_msg);
					g_free(error_msg);
#endif
				} else if (s_opt_value == 0) {
#ifdef VERBOSE_OUTPUT
					fprintf(STD_OUT, "OK\n");
#endif
					freeaddrinfo(server_info);
					return sock;
				}
#ifndef G_OS_WIN32
				fprintf(STD_ERR, "getsockopt(): %s\n", strerror(s_opt_value));
#else
				error_msg = g_win32_error_message(s_opt_value);
				fprintf(STD_ERR, "getsockopt(): %s\n", error_msg);
				g_free(error_msg);
#endif
				CLOSE_SOCK;
				break;
			}
			CLOSE_SOCK;
			break;
		} else if (i == 0) {
#ifdef VERBOSE_OUTPUT
			fprintf(STD_OUT, "OK\n");
#endif
			freeaddrinfo(server_info);
			return sock;
		} else {

#ifndef G_OS_WIN32
			fprintf(STD_ERR, "connect() error: %s\n", strerror(errno));
#else
			error_msg = g_win32_error_message(WSAGetLastError());
			fprintf(STD_ERR, "connect() error: %s\n", error_msg);
			g_free(error_msg);
#endif
			CLOSE_SOCK;
			/*redundant
			continue;*/
		}
	}
	freeaddrinfo(server_info);
	return -1;
}

#ifndef G_OS_WIN32
int writable_data_is_available_on_socket(int sock)
#else
int writable_data_is_available_on_socket(SOCKET sock)
#endif
{
	fd_set		writeset;
	struct timeval	timeout;
	int		i;

	FD_ZERO(&writeset);
	FD_SET(sock, &writeset);
	timeout.tv_sec = SEND_RECV_TIMEOUT;
	timeout.tv_usec = 0;
#ifndef G_OS_WIN32
	if ((i = select(sock + 1, NULL, &writeset, NULL, &timeout)) == -1) {
		fprintf(STD_ERR, "select() error: %s\n", strerror(errno));
#else
	if ((i = select(sock + 1, NULL, &writeset, NULL, &timeout)) == SOCKET_ERROR) {
		error_msg = g_win32_error_message(WSAGetLastError());
		fprintf(STD_ERR, "select() error: %s\n", error_msg);
		g_free(error_msg);
#endif
		return SELECT_ERROR;
	} else if (i == 0) {
		return SELECT_TIMED_OUT;
	} else {
		if (FD_ISSET(sock, &writeset))
			return SELECT_TRUE;
		else
			return SELECT_FALSE;
	}
}

#ifndef G_OS_WIN32
int readable_data_is_available_on_socket(int sock)
#else
int readable_data_is_available_on_socket(SOCKET sock)
#endif
{
	fd_set		readset;
	struct timeval	timeout;
	int		i;

	FD_ZERO(&readset);
	FD_SET(sock, &readset);
	timeout.tv_sec = SEND_RECV_TIMEOUT;
	timeout.tv_usec = 0;
#ifndef G_OS_WIN32
	if ((i = select(sock + 1, &readset, NULL, NULL, &timeout)) == -1) {
		fprintf(STD_ERR, "select() error: %s\n", strerror(errno));
#else
	if ((i = select(sock + 1, &readset, NULL, NULL, &timeout)) == SOCKET_ERROR) {
		error_msg = g_win32_error_message(WSAGetLastError());
		fprintf(STD_ERR, "select() error: %s\n", error_msg);
		g_free(error_msg);
#endif
		return SELECT_ERROR;
	} else if (i == 0) {
		return SELECT_TIMED_OUT;
	} else {
		if (FD_ISSET(sock, &readset))
			return SELECT_TRUE;
		else
			return SELECT_FALSE;
	}
}

/* return n bytes sent or -1 if error (connection closed by server or ?) */
#ifndef G_OS_WIN32
int send_full(int sock, const char *str)
{
	int	len = strlen(str), i, j = 0;

	while (writable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = send(sock, str + j, len, 0)) > 0) {
			j += i;
			len -= i;
			if (len == 0)
				break;
		} else if (i == 0) {
			/* something to do? */
		} else if (i == -1) {
			j = -1;
			if (errno == EPIPE) {
#ifdef VERBOSE_OUTPUT
				fprintf(STD_ERR, "Connection closed by server\n");
#endif
			} else
				fprintf(STD_ERR, "send() error: %s\n", strerror(errno));
			break;
		}
	}
	return j;
}
#else
int send_full(SOCKET sock, const char *str)
{
	int	len = strlen(str), i, j = 0;

	while (writable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = send(sock, str + j, len, 0)) != SOCKET_ERROR) {
			if (i > 0) {
				j += i;
				len -= i;
				if (len == 0)
					break;
			} else {
				/* something to do? */
			}
		} else {
			j = -1;
			if ((i = WSAGetLastError()) == WSAECONNRESET || i == WSAECONNABORTED ||\
					i == WSAESHUTDOWN) {
#ifdef VERBOSE_OUTPUT
				fprintf(STD_ERR, "Connection closed by server\n");
#endif
			} else {
				error_msg = g_win32_error_message(i);
				fprintf(STD_ERR, "send() error: %s\n", error_msg);
				g_free(error_msg);
			}
			break;
		}
	}
	return j;
}
#endif

/*
 * return response = recv_full(socket, &bytes_received, &status) or NULL if error
 * -> status = ok, connection_closed_by_server or recv_error
 * -> allocate memory for response (must be freed afterwards with free2() if != NULL)
 */
#ifndef G_OS_WIN32
char *recv_full(int sock, int *bytes_received, int *status)
{
	char	*response, *full_response;
	int	i;

	*bytes_received = 0;
	*status = RECV_ERROR;
	response = malloc2(RECV_CHUNK_LEN + 1);
	response[0] = '\0';
	full_response = l_str_new(response);
	while (readable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = recv(sock, response, RECV_CHUNK_LEN, 0)) > 0) {
			response[MIN(i, RECV_CHUNK_LEN)] = '\0';
			full_response = l_str_cat(full_response, response);
			*bytes_received += i;
			*status = OK;
		} else if (i == 0) {
			*status = CONNECTION_CLOSED_BY_SERVER;
#ifdef VERBOSE_OUTPUT
			fprintf(STD_ERR, "Connection closed by server\n");
#endif
			break;
		} else if (i == -1) {
			*status = RECV_ERROR;
			l_str_free(full_response);
			full_response = NULL;
#ifdef VERBOSE_OUTPUT
			fprintf(STD_ERR, "recv() error: %s\n", strerror(errno));
#endif
			break;
		}
	}
	free2(response);
	return full_response;
}
#else
char *recv_full(SOCKET sock, int *bytes_received, int *status)
{
	char	*response, *full_response;
	int	i;

	*bytes_received = 0;
	*status = RECV_ERROR;
	response = malloc2(RECV_CHUNK_LEN + 1);
	response[0] = '\0';
	full_response = l_str_new(response);
	while (readable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = recv(sock, response, RECV_CHUNK_LEN, 0)) != SOCKET_ERROR) {
			if (i > 0) {
				response[MIN(i, RECV_CHUNK_LEN)] = '\0';
				full_response = l_str_cat(full_response, response);
				*bytes_received += i;
				*status = OK;
			} else {
				*status = CONNECTION_CLOSED_BY_SERVER;
#ifdef VERBOSE_OUTPUT
				fprintf(STD_ERR, "Connection closed by server\n");
#endif
				break;
			}
		} else {
			if ((i = WSAGetLastError()) == WSAECONNRESET || i == WSAECONNABORTED ||\
					i == WSAESHUTDOWN) {
				*status = SOCK_SHOULD_BE_CLOSED;
#ifdef VERBOSE_OUTPUT
				fprintf(STD_ERR, "Connection closed by server\n");
#endif
			} else {
				*status = RECV_ERROR;
				l_str_free(full_response);
				full_response = NULL;
#ifdef VERBOSE_OUTPUT
				error_msg = g_win32_error_message(i);
				fprintf(STD_ERR, "recv() error: %s\n", error_msg);
				g_free(error_msg);
#endif
			}
			break;
		}
	}
	free2(response);
	return full_response;
}
#endif
