/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif

void set_default_options(Params *prm)
{
	char	tmp_font[FONT_MAXLEN + 1];

	prm->delay = DELAY;
	prm->shift_size = SHIFTSIZE;

	compact_font(tmp_font, FONTNAME, FONTSIZE);
	str_n_cpy(prm->font_n_s, tmp_font, FONT_MAXLEN);
	get_gdk_color_from_hexstr(FGCOLOR, &prm->fg_color, &prm->fg_color_alpha);
	get_gdk_color_from_hexstr(BGCOLOR, &prm->bg_color, &prm->bg_color_alpha);

	prm->win_x = WIN_X;
	prm->win_y = WIN_Y;
	prm->win_w = get_ticker_env()->screen_w;	/* instead of WIN_W; */
	prm->win_h = WIN_H;

	prm->windec = WINDEC;
	prm->always_on_top = ALWAYSONTOP;
	prm->win_transparency = WINTRANSPARENCY;

	prm->shadow = SHADOW;
	prm->shadow_offset_x = SHADOWOFFSET_X;
	prm->shadow_offset_y = SHADOWOFFSET_Y;
	prm->shadow_fx = SHADOWFX;

	prm->icon_in_taskbar = ICONINTASKBAR;
	prm->win_sticky = WINSTICKY;

	str_n_cpy(prm->line_delimiter, LINEDELIMITER, DELIMITER_MAXLEN);
	str_n_cpy(prm->cutline_delimiter, CUTLINEDELIMITER, DELIMITER_MAXLEN);

	prm->rss_refresh = RSSREFRESH;
	prm->rss_title = RSSTITLE;
	str_n_cpy(prm->rss_title_delimiter, RSSTITLEDELIMITER, DELIMITER_MAXLEN);
	prm->rss_description = RSSDESCRIPTION;
	str_n_cpy(prm->rss_description_delimiter, RSSDESCRIPTIONDELIMITER, DELIMITER_MAXLEN);

	prm->n_items_per_feed = NITEMSPERFEED;
	/*prm->read_in_loop = READINLOOP;*/
	prm->strip_html_tags = STRIPHTMLTAGS;
	prm->upper_case_text = UPPERCASETEXT;
	str_n_cpy(prm->homefeed, HOMEFEED, FILE_NAME_MAXLEN);

	str_n_cpy(prm->open_link_cmd, OPENLINKCMD, FILE_NAME_MAXLEN);
	str_n_cpy(prm->open_link_args, OPENLINKARGS, FILE_NAME_MAXLEN);

	prm->clock = CLOCK;
	get_gdk_color_from_hexstr(CFGCOLOR, &prm->clock_fg_color,
						&prm->clock_fg_color_alpha);
	get_gdk_color_from_hexstr(CBGCOLOR, &prm->clock_bg_color,
						&prm->clock_bg_color_alpha);
	compact_font(tmp_font, CFONTNAME, CFONTSIZE);
	str_n_cpy(prm->clock_font_n_s, tmp_font, FONT_MAXLEN);

	prm->disable_popups = DISABLEPOPUPS;
	prm->pause_on_mouseover = PAUSEONMOUSEOVER;
	prm->mouse_wheel_scroll = MOUSEWHEELSCROLL;

	prm->special_chars = SPECIALCHARS;
	prm->new_page_char = NEWPG;
	prm->tab_char = TABCH;

	prm->use_authentication = USEAUTHENTICATION;
	str_n_cpy(prm->user, USER, USER_MAXLEN);
	prm->use_proxy = USEPROXY;
	str_n_cpy(prm->proxy_host, PROXYHOST, PROXY_HOST_MAXLEN);
	str_n_cpy(prm->proxy_port, PROXYPORT, PROXY_PORT_MAXLEN);
	prm->use_proxy_authentication = USEPROXYAUTHENTICATION;
	str_n_cpy(prm->proxy_user, PROXYUSER, PROXY_USER_MAXLEN);
}

/*
 * load conf file if any & set up - returns 0 if ok
 * store options in a string array and initialize a ptr arrays that matches strings
 * we assume we have up to N_OPTION_MAX options, OPTION_MAXLEN char long max each
 */
int get_config_file_options(Params *prm)
{
	FILE		*conf_fp;
	char		options_array[N_OPTION_MAX][OPTION_MAXLEN + 2];
	char		*ptr[N_OPTION_MAX + 1];
	int		i, j = 0;

	if ((conf_fp = g_fopen(get_datafile_full_name_from_name(CONFIG_FILE), "rb")) != NULL) {
		/*
		 * parse conf file
		 */
		for (i = 0; ; i++) {
			/*
			 * here fgets() read OPTION_MAXLEN + 1 char max including
			 * '\n' then add '\0' (if I'm right...)
			 */
			if (i >= N_OPTION_MAX) {
				ptr[i] = NULL;
				warning(FALSE, 5, "Too many lines in configuration file: ",
					get_datafile_full_name_from_name(CONFIG_FILE),
					"(max = ", itoa2(N_OPTION_MAX), ")"
					"\nWon't parse further than limit\n");
				break;
			}
			if (fgets(options_array[i], OPTION_MAXLEN + 2, conf_fp) == NULL) {
				ptr[i] = NULL;
				break;
			} else {
				j = strlen(options_array[i]);
				options_array[i][j - 1] = '\0';	/* we remove trailing '\n' */
				ptr[i] = options_array[i];	/* so now options max length = OPTION_MAXLEN */
			}
		}
		if (i > 0)
			j = parse_options_array(prm, i - 1, (const char **)ptr);
		else
			j = 0;
		fclose(conf_fp);
	} else
		fprintf(STD_ERR, "Can't read configuration file: %s - %s\n",
			get_datafile_full_name_from_name(CONFIG_FILE), strerror(errno));
	return j;
}

/*
 * options array parser (NULL terminated ptrs array)
 * we assume we have up to N_OPTION_MAX options, OPTION_MAXLEN char long max each
 */
int parse_options_array(Params *prm, int n_options, const char *option[])
{
	int	i, exit_status = OK;

	if (n_options > N_OPTION_MAX) {
		n_options = N_OPTION_MAX;
		fprintf(STD_ERR,
			"Too many options (max = %d) - Won't parse further than limit\n",
			N_OPTION_MAX);
			exit_status = OPTION_TOO_MANY;
	}
	for (i = 0; i < n_options && option[i] != NULL; i++)
		if (get_name_value_from_option(prm, option[i]) != OK)
			exit_status = OPTION_ERROR;
	return exit_status;
}

int get_name_value_from_option(Params *prm, const char *option)
{
	/*
	 * options are splitted into name and value
	 * font -> fname is FONT_NAME_MAXLEN char max long, fsize FONT_SIZE_MAXLEN,
	 * plus 1 extra space so font_n_s (font name size) is up to
	 * FONT_MAXLEN = FONT_NAME_MAXLEN + 1 + FONT_SIZE_MAXLEN char long
	 * OPTION_MAXLEN = OPTION_NAME_MAXLEN + OPTION_VALUE_MAXLEN + 4 (-name="value")
	 *
	 * === if unsure, check this in tickr.h ===
	 */
	char	name[OPTION_NAME_MAXLEN + 1], value[OPTION_VALUE_MAXLEN + 1];
	char	fname[FONT_NAME_MAXLEN + 1], fsize[FONT_SIZE_MAXLEN + 1];
	char	cfname[FONT_NAME_MAXLEN + 1], cfsize[FONT_SIZE_MAXLEN + 1];
	char	tmp[OPTION_MAXLEN + 1];
	char	tmp_font[FONT_MAXLEN + 1];
	int	i, exit_status = OK;

	split_font(prm->font_n_s, fname, fsize);
	split_font(prm->clock_font_n_s, cfname, cfsize);
	/*
	 * split option into name & value if any
	 */
	name[0] = '\0';
	value[0] = '\0';
	if (option[0] == '-') {
		str_n_cpy(tmp, option + 1, OPTION_MAXLEN);
		for (i = 0; i < OPTION_MAXLEN - 1; i++)
			if (tmp[i] == '=') {
				tmp[i] = '\0';
				str_n_cpy(name, tmp, OPTION_NAME_MAXLEN);
				str_n_cpy(value, tmp + i + 1, OPTION_VALUE_MAXLEN);
				break;
			}
	} else if (option[0] != '\0') {
		fprintf(STD_ERR, "Invalid option: %s\n", option);
		return (exit_status = OPTION_INVALID);
	}
	/*
	 * check option name and get value if any
	 */
	if (strcmp(name, "delay") == 0)
		prm->delay = atoi(value);
	else if (strcmp(name, "shiftsize") == 0)
		prm->shift_size = atoi(value);
	else if (strcmp(name, "fontname") == 0)
		str_n_cpy(fname, value, FONT_NAME_MAXLEN);
	else if (strcmp(name, "fontsize") == 0) {
		str_n_cpy(fsize, value, FONT_SIZE_MAXLEN);
		if (atoi(fsize) > FONT_MAXSIZE) {
			snprintf(fsize, FONT_SIZE_MAXLEN + 1, "%3d", FONT_MAXSIZE);
			fprintf(STD_ERR, "Font size set to %d (can't be above)\n",
				FONT_MAXSIZE);
		}
	} else if (strcmp(name, "fgcolor") == 0) {
		if (!get_gdk_color_from_hexstr(value, &prm->fg_color,
						&prm->fg_color_alpha)) {
			get_gdk_color_from_hexstr(FGCOLOR, &prm->fg_color,
						&prm->fg_color_alpha);
			exit_status = OPTION_VALUE_INVALID;
		}
	} else if (strcmp(name, "bgcolor") == 0) {
		if (!get_gdk_color_from_hexstr(value, &prm->bg_color,
						&prm->bg_color_alpha)) {
			get_gdk_color_from_hexstr(BGCOLOR, &prm->bg_color,
						&prm->bg_color_alpha);
			exit_status = OPTION_VALUE_INVALID;
		}
	} else if (strcmp(name, "win_x") == 0)
		prm->win_x = atoi(value);
	else if (strcmp(name, "win_y") == 0)
		prm->win_y = atoi(value);
	else if (strcmp(name, "win_w") == 0)
		prm->win_w = atoi(value);
	/*
	 * if win_h is set and > 0, it will override requested font size
	 * with computed one
	 */
	else if (strcmp(name, "win_h") == 0) {
		prm->win_h = atoi(value);
		if (prm->win_h > 0 && prm->win_h < DRWA_HEIGHT_MIN)
			prm->win_h = DRWA_HEIGHT_MIN;
	} else if (strcmp(name, "windec") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->windec = value[0];
	} else if (strcmp(name, "alwaysontop") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->always_on_top = value[0];
	} else if (strcmp(name, "wintransparency") == 0) {
		/* compare int's (because less bug-prone) then set as gdouble */
		prm->win_transparency = atoi(value);
		if ((int)prm->win_transparency < 1) {
			prm->win_transparency = 0.1;
			fprintf(STD_ERR, "Invalid value: %s - will be set to 1\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else if ((int)prm->win_transparency > 10) {
			prm->win_transparency = 10.0;
			fprintf(STD_ERR, "Invalid value: %s - will be set to 10\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->win_transparency /= 10;
	} else if (strcmp(name, "iconintaskbar") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->icon_in_taskbar = value[0];
	} else if (strcmp(name, "winsticky") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->win_sticky = value[0];
	} else if (strcmp(name, "shadow") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->shadow = value[0];
	}
	/* should check value here */
	else if (strcmp(name, "shadowoffset_x") == 0)
		prm->shadow_offset_x = atoi(value);
	/* should check value here */
	else if (strcmp(name, "shadowoffset_y") == 0)
		prm->shadow_offset_y = atoi(value);
	else if (strcmp(name, "shadowfx") == 0)
		prm->shadow_fx = atoi(value);
	else if (strcmp(name, "linedelimiter") == 0)
		str_n_cpy(prm->line_delimiter, value, DELIMITER_MAXLEN);
	else if (strcmp(name, "cutlinedelimiter") == 0)
		str_n_cpy(prm->cutline_delimiter, value, DELIMITER_MAXLEN);
	else if (strcmp(name, "rssrefresh") == 0)
		prm->rss_refresh = atoi(value);
	else if (strcmp(name, "rsstitle") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->rss_title = value[0];
	} else if (strcmp(name, "rsstitledelimiter") == 0)
		str_n_cpy(prm->rss_title_delimiter, value, DELIMITER_MAXLEN);
	else if (strcmp(name, "rssdescription") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->rss_description = value[0];
	} else if (strcmp(name, "rssdescriptiondelimiter") == 0)
		str_n_cpy(prm->rss_description_delimiter, value, DELIMITER_MAXLEN);
	else if (strcmp(name, "nitemsperfeed") == 0)
		prm->n_items_per_feed = atoi(value);
	/*else if (strcmp(name, "readinloop") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->read_in_loop = value[0];
	}*/ else if (strcmp(name, "rmtags") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->strip_html_tags = value[0];
	} else if (strcmp(name, "uppercasetext") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->upper_case_text = value[0];

	} else if (strcmp(name, "homefeed") == 0)
		str_n_cpy(prm->homefeed, value, FILE_NAME_MAXLEN);
	else if (strcmp(name, "openlinkcmd") == 0)
		str_n_cpy(prm->open_link_cmd, value, FILE_NAME_MAXLEN);
	else if (strcmp(name, "openlinkargs") == 0)
		str_n_cpy(prm->open_link_args, value, FILE_NAME_MAXLEN);
	else if (strcmp(name, "clock") == 0) {
		if (strcmp(value, "l") != 0 && strcmp(value, "r") != 0 \
					&& strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->clock = value[0];
	} else if (strcmp(name, "clockfontname") == 0)
		str_n_cpy(cfname, value, FONT_NAME_MAXLEN);
	else if (strcmp(name, "clockfontsize") == 0)
		str_n_cpy(cfsize, value, FONT_SIZE_MAXLEN);
	else if (strcmp(name, "clockfgcolor") == 0) {
		if (!get_gdk_color_from_hexstr(value, &prm->clock_fg_color,
					&prm->clock_fg_color_alpha)) {
			get_gdk_color_from_hexstr(CFGCOLOR, &prm->clock_fg_color,
					&prm->clock_fg_color_alpha);
			exit_status = OPTION_VALUE_INVALID;
		}
	} else if (strcmp(name, "clockbgcolor") == 0) {
		if (!get_gdk_color_from_hexstr(value, &prm->clock_bg_color,
					&prm->clock_bg_color_alpha)) {
			get_gdk_color_from_hexstr(CBGCOLOR, &prm->clock_bg_color,
					&prm->clock_bg_color_alpha);
			exit_status = OPTION_VALUE_INVALID;
		}
	} else if (strcmp(name, "disablepopups") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->disable_popups = value[0];
	} else if (strcmp(name, "pauseonmouseover") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->pause_on_mouseover = value[0];
	} else if (strcmp(name, "mousewheelscroll") == 0) {
		if (strcmp(value, "s") != 0 && strcmp(value, "f") != 0 \
					&& strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->mouse_wheel_scroll = value[0];
	} else if (strcmp(name, "specialchars") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else
			prm->special_chars = value[0];
	} else if (strcmp(name, "newpgchar") == 0)
		prm->new_page_char = value[0];
	else if (strcmp(name, "tabchar") == 0)
		prm->tab_char = value[0];
	/* params in connection settings win */
	else if (strcmp(name, "useauth") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else {
			prm->use_authentication = value[0];
			set_use_authentication(value[0] == 'y' ? TRUE : FALSE);
		}
	} else if (strcmp(name, "user") == 0) {
		str_n_cpy(prm->user, value, USER_MAXLEN);
		str_n_cpy(get_http_auth_user(), value, USER_MAXLEN);
	/* this option is never saved */
	} else if (strcmp(name, "psw") == 0)
		str_n_cpy(get_http_auth_psw(), value, PSW_MAXLEN);
	else if (strcmp(name, "useproxy") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else {
			prm->use_proxy = value[0];
			set_use_proxy(value[0] == 'y' ? TRUE : FALSE);
		}
	} else if (strcmp(name, "proxyhost") == 0) {
		str_n_cpy(prm->proxy_host, value, PROXY_HOST_MAXLEN);
		str_n_cpy(get_proxy_host(), value, PROXY_HOST_MAXLEN);
	} else if (strcmp(name, "proxyport") == 0) {
		str_n_cpy(prm->proxy_port, value, PROXY_PORT_MAXLEN);
		str_n_cpy(get_proxy_port(), value, PROXY_PORT_MAXLEN);
	} else if (strcmp(name, "useproxyauth") == 0) {
		if (strcmp(value, "y") != 0 && strcmp(value, "n") != 0) {
			fprintf(STD_ERR, "Invalid value: %s\n", option);
			exit_status = OPTION_VALUE_INVALID;
		} else {
			prm->use_proxy_authentication = value[0];
			set_use_proxy_auth(value[0] == 'y' ? TRUE : FALSE);
		}
	} else if (strcmp(name, "proxyuser") == 0) {
		str_n_cpy(prm->proxy_user, value, PROXY_USER_MAXLEN);
		str_n_cpy(get_proxy_auth_user(), value, PROXY_USER_MAXLEN);
	/* this option is never saved */
	} else if (strcmp(name, "proxypsw") == 0)
		str_n_cpy(get_proxy_auth_psw(), value, PROXY_PSW_MAXLEN);
	/* connect params until here */
	/* this option is only set from command line */
	else if (strcmp(name, "instance-id") == 0)
		;/* does nothing but avoid an 'unknown option' warning/error */
	else {
		fprintf(STD_ERR, "Unknown option: %s\n", option);
		exit_status = OPTION_UNKNOWN;
	}
	compact_font(tmp_font, fname, fsize);
	str_n_cpy(prm->font_n_s, tmp_font, FONT_MAXLEN);
	compact_font(tmp_font, cfname, cfsize);
	str_n_cpy(prm->clock_font_n_s, tmp_font, FONT_MAXLEN);
	return exit_status;
}

void save_to_config_file(const Params *prm)
{
	FILE		*conf_fp;
	char		fname[FONT_NAME_MAXLEN + 1], fsize[FONT_SIZE_MAXLEN + 1];
	char		cfname[FONT_NAME_MAXLEN + 1], cfsize[FONT_SIZE_MAXLEN + 1];
	char		tmp[2 * 1024];

	if ((conf_fp = g_fopen(get_datafile_full_name_from_name(CONFIG_FILE), "wb")) != NULL) {
		split_font(prm->font_n_s, fname, fsize);
		split_font(prm->clock_font_n_s, cfname, cfsize);
		snprintf(tmp, 2 * 1024, "%s%s%s",
			"-delay=%d\n"
			"-shiftsize=%d\n"
			"-fontname=%s\n"
			"-fontsize=%d\n"
			"-fgcolor=%s\n"
			"-bgcolor=%s\n"
			"-win_x=%d\n"
			"-win_y=%d\n"
			"-win_w=%d\n"
			"-win_h=%d\n"
			"-windec=%c\n"
			"-alwaysontop=%c\n"
			"-wintransparency=%d\n"
			"-iconintaskbar=%c\n"
			"-winsticky=%c\n"
			"-shadow=%c\n"
			"-shadowoffset_x=%d\n"
			"-shadowoffset_y=%d\n"
			"-shadowfx=%d\n"
			"-linedelimiter=%s\n",
			"-cutlinedelimiter=%s\n"
			"-rssrefresh=%d\n"
			"-rsstitle=%c\n"
			"-rsstitledelimiter=%s\n"
			"-rssdescription=%c\n"
			"-rssdescriptiondelimiter=%s\n"
			"-nitemsperfeed=%d\n"
			"-rmtags=%c\n"
			"-uppercasetext=%c\n"
			"-homefeed=%s\n"
			"-openlinkcmd=%s\n"
			"-openlinkargs=%s\n"
			"-clock=%c\n"
			"-clockfontname=%s\n"
			"-clockfontsize=%d\n"
			"-clockfgcolor=%s\n"
			"-clockbgcolor=%s\n"
			"-disablepopups=%c\n"
			"-pauseonmouseover=%c\n"
			"-mousewheelscroll=%c\n",
			"-specialchars=%c\n"
			"-newpgchar=%c\n"
			"-tabchar=%c\n"
			"-useauth=%c\n"
			"-user=%s\n"
			"-useproxy=%c\n"
			"-proxyhost=%s\n"
			"-proxyport=%s\n"
			"-useproxyauth=%c\n"
			"-proxyuser=%s\n"
			);
		fprintf(conf_fp, tmp,
			prm->delay,
			prm->shift_size,
			fname, atoi(fsize),
			get_hexstr_from_gdk_color(&prm->fg_color, &prm->fg_color_alpha),
			get_hexstr_from_gdk_color(&prm->bg_color, &prm->bg_color_alpha),
			prm->win_x,
			prm->win_y,
			prm->win_w,
			prm->win_h,
			prm->windec,
			prm->always_on_top,
			(int)(prm->win_transparency * 10),
			prm->icon_in_taskbar,
			prm->win_sticky,
			prm->shadow,
			prm->shadow_offset_x,
			prm->shadow_offset_y,
			prm->shadow_fx,
			prm->line_delimiter,
			prm->cutline_delimiter,
			prm->rss_refresh,
			prm->rss_title,
			prm->rss_title_delimiter,
			prm->rss_description,
			prm->rss_description_delimiter,
			prm->n_items_per_feed,
			prm->strip_html_tags,
			prm->upper_case_text,
			prm->homefeed,
			prm->open_link_cmd,
			prm->open_link_args,
			prm->clock,
			cfname,
			atoi(cfsize),
			get_hexstr_from_gdk_color(&prm->clock_fg_color, &prm->clock_fg_color_alpha),
			get_hexstr_from_gdk_color(&prm->clock_bg_color, &prm->clock_bg_color_alpha),
			prm->disable_popups,
			prm->pause_on_mouseover,
			prm->mouse_wheel_scroll,
			prm->special_chars,
			prm->new_page_char,
			prm->tab_char,
			prm->use_authentication,
			prm->user,
			prm->use_proxy,
			prm->proxy_host,
			prm->proxy_port,
			prm->use_proxy_authentication,
			prm->proxy_user
			);
		fclose(conf_fp);
	} else
		warning(FALSE, 4, "Can't save configuration file ",
			get_datafile_full_name_from_name(CONFIG_FILE), ": ",
			strerror(errno));
}

/*
 * options colors are coded as 4 hexa values -> #rrggbbaa
 */
int get_gdk_color_from_hexstr(const char *str, GdkColor *color, guint16 *color_alpha)
{
	char tmp[3], *tailptr;
	int errflag = 0;

	if (strlen(str) != 9 && str[0] != '#')
		errflag = 1;
	else {
		str_n_cpy(tmp, str + 1, 2);
		color->red = strtoul(tmp, &tailptr, 16) * 0x100;
		if (tailptr == tmp)
		errflag = 1;

		str_n_cpy(tmp, str + 3, 2);
		color->green = strtoul(tmp, &tailptr, 16) * 0x100;
		if (tailptr == tmp)
			errflag = 1;

		str_n_cpy(tmp, str + 5, 2);
		color->blue = strtoul(tmp, &tailptr, 16) * 0x100;
		if (tailptr == tmp)
			errflag = 1;

		str_n_cpy(tmp, str + 7, 2);
		*color_alpha = strtoul(tmp, &tailptr, 16) * 0x100;
		if (tailptr == tmp)
			errflag = 1;
	}
	if (errflag != 0) {
		fprintf(STD_ERR, "Invalid color: %s\n", str);
		return FALSE;
	} else
		return TRUE;
}

/*
 * options colors are coded as 4 hexa values -> #rrggbbaa
 * *** allow up to 64 simultaneous calls ***
 */
const char *get_hexstr_from_gdk_color(const GdkColor *color, const guint16 *color_alpha)
{
	static char	str[64][10];
	static int	count = -1;

	count++;
	count &= 63;
	snprintf(str[count], 10, "#%02x%02x%02x%02x", color->red / 256, color->green / 256,
					color->blue / 256, *color_alpha / 256);
	if (strlen(str[count]) != 9)
		return NULL;
	else
		return (const  char *)str[count];
}

/*
 * space needed for font_name: FONT_NAME_MAXLEN + 1 bytes / font_size: FONT_SIZE_MAXLEN + 1 bytes
 */
void split_font(const char *font_name_size, char *font_name, char *font_size)
{
	char		tmp[FONT_MAXLEN + 1];
	int		i;

	if (font_name_size[0] == '\0') {
		font_name[0] = '\0';
		font_size[0] = '\0';
	} else {
		str_n_cpy(tmp, font_name_size, FONT_MAXLEN);
		for (i = strlen(tmp) - 1; i > 0 ; i--)
			if (tmp[i] == ' ') {
				tmp[i] = '\0';
				break;
			}
		str_n_cpy(font_name, tmp, FONT_NAME_MAXLEN);
		str_n_cpy(font_size, tmp + strlen(font_name) + 1, FONT_SIZE_MAXLEN);
	}
}

/*
 * space needed for font_name_size: FONT_NAME_MAXLEN + 1 space + FONTSIZENMAXLEN + 1 bytes
 */
void compact_font(char *font_name_size, const char *font_name, const char *font_size)
{
	if (font_name[0] == '\0' && font_size[0] == '\0') {
		font_name_size[0] = '\0';
	} else {
		str_n_cpy(font_name_size, font_name, FONT_NAME_MAXLEN);
		str_n_cat(font_name_size, " ", 1);
		str_n_cat(font_name_size, font_size, FONT_SIZE_MAXLEN);
	}
}
