-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Text_IO;
with Externals.Simple;
with Keys;
with Menus;              use Menus;
with Misc;
with Readline;

package body Configuration is

   procedure Parse_AKE (KE   : in  UBS;
                        Key   : out UBS;
                        Email  : out UBS) is
      KES : constant String := ToStr(KE);
      Sep : Natural;
   begin
      Sep := Ada.Strings.Fixed.Index(Source => KES,
                                     Pattern => ",");
      Key := ToUBS(KES(KES'First..Sep-1));
      Email := ToUBS(KES(Sep+1..KES'Last));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Parse_AKE");
         raise;
   end Parse_AKE;

   procedure Parse_Config_Line (Line   : in  String;
                                Name   : out UBS;
                                Value  : out UBS) is
      Sep : Natural;
   begin
      Sep := Ada.Strings.Fixed.Index(Source => Line,
                                     Pattern => "=");
      Name := ToUBS(Line(Line'First..Sep-1));
      Value := ToUBS(Line(Sep+1..Line'Last));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Parse_Config_Line");
         raise;
   end Parse_Config_Line;

   function Set_Two_Way (S : String) return Boolean is
   begin
      if S = "on" then
         return True;
      elsif S = "off" then
         return False;
      else
         raise Switch_Parse_Error;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Set_Two_Way");
         raise;
   end Set_Two_Way;

   function Match_Binary_Path (N : String) return Boolean is
   begin
      for I in Binaries loop
         if ToStr(Binary_Name(I)) & "-binary" = N then
            return True;
         end if;
      end loop;
      return False;
   end Match_Binary_Path;

   No_Binary_Match : exception;
   function Get_Binary_Path (N : String) return Binaries is
   begin
      for I in Binaries loop
         if ToStr(Binary_Name(I)) & "-binary" = N then
            return I;
         end if;
      end loop;
      raise No_Binary_Match;
   end Get_Binary_Path;

   procedure Read_Config_File is
      use Ada.Text_IO;
      use Misc;
      CF : File_Type;
   begin
      Open(File => CF,
           Mode => In_File,
           Name => ToStr(Topal_Directory) & "/config");
  Config_Loop:
      loop
         begin
            declare
               The_Line : constant String
                 := ToStr(Misc.Unbounded_Get_Line(CF));
               Name     : UBS;
               Value    : UBS;
            begin
               if The_Line'Length > 0
                 and then The_Line(The_Line'First) = '#' then
                  -- It's a comment.  Ignore it.
                  null;
               else
                  Parse_Config_Line(The_Line, Name, Value);
                  declare
                     N : constant String := ToStr(Name);
                     V : UBS renames Value;
                  begin
                     if N = "my-key" then
                        Config.my_key := V;
                        -- Binary command options.
                     elsif Match_Binary_Path(N)  then
                        Config.Binary(Get_Binary_Path(N)) := V;
                        -- General options.
                     elsif N = "general-options" then
                        Config.general_options := V;
                     elsif N = "gpg-options" then
                        Config.gpg_options := V;
                     elsif N = "receiving-options" then
                        Config.receiving_options := V;
                     elsif N = "sending-options" then
                        Config.sending_options := V;
                        -- Switches.
                     elsif N = "decrypt-not-cached" then
                        Config.decrypt_not_cached :=  String_To_Integer(V);
                     elsif N = "decrypt-not-cached-use-cache" then
                        Config.decrypt_not_cached_use_cache :=  String_To_Integer(V);
                     elsif N = "decrypt-cached" then
                        Config.decrypt_cached := String_To_Integer(V);
                     elsif N = "decrypt-cached-use-cache" then
                        Config.decrypt_cached_use_cache := String_To_Integer(V);
                     elsif N = "verify-not-cached" then
                        Config.verify_not_cached := String_To_Integer(V);
                     elsif N = "verify-not-cached-use-cache" then
                        Config.verify_not_cached_use_cache := String_To_Integer(V);
                     elsif N = "verify-cached" then
                        Config.verify_cached := String_To_Integer(V);
                     elsif N = "verify-cached-use-cache" then
                        Config.verify_cached_use_cache := String_To_Integer(V);
                     elsif N = "decrypt-cached-fast-continue" then
                        Config.Decrypt_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                     elsif N = "verify-cached-fast-continue" then
                        Config.Verify_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                     elsif N = "verify-not-cached-fast-continue" then
                        Config.Verify_Not_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                        -- Other oddities.
                     elsif N = "mime-viewer" then
                        Config.MIME_Viewer := String_To_Integer(V);
                     elsif N = "inline-separate-output" then
                        Config.Inline_Separate_Output := Set_Two_Way(ToStr(V));
                     elsif N = "fe-simple" then
                        Config.FE_Simple := Set_Two_Way(ToStr(V));
                     elsif N = "debug" then
                        Config.Debug := Set_Two_Way(ToStr(V));
                     elsif N = "ake" then
                        -- Break the up the key,email pair.
                        declare
                           K : UBS;
                           E : UBS;
                        begin
                           Parse_AKE(V, K, E);
                           Config.AKE_Count := Config.AKE_Count + 1;
                           UAP.Set(Config.AKE_Key, Config.AKE_Count, K);
                           UAP.Set(Config.AKE_Email, Config.AKE_Count, E);
                        end;
                     elsif N = "xk" then
                        -- This is a key to be excluded.
                        Config.XK_Count := Config.XK_Count + 1;
                        UAP.Set(Config.XK_Key, Config.XK_Count, V);
                     elsif N = "sake" then
                        -- Break the up the key,email pair.
                        declare
                           K : UBS;
                           E : UBS;
                        begin
                           Parse_AKE(V, K, E);
                           Config.SAKE_Count := Config.SAKE_Count + 1;
                           UAP.Set(Config.SAKE_Key, Config.SAKE_Count, K);
                           UAP.Set(Config.SAKE_Email, Config.SAKE_Count, E);
                        end;
                     elsif N = "sxk" then
                        -- This is a key to be excluded.
                        Config.SXK_Count := Config.SXK_Count + 1;
                        UAP.Set(Config.SXK_Key, Config.SXK_Count, V);
                     else
                        Put_Line("Bogus line: " & The_Line);
                        raise Config_Parse_Error;
                     end if;
                  end;
               end if;
            end;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Config_Loop;
         end;
      end loop Config_Loop;
      Close(CF);
   exception
      when Ada.IO_Exceptions.Name_Error =>
         -- Silently ignore the lack of a config file.
         Close(CF);
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Read_Config_File");
         raise;
   end Read_Config_File;

   -- Given the current value (V) and default config (D), Put_Line if
   --  different or make it a comment if the same.
   procedure Dump_Bin (K : in String;
                       V : in UBS;
                       D : in UBS) is
      use type UBS;
   begin
      if V = D then
         Ada.Text_IO.Put("#");
      end if;
      Ada.Text_IO.Put_Line(K & "-binary=" & ToStr(V));
   end Dump_Bin;

   procedure Dump (Overwrite_Config : in Boolean := False) is
      use Ada.Text_IO;
      use Misc;
      F : File_Type;
      Default_Config : Config_Record;
   begin
      if Overwrite_Config then
         if Externals.Simple.Test_F(ToStr(Topal_Directory) & "/config") then
            Externals.Simple.Mv_F(ToStr(Topal_Directory) & "/config",
                                  ToStr(Topal_Directory) & "/config.bak");
         end if;
         Create(File => F,
                Mode => Out_File,
                Name => ToStr(Topal_Directory) & "/config");
         Set_Output(F);
      end if;
      Default_Configuration(Default_Config);
      Put_Line("my-key=" & ToStr(Config.My_Key));
      -- Binary paths.
      for I in Binaries loop
         Dump_Bin(ToStr(Binary_Name(I)), Config.Binary(I), Default_Config.Binary(I));
      end loop;
      -- Other options.
       Put_Line("general-options=" & ToStr(Config.General_Options));
      Put_Line("gpg-options=" & ToStr(Config.Gpg_Options));
      Put_Line("receiving-options=" & ToStr(Config.Receiving_Options));
      Put_Line("sending-options=" & ToStr(Config.Sending_Options));
      Put_Line("decrypt-not-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Not_Cached)));
      Put_Line("decrypt-not-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Not_Cached_Use_Cache)));
      Put_Line("decrypt-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Cached)));
      Put_Line("decrypt-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Cached_Use_Cache)));
      Put_Line("verify-not-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Not_Cached)));
      Put_Line("verify-not-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Not_Cached_Use_Cache)));
      Put_Line("verify-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Cached)));
      Put_Line("verify-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Cached_Use_Cache)));
      Put_Line("decrypt-cached-fast-continue="
               & ToStr(Two_Way(Config.Decrypt_Cached_Fast_Continue)));
      Put_Line("verify-cached-fast-continue="
               & ToStr(Two_Way(Config.Verify_Cached_Fast_Continue)));
      Put_Line("verify-not-cached-fast-continue="
               & ToStr(Two_Way(Config.Verify_Not_Cached_Fast_Continue)));
      Put_Line("mime-viewer="
               & Trim_Leading_Spaces(Integer'Image(Config.MIME_Viewer)));
      Put_Line("inline-separate-output=" & ToStr(Two_Way(Config.Inline_Separate_Output)));
      Put_Line("fe-simple=" & ToStr(Two_Way(Config.FE_Simple)));
      Put_Line("debug=" & ToStr(Two_Way(Config.Debug)));
      for I in 1 .. Config.AKE_Count loop
         Put_Line("ake="
                  & ToStr(UAP.Value(Config.AKE_Key, I))
                  & ","
                  & ToStr(UAP.Value(Config.AKE_Email, I)));
      end loop;
      for I in 1 .. Config.XK_Count loop
         Put_Line("xk="
                  & ToStr(UAP.Value(Config.XK_Key, I)));
      end loop;
      for I in 1 .. Config.SAKE_Count loop
         Put_Line("sake="
                  & ToStr(UAP.Value(Config.SAKE_Key, I))
                  & ","
                  & ToStr(UAP.Value(Config.SAKE_Email, I)));
      end loop;
      for I in 1 .. Config.SXK_Count loop
         Put_Line("sxk="
                  & ToStr(UAP.Value(Config.SXK_Key, I)));
      end loop;
      if Overwrite_Config then
         Set_Output(Standard_Output);
         Close(F);
      end if;
   exception
      when others =>
         Set_Output(Standard_Output);
         Put_Line("Problem in Configuration.Dump, probably file related");
         Close(F);
         raise;
   end Dump;

   procedure Edit_Own_Key is
      use Ada.Text_IO;
   begin
      New_Line(2);
      Put_Line("Signing key is currently: "
               & ToStr(Config.My_Key));
      Put_Line("Do you want to replace this key?");
      Put_Line("Answer Yes to replace it; No to keep the current key.");
      if YN_Menu = Yes then
         -- Now, find all secret keys and offer them as a `select 1' list.
         declare
            SKL            : Keys.Key_List;
            Pattern        : UBS;
            New_Secret_Key : UBS;
            Use_Old_Key    : Boolean;
         begin
            Keys.Empty_Keylist(SKL);
            Pattern := ToUBS(Readline.Get_String("Type GPG search pattern: "));
            Keys.Add_Secret_Keys(Pattern, SKL, Keys.Any);
            Keys.Select_Key_From_List(SKL, New_Secret_Key, Use_Old_Key);
            if not Use_Old_Key then
               Config.My_Key := New_Secret_Key;
            end if;
         end;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Edit_Own_Key");
         raise;
   end Edit_Own_Key;

   procedure Edit_Configuration is
      use Ada.Text_IO;
      use Misc;
      Default_Config : Config_Record;
      Three_Str : constant array (Integer range 1..3) of UBS
        := (1 => ToUBS("Always"),
            2 => ToUBS("Ask"),
            3 => ToUBS("Never"));
      Five_Str : constant array (Integer range 1..5) of UBS
        := (1 => ToUBS("Use"),
            2 => ToUBS("Replace"),
            3 => ToUBS("Ask-replace"),
            4 => ToUBS("Ignore cache"),
            5 => ToUBS("Offer all"));
      MV_Str : constant array (Integer range 1..5) of UBS
        := (1 => ToUBS("Ask"),
            2 => ToUBS("Metamail"),
            3 => ToUBS("Run-Mailcap"),
            4 => ToUBS("Save"),
            5 => ToUBS("Skip"));
      Three_Values : constant array (Three_Way_Index) of Integer
        := (Always => 1,
            Ask    => 2,
            Never  => 3);
      Five_Values : constant array (Five_Way_Index) of Integer
        := (UUse        => 1,
            Replace     => 2,
            AskReplace  => 3,
            IgnoreCache => 4,
            OfferAll    => 5);
      Selection   : Edit_Index;
      Selection2O : Options_Index;
      Selection2S : Settings_Index;
   begin
      Default_Configuration(Default_Config);
  Edit_Loop:
      loop
         Ada.Text_IO.New_Line(5);
         Ada.Text_IO.Put_Line("** Configuration menu:");
         Put_Line("Configuration:             Debugging currently "
                  & ToStr(Two_Way(Config.Debug)));
         Put_Line("                           Inline separate output currently "
                  & ToStr(Two_Way(Config.Inline_Separate_Output)));
         Put_Line("                           MIME viewer is "
                  & ToStr(MV_Str(Config.MIME_Viewer)));
         Put_Line("Configuration is not saved beyond this session unless explicitly saved");
         Selection := Edit_Menu;
         case Selection is
            when Quit => -- Quit editing.
               New_Line(3);
               exit Edit_Loop;
            when Save => -- Save changes.
               Dump(Overwrite_Config => True);
               Put_Line("Configuration saved");
               New_Line(3);
            when Binary_Paths => -- Binary paths.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when GPG_Options => -- GPG options.
               Options_Loop:
               loop
               Put_Line("Options submenu (all relevant options are concatenated together):");
               Put_Line("  GPG options (all GPG operations): "
                        & NL & "    " & ToStr(Config.GPG_Options));
               Put_Line("  General options (send & receive): "
                        & NL & "    " & ToStr(Config.General_Options));
               Put_Line("  Receiving options (receive only): "
                        & NL & "    " & ToStr(Config.Receiving_Options));
               Put_Line("  Sending options (send only): "
                        & NL & "    " & ToStr(Config.Sending_Options));
               Selection2O := Options_Menu;
               case Selection2O is
                  when Quit =>
                     New_Line(3);
                     exit Options_Loop;
                  when Change_General =>
                     Readline.Add_History(ToStr(Config.General_Options));
                     Config.General_Options := ToUBS(Readline.Get_String("General options: "));
                  when Change_GPG =>
                     Readline.Add_History(ToStr(Config.GPG_Options));
                     Config.GPG_Options := ToUBS(Readline.Get_String("GPG options: "));
                  when Change_Receiving =>
                     Readline.Add_History(ToStr(Config.Receiving_Options));
                     Config.Receiving_Options := ToUBS(Readline.Get_String("Receiving options: "));
                  when Change_Sending =>
                     Readline.Add_History(ToStr(Config.Sending_Options));
                     Config.Sending_Options := ToUBS(Readline.Get_String("Sending options: "));
                  when Default_General =>
                     Config.General_Options := Default_Config.General_Options;
                  when Default_GPG =>
                     Config.GPG_Options := Default_Config.GPG_Options;
                  when Default_Receiving =>
                     Config.Receiving_Options := Default_Config.Receiving_Options;
                  when Default_Sending =>
                     Config.Sending_Options := Default_Config.Sending_Options;
                  when Default_All =>
                     Config.General_Options := Default_Config.General_Options;
                     Config.GPG_Options := Default_Config.GPG_Options;
                     Config.Receiving_Options := Default_Config.Receiving_Options;
                     Config.Sending_Options := Default_Config.Sending_Options;
               end case;
               end loop Options_Loop;
            when DV_Settings => -- Decrypt/verify settings.
               Settings_Loop:
               loop
               Put_Line("Decrypt/Verify settings submenu:");
               Put_Line("  Decrypt/cached:                  " & ToStr(Three_Str(Config.Decrypt_Cached)));
               Put_Line("  Decrypt/cached, cache usage:     " & ToStr(Five_Str(Config.Decrypt_Cached_Use_Cache)));
               Put_Line("  Decrypt/not cached:              " & ToStr(Three_Str(Config.Decrypt_Not_Cached)));
               Put_Line("  Decrypt/not cached, cache usage: " & ToStr(Three_Str(Config.Decrypt_Not_Cached_Use_Cache)));
               Put_Line("  Verify/cached:                   " & ToStr(Three_Str(Config.Verify_Cached)));
               Put_Line("  Verify/cached, cache usage:      " & ToStr(Five_Str(Config.Verify_Cached_Use_Cache)));
               Put_Line("  Verify/not cached:               " & ToStr(Three_Str(Config.Verify_Not_Cached)));
               Put_Line("  Verify/not cached, cache usage:  " & ToStr(Three_Str(Config.Verify_Not_Cached_Use_Cache)));
               Put_Line("  Decrypt/cached fast continue:    " & ToStr(Two_Way(Config.Decrypt_Cached_Fast_Continue)));
               Put_Line("  Verify/cached fast continue:     " & ToStr(Two_Way(Config.Verify_Cached_Fast_Continue)));
               Put_Line("  Verify/not cached fast continue: " & ToStr(Two_Way(Config.Verify_Not_Cached_Fast_Continue)));
               Selection2S := Settings_Menu;
               case Selection2S is
                  when Decrypt_Cached =>
                     Config.Decrypt_Cached := Three_Values(Three_Way_Menu);
                  when Decrypt_Cached_Use =>
                     Config.Decrypt_Cached_Use_Cache := Five_Values(Five_Way_Menu);
                  when Decrypt_Not_Cached =>
                     Config.Decrypt_Not_Cached := Three_Values(Three_Way_Menu);
                  when Decrypt_Not_Cached_Use =>
                     Config.Decrypt_Not_Cached_Use_Cache := Three_Values(Three_Way_Menu);
                  when Verify_Cached =>
                     Config.Verify_Cached := Three_Values(Three_Way_Menu);
                  when Verify_Cached_Use =>
                     Config.Verify_Cached_Use_Cache := Five_Values(Five_Way_Menu);
                  when Verify_Not_Cached =>
                     Config.Verify_Not_Cached := Three_Values(Three_Way_Menu);
                  when Verify_Not_Cached_Use =>
                     Config.Verify_Not_Cached_Use_Cache := Three_Values(Three_Way_Menu);
                  when Quit =>
                     New_Line(3);
                     exit Settings_Loop;
                  when All_Defaults =>
                     Config.Decrypt_Cached := Default_Config.Decrypt_Cached;
                     Config.Decrypt_Cached_Use_Cache := Default_Config.Decrypt_Cached_Use_Cache;
                     Config.Decrypt_Not_Cached := Default_Config.Decrypt_Not_Cached;
                     Config.Decrypt_Not_Cached_Use_Cache := Default_Config.Decrypt_Not_Cached_Use_Cache;
                     Config.Verify_Cached := Default_Config.Verify_Cached;
                     Config.Verify_Cached_Use_Cache := Default_Config.Verify_Cached_Use_Cache;
                     Config.Verify_Not_Cached := Default_Config.Verify_Not_Cached;
                     Config.Verify_Not_Cached_Use_Cache := Default_Config.Verify_Not_Cached_Use_Cache;
                  when Toggle_Decrypt_Cached_Fast =>
                     Config.Decrypt_Cached_Fast_Continue := not Config.Decrypt_Cached_Fast_Continue;
                  when Toggle_Verify_Cached_Fast =>
                     Config.Verify_Cached_Fast_Continue := not Config.Verify_Cached_Fast_Continue;
                  when Toggle_Verify_Not_Cached_Fast =>
                     Config.Verify_Not_Cached_Fast_Continue := not Config.Verify_Not_Cached_Fast_Continue;
               end case;
               end loop Settings_Loop;
            when Own_Key => -- Own key.
               Edit_Own_Key;
            when Key_Assoc => -- Key associations.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when Key_Excl => -- Key exclusions.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when Set_Defaults => -- Set all defaults.
               Copy_Configuration(Config, Default_Config);
               Put_Line("All configuration settings are now default");
               New_Line(3);
            when Toggle_Debug => -- Toggle debugging.
               New_Line(3);
               Config.Debug := not Config.Debug;
            when Toggle_ISO => -- Toggle inline separate output.
               New_Line(3);
               Config.Inline_Separate_Output := not Config.Inline_Separate_Output;
            when MV_Setting => -- Change MIME viewer setting.
               New_Line(3);
               Put_Line("MIME viewer is set to "
                  & ToStr(MV_Str(Config.MIME_Viewer)));
               Config.MIME_Viewer := MV_Values(MIME_Viewer_Menu1);
            when Reset => -- Restore to current saved file.
               New_Line(3);
               Copy_Configuration(Config, Default_Config);
               Read_Config_File;
               Put_Line("Read saved configuration");
               New_Line(3);
         end case;
      end loop Edit_Loop;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Edit_Configuration");
         raise;
   end Edit_Configuration;

   procedure Default_Configuration (C : in out Config_Record) is
   begin
      C.My_Key := ToUBS("");
      C.Binary(Chmod) := ToUBS("chmod");
      C.Binary(Clear) := ToUBS("clear");
      C.Binary(Date) := ToUBS("date");
      C.Binary(Diff) := ToUBS("diff");
      C.Binary(Dos2Unix) := ToUBS("dos2unix");
      C.Binary(Formail) := ToUBS("formail");
      C.Binary(File) := ToUBS("file");
      C.Binary(Gpgop) := ToUBS("gpg");
      C.Binary(Gpgsm) := ToUBS("gpgsm");
      C.Binary(Grep) := ToUBS("grep");
      C.Binary(Iconv) := ToUBS("iconv");
      C.Binary(Less) := ToUBS("less");
      C.Binary(Locale) := ToUBS("locale");
      C.Binary(Md5sum) := ToUBS("md5sum");
      C.Binary(Metamail) := ToUBS("metamail");
      C.Binary(Mimeconstruct) := ToUBS("mime-construct");
      C.Binary(Mimetool) := ToUBS("mime-tool");
      C.Binary(Mkdir) := ToUBS("mkdir");
      C.Binary(Mkfifo) := ToUBS("mkfifo");
      C.Binary(Mv) := ToUBS("mv");
      C.Binary(openssl) := ToUBS("openssl");
      C.Binary(Rm) := ToUBS("rm");
      C.Binary(Runmailcap) := ToUBS("run-mailcap");
      C.Binary(Scp) := ToUBS("scp");
      C.Binary(Sed) := ToUBS("sed");
      C.Binary(Ssh) := ToUBS("ssh");
      C.Binary(Stty) := ToUBS("stty");
      C.Binary(Tee) := ToUBS("tee");
      C.Binary(Test) := ToUBS("test");
      C.General_Options := ToUBS("");
      C.Gpg_Options := ToUBS("--no-options");
      C.Receiving_Options := ToUBS("--keyserver=wwwkeys.uk.pgp.net");
      C.Sending_Options := ToUBS("--force-v3-sigs --compress-algo 1 --cipher-algo cast5");
      C.Decrypt_Not_Cached := 2;
      C.Decrypt_Not_Cached_Use_Cache := 1;
      C.Decrypt_Cached := 2;
      C.Decrypt_Cached_Use_Cache := 1;
      C.Verify_Not_Cached := 1;
      C.Verify_Not_Cached_Use_Cache := 1;
      C.Verify_Cached := 1;
      C.Verify_Cached_Use_Cache := 1;
      C.Decrypt_Cached_Fast_Continue := False;
      C.Verify_Cached_Fast_Continue := True;
      C.Verify_Not_Cached_Fast_Continue := False;
      C.MIME_Viewer := 1;
      C.Inline_Separate_Output := False;
      C.FE_Simple := False;
      C.No_Clean := False;
      C.All_Headers := False;
      C.Read_From := False;
      C.Ask_Charset := False;
      C.Debug := False;
      UAP.Create(C.AKE_Key, Initial_KL);
      UAP.Create(C.AKE_Email, Initial_KL);
      C.AKE_Count := 0;
      UAP.Create(C.XK_Key, Initial_KL);
      C.XK_Count := 0;
      UAP.Create(C.SAKE_Key, Initial_KL);
      UAP.Create(C.SAKE_Email, Initial_KL);
      C.SAKE_Count := 0;
      UAP.Create(C.SXK_Key, Initial_KL);
      C.SXK_Count := 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Default_Configuration");
         raise;
   end Default_Configuration;

   procedure Copy_Configuration(Left  : in out Config_Record;
                                Right : in     Config_Record) is
   begin
      Left.My_Key := Right.My_Key;
      for I in Binaries loop
         Left.Binary(I) := Right.Binary(I);
      end loop;
      Left.General_Options := Right.General_Options;
      Left.Gpg_Options := Right.Gpg_Options;
      Left.Receiving_Options := Right.Receiving_Options;
      Left.Sending_Options := Right.Sending_Options;
      Left.Decrypt_Not_Cached := Right.Decrypt_Not_Cached;
      Left.Decrypt_Not_Cached_Use_Cache := Right.Decrypt_Not_Cached_Use_Cache;
      Left.Decrypt_Cached := Right.Decrypt_Cached;
      Left.Decrypt_Cached_Use_Cache := Right.Decrypt_Cached_Use_Cache;
      Left.Verify_Not_Cached := Right.Verify_Not_Cached;
      Left.Verify_Not_Cached_Use_Cache := Right.Verify_Not_Cached_Use_Cache;
      Left.Verify_Cached := Right.Verify_Cached;
      Left.Verify_Cached_Use_Cache := Right.Verify_Cached_Use_Cache;
      Left.Decrypt_Cached_Fast_Continue := Right.Decrypt_Cached_Fast_Continue;
      Left.Verify_Cached_Fast_Continue := Right.Verify_Cached_Fast_Continue;
      Left.Verify_Not_Cached_Fast_Continue := Right.Verify_Not_Cached_Fast_Continue;
      Left.MIME_Viewer := Right.MIME_Viewer;
      Left.Inline_Separate_Output := Right.Inline_Separate_Output;
      Left.FE_Simple := Right.FE_Simple;
      Left.No_Clean := Right.No_Clean;
      Left.Debug := Right.Debug;
      Left.All_Headers := Right.All_Headers;
      Left.Read_From := Right.Read_From;
      Left.Ask_Charset := Right.Ask_Charset;
      UAP.Copy(Left.AKE_Key, Right.AKE_Key);
      UAP.Copy(Left.AKE_Email, Right.AKE_Email);
      Left.AKE_Count := Right.AKE_Count;
      UAP.Copy(Left.XK_Key, Right.XK_Key);
      Left.XK_Count := Right.XK_Count;
      UAP.Copy(Left.SAKE_Key, Right.SAKE_Key);
      UAP.Copy(Left.SAKE_Email, Right.SAKE_Email);
      Left.SAKE_Count := Right.SAKE_Count;
      UAP.Copy(Left.SXK_Key, Right.SXK_Key);
      Left.SXK_Count := Right.SXK_Count;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Copy_Configuration");
         raise;
   end Copy_Configuration;

end Configuration;
