-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Characters.Latin_1;
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Expanding_Array;

pragma Elaborate_All(Expanding_Array);

package Globals is

   -- Initial number of keys to be associated.
   Initial_KL : constant Positive := 300;

   Not_Implemented : exception;

   -- Some renamings of unbounded strings.
   subtype UBS is Ada.Strings.Unbounded.Unbounded_String;

   NullUBS : UBS renames Ada.Strings.Unbounded.Null_Unbounded_String;

   function ToUBS(S : String) return UBS
     renames Ada.Strings.Unbounded.To_Unbounded_String;

   function ToStr(U : UBS) return String
     renames Ada.Strings.Unbounded.To_String;

   -- We will want to play with arrays of UBS.
   type UBS_Array is array (Integer range <>) of UBS;

   -- Pointers to UBS arrays.
   type UBS_Array_Pointer is access UBS_Array;

   -- We also want expanding arrays....
   package UBS_Array_Package is new Expanding_Array (Item => UBS);
   package UAP renames UBS_Array_Package;
   subtype UBS_Big_Array is UAP.Big_Array;

   -- Want to insert line feeds in some strings.
   NL : constant String    := Ada.Characters.Latin_1.LF & "";

   -- The result file stuff.
   Result_File : Ada.Text_IO.File_Type;

   -- What is our process ID?  (Needed for tempfiles.)
   Our_PID : Integer;

   -- Our home directory.
   Topal_Directory : UBS;

   type Binaries is
     (Chmod, Clear, Date, Diff, Dos2Unix, File,
      Formail, GPGOP, GPGSM, Grep, Iconv, Less, Locale,
      Md5sum, Metamail, Mimeconstruct, Mimetool,
      Mkdir, Mkfifo, Mv, Openssl, Rm, Runmailcap,
      Scp, Sed, Ssh, Stty, Tee, Test);

   Binary_Name : constant array(Binaries) of UBS :=
     (Chmod => ToUBS("chmod"),
      Clear => ToUBS("clear"),
      Date => ToUBS("date"),
      Diff => ToUBS("diff"),
      Dos2Unix => ToUBS("dos2unix"),
      File => ToUBS("file"),
      Formail => ToUBS("formail"),
      GPGOP => ToUBS("gpg"),
      GPGSM => ToUBS("gpgsm"),
      Grep => ToUBS("grep"),
      Iconv => ToUBS("iconv"),
      Less => ToUBS("less"),
      Locale => ToUBS("locale"),
      Md5sum => ToUBS("md5sum"),
      Metamail => ToUBS("metamail"),
      Mimeconstruct => ToUBS("mimeconstruct"),
      Mimetool => ToUBS("mimetool"),
      Mkdir => ToUBS("mkdir"),
      Mkfifo => ToUBS("mkfifo"),
      Mv => ToUBS("mv"),
      Openssl => ToUBS("openssl"),
      Rm => ToUBS("rm"),
      Runmailcap => ToUBS("runmailcap"),
      Scp => ToUBS("scp"),
      Sed => ToUBS("sed"),
      Ssh => ToUBS("ssh"),
      Stty => ToUBS("stty"),
      Tee => ToUBS("tee"),
      Test => ToUBS("test"));

   type Binaries_UBS is array (Binaries) of UBS;

   -- Configuration data types.
   type Config_Record is
      record
         My_Key : UBS;
         Binary : Binaries_UBS;
         General_Options,
           Gpg_Options,
           Receiving_Options,
           Sending_Options : UBS;

         --  decrypt-not-cached: 1 always, 2 ask, 3 never
         --  decrypt-not-cached-use-cache: 1 always, 2 ask, 3 never
         --  decrypt-cached: 1 always, 2 ask, 3 never
         --  decrypt-cached-use-cache: 1 use, 2 again-replace,
         --                            3 again-ask-replace,
         --                            4 again-never, 5 offer

         --  Ditto for verify-not-cached, verify-not-cached-use-cache,
         --  verify-cached, verify-cached-use-Cache
         Decrypt_Not_Cached,
           Decrypt_Not_Cached_Use_Cache,
           Decrypt_Cached,
           Decrypt_Cached_Use_Cache,
           Verify_Not_Cached,
           Verify_Not_Cached_Use_Cache,
           Verify_Cached,
           Verify_Cached_Use_Cache,
         -- MIME_Viewer: 1 ask, 2 use metamail, 3 use run-mailcap,
         --              4 save dummy message to ~/mail, 5 skip attachments
           MIME_Viewer : Positive;
         Decrypt_Cached_Fast_Continue,
           Verify_Cached_Fast_Continue,
           Verify_Not_Cached_Fast_Continue,
           FE_Simple,
           Inline_Separate_Output,
           No_Clean, -- Not preserved in config file.
           All_Headers, -- Not config file.
           Read_From, -- Not config file.
           Ask_Charset, -- Not config file.
           Debug : Boolean;
         -- A list of keys to be associated with email addresses.
         AKE_Key : UBS_Big_Array;
         AKE_Email : UBS_Big_Array;
         AKE_Count : Natural;
         -- A list of keys to be excluded from the key list.
         XK_Key : UBS_Big_Array;
         XK_Count : Natural;
         -- And the same again, this time for the secret key lists.
         SAKE_Key : UBS_Big_Array;
         SAKE_Email : UBS_Big_Array;
         SAKE_Count : Natural;
         -- A list of keys to be excluded from the key list.
         SXK_Key : UBS_Big_Array;
         SXK_Count : Natural;
      end record;

   Config : Config_Record;

end Globals;
