/*
 *   Copyright (C) 2008 Schutz Sacha <istdasklar@gmail.com>
 *   AND lex newton http://qgoogletranslate.googlecode.com
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   or (at your option) any later version, as published by the Free
 *   Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Sm_treet, Fifth Floor, Boston, MA  02110-1301, USA.*/


#include "ktranslatoid.h"

#include <plasma/theme.h>
#include <KDebug>
#include <QTextCodec>
#include <KTextEdit>
#include <QToolButton>
#include <KMessageBox>
#include <KPushButton>
#include <QLabel>
#include <KConfigDialog>
#include <KStandardDirs>
#include <QPainter>
#include <QTreeView>
 #include <QClipboard>
#include <Solid/Device>
#include <Solid/Networking>

KTranslatoid ::KTranslatoid(QObject *parent, const QVariantList &args)
        :  Plasma::PopupApplet(parent, args)
{
    m_textEditA = new  Plasma::TextEdit;
    m_textEditB = new  Plasma::TextEdit;
    m_buttonTranslate = new Plasma::PushButton;
    m_buttonChange = new Plasma::ToolButton;
    m_meter = new   Plasma::Meter;
    m_labelA = new Plasma::Label;
    m_labelB = new Plasma::Label;
    m_buttonA = new Plasma::ToolButton ;
    m_buttonB = new Plasma::ToolButton ;
    m_voiceForm = new VoiceForm;
    m_languageForm = new LanguageForm;
    m_popupIcon = KIcon("translator");
    m_autoPaste = true;
    m_autoTranslate = true;
    m_languageModel = new LanguageModel;

    m_http = new QHttp(this);
    m_command = m_voiceForm->commandLine();
    m_textEditA->nativeWidget()->installEventFilter(this);
    connect(m_http, SIGNAL(done(bool)), this, SLOT(done()));
    connect(m_buttonTranslate, SIGNAL(clicked()), this, SLOT(translate()));
    connect(m_buttonChange, SIGNAL(clicked()), this, SLOT(changeSens()));
    connect(m_buttonA->nativeWidget(), SIGNAL(clicked()), this, SLOT(changeButtonA()));
    connect(m_buttonB->nativeWidget(), SIGNAL(clicked()), this, SLOT(changeButtonB()));


    setBackgroundHints(DefaultBackground);
    QTextCodec::setCodecForCStrings(QTextCodec::codecForName("UTF-8"));
    setAspectRatioMode(Plasma::IgnoreAspectRatio);
    setupPopupIcon();
    constructForm();
}
//========================================================================================

KTranslatoid ::~KTranslatoid()
{
}
//====================================================================================
void KTranslatoid::constructForm()
{
    m_graphicsWidget = new QGraphicsWidget(this);

    Plasma::Frame *m_header = new Plasma::Frame;
    m_header->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    m_header->setText(i18n("Translatoid"));
    m_buttonChange->nativeWidget()->setIcon(KIcon("system-switch-user.png"));
    m_buttonChange->nativeWidget()->setToolTip(i18n("invert translation"));

    m_buttonTranslate->nativeWidget()->setIcon(KIcon("tools-wizard.png"));
    m_buttonTranslate->setText(i18n("Translate"));
    m_buttonTranslate->nativeWidget()->setToolTip(i18n("click to translate"));


    QFont font = Plasma::Theme::defaultTheme()->font(Plasma::Theme::DefaultFont);
    m_textEditA->setFont(font);
    m_textEditB->setFont(font);

    m_textEditB->nativeWidget()->setReadOnly(true);

    Plasma::ToolButton * pastButton = new Plasma::ToolButton ;
    Plasma::ToolButton * copyButton = new Plasma::ToolButton ;
    Plasma::ToolButton * clearButton = new Plasma::ToolButton ;
    Plasma::ToolButton * voiceButton = new Plasma::ToolButton ;

    voiceButton->nativeWidget()->setIcon(KIcon("text-speak.png"));
    voiceButton->setMinimumSize(QSize(20, 20));

    pastButton->setMinimumSize(QSize(20, 20));
    copyButton->setMinimumSize(QSize(20, 20));
    clearButton->setMinimumSize(QSize(20, 20));
    pastButton->nativeWidget()->setIcon(KIcon("edit-paste.png"));
    copyButton->nativeWidget()->setIcon(KIcon("edit-copy.png"));
    clearButton->nativeWidget()->setIcon(KIcon("edit-delete.png"));

    pastButton->nativeWidget()->setToolTip(i18n("Paste a selection to translate it"));
    copyButton->nativeWidget()->setToolTip(i18n("Copy the result "));
    clearButton->nativeWidget()->setToolTip(i18n("Clear the source text "));

    m_buttonA->nativeWidget()->setToolTip(i18n("Change source language"));
    m_buttonB->nativeWidget()->setToolTip(i18n("Change destination  language"));

    connect(pastButton->nativeWidget(), SIGNAL(clicked()), this, SLOT(paste()));
    connect(copyButton->nativeWidget(), SIGNAL(clicked()), this, SLOT(copy()));
    connect(clearButton->nativeWidget(), SIGNAL(clicked()), m_textEditA->nativeWidget(), SLOT(clear()));
    connect(voiceButton->nativeWidget(), SIGNAL(clicked()), this, SLOT(tts()));

    QGraphicsLinearLayout*  layoutControl = new QGraphicsLinearLayout(Qt::Horizontal);
    layoutControl->addItem(m_buttonA);
    layoutControl->addItem(m_buttonChange);
    layoutControl->addItem(m_buttonB);
    layoutControl->addItem(voiceButton);
    layoutControl->addStretch();
    layoutControl->addItem(clearButton);
    layoutControl->addItem(pastButton);
    layoutControl->addItem(copyButton);
    layoutControl->addItem(m_buttonTranslate);

    m_meter->setMinimum(0);
    m_meter->setMaximum(100);
    m_meter->setMeterType(Plasma::Meter::BarMeterHorizontal);
    m_meter->setMaximumHeight(20);

    m_buttonA->setMinimumSize(QSize(20, 20));
    m_buttonB->setMinimumSize(QSize(20, 20));
    m_buttonChange->setMinimumSize(QSize(20, 20));
    m_buttonTranslate->setMinimumSize(QSize(20, 20));
    QGraphicsLinearLayout* mainLayout = new QGraphicsLinearLayout(Qt::Vertical);

    mainLayout->addItem(m_header);
    mainLayout->addItem(m_labelA);

    mainLayout->addItem(m_textEditA);
    mainLayout->addItem(m_labelB);

    mainLayout->addItem(m_textEditB);
    mainLayout->addItem(m_meter);
    mainLayout->addItem(layoutControl);

    QGraphicsWidget * widget = new QGraphicsWidget;
    widget->setLayout(mainLayout);
  
    bigLayout = new QGraphicsLinearLayout(Qt::Horizontal);
    bigLayout->addItem(widget);
    m_graphicsWidget->setLayout(bigLayout);
    m_graphicsWidget->setPreferredSize(530, 320);

    m_textEditA->nativeWidget()->selectAll();
    m_textEditA->nativeWidget()->setHtml("Here your text source");

    if (Solid::Networking::status() != Solid::Networking::Connected) {
        kDebug() << i18n("Networking is disabled. Cannot connect to google");
    }
}
//====================================================================================
QGraphicsWidget *KTranslatoid::graphicsWidget()
{
    readData();
    return m_graphicsWidget;
}
//=============================================================================
void KTranslatoid ::translate()
{
    kDebug("Starting translation.");
    QString from, to, text;

    from = m_languageA.code;
    to =   m_languageB.code;
    text = m_textEditA->nativeWidget()->toPlainText();
    kDebug("Now we know all we need.");

    QString url = QString("/translate_a/t?client=t&sl=" + from + "&tl=" + to);
    kDebug("Url configured.");


    QHttpRequestHeader header = QHttpRequestHeader("POST", url, 1, 1);
    header.setValue("Host", "www.google.com");
    header.setValue("User-Agent", "Mozilla/5.0");
    header.setValue("Accept-Encoding", "deflate");
    header.setContentLength(text.length());
    header.setValue("Connection", "Close");
    kDebug("Headers configured.");

    QByteArray ba("text=");
    ba.append( text.toUtf8() );
    kDebug("ByteArray completed.");

    m_http->setHost("www.google.com");
    kDebug("HTTP host setted");

    m_http->request(header,ba);
    kDebug("Request started");

}
//==============================================================================
void KTranslatoid::done()
{
    kDebug("HTTP Request done.");

    QString text;
    text = text.fromUtf8( m_http->readAll() );
    kDebug("All data is readed.");

    text = text.replace(QString("\\\""),QString("\""));
    text = text.replace(QString("\\n"),QString("\n"));
    text = text.replace(QString("\n "),QString("\n"));
    text = text.replace(QString("\\x3c"),QString("<"));
    text = text.replace(QString("\\x3e"),QString(">"));

    if ( text.startsWith( QString("\"") ) ) {
        kDebug("This is a text.");
        text = text.remove( text.length()-1, 1).remove(0,1);
    } else if ( text.startsWith( QString("[") ) && text.endsWith( QString("]") ) ) {
        kDebug("This is a word.");
        QStringList tra;
        //tra = text.split(QRegExp(QString("\"(.*)\"")));
        text = text.replace(QString("]"),QString(""));
        tra = text.split(QString("["));
        text = QString("");
        for (int i=0,j=0;i<tra.count();i++)
        {
            if (tra.at(i)!="")
            {
                if (j==0)
                {
                    QString translation = tra.at(i);
                    translation = translation.replace("\"","");
                    translation = translation.replace(",","");
                    text.append( translation );
                    text.append( "\n\n") ;
                } else {
                    QString translation = tra.at(i);
                    QStringList translations = translation.split(",");
                    for (int y=0;y<translations.count();y++)
                    {
                        translation = translations.at(y);
                        translation = translation.replace("\"","");
                        if (y==0)
                        {
                            text.append( QString( translation + ": ") );
                        } else {
                            text.append( QString( "\t" + translation + "\n" ) );
                        }
                    }
                    text.append( "\n" );
                }
                j++;
            }
        }
    }

    kDebug("All operations done. Setting text to textarea.");
    m_textEditB->setText( text );
    kDebug("Done.");
    kDebug("Any else?");
}
//===================================================================================
void KTranslatoid::changeSens()
{
    updateLanguage(m_languageB, m_languageA);
}
//==========================================================================================
void KTranslatoid::updateMeter(qint64 bytesSent, qint64 bytesTotal)
{
    m_meter->setValue((bytesSent / bytesTotal)*100);
}
//====================================================================================
void KTranslatoid::changeButtonA()
{
    m_tree = new Plasma::TreeView;

    m_tree->setModel(m_languageModel);
    m_tree->nativeWidget()->setRootIsDecorated (false);

    bigLayout->insertItem(0, m_tree);
    m_tree->nativeWidget()->setColumnWidth (0, 60 );
    m_tree->nativeWidget()->setColumnWidth (1, 80 );

    m_currentButton = 1;
    m_buttonA->setEnabled(false);
    m_buttonB->setEnabled(false);
    m_buttonChange->setEnabled(false);
    m_tree->moveBy(0, 500);
    Plasma::Animator::self()->moveItem(m_tree,  Plasma::Animator::SlideInMovement, QPoint(0, 0));

    connect(m_tree->nativeWidget(),SIGNAL(clicked (const QModelIndex &)), this, SLOT(collapseLeft(const QModelIndex &)));
}
//====================================================================================
void KTranslatoid::changeButtonB()
{
    m_tree = new Plasma::TreeView;
    m_tree->setModel(m_languageModel);
    m_tree->nativeWidget()->setRootIsDecorated (false);

    bigLayout->insertItem(0, m_tree);
    m_tree->nativeWidget()->setColumnWidth (0, 60 );
    m_tree->nativeWidget()->setColumnWidth (1, 80 );
    m_currentButton = 2;
    m_buttonA->setEnabled(false);
    m_buttonB->setEnabled(false);
    m_buttonChange->setEnabled(false);
    m_tree->moveBy(0, 500);
    Plasma::Animator::self()->moveItem(m_tree,  Plasma::Animator::SlideInMovement, QPoint(0, 0));

    connect(m_tree->nativeWidget(),SIGNAL(clicked (const QModelIndex &)), this, SLOT(collapseLeft(const QModelIndex &)));
}
//=======================================================================================
void KTranslatoid::copy()
{
    m_textEditB->nativeWidget()->selectAll();
    m_textEditB->nativeWidget()->copy();
}
//=========================================================================================
void KTranslatoid::paste()
{
    m_textEditA->nativeWidget()->clear();
    m_textEditA->nativeWidget()->paste();
}

//==========================================================================================
void KTranslatoid::tts()
{

    QString qcommand = m_command;
    qcommand.replace("$RESULT", m_textEditB->nativeWidget()->toPlainText());
    qcommand.replace("$SOURCE", m_textEditA->nativeWidget()->toPlainText());


    QByteArray ba = qcommand.toUtf8();
    const char *command = ba.data();
    if (!system(command))
        kDebug() << i18n("unable to call tts program");
}
//==============================================================================================
void KTranslatoid::createConfigurationInterface(KConfigDialog *parent)
{
    m_voiceForm = new VoiceForm;
    m_languageForm = new LanguageForm;



    m_languageForm->setLanguage(m_languageA, m_languageB);
    m_languageForm->ui()->checkBoxPaste->setChecked(m_autoPaste);
    m_languageForm->ui()->checkBoxTranslate->setChecked(m_autoTranslate);


    parent->addPage(m_voiceForm, i18n("Voice settings"), KStandardDirs::locate("icon", "oxygen/64x64/apps/preferences-desktop-sound.png"));
    parent->addPage(m_languageForm, i18n("Language settings"), KStandardDirs::locate("icon", "oxygen/64x64/categories/applications-education-language.png"));

    connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
}
//================================================================================================
void KTranslatoid::configAccepted()
{

    m_command = m_voiceForm->commandLine();
    m_autoPaste = m_languageForm->ui()->checkBoxPaste->isChecked();
    m_autoTranslate = m_languageForm->ui()->checkBoxTranslate->isChecked();
    updateLanguage(m_languageForm->languageSource(), m_languageForm->languageResult());
    writeData();

}
//================================================================================================
void KTranslatoid::collapseLeft(const QModelIndex & index)
{
    kDebug()<<"collapse";
    m_tree->hide();
    bigLayout->removeItem(m_tree);


    m_buttonA->setEnabled(true);
    m_buttonB->setEnabled(true);
    m_buttonChange->setEnabled(true);

    LanguageModel * model  = static_cast <LanguageModel*>(m_tree->model());

    if (m_currentButton == 1) {
        m_buttonA->nativeWidget()->setIcon(KIcon(model->getIcon( model->index(index.row(), 0))));
        m_labelA->setText(model->getName( model->index(index.row(), 0)));
        m_languageA = model->getLanguage( model->index(index.row(), 0));
    }


    if (m_currentButton == 2) {
        m_buttonB->nativeWidget()->setIcon(KIcon(model->getIcon( model->index(index.row(), 0))));
        m_labelB->setText(model->getName( model->index(index.row(), 0)));
        m_languageB = model->getLanguage( model->index(index.row(), 0));

    }

    setupPopupIcon();
    m_currentButton = 0;
}

//============================================================================================================
void KTranslatoid::popupEvent(bool show)
{

    if (show) {
        m_textEditA->grabKeyboard();
        if (m_autoPaste) {
          QClipboard *clipboard = QApplication::clipboard();


            m_textEditA->nativeWidget()->clear();
            m_textEditA->nativeWidget()->setText(clipboard->text(QClipboard::Selection));


//             m_textEditA->nativeWidget()->paste();
        }

        if (m_autoTranslate)
            translate();

    }

    else {
        m_textEditA->ungrabKeyboard();

    }

}
//============================================================================================================
void KTranslatoid::setupPopupIcon()
{
    QPixmap pix(64, 64);
    pix.fill(Qt::transparent);
    QPainter paint(&pix);

    paint.drawPixmap(0, 0, KIcon("accessories-dictionary").pixmap(64, 64));

    QPixmap flag1 = KIcon(m_languageA.icon).pixmap(19, 19);
    QPixmap flag2 = KIcon(m_languageB.icon).pixmap(19, 19);

    QTransform transform;
    transform.shear(0.15, -0.15);
    flag1 = flag1.transformed(transform, Qt::SmoothTransformation);
    flag2 = flag2.transformed(transform, Qt::SmoothTransformation);

    paint.drawPixmap(10, 15, flag1);
    paint.drawPixmap(30, 13, flag2);
    paint.setTransform(transform, Qt::SmoothTransformation);
    paint.setPen(Qt::white);

//     paint.drawText(QRect(10,35,pix.width(),pix.height()), m_languageA.code+"  "+m_languageB.code);
    m_popupIcon = KIcon(pix);

    setPopupIcon(m_popupIcon);

}
//===============================================
void KTranslatoid::updateLanguage(Language a, Language b)
{
    m_languageA = a;
    m_languageB = b;

    m_buttonA->nativeWidget()->setIcon(KIcon(m_languageA.icon));
    m_labelA->setText(m_languageA.name);

    m_buttonB->nativeWidget()->setIcon(KIcon(m_languageB.icon));
    m_labelB->setText(m_languageB.name);

    setupPopupIcon();
}

//================================================
void KTranslatoid::readData()
{

    KConfigGroup cg = config();

    QString a = cg.readEntry("lsource", QString("fr"));
    QString b = cg.readEntry("lresult", QString("en"));
    m_command = cg.readEntry("tts", QString());
    m_autoPaste = cg.readEntry("autoPaste", true);
    m_autoTranslate = cg.readEntry("autoTranslate", true);



    Language la = m_languageModel->getLanguageFromCode(a);
    Language lb = m_languageModel->getLanguageFromCode(b);

    updateLanguage(la, lb);



}
void KTranslatoid::writeData()
{


    KConfigGroup cg = config();
    cg.writeEntry("lsource", m_languageForm->languageSource().code);
    cg.writeEntry("lresult", m_languageForm->languageResult().code);


    cg.writeEntry("tts", m_voiceForm->commandLine());
    cg.writeEntry("autoPaste", QVariant(m_languageForm->ui()->checkBoxPaste->isChecked()));
    cg.writeEntry("autoTranslate", QVariant(m_languageForm->ui()->checkBoxTranslate->isChecked()));
//


}
bool KTranslatoid::eventFilter ( QObject * obj, QEvent * event ) 
{
//Filter TextEdit Event 
//Enter  = translate
//Shift+Enter = add new line
if (obj == m_textEditA->nativeWidget())
{
  if (event->type() == QEvent::KeyPress)
  {
  QKeyEvent *keyEvent = static_cast<QKeyEvent*>(event);

      if ( keyEvent->key() == Qt::Key_Return && keyEvent->modifiers() == Qt::ShiftModifier)
      {
      m_textEditA->nativeWidget()->append("");
      return true;
      }

      if (keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter)
      {
      translate();
      return true;
      }
  }
}
  return Plasma::PopupApplet::eventFilter(obj, event);



}

#include "ktranslatoid.moc"
