# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the AddFolderButton widget."""

import os

from PyQt4 import QtCore

from twisted.internet import defer

from ubuntuone.controlpanel.gui.tests import (
    USER_HOME,
)
from ubuntuone.controlpanel.gui.qt import addfolder as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    CrashyBackend,
    CrashyBackendException,
    FakedDialog
)

# Access to a protected member
# Instance of 'ControlBackend' has no '_called' member
# pylint: disable=W0212, E1103


def set_path_on_file_dialog(folder_name=None, file_dialog=None):
    """Set a valid path in the file dialog to create a folder."""
    if folder_name is None:
        folder_name = 'Documents'
    folder = os.path.join(USER_HOME, folder_name)
    if file_dialog is None:
        file_dialog = FakedFileDialog
    file_dialog.response = QtCore.QString(folder)
    return folder


class FakedFileDialog(object):
    """Fake a file chooser dialog."""

    response = args = kwargs = None
    DontUseNativeDialog = object()

    @classmethod
    def reset(cls, *a, **kw):
        """Clear all values."""
        cls.response = cls.args = cls.kwargs = None

    @classmethod
    def getExistingDirectory(cls, *a, **kw):
        """Simulate a directory chooser."""
        cls.args = a
        cls.kwargs = kw
        return cls.response


class AddFolderButtonTestCase(BaseTestCase):
    """The test suite for the AddFolderButton."""

    class_ui = gui.AddFolderButton

    @defer.inlineCallbacks
    def setUp(self):
        yield super(AddFolderButtonTestCase, self).setUp()
        self.created_folders = []

        # default response if user does nothing
        FakedFileDialog.reset()

        self.patch(gui, 'FileDialog', FakedFileDialog)

        self.patch(FakedFileDialog, 'response', gui.QtCore.QString(''))
        self.patch(self.ui.backend, 'validate_path_for_folder', lambda p: True)
        self.patch(self.ui, 'add_folder_func', self.add_folder)

    @defer.inlineCallbacks
    def assert_does_nothing(self, method_call):
        """Nothing happens.

        Backend is not called, warning dialog is not shown, and folderCreated
        signal is not emitted.

        """
        # connect to signal to confirm it was not emitted
        self.ui.folderCreated.connect(self._set_called)

        yield method_call()

        # the backend was not called
        self.assertEqual(self.ui.backend._called, {})

        # the folderCreated signal was not emitted
        self.assertEqual(self._called, False)

    def add_folder(self, folder_path):
        """A dummy add folder func."""
        self.created_folders.append(folder_path)
        return defer.succeed(None)

    def test_default_add_folder_func(self):
        """The add_folder_func is used."""
        folder = set_path_on_file_dialog()
        yield self.ui.on_clicked()

        self.assertEqual(self.created_folders, [folder])

    @defer.inlineCallbacks
    def test_add_folder_button_clicked_opens_file_chooser(self):
        """When adding a new folder, the proper file chooser is raised."""
        yield self.ui.click()

        home_dir = yield self.ui.backend.get_home_dir()
        self.assertEqual(FakedFileDialog.args, ())
        self.assertEqual(FakedFileDialog.kwargs, {
            'options': gui.FILE_CHOOSER_OPTIONS,
            'directory': home_dir,
            'parent': self.ui,
        })

    @defer.inlineCallbacks
    def test_does_nothing_if_user_closes_file_chooser(self):
        """If the user closes the file chooser, do nothing."""
        yield self.assert_does_nothing(self.ui.click)

        # the warning dialog was not opened
        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)

    @defer.inlineCallbacks
    def test_opens_warning_if_folder_path_not_valid(self):
        """If the user chooses an invalid path, show a warning."""
        self.patch(self.ui.backend, 'validate_path_for_folder',
                   lambda p: False)
        folder_path = set_path_on_file_dialog()
        yield self.ui.click()

        args = {'folder_path': folder_path, 'home_folder': USER_HOME}
        msg = gui.FOLDER_INVALID_PATH % args
        self.assertEqual(FakedDialog.args,
                         (self.ui, '', msg, gui.CLOSE))
        self.assertEqual(FakedDialog.kwargs, {})

        yield self.assert_does_nothing(self.ui.click)

    @defer.inlineCallbacks
    def test_handles_unicode_paths(self):
        """Unicode paths are properly handled."""
        folder = set_path_on_file_dialog(u'ñoño ñandú ❤')
        yield self.ui.click()

        # no warning
        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)
        # backend called
        self.assertEqual(self.created_folders, [folder])

    @defer.inlineCallbacks
    def test_calls_backend(self):
        """The backend is called to create the new folder."""
        folder = set_path_on_file_dialog()
        yield self.ui.click()

        # no warning
        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)
        # backend called
        self.assertEqual(self.created_folders, [folder])

    @defer.inlineCallbacks
    def test_emit_folder_created_on_success(self):
        """The widget emit folderCreated on success."""
        folder = set_path_on_file_dialog()
        self.ui.folderCreated.connect(self._set_called)
        yield self.ui.click()

        self.assertEqual(self._called, ((folder,), {}))

    @defer.inlineCallbacks
    def test_emit_folder_creation_canceled_on_file_dialog_close(self):
        """The widget emit folderCreationCanceled on file dialog closed."""
        self.ui.folderCreationCanceled.connect(self._set_called)
        yield self.ui.click()

        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_emit_folder_creation_canceled_on_folder_invalid(self):
        """The widget emit folderCreationCanceled on file dialog closed."""
        self.ui.folderCreationCanceled.connect(self._set_called)
        yield self.ui.click()

        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_backend_error_is_handled(self):
        """Any error from the backend is properly handled."""
        set_path_on_file_dialog()
        self.patch(self.ui, 'backend', CrashyBackend())
        yield self.ui.click()

        self.assertTrue(self.memento.check_exception(CrashyBackendException))
