# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the devices tab."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import devices as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    SAMPLE_DEVICES_INFO,
)
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)


class DevicesPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt control panel."""

    innerclass_ui = gui.devices_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.DevicesPanel
    executed = False

    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check():
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return SAMPLE_DEVICES_INFO

        self.patch(self.ui.backend, 'device_names_info', check)

        return self.ui.load()  # trigger the info request

    def test_is_not_processing_after_info_ready(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        self.assertFalse(self.ui.is_processing)

    @defer.inlineCallbacks
    def test_info_is_requested_on_load(self):
        """The info is requested to the backend."""
        yield self.ui.load()
        self.assert_backend_called('device_names_info')

    def test_no_devices_at_startup(self):
        """The UI is reset at startup."""
        self.assertEqual(self.ui.ui.list_devices.count(), 0)

    def test_process_info(self):
        """The widget is updated with the info."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        local, remote = SAMPLE_DEVICES_INFO[0], SAMPLE_DEVICES_INFO[1:]

        local_device = self.ui.ui.local_device
        self.assertEqual(unicode(local_device.text()), local['name'])
        self.assertEqual(local_device.id, local['device_id'])

        self.assertEqual(self.ui.ui.list_devices.count(),
                         len(remote))
        for i, remote_device in enumerate(remote):
            item = self.ui.ui.list_devices.item(i)
            device = self.ui.ui.list_devices.itemWidget(item)
            self.assertEqual(unicode(device.text()), remote_device['name'])

    def test_local_device(self):
        """Test if the local_device widget is properly packed."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        local_device = self.ui.ui.local_device_layout.itemAt(0).widget()
        self.assertIs(local_device, self.ui.ui.local_device)

    def test_process_info_twice(self):
        """The widget is updated with the info."""
        self.test_process_info()
        self.test_process_info()

    def test_manage_devices_button(self):
        """Clicking the manage devices button opens the proper url."""
        self.assert_uri_hook_called(self.ui.ui.manage_devices_button,
                                    gui.EDIT_DEVICES_LINK)

    def test_local_device_removed_clears_the_widget(self):
        """When the local device was deleted, clear it."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        self.ui.ui.local_device.removed.emit()

        self.assertEqual(unicode(self.ui.ui.local_device.text()), '')
        self.assertEqual(self.ui.ui.local_device.id, None)

    def test_local_device_removed_signal(self):
        """When the local device is removed, emit localDeviceRemoved signal."""
        self.ui.localDeviceRemoved.connect(self._set_called)
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        self.ui.ui.local_device.removed.emit()

        self.assertEqual(self._called, ((), {}))
