# -*- coding: utf-8 -*-

# Author: Roberto Alsina <roberto.alsina@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the notification area icon."""

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks

import ubuntuone.controlpanel.gui.qt.gui
from ubuntuone.controlpanel.gui import (
    qt,
    IN_PROGRESS,
    IN_PROGRESS_FILE,
    RECENT_TRANSFERS,
    RECENTTRANSFERS,
    UPLOADING,
)
from ubuntuone.controlpanel.gui.qt import systray
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    FakeDesktopService,
)
from ubuntuone.controlpanel.tests import ROOT_PATH


# pylint: disable=C0103, W0212
backend = systray.backend


class FakeSDTool(object):

    """Fake SyncDaemonTool."""

    called = False

    def quit(self):
        """Fake quit."""
        self.called = True


class FakeMainWindow(QtGui.QWidget):

    """Fake Main Window."""

    def __init__(self, *args, **kwargs):
        self.args = (args, kwargs)
        super(FakeMainWindow, self).__init__()
        self.tabname = ''

    def switch_to(self, tabname):
        """Fake switch_to."""
        self.tabname = tabname


class SystrayTestCase(BaseTestCase):

    """Test the notification area icon."""

    class_ui = systray.TrayIcon

    @inlineCallbacks
    def setUp(self):
        # We need to patch the startTimer first, to avoid the timer
        # to get started on initialization.
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        yield super(SystrayTestCase, self).setUp()
        self.fake_desktop_service = FakeDesktopService()
        self.patch(QtGui, "QDesktopServices", self.fake_desktop_service)

    def assert_status_correct(self, status_bd, status_ui, action,
                              callback=None):
        """The shown status is correct."""
        expected_text = status_ui

        status = {backend.STATUS_KEY: status_bd}
        self.ui._process_status(status)

        actual_text = unicode(self.ui.status.text())
        self.assertEqual(expected_text, actual_text)

        self.assertFalse(self.ui.status.isEnabled())
        self.assertEqual(unicode(self.ui.status_action.text()), action)

        self.ui.status_action.trigger()
        self.assertFalse(self.ui.status_action.isEnabled())
        self.assert_backend_called(callback)

    def test_process_info_invalid_status(self):
        """File sync status invalid, ignore event."""
        status = {backend.STATUS_KEY: backend.FILE_SYNC_STARTING,
                  backend.MSG_KEY: qt.FILE_SYNC_STARTING,
                  'icon': backend.FILE_SYNC_STARTING}
        self.ui._process_status(status)
        self.ui._process_status(3)

        actual_text = unicode(self.ui.status.text())
        self.assertEqual(qt.FILE_SYNC_STARTING, actual_text)

    def test_status_change_handler_set(self):
        """Backend's file sync status changed callback is connected."""
        # pylint: disable=W0212
        self.assertEqual(self.ui.backend._called['add_status_changed_handler'],
                         ((self.ui._process_status,), {}))

    def test_process_info_disabled(self):
        """File sync status disabled update the label."""
        self.ui.refresh_status()
        self.assert_status_correct(status_bd=backend.FILE_SYNC_DISABLED,
                                   status_ui=qt.FILE_SYNC_DISABLED,
                                   action=qt.FILE_SYNC_ENABLE,
                                   callback='enable_files')

    def test_process_info_disconnected(self):
        """File sync status disconnected update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_DISCONNECTED,
                                   status_ui=qt.FILE_SYNC_DISCONNECTED,
                                   action=qt.FILE_SYNC_CONNECT,
                                   callback='connect_files')

    def test_process_info_error(self):
        """File sync status error update the label."""
        msg = qt.WARNING_MARKUP % qt.FILE_SYNC_ERROR
        self.assert_status_correct(status_bd=backend.FILE_SYNC_ERROR,
                                   status_ui=msg,
                                   action=qt.FILE_SYNC_RESTART,
                                   callback='restart_files')

    def test_process_info_idle(self):
        """File sync status idle update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_IDLE,
                                   status_ui=qt.FILE_SYNC_IDLE,
                                   action=qt.FILE_SYNC_DISCONNECT,
                                   callback='disconnect_files')

    def test_process_info_starting(self):
        """File sync status starting update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_STARTING,
                                   status_ui=qt.FILE_SYNC_STARTING,
                                   action=qt.FILE_SYNC_STOP,
                                   callback='stop_files')

    def test_process_info_stopped(self):
        """File sync status stopped update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_STOPPED,
                                   status_ui=qt.FILE_SYNC_STOPPED,
                                   action=qt.FILE_SYNC_START,
                                   callback='start_files')

    def test_process_info_syncing(self):
        """File sync status syncing update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_SYNCING,
                                   status_ui=qt.FILE_SYNC_SYNCING,
                                   action=qt.FILE_SYNC_DISCONNECT,
                                   callback='disconnect_files')

    def test_process_info_unknown(self):
        """File sync status unknown update the label."""
        msg = qt.WARNING_MARKUP % qt.FILE_SYNC_ERROR
        self.assert_status_correct(status_bd=backend.FILE_SYNC_UNKNOWN,
                                   status_ui=msg,
                                   action=qt.FILE_SYNC_RESTART,
                                   callback='restart_files')

    def test_backend(self):
        """Backend is created."""
        self.assertIsInstance(self.ui.backend,
                              backend.ControlBackend)

    def test_refresh_status_requested(self):
        """test refresh_status was requested on initialized."""
        data = {}

        def callback(status):
            """Fake _process_status callback."""
            data['called'] = True
            data['status'] = status

        self.patch(self.ui, "_process_status", callback)
        self.ui.refresh_status()
        self.assert_backend_called('file_sync_status')
        self.assertTrue(data['called'])
        self.assertEqual(data['status'], [])

    def test_quit(self):
        """Test the quit option in the menu."""
        # Not done on setup, because if I patch stop
        # after instantiation, it doesn't get called.
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        self.patch(systray.TrayIcon, "stop", self._set_called)
        tray = systray.TrayIcon()
        tray.quit.trigger()
        self.assertEqual(self._called, ((False,), {}))

    @inlineCallbacks
    def test_stop_sd(self):
        """Quit should call SyncDaemonTool.quit()."""
        st = FakeSDTool()
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        self.patch(systray, "SyncDaemonTool", lambda: st)
        tray = systray.TrayIcon()
        yield tray.stop()
        self.assertTrue(st.called)

    def test_restore_no_window(self):
        """Test the restore window option in the menu, with no window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        self.patch(ubuntuone.controlpanel.gui.qt.gui,
            "MainWindow", FakeMainWindow)
        tray = systray.TrayIcon()
        self.assertEqual(tray.window, None)
        tray.open_u1.trigger()
        self.assertIsInstance(tray.window, FakeMainWindow)
        self.assertTrue(tray.window.isVisible())
        self.assertEqual(tray.window.args, ((),
            {'close_callback': tray.delete_window}))

    def test_restore_window(self):
        """Test the restore window option in the menu, with a window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        self.assertFalse(tray.window.isVisible())
        tray.open_u1.trigger()
        self.assertEqual(tray.window, window)
        self.assertTrue(tray.window.isVisible())

    def test_restore_window_minimized(self):
        """Test the restore window option in the menu, with a min. window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        # This cannot be tested with the real activateWindow
        # because the un-minimization is done by the WM, so
        # it has a small delay, and it fails.
        self.patch(window, "activateWindow", self._set_called)
        tray.window = window
        tray.open_u1.trigger()
        self.assertEqual(self._called, ((), {}))

    def test_open_share_file_no_window(self):
        """Test the open_share_file option in the menu, with no window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        self.patch(ubuntuone.controlpanel.gui.qt.gui,
            "MainWindow", FakeMainWindow)
        tray = systray.TrayIcon()
        self.assertEqual(tray.window, None)
        tray.share_a_file.trigger()
        self.assertIsInstance(tray.window, FakeMainWindow)
        self.assertTrue(tray.window.isVisible())
        self.assertEqual(tray.window.tabname, 'share_links')
        self.assertEqual(tray.window.args, ((),
            {'close_callback': tray.delete_window}))

    def test_open_share_file(self):
        """Test the open_share_file option in the menu, with a window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        self.assertFalse(tray.window.isVisible())
        tray.share_a_file.trigger()
        self.assertEqual(tray.window, window)
        self.assertEqual(tray.window.tabname, 'share_links')
        self.assertTrue(tray.window.isVisible())

    def test_open_share_file_window_minimized(self):
        """Test the open_share_file option in the menu, with a min. window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        # This cannot be tested with the real activateWindow
        # because the un-minimization is done by the WM, so
        # it has a small delay, and it fails.
        self.patch(window, "activateWindow", self._set_called)
        tray.window = window
        tray.share_a_file.trigger()
        self.assertEqual(tray.window.tabname, 'share_links')
        self.assertEqual(self._called, ((), {}))

    def test_delete_window(self):
        """Test deleting an existing window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        window = FakeMainWindow()
        tray.window = window
        tray.delete_window()
        self.assertEqual(tray.window, None)
        self.assertFalse(window.isVisible())

    def test_delete_no_window(self):
        """Test deleting without an existing window."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        tray.delete_window()
        self.assertEqual(tray.window, None)

    def test_open_u1_folder_action(self):
        """Test open_u1_folder_action."""
        self.ui.open_u1_folder.trigger()
        expected_url = QtCore.QUrl(u'file://%s' % ROOT_PATH)
        self.assertEqual(self.fake_desktop_service.opened_url, expected_url)

    def test_get_more_storage_action(self):
        """Test get_more_storage."""
        self.ui.get_more_storage.trigger()
        expected_url = QtCore.QUrl(systray.GET_STORAGE_LINK)
        self.assertEqual(self.fake_desktop_service.opened_url, expected_url)

    def test_go_to_web_action(self):
        """Test go_to_web."""
        self.ui.go_to_web.trigger()
        expected_url = QtCore.QUrl(systray.DASHBOARD)
        self.assertEqual(self.fake_desktop_service.opened_url, expected_url)

    def test_get_help_action(self):
        """Test get_help_online."""
        self.ui.get_help_online.trigger()
        expected_url = QtCore.QUrl(systray.GET_SUPPORT_LINK)
        self.assertEqual(self.fake_desktop_service.opened_url, expected_url)

    def test_initialization(self):
        """Test that everything initializes correctly."""
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        tray = systray.TrayIcon()
        self.assertTrue(tray.isVisible())
        self.assertEqual(tray.window, None)
        self.assertNotEqual(tray.icon(), None)
        self.assertEqual(tray.uploading, {})
        self.assertEqual(tray.recent_transfers, {})
        self.assertIsInstance(tray.context_menu, QtGui.QMenu)
        self.assertIsInstance(tray.status, QtGui.QAction)
        self.assertIsInstance(tray.status_action, QtGui.QAction)
        self.assertIsInstance(tray.open_u1, QtGui.QAction)
        self.assertIsInstance(tray.open_u1_folder, QtGui.QAction)
        self.assertIsInstance(tray.go_to_web, QtGui.QAction)
        self.assertIsInstance(tray.get_more_storage, QtGui.QAction)
        self.assertIsInstance(tray.get_help_online, QtGui.QAction)
        self.assertIsInstance(tray.transfers, QtGui.QMenu)
        # This checks that _get_volumes_info and _process_volumes_info
        # is being called correctly on initialization.
        expected_home = u'file://%s' % ROOT_PATH
        self.assertEqual(tray.root_path, expected_home)

    def test_get_transfers_data(self):
        """Check that _get_transfers_data return the proper data."""

        result = []

        def fake_callback(data):
            """Fake callback to process data."""
            result.append(data)

        self.patch(self.ui.transfers, "_update_transfers", fake_callback)
        self.ui.transfers.get_transfers_data()
        self.assertEqual(result, [{RECENTTRANSFERS: [], UPLOADING: []}])


class TransfersMenuTestCase(BaseTestCase):

    """Test the info to be display in the transfers menu."""

    class_ui = systray.TrayIcon

    @inlineCallbacks
    def setUp(self):
        # We need to patch the startTimer first, to avoid the timer
        # to get started on initialization.
        self.patch(systray.TransfersMenu, "startTimer", lambda s, x: None)
        yield super(TransfersMenuTestCase, self).setUp()
        self.fake_desktop_service = FakeDesktopService()
        self.patch(QtGui, "QDesktopServices", self.fake_desktop_service)
        self.patch(self.ui.transfers._parent.backend, "sync_menu",
            self.fake_sync_menu)
        self.recent_transfers = []
        self.uploading = []

    def fake_sync_menu(self):
        """Fake backend sync_menu."""
        return {RECENTTRANSFERS: self.recent_transfers,
                UPLOADING: self.uploading}

    def test_load_menu(self):
        """Show the menu with just the labels."""
        self.ui.transfers.get_transfers_data()
        actions = self.ui.transfers.actions()
        self.assertEqual(actions[0].text(), RECENT_TRANSFERS)
        self.assertTrue(actions[1].isSeparator())
        self.assertEqual(actions[2].text(), IN_PROGRESS)

    def test_load_recent_transfers(self):
        """Show the menu with the recent transfers."""
        self.recent_transfers = ['file1', 'file2']
        self.ui.transfers.get_transfers_data()
        actions = self.ui.transfers.actions()
        self.assertEqual(actions[0].text(), RECENT_TRANSFERS)
        self.assertEqual(actions[1].text(), 'file1')
        self.assertEqual(actions[2].text(), 'file2')
        self.assertTrue(actions[3].isSeparator())
        self.assertEqual(actions[4].text(), IN_PROGRESS)

    def test_load_in_progress(self):
        """Show the menu with the current progress."""
        in_progress = [
            ('file1', 200, 150),
            ('file2', 400, 100),
            ('file3', 300, 200),
        ]

        # Return copy of in_progress
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        actions = self.ui.transfers.actions()
        self.assertEqual(actions[0].text(), RECENT_TRANSFERS)
        self.assertTrue(actions[1].isSeparator())
        self.assertEqual(actions[2].text(), IN_PROGRESS)

        # This also check that the files are ordered based on written
        percentage = in_progress[2][2] * 100 / in_progress[2][1]
        text = IN_PROGRESS_FILE % (in_progress[2][0], percentage)
        self.assertEqual(actions[3].label.text(), text)
        self.assertEqual(actions[3].progress.value(), percentage)

        percentage = in_progress[0][2] * 100 / in_progress[0][1]
        text = IN_PROGRESS_FILE % (in_progress[0][0], percentage)
        self.assertEqual(actions[4].label.text(), text)
        self.assertEqual(actions[4].progress.value(), percentage)

        percentage = in_progress[1][2] * 100 / in_progress[1][1]
        text = IN_PROGRESS_FILE % (in_progress[1][0], percentage)
        self.assertEqual(actions[5].label.text(), text)
        self.assertEqual(actions[5].progress.value(), percentage)

    def test_load_in_progress_refresh(self):
        """Show the menu with the current progress and refresh it."""
        in_progress = [
            ('file1', 200, 150),
            ('file2', 400, 100),
            ('file3', 300, 200),
        ]
        # Return copy of in_progress
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        actions = self.ui.transfers.actions()
        self.assertEqual(actions[0].text(), RECENT_TRANSFERS)
        self.assertTrue(actions[1].isSeparator())
        self.assertEqual(actions[2].text(), IN_PROGRESS)

        # This also check that the files are ordered based on written
        previous_actions = []
        percentage = in_progress[2][2] * 100 / in_progress[2][1]
        text = IN_PROGRESS_FILE % (in_progress[2][0], percentage)
        self.assertEqual(actions[3].text(), text)
        previous_actions.append(actions[3])

        percentage = in_progress[0][2] * 100 / in_progress[0][1]
        text = IN_PROGRESS_FILE % (in_progress[0][0], percentage)
        self.assertEqual(actions[4].text(), text)
        previous_actions.append(actions[4])

        percentage = in_progress[1][2] * 100 / in_progress[1][1]
        text = IN_PROGRESS_FILE % (in_progress[1][0], percentage)
        self.assertEqual(actions[5].text(), text)
        previous_actions.append(actions[5])

        in_progress = [
            ('file1', 200, 170),
            ('file2', 400, 300),
            ('file3', 300, 210),
        ]
        self.uploading = in_progress[:]

        self.ui.transfers.get_transfers_data()
        actions = self.ui.transfers.actions()
        current_actions = []
        self.assertEqual(actions[0].text(), RECENT_TRANSFERS)
        self.assertTrue(actions[1].isSeparator())
        self.assertEqual(actions[2].text(), IN_PROGRESS)

        # This also check that the files are ordered based on written
        percentage = in_progress[2][2] * 100 / in_progress[2][1]
        text = IN_PROGRESS_FILE % (in_progress[2][0], percentage)
        self.assertEqual(actions[3].text(), text)
        current_actions.append(actions[3])

        percentage = in_progress[0][2] * 100 / in_progress[0][1]
        text = IN_PROGRESS_FILE % (in_progress[0][0], percentage)
        self.assertEqual(actions[4].text(), text)
        current_actions.append(actions[4])

        percentage = in_progress[1][2] * 100 / in_progress[1][1]
        text = IN_PROGRESS_FILE % (in_progress[1][0], percentage)
        self.assertEqual(actions[5].text(), text)
        current_actions.append(actions[5])

        self.assertEqual(previous_actions, current_actions)

    def test_menu_not_reload(self):
        """Show the menu and test that is not reload it on refresh."""
        self.recent_transfers = ['file1', 'file2']
        self.ui.transfers.get_transfers_data()
        previous_actions = self.ui.transfers.actions()
        self.ui.transfers.get_transfers_data()
        current_actions = self.ui.transfers.actions()
        self.assertEqual(previous_actions, current_actions)

    def test_menu_reload(self):
        """Show the menu and test that is reload it on refresh."""
        self.recent_transfers = ['file1', 'file2']
        self.ui.transfers.get_transfers_data()
        previous_actions = self.ui.transfers.actions()
        self.recent_transfers = ['file1', 'file2', 'file3']

        self.ui.transfers.get_transfers_data()
        current_actions = self.ui.transfers.actions()
        self.assertNotEqual(previous_actions, current_actions)

    def test_progress_not_reload(self):
        """Show the menu and test that is not reload it on refresh."""
        in_progress = [
            ('file1', 200, 150),
            ('file2', 400, 100),
            ('file3', 300, 200),
        ]
        # Return copy of in_progress
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        previous_actions = self.ui.transfers.actions()
        in_progress = [
            ('file1', 200, 170),
            ('file2', 400, 300),
            ('file3', 300, 210),
        ]
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        current_actions = self.ui.transfers.actions()

        self.assertEqual(previous_actions, current_actions)

    def test_progress_reload(self):
        """Show the menu and test that is reload it on refresh."""
        in_progress = [
            ('file1', 200, 150),
            ('file2', 400, 100),
            ('file3', 300, 200),
        ]
        # Return copy of in_progress
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        previous_actions = self.ui.transfers.actions()
        in_progress = [
            ('file1', 200, 170),
            ('file2', 400, 300),
            ('file3', 300, 210),
            ('file4', 1000, 410),
        ]
        self.uploading = in_progress[:]
        self.ui.transfers.get_transfers_data()
        current_actions = self.ui.transfers.actions()

        self.assertNotEqual(previous_actions, current_actions)
