# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Integration tests for the webclient."""

from urlparse import urlparse, parse_qs

from twisted.internet import defer
from twisted.web import resource

from ubuntu_sso.utils.webclient.tests import BaseMockWebServer

from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.web_client import (
    UnauthorizedError,
    WebClient,
    WebClientError,
)
from ubuntuone.platform.credentials import APP_NAME


SAMPLE_KEY = "result"
SAMPLE_VALUE = "sample result"
SAMPLE_RESOURCE = '{"%s": "%s"}' % (SAMPLE_KEY, SAMPLE_VALUE)
SAMPLE_CREDENTIALS = dict(
    consumer_key="consumer_key",
    consumer_secret="consumer_secret",
    token="the_real_token",
    token_secret="the_token_secret",
)
SAMPLE_TARGET = u'http://example.com/'
SAMPLE_SIGNED = SAMPLE_TARGET + u'?oauth_nonce=36886134&' \
    'oauth_timestamp=1328544832&oauth_consumer_key=%s&' \
    'oauth_signature_method=HMAC-SHA1&next=%%2Fblah&oauth_version=1.0&' \
    'oauth_token=%s&oauth_signature=s6h0LRBiWchTADrTJWaJUSuaGpo3D' % \
    (SAMPLE_CREDENTIALS['consumer_key'], SAMPLE_CREDENTIALS['token'])


def sample_get_credentials():
    """Will return the sample credentials right now."""
    return defer.succeed(SAMPLE_CREDENTIALS)


class MockResource(resource.Resource):
    """A simple web resource."""
    isLeaf = True
    contents = ""

    # pylint: disable=C0103
    # t.w.resource methods have freeform cased names

    def getChild(self, name, request):
        """Get a given child resource."""
        if name == '':
            return self
        return resource.Resource.getChild(self, name, request)

    def render_GET(self, request):
        """Make a bit of html out of these resource's content."""
        return self.contents


class MockWebServer(BaseMockWebServer):
    """A mock webserver for the webclient tests."""

    def get_root_resource(self):
        """Get the root resource with all the children."""
        root = resource.Resource()
        devices_resource = MockResource()
        devices_resource.contents = SAMPLE_RESOURCE
        root.putChild("devices", devices_resource)
        root.putChild("throwerror", resource.NoResource())
        unauthorized = resource.ErrorPage(resource.http.UNAUTHORIZED,
                                          "Unauthrorized", "Unauthrorized")
        root.putChild("unauthorized", unauthorized)

        return root


class WebClientTestCase(TestCase):
    """Test for the webservice client."""

    timeout = 10

    @defer.inlineCallbacks
    def setUp(self):
        yield super(WebClientTestCase, self).setUp()
        self.ws = MockWebServer()
        self.addCleanup(self.ws.stop)
        self.base_iri = self.ws.get_iri()

        self.wc = WebClient(sample_get_credentials, base_url=self.base_iri)
        self.addCleanup(self.wc.shutdown)

    def test_correct_app_name(self):
        """Assert that the wc uses the correct appname."""
        self.assertEqual(APP_NAME, self.wc.wc.appname)

    @defer.inlineCallbacks
    def test_get_url(self):
        """A method is successfully called in the mock webservice."""
        result = yield self.wc.call_api("devices")
        self.assertIn(SAMPLE_KEY, result)
        self.assertEqual(SAMPLE_VALUE, result[SAMPLE_KEY])

    @defer.inlineCallbacks
    def test_get_url_error(self):
        """The errback is called when there's some error."""
        yield self.assertFailure(self.wc.call_api("throwerror"),
                                 WebClientError)

    @defer.inlineCallbacks
    def test_unauthorized(self):
        """Detect when a request failed with UNAUTHORIZED."""
        yield self.assertFailure(self.wc.call_api("unauthorized"),
                                 UnauthorizedError)


class WebClientBuildSignedIriTestCase(WebClientTestCase):
    """Test for the webservice client when signing iris."""

    # Instance of 'ParseResult' has no 'foo' member
    # pylint: disable=E1101

    @defer.inlineCallbacks
    def test_is_correct_domain(self):
        """Test that we are using the right domain."""
        signed = yield self.wc.build_signed_iri(SAMPLE_TARGET)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        self.assertEqual(parsed_signed.netloc, parsed_sample.netloc)

    @defer.inlineCallbacks
    def test_is_correct_path(self):
        """Test that we are using the right path in the URL."""
        signed = yield self.wc.build_signed_iri(SAMPLE_TARGET)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        self.assertEqual(parsed_signed.path, parsed_sample.path)

    @defer.inlineCallbacks
    def test_is_correct_scheme(self):
        """Test that we are using the right scheme."""
        signed = yield self.wc.build_signed_iri(SAMPLE_TARGET)
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)

        self.assertEqual(parsed_signed.scheme, parsed_sample.scheme)

    @defer.inlineCallbacks
    def test_correct_query(self):
        """Test the invariant parts of the signed URL."""
        signed = yield self.wc.build_signed_iri(SAMPLE_TARGET,
                                                params={'next': u'/blah'})
        parsed_signed = urlparse(signed)
        parsed_sample = urlparse(SAMPLE_SIGNED)
        signed_query = parse_qs(parsed_signed.query)
        sample_query = parse_qs(parsed_sample.query)

        for key in ('next', 'oauth_consumer_key', 'oauth_version',
                    'oauth_signature_method', 'oauth_token'):
            self.assertEqual(map(unicode, signed_query[key]),
                             sample_query[key])

    @defer.inlineCallbacks
    def test_url_with_query(self):
        """Test that we are using the right scheme."""
        signed = yield self.wc.build_signed_iri(SAMPLE_TARGET,
                                             params={'next': u'/blah?foo=bar'})
        parsed_signed = urlparse(signed)
        signed_query = parse_qs(parsed_signed.query)

        self.assertEqual(signed_query['next'], [u'/blah?foo=bar'])

    @defer.inlineCallbacks
    def test_uses_timestamper(self):
        """Test that the signed url is using the serverrelative timestamp."""
        signed = yield self.wc.build_signed_iri(u'/blah?foo=bar')
        parsed_signed = urlparse(signed)
        signed_query = parse_qs(parsed_signed.query)

        self.assertTrue(signed_query['oauth_timestamp'] is not None)
